/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "article-db.h"
#include "date.h"
#include "globals.h"
#include "log.h"
#include "nntp.h"
#include "queue.h"

#include "util.h"

gboolean must_exit = FALSE;
gboolean stop_all = FALSE;
gboolean online_prompt = TRUE;

void
nntp_disconnect (PanSocket *sock)
{
	pan_socket_putline (sock, "QUIT\r\n");
}

const gchar*
nntp_login (PanSocket *sock, server_data* sdata)
{
	const gchar* buffer;

	g_assert (sock!=NULL);

	if (pan_socket_getline (sock, &buffer))
		return _("Error reading from socket.");

	/* attempt authentication.  This isn't implemented exactly as
	 * "Common NNTP Extensions" specifies, being that we didn't
	 * wait to be asked for it, but hopefully it will work.
	 */
	if (sdata->need_auth)
	{
		if (pan_socket_putline_va (sock, "AUTHINFO user %s\r\n", sdata->username))
			return _("Error writing to socket.");
		if (pan_socket_getline (sock, &buffer))
			return _("Error reading from socket.");
		if (atoi(buffer) == 381)
		{
			/* 381 Means: More authentication information needed */
			if (pan_socket_putline_va (sock, "AUTHINFO pass %s\r\n", sdata->password))
				return _("Error writing to socket.");

			if (pan_socket_getline (sock, &buffer))
				return _("Error reading from socket.");

			/* 281 Means: Authentication accepted */
			if (atoi(buffer) != 281) {
				log_add_va ( _("Authentication for server '%s' unsuccessful."),
					buffer, sdata->name);
				return buffer;
			}
		}
		/* Else authentication not required or not understood */
	}

	if (pan_socket_putline (sock, "MODE READER\r\n"))
		return _("Error writing to socket.");
	if (pan_socket_getline (sock, &buffer))
		return _("Error reading from socket.");

	if (atoi (buffer) == 200)
		sdata->posting = TRUE;
	else if (atoi (buffer) == 201)
		sdata->posting = FALSE;
	else
		return buffer;

	return NULL;
}




gboolean
nntp_set_group (PanSocket *sock,
                const group_data *data)
{
	const gchar *buffer = NULL;

	g_assert (data);

	/* write the request */
	if (pan_socket_putline_va (sock, "GROUP %s\r\n", data->name))
		return 0;

	/* get the result string */
	if (pan_socket_getline (sock, &buffer))
		return FALSE;

	/* interpret the result string */
	return atoi(buffer) == 210;
}



/**
 * nntp_post:
 * @msg: ???
 *
 * Return value:
 **/
int
nntp_post (PanSocket *sock,
           const gchar *msg)
{
	const gchar *buffer;

	if (Pan.mute) {
		g_message ("Mute: Your Mesage will not actually be posted.");
		g_message (msg);
		return 0;
	}

	if (pan_socket_putline (sock, "POST\r\n")) {
		pan_error_dialog (_("Error writing to socket during post."));
		return -1;
	}

	if (pan_socket_getline (sock, &buffer)) {
		pan_error_dialog (_("Error reading from socket during post."));
		return -1;
	}

	if (atoi(buffer) == 340) { /* 340 Means: Send Article To Be Posted */
		if (pan_socket_putline_va (sock, "%s\r\n.\r\n", msg)) {
			pan_error_dialog (_("Error writing to socket during post."));
			return -1;
		}
		if (pan_socket_getline (sock, &buffer)) {
			pan_error_dialog (_("Error reading from socket during post."));
			return -1;
		}
		if (atoi(buffer) == 240) { /* 240 Means: Article Posted OK */
			return 0;
		}
	}

	pan_error_dialog (_("Error posting message.  Server said:\n'%s'"), buffer);
	return -1;
}

/*--------------------------------------------------------------------
 * download and decode an article
 *--------------------------------------------------------------------*/

gint
nntp_send_noop (PanSocket *sock)
{
	const char *buffer;

	if (pan_socket_putline (sock, "HELP\r\n")) {
		pan_error_dialog (_("Error writing to socket during HELP no-op."));
		return -1;
	}
	if (pan_socket_getline (sock, &buffer)) {
		pan_error_dialog (_("Error reading from socket during HELP no-op."));
		return -1;
	}
	if (atoi(buffer) == 100) { /* 100 = help text follows */
 		while (strncmp(buffer, ".\r\n", 3)) {
			if (pan_socket_getline (sock, &buffer)) {
				pan_error_dialog (_("Error reading from socket during HELP no-op."));
				return -1;
			}
		}
	}

	return 0;
}
