/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/libgnomeui.h>

#include "article.h"
#include "article-find.h"
#include "articlelist.h"
#include "fnmatch.h"
#include "globals.h"
#include "gui.h"
#include "util.h"

static GtkWidget *find_dialog;

static void
find_dialog_destroy_cb (void)
{
	pan_lock();
	gtk_widget_destroy(find_dialog);
	pan_unlock();

	find_dialog = NULL;
}

typedef struct
{
	gboolean unread_only;
	gboolean case_sensitive;

	const gchar *author;
	const gchar *subject;
} 
header_compare_data;

/**
 * Return 0 if equal
 */
static gboolean
compare_article (
	const article_data* adata,
	const header_compare_data* header )
{
	int flags = header->case_sensitive ? 0 : PAN_CASEFOLD;

	/* if it ain't an article, it ain't a match */
	if (!adata)
		return -1;

	/* make sure the unread state (if user specified) matches... */
	if (header->unread_only && article_flag_on (adata, STATE_READ))
		return -1;

	/* make sure the author (if user specified one) matches.. */
	if (header->author && fnmatch(header->author,adata->author,flags))
		return -1;

	/* make sure the subject (if user specified one) matches.. */
	if (header->subject && fnmatch(header->subject,adata->subject,flags))
		return -1;

	/* hooray, a match */
	return 0;
}

static void
find_article (
	const gchar* author,
	const gchar* subject,
	gboolean case_sensitive,
	gboolean unread_only,
	gboolean reverse )
{
	gchar *author_tmp = NULL;
	gchar *subject_tmp = NULL;
	GtkCTreeNode *sel = NULL;
	GtkCTreeNode *tmp = NULL;
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	header_compare_data header_data;

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	if (subject) {
		if (strchr(subject,'*'))
			subject_tmp = g_strdup (subject);
		else
			subject_tmp = g_strdup_printf ("*%s*", subject);
	}

	if (author) {
		if (strchr(author,'*'))
			author_tmp = g_strdup (author);
		else
			author_tmp = g_strdup_printf ("*%s*", author);
	}

	/* sanity checks */
	g_return_if_fail (articlelist_get_current_server()!=NULL);
	g_return_if_fail (articlelist_get_current_group()!=NULL);

	/* set up search function struct */
	header_data.author = author_tmp;
	header_data.subject = subject_tmp;
	header_data.unread_only = unread_only;
	header_data.case_sensitive = case_sensitive;

	/* search */
	sel = articlelist_get_selected_node();
	if (!sel)
		sel = gtk_ctree_node_nth (tree, 0);
	tmp = sel;
	for ( ;; )
	{
		const article_data* adata = NULL;

		/* next node */
		tmp = reverse
			? articlelist_node_prev (tmp, FALSE)
			: articlelist_node_next (tmp, FALSE);
		if (!tmp) { /* wrap */
			int n = reverse
				? GTK_CLIST(tree)->rows-1
				: 0;
			tmp = gtk_ctree_node_nth (tree, n);
			if (tmp==sel)
				break;
			tmp = reverse
				? articlelist_node_prev (tmp, FALSE)
				: articlelist_node_next (tmp, FALSE);
		}
		if (tmp==sel)
			break;

		/* compare */
		adata = (const article_data*)
			gtk_ctree_node_get_row_data (tree, tmp);
		if (!compare_article (adata, &header_data))
			break;
	}

	/* select the node.. */
	if (tmp!=sel)
		articlelist_set_selected_node (tmp);

	/* cleanup */
	g_free (author_tmp);
	g_free (subject_tmp);
}

static gchar* find_article__author = NULL;
static gchar* find_article__subject = NULL;
static gboolean find_article__case_sensitive = FALSE;
static gboolean find_article__unread_only = FALSE;
static gboolean find_article__reverse = FALSE;

static void
find_dialog_clicked_cb (
	GnomeDialog* dialog,
	int button,
	GtkWidget* table )
{
 	g_return_if_fail (button==0 || button==1 || button==2);
	g_return_if_fail (table!=NULL);

	if ( button==2 ) /* close button */
	{
		pan_lock();
		gtk_widget_unmap ( GTK_WIDGET(find_dialog) );
		pan_unlock();
	}
	else if (button==0 || button==1) /* "find next" button */
	{
		GtkWidget* author_entry;
		GtkWidget* subject_entry;
		GtkWidget* case_sensitive_checkbox;
		GtkWidget* unread_only_checkbox;
		gchar* author = 0;
		gchar* subject = 0;
		gboolean case_sensitive;
		gboolean unread_only;

		/* get author to search for, or NULL if none... */
		author_entry = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "author" );
		g_return_if_fail ( author_entry!=NULL );
		author = g_strdup ( gtk_entry_get_text ( GTK_ENTRY(author_entry) ) );
		g_strstrip ( author );
		if ( !*author ) {
			g_free ( author );
			author = NULL;
		}

		/* get subject to search for, or NULL if none... */
		subject_entry = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "subject" );
		g_return_if_fail ( subject_entry!=NULL );
		subject = g_strdup ( gtk_entry_get_text ( GTK_ENTRY(subject_entry) ) );
		g_strstrip ( subject );
		if ( !*subject ) {
			g_free ( subject );
			subject = NULL;
		}

		/* get case sensitivity... */
		case_sensitive_checkbox = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "case" );
		g_return_if_fail ( case_sensitive_checkbox!=NULL );
		case_sensitive = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(case_sensitive_checkbox) );

		/* get case sensitivity... */
		unread_only_checkbox = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "unread" );
		g_return_if_fail ( unread_only_checkbox!=NULL );
		unread_only = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(unread_only_checkbox) );

		/* update the find_article fields with these widget values */
		g_free ( find_article__author );
		find_article__author = author;
		g_free ( find_article__subject );
		find_article__subject = subject;
		find_article__case_sensitive = case_sensitive;
		find_article__unread_only = unread_only;

		find_article__reverse = button==0;
		articlelist_find_next_cb ( );
	}
}

void
articlelist_find_next_cb (void)
{
	/* user may have hit "find again" without loading a group first */
	if (!articlelist_get_current_server())
		return;
	if (!articlelist_get_current_group())
		return;

	/* user may have hit "find again" without hitting "find" first */
	if (!find_article__subject && !find_article__author) {
		articlelist_find_text_cb();
		return;
	}

	/* got a subject and/or an author.. let's go find 'em */
	find_article (
		find_article__author,
		find_article__subject,
		find_article__case_sensitive,
		find_article__unread_only,
		find_article__reverse );

	find_article__reverse = FALSE;
}


void
articlelist_find_text_cb (void)
{
	GtkWidget *table;
	GtkWidget *author_label, *author_entry;
	GtkWidget *subject_label, *subject_entry;
	GtkWidget *case_sensitive_checkbutton;
	GtkWidget *unread_only_checkbutton;

	/* There can be only one! (find window) */
	if (find_dialog) {
		if ( !GTK_WIDGET_MAPPED(GTK_WIDGET(find_dialog)) )
			gtk_widget_map ( GTK_WIDGET(find_dialog) );
		gdk_window_raise (GTK_WIDGET(find_dialog)->window);
		return;
	}
	
	find_dialog = gnome_dialog_new (_("Find Message"),
		GNOME_STOCK_PIXMAP_BACK,
		GNOME_STOCK_PIXMAP_FORWARD,
		GNOME_STOCK_PIXMAP_CLOSE, NULL);
	gnome_dialog_set_default ( GNOME_DIALOG(find_dialog), 0 );

	table = gtk_table_new (4, 2, FALSE);

	subject_label = gtk_label_new ( _("Subject:") );
	gtk_table_attach ( GTK_TABLE(table),
			   subject_label,
			   0, 1,
			   0, 1,
			   0,
			   0,
			   GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	subject_entry = gtk_entry_new ( );
	gtk_object_set_data (GTK_OBJECT(find_dialog), "subject", subject_entry);
	gtk_table_attach (GTK_TABLE(table),
			  subject_entry,
			  1, 2,
			  0, 1,
			  GTK_EXPAND|GTK_FILL|GTK_SHRINK,
			  0,
			  GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	author_label = gtk_label_new (_("Author:"));
	gtk_table_attach (GTK_TABLE(table),
			  author_label,
			  0, 1,
			  1, 2,
			  0,
			  0,
			  GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	author_entry = gtk_entry_new ( );
	gtk_object_set_data ( GTK_OBJECT(find_dialog), "author", author_entry );
	gtk_table_attach (GTK_TABLE(table),
			  author_entry,
			  1, 2,
			  1, 2,
			  GTK_EXPAND|GTK_FILL|GTK_SHRINK,
			  0,
			  GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	unread_only_checkbutton =
		gtk_check_button_new_with_label (_("Unread Only"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(unread_only_checkbutton), FALSE);
	gtk_object_set_data (GTK_OBJECT(find_dialog),
		"unread", unread_only_checkbutton );
	gtk_table_attach ( GTK_TABLE(table),
			   unread_only_checkbutton,
			   0, 2,
			   2, 3,
			   GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			   GTK_EXPAND | GTK_FILL,
			   GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	case_sensitive_checkbutton = gtk_check_button_new_with_label ( "Case Sensitive" );
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(case_sensitive_checkbutton), FALSE );
	gtk_object_set_data ( GTK_OBJECT(find_dialog), "case", case_sensitive_checkbutton );
	gtk_table_attach ( GTK_TABLE(table),
			   case_sensitive_checkbutton,
			   0, 2,
			   3, 4,
			   GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			   GTK_EXPAND | GTK_FILL,
			   GNOME_PAD_SMALL, GNOME_PAD_SMALL );

	
//	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(find_dialog)->vbox), table );
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (find_dialog)->vbox), table, FALSE, FALSE, 0);
	
	gtk_signal_connect (GTK_OBJECT(find_dialog), "clicked",
			    GTK_SIGNAL_FUNC(find_dialog_clicked_cb),
			    table);
	gtk_signal_connect (GTK_OBJECT(find_dialog), "destroy",
			    GTK_SIGNAL_FUNC(find_dialog_destroy_cb), NULL);

	gtk_widget_grab_focus (subject_entry);
	gui_popup_draw (find_dialog, Pan.window);
}


