/*
 ui-channels.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static void channel_creation_finish(WINDOW_REC *window, CHANNEL_REC *channel)
{
    if (CHANNEL_PARENT(channel) == NULL)
    {
	/* Add the channel to channels list in window */
	channel->ui_data = window;
	window->channels = g_list_append(window->channels, channel);
    }

    signal_emit("gui channel init", 1, channel);

    if (g_list_length(window->channels) == 1)
    {
        /* first channel in this window */
        window->active = channel;
        signal_emit("channel focused", 1, channel);
    }

    if (window->active == channel)
        signal_emit("channel server changed", 1, channel);
}

static gboolean signal_channel_created(CHANNEL_REC *channel)
{
    WINDOW_REC *window, *window2;
    CHANNEL_REC *destroy;
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    /* first check if there's any windows without channels, this happens when
       last channel is closed in window but window isn't destroyed and empty
       channel is just being created .. (when kicked from channel)

       .. also check if there's any empty windows that we could use.. */
    window = window2 = NULL;
    for (tmp = g_list_first(windows); tmp != NULL; tmp = tmp->next)
    {
        WINDOW_REC *rec = tmp->data;

        if (rec->channels == NULL)
        {
            /* found, use this window.. */
            window = rec;
            break;
        }

        if (window2 == NULL && g_list_length(rec->channels) == 1)
        {
            CHANNEL_REC *chan = rec->channels->data;

            if (chan->type == CHANNEL_TYPE_EMPTY && chan->level == 0)
                window2 = rec;
        }
    }

    destroy = NULL;
    if (window == NULL && window2 != NULL && channel->type != CHANNEL_TYPE_EMPTY)
    {
        /* use the empty window */
        window = window2;
        destroy = window->channels->data;
    }

    if (cur_channel != NULL && channel->type != CHANNEL_TYPE_EMPTY &&
        cur_channel->type == CHANNEL_TYPE_EMPTY && cur_channel->level == 0)
    {
        /* window is empty, use it. */
        window = CHANNEL_PARENT(cur_channel);
        destroy = cur_channel;
    }
    else if (window == NULL)
    {
        /* create new window to use */
        window = ui_window_create(channel);
    }

    channel_creation_finish(window, channel);

    if (destroy != NULL)
    {
        /* destroy the empty channel */
        channel_destroy(destroy);
    }
    return TRUE;
}

static gboolean signal_channel_created_curwin(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    channel_creation_finish(CHANNEL_PARENT(cur_channel), channel);
    signal_emit("gui channel open", 1, channel);
    return FALSE;
}

static gboolean signal_channel_destroyed(CHANNEL_REC *channel)
{
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    window->channels = g_list_remove(window->channels, channel);

    if (window->channels != NULL)
    {
	signal_emit("channel child destroyed", 1, channel);
        if (window->active == channel)
        {
            window->active = window->channels->data;
            signal_emit("channel focused", 1, window->active);
        }
    }
    else
    {
        /* no channels left in window .. what do I do? */
        if ((!channel->left || channels == NULL) && !CHANNEL_PARENT(channel)->destroying)
        {
            /* kicked / disconnected / closed last channel, create empty
               channel into window */
	    CHANNEL_REC *new_channel;

            new_channel = channel_create(channel->server, "(empty)", CHANNEL_TYPE_EMPTY);
            new_channel->new_data = channel->new_data;
        }
        else
        {
            /* window is useless, destroy it. */
            ui_window_destroy(window);
        }
    }

    channel->ui_data = NULL;
    if (cur_channel == channel)
    {
	/* Still using this channel?! Change to anything else but this! */
	cur_channel = channels == NULL ? NULL : channels->data;
    }

    return TRUE;
}

static gboolean signal_channel_focused(CHANNEL_REC *channel)
{
    if (g_list_length(CHANNEL_PARENT(channel)->channels) > 1)
    {
	printformat(channel->server, channel->name, MSGLEVEL_CLIENTNOTICE,
		    IRCTXT_TALKING_IN, channel->name);
    }
    return TRUE;
}

static gboolean sig_server_looking(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    /* try to keep some server assigned to channels.. */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

	if (rec->server == NULL)
	{
	    rec->server = server;
	    signal_emit("channel server changed", 1, rec);
	}
    }

    return TRUE;
}

static gboolean cmd_wjoin(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    signal_add("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    signal_emit("command join", 3, data, server, channel);
    signal_remove("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    return TRUE;
}

void ui_channels_init(void)
{
    signal_add("channel created", (SIGNAL_FUNC) signal_channel_created);
    signal_add("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_add("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_add("server looking", (SIGNAL_FUNC) sig_server_looking);
    command_bind("wjoin", NULL, (SIGNAL_FUNC) cmd_wjoin);
}

void ui_channels_deinit(void)
{
    signal_remove("channel created", (SIGNAL_FUNC) signal_channel_created);
    signal_remove("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_remove("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_remove("server looking", (SIGNAL_FUNC) sig_server_looking);
    command_unbind("wjoin", (SIGNAL_FUNC) cmd_wjoin);
}
