/*
 dialog-log.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

gchar *level_button_label[] =
{
    N_("Crap"),
    N_("Private messages"),
    N_("Public messages"),
    N_("Notices"),
    N_("Server notices"),
    N_("CTCP messages"),
    N_("Actions (/me)"),
    N_("Joins"),
    N_("Parts"),
    N_("Quits"),
    N_("Kicks"),
    N_("Modes"),
    N_("Server modes"),
    N_("Topics"),
    N_("Wallops"),
    N_("Invites"),
    N_("Nick changes"),
    N_("Pongs"),
    N_("DCC messages"),
    N_("Hilighted messages"),
    N_("Client notices"),
    N_("Client crap"),
    N_("Client errors"),
};

gint level_button_position[] =
{
    MSGLEVEL_CRAP,
    MSGLEVEL_MSGS,
    MSGLEVEL_PUBLIC,
    MSGLEVEL_NOTICES,
    MSGLEVEL_SNOTES,
    MSGLEVEL_CTCPS,
    MSGLEVEL_ACTIONS,
    MSGLEVEL_JOINS,
    MSGLEVEL_PARTS,
    MSGLEVEL_QUITS,
    MSGLEVEL_KICKS,
    MSGLEVEL_MODES,
    MSGLEVEL_SMODES,
    MSGLEVEL_TOPICS,
    MSGLEVEL_WALLOPS,
    MSGLEVEL_INVITES,
    MSGLEVEL_NICKS,
    MSGLEVEL_PONGS,
    MSGLEVEL_DCC,
    MSGLEVEL_HILIGHT,
    MSGLEVEL_CLIENTNOTICE,
    MSGLEVEL_CLIENTCRAP,
    MSGLEVEL_CLIENTERROR,
};

#define LEVEL_BUTTONS (sizeof(level_button_position)/sizeof(level_button_position[0]))

GtkWidget *level_button[LEVEL_BUTTONS];

static GtkWidget *dialog;

static void save_log_setup(void)
{
    proplist_t pkey, lprop, iprop, sublprop, subiprop;
    GList *tmp, *sub;

    /* First clear the log entries.. */
    pkey = PLMakeString("log");
    PLRemoveDictionaryEntry(cprop, pkey);

    lprop = NULL;
    if (g_list_length(logs) > 0)
    {
	/* Create an array of log entries */
	lprop = PLMakeArrayFromElements(NULL);
	cprop = PLInsertDictionaryEntry(cprop, pkey, lprop);
    }
    PLRelease(pkey);

    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;

	/* create dictionary for log entry and add it to array */
	iprop = PLMakeDictionaryFromEntries(NULL, NULL);
	lprop = PLAppendArrayElement(lprop, iprop);

	config_set_str(iprop, "file", rec->fname);
	config_set_int(iprop, "level", rec->level);
	config_set_bool(iprop, "autoopen", rec->autoopen_log);

	/* Save the subitems */
	if (g_list_length(rec->items) == 0)
	{
	    /* none, don't save empty lists */
	    continue;
	}

	pkey = PLMakeString("items");
	sublprop = PLMakeArrayFromElements(NULL);
	iprop = PLInsertDictionaryEntry(iprop, pkey, sublprop);
	PLRelease(pkey);

        for (sub = g_list_first(rec->items); sub != NULL; sub = sub->next)
        {
            LOG_ITEM_REC *item = sub->data;

	    subiprop = PLMakeDictionaryFromEntries(NULL, NULL);
	    sublprop = PLAppendArrayElement(sublprop, subiprop);
	    config_set_str(subiprop, "name", item->name);
	    config_set_int(subiprop, "level", item->level);
        }
    }
    PLSave(cprop, TRUE);
}

static void load_log_setup(void)
{
    proplist_t lprop, iprop, sublprop, subiprop;
    gint num, subnum, max, submax;
    gchar *name;
    LOG_REC *rec;

    lprop = config_get_prop(cprop, "log");
    max = lprop == NULL ? 0 : PLGetNumberOfElements(lprop);
    for (num = 0; num < max; num++)
    {
	iprop = PLGetArrayElement(lprop, num);

	name = config_get_str(iprop, "file", NULL);
	if (name == NULL) continue;
	rec = log_create_with_level(name, config_get_int(iprop, "level", MSGLEVEL_ALL));
	rec->autoopen_log = config_get_bool(iprop, "autoopen", FALSE);

	/* Get the subitems */
	sublprop = config_get_prop(iprop, "items");
	submax = sublprop == NULL ? 0 : PLGetNumberOfElements(sublprop);
        for (subnum = 0; subnum < submax; subnum++)
        {
	    subiprop = PLGetArrayElement(sublprop, subnum);

	    name = config_get_str(subiprop, "name", NULL);
	    if (name == NULL) continue;
	    log_append_item(rec, name, config_get_int(subiprop, "level", MSGLEVEL_ALL));
        }
    }
}

static void logs_autoopen(void)
{
    GList *tmp;

    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;

        if (rec->autoopen_log) log_file_open(rec);
    }
}

static void get_node_data(GtkCTree *ctree, GtkCTreeNode *node, LOG_REC **log, LOG_ITEM_REC **item)
{
    GtkCTreeRow *row;

    row = node->list.data;
    if (log != NULL) *log = gtk_ctree_node_get_row_data(ctree, row->parent == NULL ? node : row->parent);
    if (item != NULL) *item = row->parent == NULL ? NULL : gtk_ctree_node_get_row_data(ctree, node);
}

static void redraw_row(LOG_REC *log, LOG_ITEM_REC *item)
{
    GtkWidget *fileentry, *chanentry, *autolog;
    gint level, n;

    g_return_if_fail(log != NULL);

    fileentry = gtk_object_get_data(GTK_OBJECT(dialog), "fileentry");
    chanentry = gtk_object_get_data(GTK_OBJECT(dialog), "chanentry");
    autolog = gtk_object_get_data(GTK_OBJECT(dialog), "autolog");

    gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "startlog"), log->handle == -1);
    gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "stoplog"), log->handle != -1);

    gtk_widget_set_sensitive(fileentry, item == NULL);
    gtk_widget_set_sensitive(chanentry, item != NULL);
    gtk_widget_set_sensitive(autolog, item == NULL);

    gtk_entry_set_text(GTK_ENTRY(fileentry), log->fname);
    gtk_entry_set_text(GTK_ENTRY(chanentry), item == NULL ? "" : item->name);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autolog), log->autoopen_log);

    level = item == NULL ? log->level : item->level;
    for (n = 0; n < LEVEL_BUTTONS; n++)
    {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(level_button[n]),
                                     level & level_button_position[n]);
    }
}

static void get_log_texts(LOG_REC *rec, gchar **texts)
{
    texts[0] = rec->handle == -1 ? _("No") : _("Yes");
    texts[1] = rec->fname;
    texts[2] = bits2level(rec->level);
}

static void get_log_item_texts(LOG_ITEM_REC *rec, gchar **texts)
{
    texts[0] = channel_find_any(rec->name) == NULL ? "" : _("Yes");
    texts[1] = rec->name;
    texts[2] = bits2level(rec->level);
}

static void insert_log_item(GtkCTree *ctree, LOG_REC *log)
{
    GtkCTreeNode *node, *subnode;
    gchar *texts[3];
    GList *tmp;

    g_return_if_fail(ctree != NULL);
    g_return_if_fail(log != NULL);

    gtk_clist_freeze(GTK_CLIST(ctree));
    get_log_texts(log, texts);
    node = gtk_ctree_insert_node(ctree, NULL, NULL, texts, 0,
                                 NULL, NULL, NULL, NULL, FALSE, TRUE);
    g_free(texts[2]);
    gtk_ctree_node_set_row_data(ctree, node, log);

    subnode = NULL;
    for (tmp = g_list_first(log->items); tmp != NULL; tmp = tmp->next)
    {
        LOG_ITEM_REC *rec = tmp->data;

        get_log_item_texts(rec, texts);
        subnode = gtk_ctree_insert_node(ctree, node, NULL, texts, 0,
                              NULL, NULL, NULL, NULL, FALSE, TRUE);
        gtk_ctree_node_set_row_data(ctree, subnode, rec);
        g_free(texts[2]);
    }
    gtk_clist_thaw(GTK_CLIST(ctree));

    gtk_ctree_unselect(ctree, subnode ? subnode : node);
    gtk_ctree_select(ctree, subnode ? subnode : node);
}

static void sig_unselect_row(GtkCTree *ctree, GtkCTreeNode *row, gint column); /*FIXME*/

static void remove_log_item(GtkCTree *ctree, LOG_REC *log)
{
    GtkCTreeNode *node;

    g_return_if_fail(ctree != NULL);
    g_return_if_fail(log != NULL);

    node = gtk_ctree_find_by_row_data(ctree, NULL, log);
    if (node != NULL)
    {
        gtk_ctree_remove_node(ctree, node);
        /*FIXME: tree_unselect_row doesn't work so use this for now... */
        if (GTK_CLIST(ctree)->row_list == NULL)
            sig_unselect_row(ctree, NULL, 0);
    }
}

static void redraw_ctree_node(GtkCTree *ctree, GtkCTreeNode *node)
{
    LOG_REC *log;
    LOG_ITEM_REC *item;
    gchar *texts[3];

    g_return_if_fail(ctree != NULL);
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, &item);
    if (item == NULL)
        get_log_texts(log, texts);
    else
        get_log_item_texts(item, texts);

    gtk_ctree_node_set_text(ctree, node, 0, texts[0]);
    gtk_ctree_node_set_text(ctree, node, 1, texts[1]);
    gtk_ctree_node_set_text(ctree, node, 2, texts[2]);

    g_free(texts[2]);
}

static void sig_new_log_file(void)
{
    log_create(_("<new log file>"), "");
}

static void sig_new_log_entry(GtkWidget *button, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;

    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, NULL);
    log_append_item(log, _("<new item>"), MSGLEVEL_ALL);
}

static void sig_delete_entry(GtkWidget *button, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;
    LOG_ITEM_REC *item;

    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, &item);
    if (item == NULL)
    {
        /* remove whole log file */
        log_file_destroy(log);
    }
    else
    {
        /* remove log entry */
        log_remove_item(log, item->name);
    }
}

static void sig_start_log(GtkWidget *button, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;

    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, NULL);
    log_file_open(log);
}

static void sig_stop_log(GtkWidget *button, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;

    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, NULL);
    log_file_close(log);
}

static void sig_level_button_pressed(GtkWidget *widget, gpointer levelptr)
{
    GtkCTree *ctree;
    GtkCTreeNode *node;
    LOG_REC *log;
    LOG_ITEM_REC *item;
    gboolean active, changed;
    gint level, *levelp;

    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");
    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    g_return_if_fail(node != NULL);

    get_node_data(ctree, node, &log, &item);

    active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    level = GPOINTER_TO_INT(levelptr);
    levelp = item != NULL ? &item->level : &log->level;

    /* log file level */
    changed = ((*levelp & level) != 0) != active;
    if (changed)
    {
        *levelp ^= level;
        redraw_ctree_node(ctree, node);
    }
}

static void sig_clear_all_levels(GtkWidget *button, GtkCTree *ctree)
{
    gint n;

    for (n = 0; n < LEVEL_BUTTONS; n++)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(level_button[n]), FALSE);
}

static void sig_select_all_levels(GtkWidget *button, GtkCTree *ctree)
{
    gint n;

    for (n = 0; n < LEVEL_BUTTONS; n++)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(level_button[n]), TRUE);
}

static void sig_fileentry_changed(GtkWidget *entry, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;
    LOG_ITEM_REC *item;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    if (node == NULL) return;

    get_node_data(ctree, node, &log, &item);
    if (item == NULL)
    {
        g_free(log->fname);
        log->fname = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
        redraw_ctree_node(ctree, node);
    }
}

static void sig_chanentry_changed(GtkWidget *entry, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_ITEM_REC *item;

    g_return_if_fail(ctree != NULL);
    g_return_if_fail(entry != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    if (node == NULL) return;

    get_node_data(ctree, node, NULL, &item);
    if (item != NULL)
    {
        g_free(item->name);
        item->name = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
        redraw_ctree_node(ctree, node);
    }
}

static void sig_autolog_clicked(GtkWidget *button, GtkCTree *ctree)
{
    GtkCTreeNode *node;
    LOG_REC *log;
    LOG_ITEM_REC *item;

    g_return_if_fail(ctree != NULL);

    node = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    if (node == NULL) return;

    get_node_data(ctree, node, &log, &item);
    if (item == NULL)
        log->autoopen_log = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
}

static void sig_select_row(GtkCTree *ctree, GtkCTreeNode *node, gint column)
{
    GtkCTreeNode *old;
    LOG_REC *log;
    LOG_ITEM_REC *item;

    g_return_if_fail(ctree != NULL);

    old = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    if (old == node) return;

    if (old == NULL)
    {
        /* set all widgets sensitive */
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "newentry_button"), TRUE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "delete_button"), TRUE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "level_table"), TRUE);
    }

    gtk_object_set_data(GTK_OBJECT(dialog), "selection", node);

    get_node_data(ctree, node, &log, &item);
    if (log != NULL) redraw_row(log, item);
}

static void sig_unselect_row(GtkCTree *ctree, GtkCTreeNode *row, gint column)
{
    GtkCTreeNode *old;

    g_return_if_fail(ctree != NULL);

    old = gtk_object_get_data(GTK_OBJECT(dialog), "selection");
    if (old != NULL)
    {
        /* set all widgets unsensitive */
        gtk_object_set_data(GTK_OBJECT(dialog), "selection", NULL);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "newentry_button"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "delete_button"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "startlog"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "stoplog"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "level_table"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "fileentry"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "chanentry"), FALSE);
        gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "autolog"), FALSE);
        gtk_entry_set_text(gtk_object_get_data(GTK_OBJECT(dialog), "fileentry"), "");
        gtk_entry_set_text(gtk_object_get_data(GTK_OBJECT(dialog), "chanentry"), "");
    }
}

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    save_log_setup();
    gtk_widget_destroy(*dialog);
    *dialog = NULL;
}

void dialog_log(void)
{
    gchar *titles[] = { N_("Open"), N_("Name"), N_("Level") };
    GtkWidget *scrollbox, *frame, *level_table, *vbox, *hbox, *hbox2;
    GtkWidget *ctree, *fileentry, *chanentry, *button;
    GtkWidget *newentry_button, *delete_button, *startlog_button, *stoplog_button, *autolog_button;
    GList *tmp;
    gint n;

    if (dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(dialog->window);
        return;
    }

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    /* Create notify clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scrollbox, TRUE, TRUE, 0);

    ctree = gtk_ctree_new_with_titles(3, 1, titles);
    gtk_widget_set_usize(ctree, 300, -1);
    gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_BROWSE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(ctree), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(ctree), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(ctree), 2, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), ctree);

    /* create log management buttons */
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);

    button = gtk_button_new_with_label(_("New log file"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_new_log_file), NULL);
    gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, TRUE, 0);

    newentry_button = gtk_button_new_with_label(_("New log entry"));
    gtk_signal_connect(GTK_OBJECT(newentry_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_new_log_entry), ctree);
    gtk_widget_set_sensitive(newentry_button, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox2), newentry_button, TRUE, TRUE, 0);

    delete_button = gtk_button_new_with_label(_("Delete"));
    gtk_signal_connect(GTK_OBJECT(delete_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_delete_entry), ctree);
    gtk_widget_set_sensitive(delete_button, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox2), delete_button, TRUE, TRUE, 0);

    /* start/stop logging buttons */
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);

    startlog_button = gtk_button_new_with_label(_("Start logging"));
    gtk_signal_connect(GTK_OBJECT(startlog_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_start_log), ctree);
    gtk_widget_set_sensitive(startlog_button, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox2), startlog_button, TRUE, TRUE, 0);

    stoplog_button = gtk_button_new_with_label(_("Stop logging"));
    gtk_signal_connect(GTK_OBJECT(stoplog_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_stop_log), ctree);
    gtk_widget_set_sensitive(stoplog_button, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox2), stoplog_button, TRUE, TRUE, 0);

    /* create logging level list */
    frame = gtk_frame_new(NULL);
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);

    level_table = gtk_table_new(1, 1, FALSE);
    gtk_widget_set_sensitive(level_table, FALSE);
    gtk_container_border_width(GTK_CONTAINER(level_table), 3);
    gtk_container_add(GTK_CONTAINER(frame), level_table);

    for (n = 0; n < LEVEL_BUTTONS; n++)
    {
        level_button[n] = gtk_check_button_new_with_label(level_button_label[n]);
        gtk_signal_connect(GTK_OBJECT(level_button[n]), "clicked",
                           GTK_SIGNAL_FUNC(sig_level_button_pressed),
                           GINT_TO_POINTER(level_button_position[n]));
        gtk_table_attach_defaults(GTK_TABLE(level_table), level_button[n], n&1, (n&1)+1, n/2, n/2+1);
    }

    button = gtk_button_new_with_label(_("Clear all"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_clear_all_levels), ctree);
    gtk_table_attach_defaults(GTK_TABLE(level_table), button, 0, 1, 11, 12);
    button = gtk_button_new_with_label(_("Select all"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_select_all_levels), ctree);
    gtk_table_attach_defaults(GTK_TABLE(level_table), button, 1, 2, 11, 12);

    /* auto-open log button */
    autolog_button = gtk_check_button_new_with_label(_("Open log at startup"));
    gtk_signal_connect(GTK_OBJECT(autolog_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_autolog_clicked), ctree);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), autolog_button, FALSE, FALSE, 0);

    /* entry widgets for log file name and subitems name */
    fileentry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Log file name:"), NULL, FALSE);
    gtk_signal_connect(GTK_OBJECT(fileentry), "changed",
                       GTK_SIGNAL_FUNC(sig_fileentry_changed), ctree);
    chanentry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Channel/nick:"), NULL, FALSE);
    gtk_signal_connect(GTK_OBJECT(chanentry), "changed",
                       GTK_SIGNAL_FUNC(sig_chanentry_changed), ctree);
    gtk_widget_set_sensitive(fileentry, FALSE);
    gtk_widget_set_sensitive(chanentry, FALSE);

    gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(ctree), "tree_unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_object_set_data(GTK_OBJECT(dialog), "ctree", ctree);
    gtk_object_set_data(GTK_OBJECT(dialog), "newentry_button", newentry_button);
    gtk_object_set_data(GTK_OBJECT(dialog), "delete_button", delete_button);
    gtk_object_set_data(GTK_OBJECT(dialog), "level_table", level_table);
    gtk_object_set_data(GTK_OBJECT(dialog), "fileentry", fileentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "chanentry", chanentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "autolog", autolog_button);
    gtk_object_set_data(GTK_OBJECT(dialog), "startlog", startlog_button);
    gtk_object_set_data(GTK_OBJECT(dialog), "stoplog", stoplog_button);

    gtk_clist_freeze(GTK_CLIST(ctree));
    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
        insert_log_item(GTK_CTREE(ctree), tmp->data);
    gtk_clist_thaw(GTK_CLIST(ctree));
    gtk_widget_show_all(dialog);
}

static gboolean sig_log_changed(LOG_REC *log)
{
    GtkCTree *ctree;

    g_return_val_if_fail(log != NULL, FALSE);

    if (dialog == NULL) return TRUE;
    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");

    /* don't update these if we are removing the whole log file */
    if (g_list_find(logs, log) != NULL)
    {
        remove_log_item(ctree, log);
        insert_log_item(ctree, log);
    }
    return TRUE;
}

static void close_func(GtkCTree *ctree, GtkCTreeNode *node)
{
    g_return_if_fail(ctree != NULL);
    g_return_if_fail(node != NULL);

    gtk_ctree_node_set_text(ctree, node, 0, "");
}

static gboolean sig_log_closed(LOG_REC *log)
{
    GtkCTree *ctree;
    GtkCTreeNode *node;

    g_return_val_if_fail(log != NULL, FALSE);
    if (dialog == NULL) return TRUE;

    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");
    node = gtk_ctree_find_by_row_data(ctree, NULL, log);
    if (node != NULL)
    {
        gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) close_func, NULL);
        gtk_ctree_node_set_text(ctree, node, 0, _("No"));

        if (node == gtk_object_get_data(GTK_OBJECT(dialog), "selection"))
        {
            gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "startlog"), TRUE);
            gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(dialog), "stoplog"), FALSE);
        }
    }
    return TRUE;
}

static gboolean sig_log_destroyed(LOG_REC *log)
{
    GtkCTree *ctree;

    g_return_val_if_fail(log != NULL, FALSE);

    if (dialog == NULL) return TRUE;
    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");
    remove_log_item(ctree, log);
    return TRUE;
}

static GList *log_find_channels(gchar *channel)
{
    GList *tmp, *sub, *ret;

    g_return_val_if_fail(channel != NULL, NULL);

    ret = NULL;
    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;

        for (sub = g_list_first(rec->items); sub != NULL; sub = sub->next)
        {
            LOG_ITEM_REC *subrec = sub->data;

            if (g_strcasecmp(subrec->name, channel) == 0)
            {
                ret = g_list_append(ret, rec);
                break;
            }
        }
    }

    return ret;
}

static gboolean sig_channel_updated(CHANNEL_REC *channel)
{
    GtkCTree *ctree;
    GList *tmp, *list;

    g_return_val_if_fail(channel != NULL, FALSE);
    if (dialog == NULL) return TRUE;

    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");
    list = log_find_channels(channel->name);
    for (tmp = g_list_first(list); tmp != NULL; tmp = tmp->next)
    {
        remove_log_item(ctree, tmp->data);
        insert_log_item(ctree, tmp->data);
    }
    g_list_free(list);
    return TRUE;
}

void dialog_log_init(void)
{
    dialog = NULL;
    load_log_setup();
    logs_autoopen();
    signal_add("log opened", (SIGNAL_FUNC) sig_log_changed);
    signal_add("log closed", (SIGNAL_FUNC) sig_log_closed);
    signal_add("log created", (SIGNAL_FUNC) sig_log_changed);
    signal_add("log destroyed", (SIGNAL_FUNC) sig_log_destroyed);
    signal_add("log item created", (SIGNAL_FUNC) sig_log_changed);
    signal_add("log item destroyed", (SIGNAL_FUNC) sig_log_changed);
    signal_add("channel created", (SIGNAL_FUNC) sig_channel_updated);
    signal_add("channel destroyed", (SIGNAL_FUNC) sig_channel_updated);
}

void dialog_log_deinit(void)
{
    save_log_setup();
    signal_remove("log opened", (SIGNAL_FUNC) sig_log_changed);
    signal_remove("log closed", (SIGNAL_FUNC) sig_log_closed);
    signal_remove("log created", (SIGNAL_FUNC) sig_log_changed);
    signal_remove("log destroyed", (SIGNAL_FUNC) sig_log_destroyed);
    signal_remove("log item created", (SIGNAL_FUNC) sig_log_changed);
    signal_remove("log item destroyed", (SIGNAL_FUNC) sig_log_changed);
    signal_remove("channel created", (SIGNAL_FUNC) sig_channel_updated);
    signal_remove("channel destroyed", (SIGNAL_FUNC) sig_channel_updated);
}
