/***

IPTraf
An IP Network Statistics Utility
   		
Written by Gerard Paul Java <riker@mozcom.com>
Copyright (c) Gerard Paul Java 1997-1999
   	
Version 2.1.1
Main Module

This software is open-source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <panel.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "dirs.h"
#include "deskman.h"
#include "menurt.h"
#include "stdwinset.h"
#include "utfdefs.h"
#include "othfilter.h"
#include "utfilter.h"
#include "ifstats.h"
#include "serv.h"
#include "options.h"
#include "externs.h"
#include "log.h"
#include "attrs.h"
#include "error.h"

#define WITHALL 1
#define WITHOUTALL 0

#define ALLSPEC "all"

int exitloop = 0;
int daemonized = 0;

extern void srpromisc(int mode);
extern void about();
extern void masq_warning();

void removetags()
{
    unlink(IPTIDFILE);
    unlink(IPMONIDFILE);
    unlink(GSTATIDFILE);
    unlink(DSTATIDFILE);
    unlink(TCPUDPIDFILE);
    unlink(LANMONIDFILE);
    unlink(TCPFLTIDFILE);
    unlink(UDPFLTIDFILE);
    unlink(OTHFLTIDFILE);
    unlink(PKTSIZEIDFILE);
}

/*
 * Handlers for the TERM signal and HUP signals.  There's nothing we can do
 * for the KILL.
 */

void termhandler()
{
    erase();
    refresh();
    endwin();
    removetags();
    fprintf(stderr, "Received TERM signal\n");
    fprintf(stderr, "IPTraf terminated\n\n");
    exit(0);
}

/*
 * Handler for the hangup signal.  Simply removes all tags.
 */

void huphandler()
{
    removetags();
    exit(0);
}

/* 
 * Handler for the SIGSEGV, Segmentation Fault.  Tries to clear the screen
 * and issue a better message than "Segmentation fault".
 */

void segvhandler()
{
    erase();
    refresh();
    endwin();
    removetags();
    fprintf(stderr, "Fatal error: unable to allocate memory for a critical function\n");
    fprintf(stderr, "IPTraf terminated abnormally\n\n");
    exit(2);
}

void usr2handler()
{
    exitloop = 1;
}


int check_ip_masquerade(void)
{
    int fd;

    fd = open("/proc/net/ip_masquerade", O_RDONLY);

    if (fd >= 0) {
	close(fd);
	return 1;
    }
    return 0;
}

void init_break_menu(struct MENU *break_menu)
{
    initmenu(break_menu, 5, 20, (LINES - 5) / 2, COLS / 2);
    additem(break_menu, " By packet ^s^ize", "Displays packet counts by packet size range");
    additem(break_menu, " By TCP/UDP ^p^ort", "Displays packet and byte counts by service port");
    additem(break_menu, " E^x^it menu", "Return to main menu");
}

/*
 * Get the ball rolling: The program interface routine.
 */

void program_interface(struct OPTIONS *options,
		       int opt, int dontwarn, char *optarg,
		       int facilitytime, int is_first_instance)
{
    struct MENU menu;
    struct MENU break_menu;

    int endloop = 0;
    int row = 1;
    int break_row = 1;
    int aborted;
    int break_aborted;
    int faborted;
    int filtered = 0;
    struct filterlist fl;
    struct othpoptions ofilter;
    char ifname[10];
    char *ifptr = NULL;
    struct porttab *ports;

    draw_desktop();
    attrset(STDATTR);
    move(0, 1);
    printw("IPTraf");

    loadsavedtcpfilter(&fl, &filtered);
    loadothfilter(&ofilter);
    indicate("");
    loadaddports(&ports);

    /*
     * Issue masquerading warning onscreen (intaractive mode) or in the log
     * (daemonized mode).
     */

    if ((!dontwarn) && (check_ip_masquerade())) {
	if (!daemonized)
	    masq_warning();
	else
	    write_daemon_err("Warning: IP masquerading enabled, you may get strange results");
    }
    if (opt == 0) {
	about();

	initmenu(&menu, 14, 35, (LINES - 14) / 2, (COLS - 35) / 2);

	additem(&menu, " IP traffic ^m^onitor", "Displays current IP traffic information");
	additem(&menu, " General interface ^s^tatistics", "Displays some statistics for attached interfaces");
	additem(&menu, " ^D^etailed interface statistics", "Displays more statistics for a selected interface");
	additem(&menu, " Statistical ^b^reakdowns", "Facilities for traffic counts by packet size or TCP/UDP port");
	additem(&menu, " ^L^AN station monitor", "Displays statistics on detected LAN stations");
	additem(&menu, NULL, NULL);
	additem(&menu, " ^T^CP display filters", "Manages TCP display filters");
	additem(&menu, " Other ^p^rotocol filters", "Select which non-TCP protocols to display");
	additem(&menu, NULL, NULL);
	additem(&menu, " C^o^nfigure", "Set various program options");
	additem(&menu, NULL, NULL);
	additem(&menu, " E^x^it", "Exits program");

	endloop = 0;

	do {
	    showmenu(&menu);
	    operatemenu(&menu, &row, &aborted);

	    switch (row) {
	    case 1:
		selectiface(ifname, WITHALL, &aborted);
		if (!aborted) {
		    if (strcmp(ifname, "") != 0)
			ifptr = ifname;
		    else
			ifptr = NULL;

		    ipmon(options,
		    filtered, &fl, &ofilter, options->timeout, 0, ifptr);
		}
		break;
	    case 2:
		ifstats(options, 0);
		break;
	    case 3:
		selectiface(ifname, WITHOUTALL, &aborted);
		if (!aborted)
		    detstats(ifname, options, 0);
		break;
	    case 4:
		break_row = 1;
		init_break_menu(&break_menu);
		showmenu(&break_menu);
		operatemenu(&break_menu, &break_row, &break_aborted);

		switch (break_row) {
		case 1:
		    selectiface(ifname, WITHOUTALL, &aborted);
		    if (!aborted)
			packet_size_breakdown(options, ifname, 0);
		    break;
		case 2:
		    selectiface(ifname, WITHOUTALL, &aborted);
		    if (!aborted)
			servmon(ifname, ports, options, 0);
		    break;
		case 4:
		    break;
		}
		destroymenu(&break_menu);
		break;
	    case 5:
		selectiface(ifname, WITHALL, &aborted);
		if (!aborted) {
		    if (strcmp(ifname, "") != 0)
			ifptr = ifname;
		    else
			ifptr = NULL;
		    hostmon(options, 0, ifptr);
		}
		break;
	    case 7:
		tcpfilterselect(&fl, &filtered, &faborted);
		break;
	    case 8:
		othfilterselect(&ofilter);
		saveothfilter(&ofilter);
		break;
	    case 10:
		setoptions(options, &ports, is_first_instance);
		saveoptions(options);
		break;
	    case 12:
		endloop = 1;
		break;
	    }
	} while (!endloop);

	destroymenu(&menu);
    } else {
	switch (opt) {
	case 'i':
	    if ((strcmp(optarg, ALLSPEC) == 0) || (strcmp(optarg, "") == 0))
		ifptr = NULL;
	    else
		ifptr = optarg;

	    ipmon(options,
		  filtered, &fl, &ofilter, options->timeout, facilitytime, ifptr);
	    break;
	case 'g':
	    ifstats(options, facilitytime);
	    break;
	case 'd':
	    detstats(optarg, options, facilitytime);
	    break;
	case 's':
	    servmon(optarg, ports, options, facilitytime);
	    break;
	case 'z':
	    packet_size_breakdown(options, optarg, facilitytime);
	    break;
	case 'l':
	    if ((strcmp(optarg, ALLSPEC) == 0) || (strcmp(optarg, "") == 0))
		ifptr = NULL;
	    else
		ifptr = optarg;

	    hostmon(options, facilitytime, ifptr);
	    break;
	}
    }

    destroyporttab(ports);
    erase();
    update_panels();
    doupdate();
}


/*
 * Command-line help facility.
 */

void commandhelp()
{
    printf("\nSyntax:\n");
    printf("    iptraf [ -f ][ -q ][ { -i iface | -g | -d iface | -s iface | -z iface |\n");
    printf("           -l iface } [ -B ] [-t timeout ]]\n\n");
    printf("Issue the iptraf command with no parameters for menu-driven operation.\n");
    printf("These options can also be supplied to the command:\n\n");
    printf("-i iface    - start the IP traffic monitor (use \"-i all\" for all interfaces)\n");
    printf("-g          - start the general interface statistics\n");
    printf("-d iface    - start the detailed statistics facility on an interface\n");
    printf("-s iface    - start the TCP and UDP monitor on an interface\n");
    printf("-z iface    - shows the packet size counts on an interface\n");
    printf("-l iface    - start the LAN station monitor (\"-l all\" for all LAN interfaces)\n");
    printf("-B          - Run in background (use only with one of the above parameters)\n");
    printf("-t timeout  - when used with one of the above parameters, tells\n");
    printf("              the facility to run only for the specified number of\n");
    printf("              minutes (timeout)\n");
    printf("-f          - Remove all lock files.  This will allow multiple\n");
    printf("              instances of facilities in different IPTraf processes.\n");
    printf("              Use with great caution.  Normally used to recover from\n");
    printf("              an abnormal termination.\n");
    printf("-q          - suppress IP masquerading warning on startup\n\n");
    printf("IPTraf %s Copyright (c) Gerard Paul Java 1997-1999\n", VERSION);
}

int first_instance()
{
    int fd;

    fd = open(IPTIDFILE, O_RDONLY);

    if (fd < 0)
	return !0;
    else {
	close(fd);
	return 0;
    }
}

void mark_first_instance()
{
    int fd;

    fd = open(IPTIDFILE, O_CREAT | O_WRONLY | O_TRUNC, S_IRUSR | S_IWUSR);
    if (fd < 0) {
	fprintf(stderr, "\nWarning: unable to tag this process\r\n");
	fprintf(stderr, "Press Enter to continue\r\n");
	getchar();
	return;
    }
    close(fd);
}

/*
 * Main routine
 */

int main(int argc, char **argv)
{
    struct OPTIONS options;
    extern char *optarg;
    extern int optind;
    extern int opterr;
    extern int optopt;
    int opt = 0;
    int command = 0;
    int is_first_instance;
    char keyparm[12];
    int facilitytime = 0;
    int dontwarn = 0;

#ifndef ALLOWUSERS
    if (getuid() != 0) {
	fprintf(stderr, "\nIPTraf Version %s\n", VERSION);
	fprintf(stderr, "Copyright (c) Gerard Paul Java 1997-1999\n\n");
	fprintf(stderr, "This program can be run only by the system administrator\n\n");
	exit(1);
    }
#endif

    /*
     * Parse command line
     */

    if (argc > 1) {
	do {
	    opterr = 0;
	    opt = getopt(argc, argv, "i::gd:s:z:l::hfqt:B");

	    if (opt == 'h') {
		commandhelp();
		exit(0);
	    } else if (opt == 'f') {
		removetags();
	    } else if (opt == 't') {
		facilitytime = atoi(optarg);
		if (facilitytime == 0) {
		    fprintf(stderr, "\nInvalid time value\n\n");
		    exit(1);
		}
	    } else if (opt == 'q') {
		dontwarn = 1;
	    } else if (opt == 'B') {	/* daemonization sequence */
		daemonized = 1;
	    } else if (opt == '?') {
		fprintf(stderr, "\nInvalid option or missing parameter, use iptraf -h for help\n\n");
		exit(1);
	    } else if (opt != -1) {
		if (optarg != 0) {
		    bzero(keyparm, 12);
		    strncpy(keyparm, optarg, 11);
		} else
		    strcpy(keyparm, "");
		    
		command = opt;
	    }
	} while ((opt != '?') && (opt != -1));
    }
    is_first_instance = first_instance();

    if (getenv("TERM") == NULL) {
	fprintf(stderr, "Your TERM variable is not set.\n");
	fprintf(stderr, "Please set it to an appropriate value.\n");
	exit(1);
    }
    loadoptions(&options);

    /*
     * If a facility is directly invoked from the command line, check for
     * a daemonization request
     */

    if ((daemonized) && (command != 0)) {
	switch (fork()) {
	case 0:		/* child */
	    setsid();
	    freopen("/dev/null", "w", stdout);	/* redirect std output */
	    freopen("/dev/null", "r", stdin);
	    freopen("/dev/null", "w", stderr);
	    signal(SIGUSR2, (void *) usr2handler);
	    options.logging = 1;               /* daemon mode implies logging */
	    break;
	case -1:		/* error */
	    fprintf(stderr, "\nFork error, IPTraf cannot run in background\n\n");
	    exit(1);
	default:		/* parent */
	    exit(0);
	}
    }
#ifdef SIMDAEMON
    daemonized = 1;
    freopen("/dev/null", "w", stdout);	/* redirect std output */
    freopen("/dev/null", "r", stdin);
    freopen("/dev/null", "w", stderr);
#endif

    initscr();

    mark_first_instance();

    if ((LINES < 24) || (COLS < 80)) {
	endwin();
	fprintf(stderr, "\nThis program requires a screen size of at least 80 columns by 24 lines\n");
	fprintf(stderr, "Please resize your window\n\n");
	exit(1);
    }
    signal(SIGTERM, (void *) termhandler);
    signal(SIGHUP, (void *) huphandler);
    signal(SIGSEGV, (void *) segvhandler);
    signal(SIGTSTP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGUSR1, SIG_IGN);

    start_color();
    standardcolors(options.color);
    noecho();
    nonl();
    cbreak();

#ifndef DEBUG
    curs_set(0);
#endif

    if (is_first_instance)
	srpromisc(options.promisc);

    program_interface(&options, command, dontwarn,
		      keyparm, facilitytime, is_first_instance);

    if (is_first_instance)
	srpromisc(0);

    endwin();

    if (is_first_instance)
	unlink(IPTIDFILE);

    return (0);
}
