#include "hx_types.h"
#include <sys/time.h>
#include <sys/socket.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include "htlc.h"
#include "hxlib.h"
#include "input.h"
#include "screen.h"
#include "term.h"
#include "hx.h"
#include "xmalloc.h"

int hx_sock = -1;

extern void rcv_hdr (void);

u_int8_t *hx_buf = 0;
u_int32_t hx_pos = 0, hx_len = SIZEOF_HX_HDR;
void (*hx_fun)(void) = rcv_hdr;

void hx_read (void);

void
hx_read (void)
{
	int r;

	r = recv(hx_sock, &(hx_buf[hx_pos]), hx_len, 0);
	if (r < 1 && errno != EINTR) {
		hx_conn_close();
		return;
	}
	hx_pos += r;
	hx_len -= r;
	if (!hx_len) {
		if (hx_fun)
			hx_fun();
		else
			hx_reset();
	}
}

#ifdef __BEOS__
#include <be/kernel/OS.h>

thread_id input_threadid;

int32 input_thread (void *p);

int32
input_thread (void *p)
{
	sigset_t mask;
	sigfillset(&mask);
	sigprocmask(SIG_SETMASK,&mask,0);
	for (;;)
		input_getchar();
}
#endif

void
hx_loop (void)
{
	fd_set rfds;

#ifdef __BEOS__
	thread_id thr;

	hx_reset();

	thr = spawn_thread(input_thread, "input", B_NORMAL_PRIORITY, 0);
	if (thr < B_NO_ERROR) {
		curscr_printf("\nspawn_thread: %ld", thr);
		hx_exit(-1);
	}
	input_threadid = thr;
	resume_thread(thr);

	for (;;) {
		if (hx_sock >= 0) {
			FD_ZERO(&rfds);
			FD_SET(hx_sock, &rfds);
			if (select(hx_sock+1, &rfds, 0, 0, 0) <= 0) {
				if (errno != EINTR && errno != B_BAD_SEM_ID) {
					curscr_printf("\nselect: %s", strerror(errno));
					hx_exit(-1);
				}
				continue;
			}
			hx_read();
		} else {
			sleep(3);
		}
	}
#else /* !BEOS */
	hx_reset();

	for (;;) {
		register int n;

		FD_ZERO(&rfds);
		FD_SET(0, &rfds);
		if (hx_sock >= 0)
			FD_SET((unsigned)hx_sock, &rfds);
		n = select(FD_SETSIZE, &rfds, 0, 0, 0);
		if (n <= 0) {
			if (errno != EINTR) {
				curscr_printf("\nselect: %s", strerror(errno));
				hx_exit(-1);
			}
			continue;
		}
		if (FD_ISSET(0, &rfds))
			input_getchar();
		if (hx_sock > 1 && FD_ISSET((unsigned)hx_sock, &rfds))
			hx_read();
	}
#endif /* __BEOS__ */
}

void
hx_set (void (*fun)(void), u_int32_t len)
{
	hx_len += len;
	hx_buf = xrealloc(hx_buf, 8 + hx_len + hx_pos);
	hx_fun = fun;
}

void
hx_reset (void)
{
	hx_buf = xrealloc(hx_buf, 8 + SIZEOF_HX_HDR);
	hx_pos = 0;
	hx_len = SIZEOF_HX_HDR;
	hx_fun = rcv_hdr;
}

void
hx_exit (int st)
{
	term_reset();
	exit(st);
}
