/* gnome-napster, a GNOME Napster client.
 * Copyright (C) 1999 Evan Martin <eeyem@u.washington.edu>
 */
#include <gnome.h>
#include "songinfo.h"
#include "gtkflist.h"
#include "host.h"
#include "ui.h"
#include "lnap.h"
#include "gconfig.h"
#include "util.h"
#include "download.h"

static GtkWidget *clist, *hostlist, *controlbox;
static GList *songlist = NULL;
static int songcount = 0;

enum {
	COL_NAME, COL_SIZE, COL_BITRATE, COL_TIME, COL_COUNT
};
static gchar* clist_titles[] = { "Song Name", "Size", "kbps", "Length" };

static void browsehost(char *host) {
	char *add[COL_COUNT];

	songlist = g_list_free_all(songlist);
	songcount = 0;

	memset(add, 0, sizeof(add));
	add[0] = g_strdup_printf("Requesting file list from '%s'...", host);
	gtk_clist_clear(GTK_CLIST(clist));
	gtk_clist_append(GTK_CLIST(clist), add);
	g_free(add[0]);

	nap_sendpacket(nap_socknap, NM_BROWSE_REQUEST, strlen(host), host);
}

static void browse_cb(GtkWidget *w, gpointer d) {
	char *host;
	GList *sel;

	sel = GTK_CLIST(hostlist)->selection;
	if (sel == NULL) return;
	gtk_clist_get_text(GTK_CLIST(hostlist), (int)sel->data, 0, &host);

	browsehost(host);
}

static void
list_update(int row) {
	ssong_info *i = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	char *text = short_path(i->title, i->pathshift);
	gtk_clist_set_text(GTK_CLIST(clist), row, COL_NAME, text);
}

static void 
list_add(ssong_info *i) {
	gchar* add[COL_COUNT];
	int row;

	add[COL_NAME] = short_path(i->title, i->pathshift);
	add[COL_SIZE] = g_strdup_printf("%lu", i->size);
	add[COL_TIME] = g_strdup_printf("%u:%02u", i->time/60, i->time % 60);
	add[COL_BITRATE] = g_strdup_printf("%d", i->bitrate);
	row = gtk_clist_append(GTK_CLIST(clist), add);
	gtk_clist_set_row_data(GTK_CLIST(clist), row, i);
};

static void list_statustext(char *text) {
	char *add[COL_COUNT];
	memset(add, 0, sizeof(add));
	add[0] = text;
	gtk_clist_clear(GTK_CLIST(clist));
	gtk_clist_append(GTK_CLIST(clist), add);
}

void host_notfound(char *text) {
	char *add = g_strdup_printf("Host '%s' not found.", text);
	list_statustext(add);
	g_free(add);
}

void host_result(char *text) {
	ssong_info *i;
	char buf[50];

	i = songinfo_new();
	sscanf(text, "%s \"%[^\"]\" %s %lu %d %d %d",
			i->host, i->title, i->checksum, &i->size, 
			&i->bitrate, &i->samprate, &i->time);

	i->pathshift = gnome_config_get_int(CONFIG_PREFIX "/Interface/PathDepth=1");

	songcount++;
	snprintf(buf, 50, "Got %d results...", songcount);
	gtk_clist_set_text(GTK_CLIST(clist), 0, COL_NAME, buf);

	songlist = g_list_append(songlist, i);
}

void host_complete(void) {
	GList *l = songlist;

	list_statustext("Updating list...");
	gtk_clist_freeze(GTK_CLIST(clist));
	gtk_clist_clear(GTK_CLIST(clist));
	while (l != NULL) {
		list_add(l->data);
		l = l->next;
	}
	gtk_clist_thaw(GTK_CLIST(clist));
}

void host_addhost(char *name) {
	char *add[1] = { name };
	gtk_clist_append(GTK_CLIST(hostlist), add);
}

void host_login_status(int loggedin) {
	gtk_widget_set_sensitive(controlbox, loggedin);
}

/* implement column sorting when selecting column headings */
static void 
sort_col(GtkCList *clist, gint column, gint width, gpointer user_data)
{
	switch (column) {
		case COL_SIZE:
			gtk_clist_set_compare_func(clist, songinfo_compare_size);
			break;
		case COL_BITRATE:
			gtk_clist_set_compare_func(clist, songinfo_compare_bitrate);
			break;
		case COL_TIME:
			gtk_clist_set_compare_func(clist, songinfo_compare_time);
			break;
		case COL_NAME:
		default:
			gtk_clist_set_compare_func(clist, NULL); /* default compare func */
	}

	if (column == clist->sort_column) {
		/* switch sort type */
		gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING + GTK_SORT_ASCENDING
				- clist->sort_type);
	} else {
		/* select column to sort */
		gtk_clist_set_sort_column(clist, column);
	}

	/* go for it */
	gtk_clist_sort(clist);
}

static void getfiles_cb() {
	GList *sel;
	ssong_info *i;

	sel = GTK_CLIST(clist)->selection;
	if (sel == NULL) return;
	while (sel != NULL) {
		i = gtk_clist_get_row_data(GTK_CLIST(clist), (int)sel->data);
		if (i != NULL) /* NULL possible if they select "no matches found" */
			download_enqueue(i->host, i->title, i->pathshift);
		sel = sel->next;
	}
}

static void shift_foreach_cb(gpointer data, gpointer d) {
	int row = (int)data;
	int shift = (int)d;
	ssong_info *i = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	if (i == NULL) return;

	if (i->pathshift + shift < 0) i->pathshift = 0;
	else i->pathshift += shift;
	list_update(row);
}

static void shift_cb(GtkWidget *w, gpointer d) {
	if (GTK_CLIST(clist)->selection)
		g_list_foreach(GTK_CLIST(clist)->selection, shift_foreach_cb, d);
}

static gint button_press_cb(GtkWidget *w, GdkEventButton *e, gpointer d) {
	/*GtkWidget *menu;
	ssong_info *i;
	int row, col;*/

	if (e->type == GDK_2BUTTON_PRESS) {
		getfiles_cb();
		return TRUE;
	} /*else if (e->type == GDK_BUTTON_PRESS && e->button == 3) {
		gtk_clist_get_selection_info(GTK_CLIST(clist), e->x, e->y,
				&row, &col);
		i = gtk_clist_get_row_data(GTK_CLIST(clist), row);
		if (i == NULL) return FALSE;
		menu = gnome_popup_menu_new(popup_menu);
		gnome_popup_menu_do_popup_modal(menu, NULL, NULL, e, i->host);
		gtk_widget_destroy(menu);
		return TRUE;
	}*/

	return FALSE;
}


GtkWidget*
create_host(void) {
	GtkWidget *mainbox, *paned;
	GtkWidget *hostvbox, *scroll;
	GtkWidget *filevbox;
	GtkWidget *hbox, *b;
	static gchar *host_title[] = { "Host" };

	mainbox = gtk_vbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(mainbox), 10);

	paned = gtk_hpaned_new();
	gtk_paned_set_gutter_size(GTK_PANED(paned), 20);

	hostvbox = gtk_vbox_new(FALSE, 10);
		hostlist = gtk_flist_new_with_titles(1, host_title);
	gtk_box_pack_start(GTK_BOX(hostvbox), hostlist, TRUE, TRUE, 0);
		controlbox = gtk_vbox_new(FALSE, 10);
			b = gtk_button_new_with_label("Browse Host");
			gtk_signal_connect(GTK_OBJECT(b), "clicked",
					GTK_SIGNAL_FUNC(browse_cb), NULL);
		gtk_box_pack_start(GTK_BOX(controlbox), b, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hostvbox), controlbox, FALSE, FALSE, 0);

	filevbox = gtk_vbox_new(FALSE, 10);
	scroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), 
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		clist = gtk_flist_new_with_titles(sizeof(clist_titles)/sizeof(char*), 
				clist_titles);
		hbox = gtk_hbox_new(FALSE, 0);
				b = songinfo_arrow_button_new(app, TRUE);
				gtk_signal_connect(GTK_OBJECT(b), "clicked",
						GTK_SIGNAL_FUNC(shift_cb), (gpointer)-1);
			gtk_box_pack_start(GTK_BOX(hbox), b, FALSE, FALSE, 0);
				b = songinfo_arrow_button_new(app, FALSE);
				gtk_signal_connect(GTK_OBJECT(b), "clicked",
						GTK_SIGNAL_FUNC(shift_cb), (gpointer)1);
			gtk_box_pack_start(GTK_BOX(hbox), b, FALSE, FALSE, 0);

			gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(clist_titles[0]), 
					FALSE, FALSE, 5);
		gtk_widget_show_all(hbox);
		gtk_clist_set_column_widget(GTK_CLIST(clist), 0, hbox);
		gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_MULTIPLE);
		gtk_clist_set_column_justification(GTK_CLIST(clist), COL_SIZE,
				GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification(GTK_CLIST(clist), COL_BITRATE,
				GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification(GTK_CLIST(clist), COL_TIME,
				GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_width(GTK_CLIST(clist), COL_NAME, 230);
		gtk_clist_set_column_width(GTK_CLIST(clist), COL_SIZE, 50);
		gtk_clist_set_column_width(GTK_CLIST(clist), COL_BITRATE, 25);
		gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
				GTK_SIGNAL_FUNC(button_press_cb), NULL);
		gtk_signal_connect(GTK_OBJECT(clist), "click-column",
				GTK_SIGNAL_FUNC(sort_col), NULL);
		gtk_clist_set_sort_column(GTK_CLIST(clist), 0);	/* sort on title by default */
		gtk_clist_set_auto_sort(GTK_CLIST(clist), TRUE);
	gtk_container_add(GTK_CONTAINER(scroll), clist);
	gtk_box_pack_start(GTK_BOX(filevbox), scroll, TRUE, TRUE, 0);
		hbox = gtk_hbox_new(FALSE, 10);
			b = gtk_button_new_with_label("Get File(s)");
		gtk_box_pack_end(GTK_BOX(hbox), b, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(filevbox), hbox, FALSE, FALSE, 0);

	gtk_paned_add1(GTK_PANED(paned), hostvbox);
	gtk_paned_add2(GTK_PANED(paned), filevbox);

	gtk_box_pack_start(GTK_BOX(mainbox), paned, TRUE, TRUE, 0);

	return mainbox;	
}


