/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module defines the structure of compiled Malaga files. */

/* constants ================================================================*/

#define MALAGA_LEN 6 /* length of magic code at the beginning of Malaga file */

/* values for <file_type> */
#define SYMBOL_FILE 0
#define RULE_FILE 1
#define LEXICON_FILE 2

/* versions of compiled Malaga file types */
#define SYMBOL_CODE_VERSION 6
#define RULE_CODE_VERSION 38
#define LEXICON_CODE_VERSION 11

/* types ====================================================================*/

typedef struct /* the common header of every Malaga file */
{
  char malaga[MALAGA_LEN]; /* "MALAGA" to recognise Malaga files */
  short_t file_type; /* SYMBOL_FILE, RULE_FILE or LEXICON_FILE */
  int_t code_version; /* only load code of the current version */
  time_t symbol_compile_time; /* time when symbol file was compiled */
} common_header_t;

/*---------------------------------------------------------------------------*/

typedef struct /* the format of a Malaga rule file */
{
  common_header_t common_header;
  int_t initial_rule_set; /* index of the initial rule set in rule_set */
  int_t initial_cat; /* value index of the initial combi rule category */
  int_t robust_rule; /* rule number of robust_rule */
  int_t pruning_rule; /* rule number of pruning_rule */
  int_t allo_rule; /* rule number of allo_rule */
  int_t input_filter; /* rule number of input_filter */
  int_t output_filter; /* rule number of output_filter */
  int_t rules_size; /* number of rules in this file */
  int_t rule_sets_size; /* size of rule set table */
  int_t instrs_size; /* number of instructions in this file */
  int_t values_size; /* size of Malaga value table */
  int_t src_lines_size; /* number of correspondences
			 * between source lines and rule code */
  int_t vars_size; /* number of variable names */
  int_t var_scopes_size; /* number of variable scopes */
  int_t strings_size; /* size of string table */

  /* the following blocks are of dynamic size: */
  /* rule_t rules [rules_size]; */
  /* int_t rule_sets [rule_sets_size]; */
  /* instr_t instrs [instrs_size]; */
  /* cell_t values [values_size]; */
  /* src_line_t src_lines [src_lines_size]; */
  /* var_t vars [vars_size]; */
  /* var_scope_t var_scopes [var_scopes_size]; */
  /* char strings [strings_size]; */
} rule_header_t;

/*---------------------------------------------------------------------------*/

typedef struct /* an entry in the symbol table */ 
{
  int_t name; /* string pool index to category name */ 
  int_t atoms; /* <values> index to list of the atomic symbols
                 * of a multi-symbol (or -1) */
} symbol_entry_t;

typedef struct /* the format of a Malaga symbol file */
{
  common_header_t common_header;
  int_t symbols_size; /* number of symbols in this file */
  int_t values_size; /* size of Malaga value table
		      * (used to define multi-symbols) */
  int_t strings_size; /* size of string table (used to define symbol names) */

  /* the following blocks are of dynamic size:
   * symbol_entry_t symbols [symbols_size];
   * cell_t values [values_size];
   * char strings [strings_size]; */
} symbol_header_t;

/*---------------------------------------------------------------------------*/

typedef struct /* the format of a Malaga lexicon file */
{ 
  common_header_t common_header;
  int_t trie_size; /* size of trie table */
  int_t trie_root; /* index of root node in <trie>. */
  int_t cat_lists_size; /* size of cat lists table */
  int_t values_size; /* size of value table */

  /* the following blocks are of dynamic size:
   * int_t trie [trie_size];
   * int_t cat_lists [cat_lists_size];
   * cell_t values [values_size]; */
} lexicon_header_t;

/* functions ================================================================*/

extern void check_header (common_header_t *header, 
			  string_t file_name, 
			  int_t file_type, 
			  int_t code_version);
/* Check if <header> is of <file_type> and <code_version>.
 * <file_name> is needed for error messages. */

extern void set_header (common_header_t *header, 
			int_t file_type,
  			int_t code_version);
/* Set header to be of <file_type> and <code_version>. */

/* end of file ==============================================================*/
