/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include "avl.h"
#include "common.h"
#include "error.h"
#include "expr.h"
#include "misc.h"
#include "var.h"
#include "exprP.h"
#include "str.h"
#include "lexer.h"
#include "lexerP.h"
#include "vector.h"
#include "vfm.h"

/* Declarations. */

/* lowest precedence */
static exprtype parse_or (any_node ** n);
static exprtype parse_and (any_node ** n);
static exprtype parse_not (any_node ** n);
static exprtype parse_rel (any_node ** n);
static exprtype parse_add (any_node ** n);
static exprtype parse_mul (any_node ** n);
static exprtype parse_neg (any_node ** n);
static exprtype parse_exp (any_node ** n);
static exprtype parse_primary (any_node ** n);
static exprtype parse_function (any_node ** n);
/* highest precedence */

#if DEBUGGING
static void debug_print_tree (any_node *, int);
#endif

#if GLOBAL_DEBUGGING
static void debug_print_postfix (expression *);
#endif

/* Utility functions. */

static const char *
exprtypename (exprtype t)
{
  switch (t)
    {
    case EX_ERROR:
      return _("error");
    case EX_BOOLEAN:
      return _("Boolean");
    case EX_NUMERIC:
      return _("numeric");
    case EX_STRING:
      return _("string");
    default:
      assert (0);
    }
#if __GNUC__ || __BORLANDC__
  return 0;
#endif
}

static const char *
typename (int t)
{
  switch (t)
    {
    case NUMERIC:
      return _("numeric");
    case ALPHA:
      return _("string");
    default:
      assert (0);
    }
#if __GNUC__ || __BORLANDC__
  return 0;
#endif
}

static void
make_bool (any_node ** n)
{
  any_node *c;
  c = xmalloc (sizeof (nonterm_node));
  c->nnt.type = OP_NUM_TO_BOOL;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  *n = c;
}

void
free_node (any_node * n)
{
  if (n->type < OP_TERMINAL)
    {
      int i;

      for (i = 0; i < n->nnt.n; i++)
	free_node (n->nnt.arg[i]);
    }
  free (n);
}

void
free_expression (expression * e)
{
  if (!e)
    return;
  free (e->op);
  free (e->vars);
  free (e->dbl);
  free (e->str);
  free (e->stack);
  free (e->str_stk);
  free (e);
}

static void init_functab (void);

expression *
parse_expression (int flags)
{
  expression *e;
  any_node *n;
  int type;

  init_functab ();

  type = parse_or (&n);
  if (type == EX_ERROR)
    return NULL;

  /* Enforce PXP_BOOLEAN flag. */
  if (flags & PXP_BOOLEAN)
    {
      if (type == EX_STRING)
	{
	  free_node (n);
	  msg (SE, _("A Boolean expression was expected in a place "
		     "where a string expression was supplied."));
	  return NULL;
	}
      else if (type == EX_NUMERIC)
	{
	  any_node *m = xmalloc (sizeof (nonterm_node));
	  m->nnt.type = OP_NUM_TO_BOOL;
	  m->nnt.n = 1;
	  m->nnt.arg[0] = n;
	  n = m;
	}
    }
  
  /* Enforce PXP_NUMERIC flag. */
  if (flags & PXP_NUMERIC)
    if (type != EX_NUMERIC)
      {
	msg (SE, _("A numeric expression was expected in a place "
	     "where one was not supplied."));
	return NULL;
      }

#if DEBUGGING && 0
  debug_print_tree (n, 0);
#endif

  n = (any_node *) optimize_expression ((nonterm_node *) n);
#if DEBUGGING && 0
  debug_print_tree (n, 0);
#endif

  e = xmalloc (sizeof (expression));
  e->type = type;
  dump_expression (n, e);
  free_node (n);

#if GLOBAL_DEBUGGING
#if !DEBUGGING
  if (flags & PXP_DUMP)
#endif
    debug_print_postfix (e);
#endif

  return e;
}

/* Recursive-descent expression parser. */

/* Parses the OR level. */
static exprtype
parse_or (any_node ** n)
{
  char typ[] = N_("The OR operator cannot take string operands.");
  any_node *c;
  int type = parse_and (n);

  if (type == EX_ERROR || token != OR)
    return type;
  if (type == EX_STRING)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }
  else if (type == EX_NUMERIC)
    make_bool (n);

  c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
  c->nnt.type = OP_OR;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  while (1)
    {
      get_token ();
      type = parse_and (n);
      if (type == EX_ERROR)
	goto fail;
      else if (type == EX_STRING)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      else if (type == EX_NUMERIC)
	make_bool (n);
      c->nnt.arg[c->nnt.n++] = *n;

      if (token != OR)
	break;
      c = xrealloc (c, (sizeof (nonterm_node)
			+ (c->nnt.n + 1) * sizeof (any_node *)));
    }
  *n = c;
  return EX_BOOLEAN;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses the AND level. */
static exprtype
parse_and (any_node ** n)
{
  static char typ[] = N_("The AND operator cannot take string operands.");
  any_node *c;
  int type = parse_not (n);

  if (type == EX_ERROR)
    return EX_ERROR;
  if (token != AND)
    return type;
  if (type == EX_STRING)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }
  else if (type == EX_NUMERIC)
    make_bool (n);

  c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
  c->nnt.type = OP_AND;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  while (1)
    {
      get_token ();
      type = parse_not (n);
      if (type == EX_ERROR)
	goto fail;
      else if (type == EX_STRING)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      else if (type == EX_NUMERIC)
	make_bool (n);
      c->nnt.arg[c->nnt.n++] = *n;

      if (token != AND)
	break;
      c = xrealloc (c, (sizeof (nonterm_node)
			+ (c->nnt.n + 1) * sizeof (any_node *)));
    }
  *n = c;
  return EX_BOOLEAN;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses the NOT level. */
static exprtype
parse_not (any_node ** n)
{
  static char typ[] = N_("The NOT operator cannot take a string operand.");
  any_node *c;
  int not = 0;
  int type;

  while (match_tok (NOT))
    not ^= 1;
  type = parse_rel (n);
  if (!not || type == EX_ERROR)
    return type;

  if (type == EX_STRING)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }
  else if (type == EX_NUMERIC)
    make_bool (n);

  c = xmalloc (sizeof (nonterm_node));
  c->nnt.type = OP_NOT;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  *n = c;
  return EX_BOOLEAN;
}

static exprtype
parse_rel (any_node ** n)
{
  static char typ[] = N_("Strings cannot be compared with numeric or Boolean "
			 "values with the relational operators "
			 "= >= > <= < <>.");
  any_node *c;
  int type = parse_add (n);

  if (type == EX_ERROR)
    return EX_ERROR;
  if (token == '=')
    token = EQ;
  if (token < EQ || token > NE)
    return type;

  while (1)
    {
      int t;

      c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
      switch (token)
	{
	case EQ:
	  c->nnt.type = OP_EQ;
	  break;
	case GE:
	  c->nnt.type = OP_GE;
	  break;
	case GT:
	  c->nnt.type = OP_GT;
	  break;
	case LE:
	  c->nnt.type = OP_LE;
	  break;
	case LT:
	  c->nnt.type = OP_LT;
	  break;
	case NE:
	  c->nnt.type = OP_NE;
	  break;
	default:
	  assert (0);
	}
      c->nnt.n = 1;
      c->nnt.arg[0] = *n;
      get_token ();

      t = parse_add (n);
      if (t == EX_ERROR)
	goto fail;
      if (t == EX_BOOLEAN && type == EX_NUMERIC)
	make_bool (&c->nnt.arg[0]);
      else if (t == EX_NUMERIC && type == EX_BOOLEAN)
	make_bool (n);
      else if (t == EX_STRING && type == EX_STRING)
	switch (c->nnt.type)
	  {
	  case OP_EQ:
	    c->nnt.type = OP_STRING_EQ;
	    break;
	  case OP_GE:
	    c->nnt.type = OP_STRING_GE;
	    break;
	  case OP_GT:
	    c->nnt.type = OP_STRING_GT;
	    break;
	  case OP_LE:
	    c->nnt.type = OP_STRING_LE;
	    break;
	  case OP_LT:
	    c->nnt.type = OP_STRING_LT;
	    break;
	  case OP_NE:
	    c->nnt.type = OP_STRING_NE;
	    break;
	  default:
	    assert (0);
	  }
      else if (t != type)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      c->nnt.arg[c->nnt.n++] = *n;
      *n = c;

      if (token == '=')
	token = EQ;
      if (token < EQ || token > NE)
	break;

      type = EX_BOOLEAN;
    }
  return EX_BOOLEAN;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses the addition and subtraction level. */
static exprtype
parse_add (any_node ** n)
{
  static char typ[] = N_("The `+' and `-' operators may only be used with "
			 "numeric operands.");
  any_node *c;
  int type;
  int op;

  type = parse_mul (n);
  convert_negative_to_dash ();
  if (type == EX_ERROR || (token != '+' && token != '-'))
    return type;
  if (type != EX_NUMERIC)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }

  c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
  c->nnt.type = OP_PLUS;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  while (1)
    {
      op = token;
      get_token ();

      type = parse_mul (n);
      if (type == EX_ERROR)
	goto fail;
      else if (type != EX_NUMERIC)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      c->nnt.arg[c->nnt.n] = *n;
      convert_negative_to_dash ();
      if (op == '-')
	{
	  *n = xmalloc (sizeof (nonterm_node));
	  (*n)->nnt.type = OP_NEG;
	  (*n)->nnt.n = 1;
	  (*n)->nnt.arg[0] = c->nnt.arg[c->nnt.n];
	  c->nnt.arg[c->nnt.n] = *n;
	}
      c->nnt.n++;

      if (token != '+' && token != '-')
	break;
      c = xrealloc (c, (sizeof (nonterm_node)
			+ (c->nnt.n + 1) * sizeof (any_node *)));
    }
  *n = c;
  return EX_NUMERIC;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses the multiplication and division level. */
static exprtype
parse_mul (any_node ** n)
{
  static char typ[] = N_("The `*' and `/' operators may only be used with "
			 "numeric operands.");
  any_node *c;
  int type;
  int op;

  type = parse_neg (n);
  if (type == EX_ERROR || (token != '*' && token != '/'))
    return type;
  if (type != EX_NUMERIC)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }

  c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
  c->nnt.type = OP_MUL;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  while (1)
    {
      op = token;
      get_token ();

      type = parse_neg (n);
      if (type == EX_ERROR)
	goto fail;
      else if (type != EX_NUMERIC)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      c->nnt.arg[c->nnt.n] = *n;
      if (op == '/')
	{
	  *n = xmalloc (sizeof (nonterm_node));
	  (*n)->nnt.type = OP_INV;
	  (*n)->nnt.n = 1;
	  (*n)->nnt.arg[0] = c->nnt.arg[c->nnt.n];
	  c->nnt.arg[c->nnt.n] = *n;
	}
      c->nnt.n++;

      if (token != '*' && token != '/')
	break;
      c = xrealloc (c, (sizeof (nonterm_node)
			+ (c->nnt.n + 1) * sizeof (any_node *)));
    }
  *n = c;
  return EX_NUMERIC;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses the unary minus level. */
static exprtype
parse_neg (any_node ** n)
{
  static char typ[] = N_("The unary minus (-) operator can only "
			 "take a numeric operand.");
  any_node *c;
  int neg = 0;
  int type;

  for (;;)
    {
      convert_negative_to_dash ();
      if (!match_tok ('-'))
	break;
      neg ^= 1;
    }
  type = parse_exp (n);
  if (!neg || type == EX_ERROR)
    return type;
  if (type != EX_NUMERIC)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }

  c = xmalloc (sizeof (nonterm_node));
  c->nnt.type = OP_NEG;
  c->nnt.n = 1;
  c->nnt.arg[0] = *n;
  *n = c;
  return EX_NUMERIC;
}

static exprtype
parse_exp (any_node ** n)
{
  static char typ[] = N_("Both operands to the ** operator must be numeric.");
  any_node *c;
  int type = parse_primary (n);

  if (type == EX_ERROR || token != EXP)
    return type;
  if (type != EX_NUMERIC)
    {
      free_node (*n);
      return msg (SE, gettext (typ));
    }

  while (1)
    {
      c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
      c->nnt.type = OP_POW;
      c->nnt.n = 1;
      c->nnt.arg[0] = *n;
      get_token ();

      type = parse_primary (n);
      if (type == EX_ERROR)
	goto fail;
      else if (type != EX_NUMERIC)
	{
	  msg (SE, gettext (typ));
	  goto fail;
	}
      c->nnt.arg[c->nnt.n++] = *n;
      *n = c;

      if (token != EXP)
	break;
    }
  return EX_NUMERIC;

fail:
  free_node (c);
  return EX_ERROR;
}

/* Parses system variables. */
static exprtype
parse_sysvar (any_node ** n)
{
  if (streq (tokstr, "$CASENUM"))
    {
      *n = xmalloc (sizeof (casenum_node));
      (*n)->cas.type = OP_CASENUM;
      return EX_NUMERIC;
    }
  else
    {
      double d;

      if (streq (tokstr, "$SYSMIS"))
	d = SYSMIS;
      else if (streq (tokstr, "$JDATE"))
	{
	  struct tm *time;
	  time = localtime (&last_vfm_invocation);
	  d = yrmoda (time->tm_year + 1900, time->tm_mon + 1, time->tm_mday);
	}
      else if (streq (tokstr, "$DATE"))
	{
	  static const char *months[12] =
	    {
	      "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
	      "JUL", "AUG", "SEP", "OCT", "NOV", "DEC",
	    };

	  struct tm *time;
	  char temp_buf[10];

	  time = localtime (&last_vfm_invocation);
	  sprintf (temp_buf, "%02d %s %02d", abs (time->tm_mday) % 100,
		months[abs (time->tm_mon) % 12], abs (time->tm_year) % 100);

	  *n = xmalloc (sizeof (str_con_node) + 8);
	  (*n)->stc.type = OP_STR_CON;
	  (*n)->stc.len = 9;
	  memcpy ((*n)->stc.s, temp_buf, 9);
	  return EX_STRING;
	}
      else if (streq (tokstr, "$TIME"))
	{
	  struct tm *time;
	  time = localtime (&last_vfm_invocation);
	  d = (yrmoda (time->tm_year + 1900, time->tm_mon + 1,
		       time->tm_mday) * 60. * 60. * 24.
	       + time->tm_hour * 60 * 60.
	       + time->tm_min * 60.
	       + time->tm_sec);
	}
      else if (streq (tokstr, "$LENGTH"))
	{
	  msg (SW, _("Use of $LENGTH is obsolete, returning default of 66."));
	  d = 66.0;
	}
      else if (streq (tokstr, "$WIDTH"))
	{
	  msg (SW, _("Use of $WIDTH is obsolete, returning default of 131."));
	  d = 131.0;
	}
      else
	{
	  msg (SE, _("Unknown system variable %s."), tokstr);
	  return EX_ERROR;
	}

      *n = xmalloc (sizeof (num_con_node));
      (*n)->nmc.type = OP_NUM_CON;
      (*n)->nmc.value = d;
      return EX_NUMERIC;
    }
}

/* Parses numbers, varnames, etc.--the PRIMARY level. */
static exprtype
parse_primary (any_node ** n)
{
  switch (token)
    {
    case ID:
      {
	variable *v;

	if (lookahead () == '(')
	  return parse_function (n);

	/* By this point, the current token must be a user variable or
	   a system variable. */
	if (tokstr[0] == '$')
	  {
	    exprtype type = parse_sysvar (n);
	    get_token ();
	    return type;
	  }
	v = find_variable (tokstr);
	get_token ();
	if (!v)
	  {
	    syntax_error (_("expecting variable name"));
	    return EX_ERROR;
	  }
	*n = xmalloc (sizeof (var_node));
	(*n)->var.type = v->type == NUMERIC ? OP_NUM_VAR : OP_STR_VAR;
	(*n)->var.v = v;
	return (v->type == NUMERIC ? EX_NUMERIC : EX_STRING);
      }
    case NUM:
      *n = xmalloc (sizeof (num_con_node));
      (*n)->nmc.type = OP_NUM_CON;
      (*n)->nmc.value = tokval;
      get_token ();
      return EX_NUMERIC;
    case STRING:
      {
	int len = strlen (tokstr);
	*n = xmalloc (sizeof (str_con_node) + (len - 1) * sizeof (char));
	(*n)->stc.type = OP_STR_CON;
	(*n)->stc.len = len;
	memcpy ((*n)->stc.s, tokstr, len);
	get_token ();
	return EX_STRING;
      }
    case '(':
      {
	exprtype t;
	get_token ();
	t = parse_or (n);
	if (!match_tok (')'))
	  {
	    msg (SE, _("Missing right parenthesis in expression."));
	    free_node (*n);
	    return EX_ERROR;
	  }
	return t;
      }
    default:
      syntax_error (_("in expression"));
      return EX_ERROR;
    }
}

/* Individual function parsing. */

typedef struct function function;
struct function
  {
    const char *s;
    int t;
    exprtype (*func) (function *, int, any_node **);
    const char *desc;
  };

static avl_tree *functree;

static int get_num_args (function *, int, any_node **);

static exprtype
unary_func (function * f, unused int x, any_node ** n)
{
  double divisor;
  nonterm_node *c;

  if (!get_num_args (f, 1, n))
    return EX_ERROR;
  switch (f->t)
    {
    case OP_CTIME_DAYS:
      divisor = 1 / 60. / 60. / 24.;
      goto multiply;
    case OP_CTIME_HOURS:
      divisor = 1 / 60. / 60.;
      goto multiply;
    case OP_CTIME_MINUTES:
      divisor = 1 / 60.;
      goto multiply;
    case OP_TIME_DAYS:
      divisor = 60. * 60. * 24.;
      goto multiply;

    case OP_CTIME_SECONDS:
      c = &(*n)->nnt;
      *n = (*n)->nnt.arg[0];
      free (c);
      return EX_NUMERIC;
    }
  return EX_NUMERIC;

multiply:
  /* Arrive here when we encounter an operation that is just a
     glorified version of a multiplication or division.  Converts the
     operation directly into that multiplication. */
  c = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[1]));
  c->type = OP_MUL;
  c->n = 2;
  c->arg[0] = (*n)->nnt.arg[0];
  c->arg[1] = xmalloc (sizeof (num_con_node));
  c->arg[1]->nmc.type = OP_NUM_CON;
  c->arg[1]->nmc.value = divisor;
  free (*n);
  *n = (any_node *) c;
  return EX_NUMERIC;
}

static exprtype
binary_func (function * f, unused int x, any_node ** n)
{
  if (!get_num_args (f, 2, n))
    return EX_ERROR;
  return EX_NUMERIC;
}

static exprtype
ternary_func (function * f, unused int x, any_node ** n)
{
  if (!get_num_args (f, 3, n))
    return EX_ERROR;
  return EX_NUMERIC;
}

static exprtype
MISSING_func (function * f, unused int x, any_node ** n)
{
  if (token == ID && is_varname (tokstr) && lookahead () == ')')
    {
      sys_node *c = xmalloc (sizeof (sys_node));
      c->v = parse_variable ();
      c->type = c->v->type == ALPHA ? OP_STR_MIS : OP_NUM_SYS;
      *n = (any_node *) c;
      return EX_BOOLEAN;
    }
  if (!get_num_args (f, 1, n))
    return EX_ERROR;
  return EX_BOOLEAN;
}

static exprtype
SYSMIS_func (unused function * f, unused int x, any_node ** n)
{
  exprtype t;
  
  if (token == ID && is_varname (tokstr) && lookahead () == ')')
    {
      variable *v;
      v = parse_variable ();
      if (v->type == ALPHA)
	{
	  num_con_node *c = xmalloc (sizeof (num_con_node));
	  c->type = OP_NUM_CON;
	  c->value = 0;
	  return EX_BOOLEAN;
	}
      else
	{
	  sys_node *c = xmalloc (sizeof (sys_node));
	  c->type = OP_NUM_SYS;
	  c->v = v;
	  return EX_BOOLEAN;
	}
    }
  
  t = parse_or (n);
  if (t == EX_ERROR)
    return t;
  else if (t == EX_NUMERIC)
    {
      /* Return SYSMIS expression. */
      any_node *m = xmalloc (sizeof (nonterm_node));
      m->nnt.type = OP_SYSMIS;
      m->nnt.n = 1;
      m->nnt.arg[0] = *n;
      *n = m;
      return EX_BOOLEAN;
    }
  else /* EX_STRING or EX_BOOLEAN */
    {
      /* Return constant `true' value. */
      free_node (*n);
      *n = xmalloc (sizeof (num_con_node));
      (*n)->nmc.type = OP_NUM_CON;
      (*n)->nmc.value = 1.0;
      return EX_BOOLEAN;
    }
}

static exprtype
VALUE_func (unused function * f, unused int x, any_node ** n)
{
  variable *v = parse_variable ();

  if (!v)
    return EX_ERROR;
  if (v->type == NUMERIC)
    {
      *n = xmalloc (sizeof (val_node));
      (*n)->val.type = OP_NUM_VAL;
      (*n)->val.v = v;
      return EX_NUMERIC;
    }
  else
    {
      *n = xmalloc (sizeof (var_node));
      (*n)->var.type = OP_STR_VAR;
      (*n)->var.v = v;
      return EX_STRING;
    }
}

static exprtype
LAG_func (unused function * f, unused int x, any_node ** n)
{
  variable *v = parse_variable ();
  int nlag = 1;

  if (!v)
    return EX_ERROR;
  if (match_tok (','))
    {
      if (token != NUM || tokint == NOT_LONG || tokint <= 0 || tokint > 1000)
	return msg (SE, _("Argument 2 to LAG must be a small positive "
		    "constant integer."));
      nlag = tokint;
      get_token ();
    }
  n_lag = max (nlag, n_lag);
  *n = xmalloc (sizeof (lag_node));
  (*n)->lag.type = (v->type == NUMERIC ? OP_NUM_LAG : OP_STR_LAG);
  (*n)->lag.v = v;
  (*n)->lag.lag = nlag;
  return (v->type == NUMERIC ? EX_NUMERIC : EX_STRING);
}

/* This screwball function parses n-ary operators:
   1. NMISS, NVALID, SUM, MEAN, MIN, MAX: any number of (numeric) arguments.
   2. SD, VARIANCE, CFVAR: at least two (numeric) arguments.
   3. RANGE: An odd number of arguments, but at least three.
   All arguments must be the same type.
   4. ANY: At least two arguments.  All arguments must be the same type.
 */
static exprtype
nary_num_func (function * f, int min_args, any_node ** n)
{
  /* Argument number of current argument (used for error messages). */
  int argn = 1;

  /* Number of arguments. */
  int nargs;

  /* Number of arguments allocated. */
  int m = 16;

  /* Type of arguments. */
  int type = (f->t == OP_ANY || f->t == OP_RANGE) ? -1 : NUMERIC;

  *n = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[15]));
  (*n)->nnt.type = f->t;
  (*n)->nnt.n = 0;
  while (1)
    {
      /* Special case: vara TO varb. */

      /* FIXME: Is this condition failsafe?  Can we _ever_ have two
         juxtaposed identifiers otherwise?  */
      if (token == ID && is_varname (tokstr) && toupper (lookahead ()) == 'T')
	{
	  variable **v;
	  int nv;
	  int j;
	  int opts = PV_SINGLE;

	  if (type == NUMERIC)
	    opts |= PV_NUMERIC;
	  else if (type == ALPHA)
	    opts |= PV_STRING;
	  if (!parse_variables (NULL, &v, &nv, opts))
	    goto fail;
	  if (nv + (*n)->nnt.n >= m)
	    {
	      m += nv + 16;
	      *n = xrealloc (*n, (sizeof (nonterm_node)
				  + (m - 1) * sizeof (any_node *)));
	    }
	  if (type == -1)
	    {
	      type = v[0]->type;
	      for (j = 1; j < nv; j++)
		if (type != v[j]->type)
		  {
		    msg (SE, _("Type mismatch in argument %d of %s, which was "
			 "expected to be of %s type.  It was actually "
			 "of %s type. "), argn, f->s, typename (type),
			 typename (v[j]->type));
		    free (v);
		    goto fail;
		  }
	    }
	  for (j = 0; j < nv; j++)
	    {
	      any_node **c = &(*n)->nnt.arg[(*n)->nnt.n++];
	      *c = xmalloc (sizeof (var_node));
	      (*c)->var.type = (type == NUMERIC ? OP_NUM_VAR : OP_STR_VAR);
	      (*c)->var.v = v[j];
	    }
	}
      else
	{
	  any_node *c;
	  int t = parse_or (&c);

	  if (t == EX_ERROR)
	    goto fail;
	  if (t == EX_BOOLEAN)
	    {
	      free_node (c);
	      msg (SE, _("%s cannot take Boolean operands."), f->s);
	      goto fail;
	    }
	  if (type == -1)
	    {
	      if (t == EX_NUMERIC)
		type = NUMERIC;
	      else if (t == EX_STRING)
		type = ALPHA;
	    }
	  else if ((t == EX_NUMERIC) ^ (type == NUMERIC))
	    {
	      free_node (c);
	      msg (SE, _("Type mismatch in argument %d of %s, which was "
		   "expected to be of %s type.  It was actually "
	      "of %s type. "), argn, f->s, typename (type), exprtypename (t));
	      goto fail;
	    }
	  printf ("t=%s type=%s\n", t == EX_NUMERIC ? "EX_NUMERIC" : "EX_STRING",
		  type == NUMERIC ? "NUMERIC" : "ALPHA");
	  if ((*n)->nnt.n + 1 >= m)
	    {
	      m += 16;
	      *n = xrealloc (*n, (sizeof (nonterm_node)
				  + (m - 1) * sizeof (any_node *)));
	    }
	  (*n)->nnt.arg[(*n)->nnt.n++] = c;
	}

      if (token == ')')
	break;
      if (!match_tok (','))
	{
	  syntax_error (_("in function call"));
	  goto fail;
	}

      argn++;
    }
  *n = xrealloc (*n, (sizeof (nonterm_node)
		      + ((*n)->nnt.n) * sizeof (any_node *)));

  nargs = (*n)->nnt.n;
  if (f->t == OP_RANGE)
    {
      if (nargs < 3 || (nargs & 1) == 0)
	return msg (SE, _("RANGE requires an odd number of arguments, but "
		    "at least three."));
    }
  else if (f->t == OP_SD || f->t == OP_VARIANCE
	   || f->t == OP_CFVAR || f->t == OP_ANY)
    {
      if (nargs < 2)
	return msg (SE, _("%s requires at least two arguments."), f->s);
    }

  if (f->t == OP_CFVAR || f->t == OP_SD || f->t == OP_VARIANCE)
    min_args = max (min_args, 2);
  else
    min_args = max (min_args, 1);

  /* Yes, this is admittedly a terrible crock, but it works. */
  (*n)->nnt.arg[(*n)->nnt.n] = (any_node *) min_args;

  if (min_args > nargs)
    return msg (SE, _("%s.%d requires at least %d arguments."),
		f->s, min_args, min_args);

  if (f->t == OP_ANY || f->t == OP_RANGE)
    {
      if (type == STRING)
	f->t++;
      return EX_BOOLEAN;
    }
  else
    return EX_NUMERIC;

fail:
  free_node (*n);
  return EX_ERROR;
}

static exprtype
CONCAT_func (unused function * f, unused int x, any_node ** n)
{
  int m = 0;

  int type;

  *n = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[15]));
  (*n)->nnt.type = OP_CONCAT;
  (*n)->nnt.n = 0;
  while (1)
    {
      if ((*n)->nnt.n >= m)
	{
	  m += 16;
	  *n = xrealloc (*n, sizeof (nonterm_node) + (m - 1) * sizeof (any_node *));
	}
      type = parse_or (&(*n)->nnt.arg[(*n)->nnt.n]);
      if (type == EX_ERROR)
	goto fail;
      if (type != EX_STRING)
	{
	  msg (SE, _("Argument %d to CONCAT is type %s.  All arguments "
	       "to CONCAT must be strings."),
	       (*n)->nnt.n + 1, exprtypename (type));
	  goto fail;
	}
      (*n)->nnt.n++;

      if (!match_tok (','))
	break;
    }
  *n = xrealloc (*n, (sizeof (nonterm_node)
		      + ((*n)->nnt.n - 1) * sizeof (any_node *)));
  return EX_STRING;

fail:
  free (*n);
  return EX_ERROR;
}

/* Parses a string function according to f->desc.  f->desc[0] is the
   return type of the function.  Succeeding characters represent
   successive args.  Optional args are separated from the required
   args by a slash (`/').  Codes are `n', numeric arg; `s', string
   arg; and `f', format spec (this must be the last arg).  If the
   optional args are included, the type becomes f->t+1. */
static exprtype
generic_str_func (function *f, unused int x, any_node ** n)
{
  int max_args = 0;
  int type;
  const char *cp;

  /* Count max number of arguments. */
  cp = &f->desc[1];
  while (*cp)
    {
      if (*cp == 'n' || *cp == 's')
	max_args++;
      else if (*cp == 'f')
	max_args += 3;
      cp++;
    }
  cp = &f->desc[1];

  *n = xmalloc (sizeof (nonterm_node) + (max_args - 1) * sizeof (any_node *));
  (*n)->nnt.type = f->t;
  (*n)->nnt.n = 0;
  while (1)
    {
      if (*cp == 'n' || *cp == 's')
	{
	  int t = *cp == 'n' ? EX_NUMERIC : EX_STRING;
	  type = parse_or (&(*n)->nnt.arg[(*n)->nnt.n]);

	  if (type == EX_ERROR)
	    goto fail;
	  if (type != t)
	    {
	      msg (SE, _("Argument %d to %s was expected to be of %s type.  "
		   "It was actually of type %s."), (*n)->nnt.n + 1,
		   f->s, *cp == 'n' ? _("numeric") : _("string"),
		   exprtypename (type));
	      goto fail;
	    }
	  (*n)->nnt.n++;
	}
      else if (*cp == 'f')
	{
	  /* This is always the very last argument.  Also, this code
	     is a crock.  However, it works. */
	  fmt_spec fmt;

	  if (!parse_format_specifier (&fmt, 0))
	    goto fail;
	  if (formats[fmt.type].cat & FCAT_STRING)
	    {
	      msg (SE, _("%s is not a numeric format."), fmt_to_string (&fmt));
	      goto fail;
	    }
	  (*n)->nnt.arg[(*n)->nnt.n + 0] = (any_node *) fmt.type;
	  (*n)->nnt.arg[(*n)->nnt.n + 1] = (any_node *) fmt.w;
	  (*n)->nnt.arg[(*n)->nnt.n + 2] = (any_node *) fmt.d;
	  break;
	}
      else
	assert (0);

      if (*++cp == 0)
	break;
      if (*cp == '/')
	{
	  cp++;
	  if (match_tok (','))
	    {
	      (*n)->nnt.type++;
	      continue;
	    }
	  else
	    break;
	}
      else if (!match_tok (','))
	{
	  msg (SE, _("Too few arguments to function %s."), f->s);
	  goto fail;
	}
    }

  return f->desc[0] == 'n' ? EX_NUMERIC : EX_STRING;

fail:
  free (*n);
  return EX_ERROR;
}

/* General function parsing. */

static int
get_num_args (function * f, int num_args, any_node ** n)
{
  exprtype t;
  int i;

  *n = xmalloc (sizeof (nonterm_node) + (num_args - 1) * sizeof (any_node *));
  (*n)->nnt.type = f->t;
  (*n)->nnt.n = 0;
  for (i = 0;;)
    {
      t = parse_or (&(*n)->nnt.arg[i]);
      if (t == EX_ERROR)
	goto fail;
      (*n)->nnt.n++;
      if (t != EX_NUMERIC)
	{
	  msg (SE, _("Type mismatch in argument %d of %s, which was expected "
	       "to be numeric.  It was actually type %s."),
	       i + 1, f->s, exprtypename (t));
	  goto fail;
	}
      if (++i >= num_args)
	return 1;
      if (!match_tok (','))
	{
	  msg (SE, _("Missing comma following argument %d of %s."), i + 1, f->s);
	  goto fail;
	}
    }

fail:
  free_node (*n);
  return 0;
}

static int
func_cmp (const void *a, const void *b, unused void *foo)
{
  return strcmp (*(char **) a, *(char **) b);
}

static void
init_functab (void)
{
  static function functab[] =
  {
    {"ABS", OP_ABS, unary_func, NULL},
    {"ACOS", OP_ARCOS, unary_func, NULL},
    {"ARCOS", OP_ARCOS, unary_func, NULL},
    {"ARSIN", OP_ARSIN, unary_func, NULL},
    {"ARTAN", OP_ARTAN, unary_func, NULL},
    {"ASIN", OP_ARSIN, unary_func, NULL},
    {"ATAN", OP_ARTAN, unary_func, NULL},
    {"COS", OP_COS, unary_func, NULL},
    {"EXP", OP_EXP, unary_func, NULL},
    {"LG10", OP_LG10, unary_func, NULL},
    {"LN", OP_LN, unary_func, NULL},
    {"MOD10", OP_MOD10, unary_func, NULL},
    {"NORMAL", OP_NORMAL, unary_func, NULL},
    {"RND", OP_RND, unary_func, NULL},
    {"SIN", OP_SIN, unary_func, NULL},
    {"SQRT", OP_SQRT, unary_func, NULL},
    {"TAN", OP_TAN, unary_func, NULL},
    {"TRUNC", OP_TRUNC, unary_func, NULL},
    {"UNIFORM", OP_UNIFORM, unary_func, NULL},

    {"TIME.DAYS", OP_TIME_DAYS, unary_func, NULL},
    {"TIME.HMS", OP_TIME_HMS, ternary_func, NULL},

    {"CTIME.DAYS", OP_CTIME_DAYS, unary_func, NULL},
    {"CTIME.HOURS", OP_CTIME_HOURS, unary_func, NULL},
    {"CTIME.MINUTES", OP_CTIME_MINUTES, unary_func, NULL},
    {"CTIME.SECONDS", OP_CTIME_SECONDS, unary_func, NULL},

    {"DATE.DMY", OP_DATE_DMY, ternary_func, NULL},
    {"DATE.MDY", OP_DATE_MDY, ternary_func, NULL},
    {"DATE.MOYR", OP_DATE_MOYR, binary_func, NULL},
    {"DATE.QYR", OP_DATE_QYR, binary_func, NULL},
    {"DATE.WKYR", OP_DATE_WKYR, binary_func, NULL},
    {"DATE.YRDAY", OP_DATE_YRDAY, binary_func, NULL},

    {"XDATE.DATE", OP_XDATE_DATE, unary_func, NULL},
    {"XDATE.HOUR", OP_XDATE_HOUR, unary_func, NULL},
    {"XDATE.JDAY", OP_XDATE_JDAY, unary_func, NULL},
    {"XDATE.MDAY", OP_XDATE_MDAY, unary_func, NULL},
    {"XDATE.MINUTE", OP_XDATE_MINUTE, unary_func, NULL},
    {"XDATE.MONTH", OP_XDATE_MONTH, unary_func, NULL},
    {"XDATE.QUARTER", OP_XDATE_QUARTER, unary_func, NULL},
    {"XDATE.SECOND", OP_XDATE_SECOND, unary_func, NULL},
    {"XDATE.TDAY", OP_XDATE_TDAY, unary_func, NULL},
    {"XDATE.TIME", OP_XDATE_TIME, unary_func, NULL},
    {"XDATE.WEEK", OP_XDATE_WEEK, unary_func, NULL},
    {"XDATE.WKDAY", OP_XDATE_WKDAY, unary_func, NULL},
    {"XDATE.YEAR", OP_XDATE_YEAR, unary_func, NULL},

    {"MISSING", OP_SYSMIS, MISSING_func, NULL},
    {"MOD", OP_MOD, binary_func, NULL},
    {"SYSMIS", OP_SYSMIS, SYSMIS_func, NULL},
    {"VALUE", OP_NUM_VAL, VALUE_func, NULL},
    {"LAG", OP_NUM_LAG, LAG_func, NULL},
    {"YRMODA", OP_YRMODA, ternary_func, NULL},

    {"ANY", OP_ANY, nary_num_func, NULL},
    {"CFVAR", OP_CFVAR, nary_num_func, NULL},
    {"MAX", OP_MAX, nary_num_func, NULL},
    {"MEAN", OP_MEAN, nary_num_func, NULL},
    {"MIN", OP_MIN, nary_num_func, NULL},
    {"NMISS", OP_NMISS, nary_num_func, NULL},
    {"NVALID", OP_NVALID, nary_num_func, NULL},
    {"RANGE", OP_RANGE, nary_num_func, NULL},
    {"SD", OP_SD, nary_num_func, NULL},
    {"SUM", OP_SUM, nary_num_func, NULL},
    {"VARIANCE", OP_VARIANCE, nary_num_func, NULL},

    {"CONCAT", OP_CONCAT, CONCAT_func, NULL},
    {"INDEX", OP_INDEX, generic_str_func, "nss/n"},
    {"RINDEX", OP_RINDEX, generic_str_func, "nss/n"},
    {"LENGTH", OP_LENGTH, generic_str_func, "ns"},
    {"LOWER", OP_LOWER, generic_str_func, "ss"},
    {"UPCAS", OP_UPPER, generic_str_func, "ss"},
    {"LPAD", OP_LPAD, generic_str_func, "ssn/s"},
    {"RPAD", OP_RPAD, generic_str_func, "ssn/s"},
    {"LTRIM", OP_LTRIM, generic_str_func, "ss/s"},
    {"RTRIM", OP_RTRIM, generic_str_func, "ss/s"},
    {"NUMBER", OP_NUMBER, generic_str_func, "ns/f"},
    {"STRING", OP_STRING, generic_str_func, "snf"},
    {"SUBSTR", OP_SUBSTR, generic_str_func, "ssn/n"},

    {NULL, 0, NULL, NULL},
  };
  function *f;
  static int inited;

  if (inited)
    return;
  inited = 1;
  functree = avl_create (NULL, func_cmp, NULL);
  for (f = functab; f->s; f++)
    avl_force_insert (functree, f);
}

static exprtype
parse_function (any_node ** n)
{
  function *fp, f;
  char fname[32], *cp;
  exprtype t;
  int min_args;
  vector *v;

  /* Check for a vector with this name. */
  v = find_vector (tokstr);
  if (v)
    {
      get_token ();
      assert (token == '(');
      get_token ();

      *n = xmalloc (sizeof (nonterm_node) + sizeof (any_node *[2]));
      (*n)->nnt.type = (v->v[0]->type == NUMERIC
			? OP_VEC_ELEM_NUM : OP_VEC_ELEM_STR);
      (*n)->nnt.n = 1;

      t = parse_or (&(*n)->nnt.arg[0]);
      if (t == EX_ERROR)
	goto fail;
      if (t != EX_NUMERIC)
	{
	  msg (SE, _("The index value after a vector name must be numeric."));
	  goto fail;
	}

      if (!match_tok (')'))
	{
	  msg (SE, _("`)' expected after a vector index value."));
	  goto fail;
	}
      ((*n)->nnt.arg[1]) = (any_node *) v->index;

      return v->v[0]->type == NUMERIC ? EX_NUMERIC : EX_STRING;
    }

  strmaxcpy (fname, toklongstr, 32);
  cp = strrchr (fname, '.');
  if (cp && isdigit ((unsigned char) cp[1]))
    {
      min_args = atoi (&cp[1]);
      *cp = 0;
    }
  else
    min_args = 0;

  get_token ();
  force_match ('(');

  f.s = fname;
  fp = avl_find (functree, &f);
  if (!fp)
    return msg (SE, _("There is no function named %s."), fname);
  if (min_args && fp->func != nary_num_func)
    return msg (SE, _("Function %s may not be given a minimum number of "
		"arguments."), fname);
  t = fp->func (fp, min_args, n);
  if (t == EX_ERROR)
    return EX_ERROR;
  if (!match_tok (')'))
    {
      msg (SE, _("Missing right parenthesis after %s function."), fname);
      goto fail;
    }

  return t;

fail:
  free (*n);
  return EX_ERROR;
}

#if GLOBAL_DEBUGGING
#define op(a,b,c,d) {a,b,c,d}
#else
#define op(a,b,c,d) {b,c,d}
#endif

#define varies 0

op_desc ops[OP_SENTINEL + 1] =
{
  op ("!?ERROR?!", 000, 0, 0),

  op ("plus", 001, varies, 1),
  op ("mul", 011, varies, 1),
  op ("pow", 010, -1, 0),
  op ("and", 010, -1, 0),
  op ("or", 010, -1, 0),
  op ("not", 000, 0, 0),
  op ("eq", 000, -1, 0),
  op ("ge", 000, -1, 0),
  op ("gt", 000, -1, 0),
  op ("le", 000, -1, 0),
  op ("lt", 000, -1, 0),
  op ("ne", 000, -1, 0),

  op ("string-eq", 000, -1, 0),
  op ("string-ge", 000, -1, 0),
  op ("string-gt", 000, -1, 0),
  op ("string-le", 000, -1, 0),
  op ("string-lt", 000, -1, 0),
  op ("string-ne", 000, -1, 0),

  op ("neg", 000, 0, 0),
  op ("abs", 000, 0, 0),
  op ("arcos", 000, 0, 0),
  op ("arsin", 000, 0, 0),
  op ("artan", 000, 0, 0),
  op ("cos", 000, 0, 0),
  op ("exp", 000, 0, 0),
  op ("lg10", 000, 0, 0),
  op ("ln", 000, 0, 0),
  op ("mod10", 000, 0, 0),
  op ("rnd", 000, 0, 0),
  op ("sin", 000, 0, 0),
  op ("sqrt", 000, 0, 0),
  op ("tan", 000, 0, 0),
  op ("trunc", 000, 0, 0),

  op ("any", 011, varies, 1),
  op ("any-string", 001, varies, 1),
  op ("cfvar", 013, varies, 2),
  op ("max", 013, varies, 2),
  op ("mean", 013, varies, 2),
  op ("min", 013, varies, 2),
  op ("nmiss", 011, varies, 1),
  op ("nvalid", 011, varies, 1),
  op ("range", 011, varies, 1),
  op ("range-string", 001, varies, 1),
  op ("sd", 013, varies, 2),
  op ("sum", 013, varies, 2),
  op ("variance", 013, varies, 2),

  op ("time_hms", 000, -2, 0),
  op ("ctime_days?!", 000, 0, 0),
  op ("ctime_hours?!", 000, 0, 0),
  op ("ctime_minutes?!", 000, 0, 0),
  op ("ctime_seconds?!", 000, 0, 0),
  op ("time_days?!", 000, 0, 0),

  op ("date_dmy", 000, -2, 0),
  op ("date_mdy", 000, -2, 0),
  op ("date_moyr", 000, -1, 0),
  op ("date_qyr", 000, -1, 0),
  op ("date_wkyr", 000, -1, 0),
  op ("date_yrday", 000, -1, 0),
  op ("yrmoda", 000, -2, 0),

  op ("xdate_date", 000, 0, 0),
  op ("xdate_hour", 000, 0, 0),
  op ("xdate_jday", 000, 0, 0),
  op ("xdate_mday", 000, 0, 0),
  op ("xdate_minute", 000, 0, 0),
  op ("xdate_month", 000, 0, 0),
  op ("xdate_quarter", 000, 0, 0),
  op ("xdate_second", 000, 0, 0),
  op ("xdate_tday", 000, 0, 0),
  op ("xdate_time", 000, 0, 0),
  op ("xdate_week", 000, 0, 0),
  op ("xdate_wkday", 000, 0, 0),
  op ("xdate_year", 000, 0, 0),

  op ("concat", 001, varies, 1),
  op ("index-2", 000, -1, 0),
  op ("index-3", 000, -2, 0),
  op ("rindex-2", 000, -1, 0),
  op ("rindex-3", 000, -2, 0),
  op ("length", 000, 0, 0),
  op ("lower", 000, 0, 0),
  op ("upcas", 000, 0, 0),
  op ("lpad-2", 010, -1, 0),
  op ("lpad-3", 010, -2, 0),
  op ("rpad-2", 010, -1, 0),
  op ("rpad-3", 010, -2, 0),
  op ("ltrim-1", 000, 0, 0),
  op ("ltrim-2", 000, -1, 0),
  op ("rtrim-1", 000, 0, 0),
  op ("rtrim-2", 000, -1, 0),
  op ("number-1", 010, 0, 0),
  op ("number-2", 014, 0, 3),
  op ("string", 004, 0, 3),
  op ("substr-2", 010, -1, 0),
  op ("substr-3", 010, -2, 0),

  op ("inv", 000, 0, 0),
  op ("square", 000, 0, 0),
  op ("num-to-Bool", 000, 0, 0),

  op ("mod", 010, -1, 0),
  op ("normal", 000, 0, 0),
  op ("uniform", 000, 0, 0),
  op ("sysmis", 010, 0, 0),
  op ("vec-elem-num", 002, 0, 1),
  op ("vec-elem-str", 002, 0, 1),

  op ("!?TERMINAL?!", 000, 0, 0),
  op ("num-con", 000, +1, 0),
  op ("str-con", 000, +1, 0),
  op ("num-var", 000, +1, 0),
  op ("str-var", 000, +1, 0),
  op ("num-lag", 000, +1, 1),
  op ("str-lag", 000, +1, 1),
  op ("num-sys", 000, +1, 1),
  op ("num-val", 000, +1, 1),
  op ("str-mis", 000, +1, 1),
  op ("$casenum", 000, +1, 0),
  op ("!?SENTINEL?!", 000, 0, 0),
};

#undef op
#undef varies

/* Debug output. */

#if DEBUGGING
static void
print_type (any_node * n)
{
  char *s = NULL;
  int len;

  s = ops[n->type].name;
  len = strlen (s);
  if (ops[n->type].flags & OP_MIN_ARGS)
    printf ("%s.%d\n", s, (int) n->nnt.arg[n->nnt.n]);
  else if (ops[n->type].flags & OP_FMT_SPEC)
    {
      fmt_spec f;
      f.type = (int) n->nnt.arg[n->nnt.n + 0];
      f.w = (int) n->nnt.arg[n->nnt.n + 1];
      f.d = (int) n->nnt.arg[n->nnt.n + 2];
      printf ("%s(%s)\n", s, fmt_to_string (&f));
    }
  else
    printf ("%s\n", s);
}

static void
debug_print_tree (any_node * n, int level)
{
  int i;
  for (i = 0; i < level; i++)
    printf ("  ");
  if (n->type < OP_TERMINAL)
    {
      print_type (n);
      for (i = 0; i < n->nnt.n; i++)
	debug_print_tree (n->nnt.arg[i], level + 1);
    }
  else
    {
      switch (n->type)
	{
	case OP_TERMINAL:
	  printf (_("!!TERMINAL!!"));
	  break;
	case OP_NUM_CON:
	  if (n->nmc.value == SYSMIS)
	    printf ("SYSMIS");
	  else
	    printf ("%f", n->nmc.value);
	  break;
	case OP_STR_CON:
	  printf ("\"%.*s\"", n->stc.len, n->stc.s);
	  break;
	case OP_NUM_VAR:
	case OP_STR_VAR:
	  printf ("%s", n->var.v->name);
	  break;
	case OP_NUM_LAG:
	case OP_STR_LAG:
	  printf ("LAG(%s,%d)", n->lag.v->name, n->lag.lag);
	  break;
	case OP_NUM_SYS:
	  printf ("SYSMIS(%s)", n->sys.v->name);
	  break;
	case OP_NUM_VAL:
	  printf ("VALUE(%s)", n->val.v->name);
	  break;
	case OP_SENTINEL:
	  printf (_("!!SENTINEL!!"));
	  break;
	default:
	  printf (_("!!ERROR%d!!"), n->type);
	  assert (0);
	}
      printf ("\n");
    }
}
#endif /* DEBUGGING */

#if GLOBAL_DEBUGGING
static void
debug_print_postfix (expression * e)
{
  operator *o;
  double *dbl = e->dbl;
  unsigned char *str = e->str;
  variable **v = e->vars;
  int t;

  debug_printf ((_("postfix:")));
  for (o = e->op; *o != OP_SENTINEL;)
    {
      t = *o++;
      if (t < OP_TERMINAL)
	{
	  debug_printf ((" %s", ops[t].name));

	  if (ops[t].flags & OP_VAR_ARGS)
	    {
	      debug_printf (("(%d)", *o));
	      o++;
	    }
	  if (ops[t].flags & OP_MIN_ARGS)
	    {
	      debug_printf ((".%d", *o));
	      o++;
	    }
	  if (ops[t].flags & OP_FMT_SPEC)
	    {
	      fmt_spec f;
	      f.type = (int) *o++;
	      f.w = (int) *o++;
	      f.d = (int) *o++;
	      debug_printf (("(%s)", fmt_to_string (&f)));
	    }
	}
      else if (t == OP_NUM_CON)
	{
	  if (*dbl == SYSMIS)
	    debug_printf ((" SYSMIS"));
	  else
	    debug_printf ((" %f", *dbl));
	  dbl++;
	}
      else if (t == OP_STR_CON)
	{
	  debug_printf ((" \"%.*s\"", *str, &str[1]));
	  str += str[0] + 1;
	}
      else if (t == OP_NUM_VAR || t == OP_STR_VAR)
	{
	  debug_printf ((" %s", (*v)->name));
	  v++;
	}
      else if (t == OP_NUM_SYS)
	{
	  debug_printf ((" SYSMIS(#%d)", *o));
	  o++;
	}
      else if (t == OP_NUM_VAL)
	{
	  debug_printf ((" VALUE(#%d)", *o));
	  o++;
	}
      else if (t == OP_NUM_LAG || t == OP_STR_LAG)
	{
	  debug_printf ((" LAG(%s,%d)", (*v)->name, *o));
	  o++;
	  v++;
	}
      else
	{
	  printf ("debug_print_postfix(): %d\n", t);
	  assert (0);
	}
    }
  debug_putc ('\n', stdout);
}
#endif /* GLOBAL_DEBUGGING */
