;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <1999-05-28 18:19:17 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'elmo2)
;; from x-face.el
(unless (and (fboundp 'defgroup)
             (fboundp 'defcustom))
  (require 'backquote)
  (defmacro defgroup (&rest args))
  (defmacro defcustom (symbol value &optional doc &rest args)
    (let ((doc (concat "*" (or doc ""))))
      (` (defvar (, symbol) (, value) (, doc)))))
  )

(require 'wl-vars)
(require 'wl-util)

(if wl-on-xemacs
    (require 'wl-xmas)
  (if wl-on-nemacs
      (require 'wl-nemacs)
    (require 'wl-mule)))

(require 'wl-folder)
(require 'wl-summary)
(require 'wl-thread)
(require 'wl-address)

(wl-draft-mode-setup)
(require 'wl-draft)
(wl-draft-key-setup)

(require 'wl-demo)
(require 'wl-highlight)

(eval-when-compile 
  (unless wl-on-nemacs
    (require 'wl-fldmgr))
  (if wl-use-semi
      (progn
	(require 'wl-mime)
	;; dummy.
	(defun tm-wl-setup ()))
    (require 'tm-wl)
    (defun wl-mime-setup ())
    ))

(defun wl-toggle-plugged (&optional arg)
  (interactive)
  (if wl-plugged
      (elmo-quit)) ; Disconnect current connection.
  (cond
   ((eq arg 'on)
    (setq wl-plugged t))
   ((eq arg 'off)
    (setq wl-plugged nil))
   (t (setq wl-plugged (null wl-plugged))))
  (setq elmo-plugged wl-plugged)
  (save-excursion
    (mapcar 
     (function 
      (lambda (x)
	(set-buffer x)
	(wl-summary-msgdb-save)
	;; msgdb is saved, but cache is not saved yet.
	(wl-summary-set-message-modified)))
     (wl-collect-summary)))
  (if wl-plugged
      (progn
	;; flush queue!!
	(setq wl-plug-state-indicator wl-plug-state-indicator-on)
	(elmo-dop-queue-flush)
	(if (and wl-draft-enable-queuing
		 wl-auto-flush-queue)
	    (wl-draft-queue-flush))
	(if (eq major-mode 'wl-summary-mode)
	    (wl-summary-flush-pending-append-operations))
	(run-hooks wl-plugged-hook))
    (setq wl-plug-state-indicator wl-plug-state-indicator-off)
    (run-hooks wl-unplugged-hook)
    )
  (force-mode-line-update t)
  )

(defun wl-collect-summary ()
  (let (result)
    (mapcar
     (function (lambda (x)
		 (if (and (string-match "^Summary"
					(buffer-name x))
			  (save-excursion
			    (set-buffer x)
			    (equal major-mode 'wl-summary-mode)))
		     (setq result (nconc result (list x))))))
     (buffer-list))
    result))

(defun wl-save-status ()
  (message "Saving summary and folder status...")
  (let (summary-buf)
    (save-excursion
      (let ((summaries (wl-collect-summary)))
	(mapcar 
	 (function
	  (lambda (x)
	    (set-buffer x)
	    (wl-summary-save-status)
	    (kill-buffer x)
	    ))
	 summaries))))
  (if (and wl-folder-info-save
	   wl-folder-info-alist-modified)
      (wl-folder-info-save))
  (message "Saving summary and folder status...done."))

(defun wl-exit ()
  (interactive)
  (when (or (not wl-interactive-exit)
	    (y-or-n-p "Quit Wanderlust?"))
    (elmo-quit)
    (run-hooks 'wl-exit-hook)
    (if (get-buffer wl-folder-buffer-name)
 	(kill-buffer (get-buffer wl-folder-buffer-name)))
    (wl-save-status)
    (if (get-buffer wl-message-buf-name)
	(kill-buffer (get-buffer wl-message-buf-name)))
    (if (get-buffer wl-original-buf-name)
	(kill-buffer (get-buffer wl-original-buf-name)))
    (if (fboundp 'mmelmo-cleanup-entity-buffers)
	 (mmelmo-cleanup-entity-buffers))
    (wl-folder-cleanup-variables)
    (elmo-cleanup-variables)
    ;(delete-other-windows)
    (setq wl-init nil)
    t
    )
  (message "");; empty minibuffer.
  )

(defun wl-init (&optional arg)
  (when (not wl-init)
    (load wl-init-file t nil t)
    (message "Initializing ...")
    ;; Plug status.
    (setq elmo-plugged wl-plugged)
    (setq wl-plug-state-indicator 
	  (if wl-plugged
	      wl-plug-state-indicator-on
	    wl-plug-state-indicator-off))
    (let (succeed demo-buf)
      (if wl-demo
	  (setq demo-buf (wl-demo)))
      (if (not wl-on-nemacs)
	  (add-hook 'kill-emacs-hook 'wl-save-status t))
      (unwind-protect
	  (progn
	    (wl-address-setup)    
	    (wl-draft-setup)
	    (if wl-use-semi
		(progn
		  (require 'wl-mime)
		  (setq elmo-use-semi t)
		  (wl-mime-setup))
	      (require 'tm-wl)
	      (setq elmo-use-semi nil)
	      (tm-wl-setup))
	    ;; for speed up
	    (if elmo-use-semi
		(defalias 'elmo-eword-decode-string 'eword-decode-string)
	      (defalias 'elmo-eword-decode-string 'mime-eword/decode-string))
	    (fset 'wl-summary-from-func-internal 
		  (symbol-value 'wl-summary-from-func))
	    (fset 'wl-summary-subject-func-internal 
		  (symbol-value 'wl-summary-subject-func))
	    (fset 'wl-summary-subject-filter-func-internal 
		  (symbol-value 'wl-summary-subject-filter-func))
	    (setq elmo-no-from wl-summary-no-from-message)
	    (setq elmo-no-subject wl-summary-no-subject-message)
	    (setq succeed t)
	    (progn
	      (message "Checking environment...")
	      (wl-check-environment arg)
	      (message "Checking environment...done."))
	    )
	(if demo-buf
	    (kill-buffer demo-buf))
	(if succeed
	    (setq wl-init t))
	(run-hooks 'wl-init-hook)))))

(defun wl-check-environment (no-check-folder)
  (if (null wl-from)
      (error "Please set wl-from variable."))
  (when (not no-check-folder)
    (if (not (eq (elmo-folder-get-type wl-draft-folder) 'localdir))
	(error "%s is not allowed for draft folder." wl-draft-folder))
    (unless (elmo-folder-exists-p wl-draft-folder)
      (if (y-or-n-p 
	   (format "Draft Folder %s does not exit, create it?" 
		   wl-draft-folder))
	  (elmo-create-folder wl-draft-folder)
	(error "Draft Folder is not created.")))
    (if (and wl-draft-enable-queuing
	     (not (elmo-folder-exists-p wl-queue-folder)))
	(if (y-or-n-p 
	     (format "Queue Folder %s does not exit, create it?" 
		     wl-queue-folder))
	    (elmo-create-folder wl-queue-folder)
	  (error "Queue Folder is not created.")))
    (unless (elmo-folder-exists-p wl-trash-folder)
      (if (y-or-n-p 
	   (format "Trash Folder %s does not exit, create it?" 
		   wl-trash-folder))
	  (elmo-create-folder wl-trash-folder)
	(error "Trash Folder is not created."))))
  (unless (file-exists-p wl-tmp-dir)
    (if (y-or-n-p 
	 (format "Temp directory %s does not exit, create it now?" 
		 wl-tmp-dir))
	(make-directory wl-tmp-dir)
      (error "Temp directory is not created."))))

;;;###autoload
(defun wl (&optional arg)
  "Start Wanderlust -- Yet Another Message Interface On Emacsen.
If prefix argument is specified, folder checkings are skipped."
  (interactive "P")
  (unwind-protect
      (wl-init arg)
    (wl-folder arg))
  (run-hooks 'wl-hook)
  )

;; Define some autoload functions WL might use.
(eval-and-compile
  ;; This little mapcar goes through the list below and marks the
  ;; symbols in question as autoloaded functions.
  (mapcar
   (function
    (lambda (package)
      (let ((interactive (nth 1 (memq ':interactive package))))
	(mapcar
	 (function
	  (lambda (function)
	    (let (keymap)
	      (when (consp function)
		(setq keymap (car (memq 'keymap function)))
		(setq function (car function)))
	      (autoload function (car package) nil interactive keymap))))
	 (if (eq (nth 1 package) ':interactive)
	     (cdddr package)
	   (cdr package))))))
   '(("wl-fldmgr" :interactive t
      wl-fldmgr-save-folders
      wl-fldmgr-make-multi wl-fldmgr-make-group wl-fldmgr-make-access-group
      wl-fldmgr-make-filter wl-fldmgr-set-petname
      wl-fldmgr-add wl-fldmgr-copy wl-fldmgr-cut wl-fldmgr-yank
      wl-fldmgr-copy-region wl-fldmgr-cut-region wl-fldmgr-sort
      wl-fldmgr-access-display-normal wl-fldmgr-access-display-all
      wl-fldmgr-clear-cut-entity-list wl-fldmgr-reconst-entity-hashtb
      wl-fldmgr-unsubscribe wl-fldmgr-unsubscribe-region)
     ("wl-fldmgr"
      (wl-fldmgr-mode-map keymap)
      wl-fldmgr-add-entity-hashtb)
     ("wl-expire" :interactive t
      wl-summary-expire wl-folder-expire-current-entity
      wl-summary-archive wl-folder-archive-current-entity))))

;; for backward compatibility
(defalias 'wl-summary-from-func-petname 'wl-summary-default-from)
 
(provide 'wl)
