/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysmouse.c,v 1.6 1999/07/15 18:42:02 slouken Exp $";
#endif

#include <stdlib.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "SDL_error.h"
#include "SDL_mouse.h"
#include "SDL_sysmouse_c.h"
#include "SDL_events_c.h"
#include "SDL_lowvideo.h"


/* The implementation dependent data for the window manager cursor */
struct WMcursor {
	Cursor x_cursor;
};


void SDL_FreeWMCursor(SDL_Surface *screen, WMcursor *cursor)
{
	if ( SDL_Display != NULL ) {
		SDL_Lock_EventThread();
		XFreeCursor(SDL_Display, cursor->x_cursor);
		XSync(SDL_Display, False);
		SDL_Unlock_EventThread();
	}
	free(cursor);
}

WMcursor *SDL_CreateWMCursor(SDL_Surface *screen,
		Uint8 *data, Uint8 *mask, int w, int h, int hot_x, int hot_y)
{
	WMcursor *cursor;
	XGCValues GCvalues;
	GC        GCcursor;
	XImage *data_image, *mask_image;
	Pixmap  data_pixmap, mask_pixmap;
	int       clen, i;
	char     *x_data, *x_mask;
	static XColor black = {  0,  0,  0,  0 };
	static XColor white = { 0xffff, 0xffff, 0xffff, 0xffff };

	/* Allocate the cursor memory */
	cursor = (WMcursor *)malloc(sizeof(WMcursor));
	if ( cursor == NULL ) {
		SDL_OutOfMemory();
		return(NULL);
	}

	/* Mix the mask and the data */
	clen = (w/8)*h;
	x_data = (char *)malloc(clen);
	if ( x_data == NULL ) {
		free(cursor);
		SDL_OutOfMemory();
		return(NULL);
	}
	x_mask = (char *)malloc(clen);
	if ( x_mask == NULL ) {
		free(cursor);
		free(x_data);
		SDL_OutOfMemory();
		return(NULL);
	}
	for ( i=0; i<clen; ++i ) {
		/* The mask is OR'd with the data to turn inverted color
		   pixels black since inverted color cursors aren't supported
		   under X11.
		 */
		x_mask[i] = data[i] | mask[i];
		x_data[i] = data[i];
	}

	/* Prevent the event thread from running while we use the X server */
	SDL_Lock_EventThread();

	/* Create the data image */
	data_image = XCreateImage(SDL_Display, 
			DefaultVisual(SDL_Display, DefaultScreen(SDL_Display)),
					1, XYBitmap, 0, x_data, w, h, 8, w/8);
	data_image->byte_order = MSBFirst;
	data_image->bitmap_bit_order = MSBFirst;
	data_pixmap = XCreatePixmap(SDL_Display,
				DefaultRootWindow(SDL_Display), w, h, 1);

	/* Create the data mask */
	mask_image = XCreateImage(SDL_Display, 
			DefaultVisual(SDL_Display, DefaultScreen(SDL_Display)),
					1, XYBitmap, 0, x_mask, w, h, 8, w/8);
	mask_image->byte_order = MSBFirst;
	mask_image->bitmap_bit_order = MSBFirst;
	mask_pixmap = XCreatePixmap(SDL_Display,
				DefaultRootWindow(SDL_Display), w, h, 1);

	/* Create the graphics context */
	GCvalues.function = GXcopy;
	GCvalues.foreground = ~0;
	GCvalues.background =  0;
	GCvalues.plane_mask = AllPlanes;
	GCcursor = XCreateGC(SDL_Display, data_pixmap,
			(GCFunction|GCForeground|GCBackground|GCPlaneMask),
								&GCvalues);

	/* Blit the images to the pixmaps */
	XPutImage(SDL_Display, data_pixmap, GCcursor, data_image,
							0, 0, 0, 0, w, h);
	XPutImage(SDL_Display, mask_pixmap, GCcursor, mask_image,
							0, 0, 0, 0, w, h);
	XFreeGC(SDL_Display, GCcursor);
	/* These free the x_data and x_mask memory pointers */
	XDestroyImage(data_image);
	XDestroyImage(mask_image);

	/* Create the cursor */
	cursor->x_cursor = XCreatePixmapCursor(SDL_Display, data_pixmap,
				mask_pixmap, &black, &white, hot_x, hot_y);

	/* Release the event thread */
	XSync(SDL_Display, False);
	SDL_Unlock_EventThread();

	return(cursor);
}

int SDL_ShowWMCursor(SDL_Surface *screen, WMcursor *cursor)
{
	/* Don't do anything if the display is gone or fullscreen ... */
	if ( (SDL_Display == NULL) ||
	     ((screen->flags&SDL_FULLSCREEN) == SDL_FULLSCREEN) ) {
		return(0);
	}

	/* Set the X11 cursor cursor, or blank if cursor is NULL */
	SDL_Lock_EventThread();
	if ( cursor == NULL ) {
		if ( SDL_BlankCursor != NULL ) {
			XDefineCursor(SDL_Display, SDL_Window,
					SDL_BlankCursor->x_cursor);
		}
	} else {
		XDefineCursor(SDL_Display, SDL_Window, cursor->x_cursor);
	}
	XSync(SDL_Display, False);
	SDL_Unlock_EventThread();
	return(1);
}

void SDL_WarpWMCursor(SDL_Surface *screen, Uint16 x, Uint16 y)
{
	if ( (screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		SDL_PrivateMouseMotion(0, 0, screen->w/2, screen->h/2);
	} else {
		SDL_Lock_EventThread();
		XWarpPointer(SDL_Display, None, SDL_Window, 0, 0, 0, 0, x, y);
		XSync(SDL_Display, False);
		SDL_Unlock_EventThread();
	}
}

