/********************************************************************************
*                                                                               *
*                             I m a g e    O b j e c t                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXImage.cpp,v 1.5 1999/11/04 20:46:38 jeroen Exp $                       *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXDC.h"
#include "FXDCWindow.h"


/*
  To do:
  - Need API's to set/clear clipping bitmaps ``masks''
  - FXImage::create() renders rgb[a] data into X image
  - FXIcon::create() ditto, and also into X shape bitmap
  - Subclasses read in from particular format; like
     
      FXGIFImage(FXApp*,const FXuchar pixels,...)
      
    So you can:
    
      new FXLabel(dialog,NULL,new FXGIFImage(app,doc_icon),LAYOUT_FILL_X,...)
      
    Which should be much easier to use.
    
  - Use XSHM just added.
  - Issue: should FXImage return the DC for drawing onto the X-Server resident
    pixmap, or the client-side image bits?
    
    My idea is it should be the latter:
    
      - Allows even richer set of drawing primitives, as everything is
        drawn in software.
      - Very useful to generate off-screen renderings, e.g. during printing.
      - Allows for building and running true-color drawing programs on
        low-end graphics hardware.
      - The only drawback I can see is it will be a fairly large implementation
        effort...
        
  - Need to be able to re-render subpart of image?
  - If IMAGE_KEEP, repeated rendering is usually desired; should we
    hang on to XImage, and the shared memory segment in that case?
    How about shared pixmaps...
    
  - Under Windows, we probably need to:
  
    hBitmap=CreateBitmapIndirect(&bitmap);
    hdcMem=CreateCompatibleDC(hdc);
    SelectObject(hdcMem,hBitmap);
  
    This will produce a DDB bitmap which lives in GDI [not client].

  - Slight change in interpretation of IMAGE_OWNED flag:- if passed, the
    FXImage will own client-side pixel buffer, otherwise it will not; if
    no pixel-buffer has been passed and IMAGE_OWNED *is* passed, a pixel
    buffer will be allocated [and cleared to zero].
    No pixel buffer will be allocated if neither IMAGE_OWNED nor pixels
    are passed.
      
  - We should implement shared pixmaps.
  - When using shared image/pixmaps, if IMAGE_KEEP is set, hang on to pixel buffer.
  - Need resize, rotate member functions.
*/


// RGB Ordering code
enum {
  RGB = 7,   // RGB 111      > | R  G  B
  BGR = 0,   // BGR 000      --+--------
  RBG = 6,   // RBG 110      R | x  4  2
  GBR = 1,   // GBR 001      G |    x  1
  BRG = 4,   // BRG 100      B |       x
  GRB = 3    // GRB 011
  };


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXImage,FXDrawable,NULL,0)


// For deserialization
FXImage::FXImage(){
  data=NULL;
  options=0;
  }


// Initialize
FXImage::FXImage(FXApp* a,const void *pix,FXuint opts,FXint w,FXint h):FXDrawable(a,w,h){
  FXTRACE((100,"FXImage::FXImage %08x\n",this));
  visual=getApp()->getDefaultVisual();
  if(!pix && (opts&IMAGE_OWNED)){
    FXuint size=width*height;
    if(options&IMAGE_ALPHA) size*=4; else size*=3;
    FXCALLOC(&pix,FXuchar,size);
    }
  data=(FXuchar*)pix;
  //opts|=(IMAGE_SHMI|IMAGE_SHMP);
  options=opts;
#ifdef FX_NATIVE_WIN32
  hdcmem=NULL;
#endif
  }


// Create image
void FXImage::create(){
  if(!xid){
    FXTRACE((100,"%s::create %08x\n",getClassName(),this));
    
#ifndef FX_NATIVE_WIN32
    
    // App should exist
    if(!getApp()->display){ fxerror("%s::create: trying to create image before opening display.\n",getClassName()); }
  
    // Initialize visual
    visual->init();

    // Get depth (should use visual!!)
    int dd=visual->getDepth();
    
    // Make pixmap
    xid=XCreatePixmap(getApp()->display,XDefaultRootWindow(getApp()->display),width,height,dd);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }
  
#else

    // Initialize visual
    visual->init();

    // Create a memory DC compatible with current display
    HDC hdc=::GetDC(GetDesktopWindow());
    hdcmem=CreateCompatibleDC(hdc);
    if(!hdcmem){ fxerror("%s::create: unable to create image.\n",getClassName()); }
    xid=CreateCompatibleBitmap(hdc,width,height);
    ::ReleaseDC(GetDesktopWindow(),hdc);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }

#endif

    // Render pixels
    render();

    // Zap data
    if(!(options&IMAGE_KEEP) && (options&IMAGE_OWNED)){
      options&=~IMAGE_OWNED;
      FXFREE(&data);
      }
    }
  }


// Detach image
void FXImage::detach(){
  if(xid){
    FXTRACE((100,"%s::detach %08x\n",getClassName(),this));
    xid=0;
    }
  }


// Destroy image
void FXImage::destroy(){
  if(xid){
    FXTRACE((100,"%s::destroy %08x\n",getClassName(),this));
#ifndef FX_NATIVE_WIN32
    XFreePixmap(getApp()->display,xid);
#else
    DeleteObject(xid);
#endif
    xid=0;
    }
  }


// // Restore image from drawable
// int FXImage::restore(){
//   if(xid){
//     if(image) XDestroyImage(image); 
//     image=XGetImage(getApp()->display,xid,0,0,width,height,AllPlanes,ZPixmap);
//     }
//   return image!=0;
//   }



#ifndef FX_NATIVE_WIN32

// True color
void FXImage::render_true(XImage *xim,FXuchar *img,FXuint step){
  register FXuint val,r,g,b,jmp,rgborder;
  register FXuchar *pix;
  register FXint w,h;
  FXuint bits;
  rgborder=visual->getRGBOrder();
  pix=(FXuchar*)xim->data;
  bits=xim->bits_per_pixel;
  if(xim->byte_order==MSBFirst) bits|=0x80; 
  switch(bits){
    
    case 0x08:
    case 0x88:                                        // MSB/LSB 8-bit true color (rare)
      FXTRACE((150,"True MSB/LSB 8bpp render\n"));
      jmp=xim->bytes_per_line-width;
      h=height-1;
      do{
        w=width-1;
        do{
          val=visual->pixel_true_dither(img[0],img[1],img[2],w,h); img+=step;
          *pix++=(FXuchar)val;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x8f:                                        // MSB 15-bit true color (5,5,5)
    case 0x90:                                        // MSB 16-bit true color (5,6,5)
      FXTRACE((150,"True MSB 16bpp 5,6,5/5,5,5 render\n"));
      jmp=xim->bytes_per_line-(width<<1);
      h=height-1;
      do{
        w=width-1;
        do{
          val=visual->pixel_true_dither(img[0],img[1],img[2],w,h); img+=step;
          pix[0]=(FXuchar)(val>>8); pix[1]=(FXuchar)val; pix+=2;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x0f:                                        // LSB 15-bit true color (5,5,5)
    case 0x10:                                        // LSB 16-bit true color (5,6,5) (very common!)
      FXTRACE((150,"True LSB 16bpp 5,6,5/5,5,5 render\n"));
      jmp=xim->bytes_per_line-(width<<1);
      h=height-1;
      do{
        w=width-1;
        do{
          val=visual->pixel_true_dither(img[0],img[1],img[2],w,h); img+=step;
          pix[0]=(FXuchar)val; pix[1]=(FXuchar)(val>>8); pix+=2;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x18:                                        // LSB 24-bit true color
      rgborder^=7;
    
    case 0x98:                                        // MSB 24-bit true color
      FXTRACE((150,"True MSB/LSB 24bpp render\n"));
      jmp=xim->bytes_per_line-(width*3);
      switch(rgborder){
        case RGB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=r; pix[1]=g; pix[2]=b; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BGR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=b; pix[1]=g; pix[2]=r; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case RBG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=r; pix[1]=b; pix[2]=g; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GBR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=g; pix[1]=b; pix[2]=r; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BRG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=b; pix[1]=r; pix[2]=g; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GRB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=g; pix[1]=r; pix[2]=b; pix+=3;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        }
      break;
    case 0x20:                                        // LSB 32-bit true color
      FXTRACE((150,"True LSB 32bpp render\n"));
      jmp=xim->bytes_per_line-(width<<2);
      switch(rgborder){
        case RGB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=b; pix[1]=g; pix[2]=r; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BGR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=r; pix[1]=g; pix[2]=b; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case RBG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=g; pix[1]=b; pix[2]=r; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GBR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=r; pix[1]=b; pix[2]=g; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BRG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=g; pix[1]=r; pix[2]=b; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GRB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[0]=b; pix[1]=r; pix[2]=g; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        }
      break;
      
    case 0xa0:                                        // MSB 32-bit true color
      FXTRACE((150,"True MSB 32bpp render\n"));
      jmp=xim->bytes_per_line-(width<<2);
      switch(rgborder){
        case RGB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=r; pix[2]=g; pix[3]=b; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BGR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=b; pix[2]=g; pix[3]=r; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case RBG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=r; pix[2]=b; pix[3]=g; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GBR:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=g; pix[2]=b; pix[3]=r; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case BRG:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=b; pix[2]=r; pix[3]=g; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        case GRB:
          h=height-1;
          do{
            w=width-1;
            do{
              r=img[0]; g=img[1]; b=img[2]; img+=step;
              pix[1]=g; pix[2]=r; pix[3]=b; pix+=4;
              }
            while(--w>=0);
            pix+=jmp;
            }
          while(--h>=0);
          break;
        }
      break;

    default:                                          // Unsupported mode
      fxerror("%s::render: unimplemented true-color mode: %dbpp.\n",xim->bits_per_pixel,getClassName());
      break;
    }
  }


// Render index-color mode
void FXImage::render_index(XImage *xim,FXuchar *img,FXuint step){
  register FXuint val,jmp,half;
  register FXuchar *pix;
  register FXint w,h;
  FXuint bits;
  pix=(FXuchar*)xim->data;
  bits=xim->bits_per_pixel;
  if(xim->byte_order==MSBFirst) bits|=0x80;
  switch(bits){

    case 0x08:                                        // MSB/LSB 8-bit pseudocolor
    case 0x88:
      FXTRACE((150,"Index MSB/LSB 8bpp render\n"));
      jmp=xim->bytes_per_line-width;
      h=height-1;
      do{
        w=width-1;
        do{
          val=visual->pixel_index_dither(img[0],img[1],img[2],w,h); img+=step;
          *pix++=(FXuchar)val;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x84:                                        // MSB 4-bit pseudocolor 
      FXTRACE((150,"Index MSB 4bpp render\n"));
      jmp=xim->bytes_per_line-width;
      half=0;
      h=height-1;
      do{
        w=width-1;
        half=0;
        do{
          val=0x0f&visual->pixel_index_dither(img[0],img[1],img[2],w,h); img+=step;
          if(half) *pix++|=val; 
          else *pix=val<<4; 
          half^=1;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x04:                                        // LSB 4-bit pseudocolor 
      FXTRACE((150,"Index LSB 4bpp render\n"));
      jmp=xim->bytes_per_line-width;
      half=0;
      h=height-1;
      do{
        w=width-1;
        half=0;
        do{
          val=0x0f&visual->pixel_index_dither(img[0],img[1],img[2],w,h); img+=step;
          if(half) *pix++|=val<<4; 
          else *pix=val; 
          half^=1;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    case 0x82:                                        // MSB 2-bit pseudocolor
      FXTRACE((150,"Index MSB 2bpp render\n"));
      jmp=xim->bytes_per_line-width;
      h=height-1;
      do{
        w=width-1;
        half=0;
        do{
          val=0x03&visual->pixel_index_dither(img[0],img[1],img[2],w,h); img+=step;
          if(half==0) *pix=val<<6; 
          else if(half==1) *pix|=val<<4; 
          else if(half==2) *pix|=val<<2; 
          else *pix++|=val; 
          half=(half+1)&3;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;
      
    case 0x02:                                        // LSB 2-bit pseudocolor
      FXTRACE((150,"Index LSB 2bpp render\n"));
      jmp=xim->bytes_per_line-width;
      h=height-1;
      do{
        w=width-1;
        half=0;
        do{
          val=0x03&visual->pixel_index_dither(img[0],img[1],img[2],w,h); img+=step;
          if(half==0) *pix=val; 
          else if(half==1) *pix|=val<<2; 
          else if(half==2) *pix|=val<<4; 
          else *pix++|=val<<6; 
          half=(half+1)&3;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;

    default:
      fxerror("%s::render: unimplemented pseudo-color depth: %dbpp.\n",xim->bits_per_pixel,getClassName());
      break;
    } 
  }


// Render gray mode
void FXImage::render_gray(XImage *xim,FXuchar *img,FXuint step){
  register FXuint val,jmp;
  register FXuchar *pix;
  register FXint x,y,w,h;
  FXuint bits;
  pix=(FXuchar*)xim->data;
  bits=xim->bits_per_pixel;
  if(xim->byte_order==MSBFirst) bits|=0x80; 
  switch(bits){
    
    case 0x08:
    case 0x88:                                        // MSB/LSB 8-bit gray color (works!)
      FXTRACE((150,"Gray MSB/LSB 8bpp render\n"));
      jmp=xim->bytes_per_line-width;
      h=height-1;
      do{
        w=width-1;
        do{
          val=visual->pixel_gray_dither(img[0],img[1],img[2],w,h); img+=step;
          *pix++=(FXuchar)val;
          }
        while(--w>=0);
        pix+=jmp;
        }
      while(--h>=0);
      break;
      
    case 0x01:
    case 0x81:                                        // Monochrome 1-bit per pixel
      FXTRACE((150,"Monochrome MSB/LSB 1bpp render\n"));
      jmp=xim->bytes_per_line-width;
      for(y=0; y<height; y++){
        for(x=0; x<width; x++){
          val=visual->pixel_gray_dither(img[0],img[1],img[2],x,y); img+=step;
          XPutPixel(xim,x,y,val);
          }
        }
      break;
  
    default:
      fxerror("%s::render: unimplemented gray-color depth: %dbpp.\n",xim->bits_per_pixel,getClassName());
      break;
    }
  }

#endif

#ifndef FX_NATIVE_WIN32

// Render into pixmap
void FXImage::render(){
#ifdef HAVE_XSHM
  XShmSegmentInfo shminfo;
#endif
  register FXuint step=3;
  register FXbool shmi=FALSE;
  register XImage *xim=NULL;
  register Visual *vis;
  register int dd;
  XGCValues values;
  GC gc;
  
  FXTRACE((100,"%s::render image %0x8\n",getClassName(),this));

  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<2 || height<2){ fxerror("%s::render: illegal image size.\n",getClassName()); }

  // Make GC
  values.foreground=BlackPixel(getApp()->display,DefaultScreen(getApp()->display));
  values.background=WhitePixel(getApp()->display,DefaultScreen(getApp()->display));
  gc=XCreateGC(getApp()->display,xid,GCForeground|GCBackground,&values);

  // Just leave if black if no data
  if(data){
    
    // Have alpha?
    if(options&IMAGE_ALPHA) step=4;
    
    // Get Visual
    vis=visual->visual;
  
    dd=visual->getDepth();
    
    // Turn it on iff both supported and desired
#ifdef HAVE_XSHM
    if(options&IMAGE_SHMI) shmi=getApp()->shmi;
#endif
    
    // First try XShm
#ifdef HAVE_XSHM
    if(shmi){
      xim=XShmCreateImage(getApp()->display,vis,dd,(dd==1)?XYPixmap:ZPixmap,NULL,&shminfo,width,height);
      if(!xim){ shmi=0; }
      if(shmi){
        shminfo.shmid=shmget(IPC_PRIVATE,xim->bytes_per_line*xim->height,IPC_CREAT|0777);
        if(shminfo.shmid==-1){ XDestroyImage(xim); xim=NULL; shmi=0; }
        if(shmi){
          shminfo.shmaddr=xim->data=(char*)shmat(shminfo.shmid,0,0);
          shminfo.readOnly=FALSE;
          XShmAttach(getApp()->display,&shminfo);
          FXTRACE((150,"RGBPixmap XSHM attached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
          }
        }
      }
#endif
    
    // Try the old fashioned way
    if(!shmi){
      xim=XCreateImage(getApp()->display,vis,dd,(dd==1)?XYPixmap:ZPixmap,0,NULL,width,height,32,0);
      if(!xim){ fxerror("%s::render: unable to render image.\n",getClassName()); }

      // Try create temp pixel store
      xim->data=(char*)malloc(xim->bytes_per_line*height);

      // Failed completely
      if(!xim->data){ fxerror("%s::render: unable to allocate memory.\n",getClassName()); }
      }
    
    // Should have succeeded
    FXASSERT(xim);
    
    FXTRACE((150,"im format = %d\n",xim->format));
    FXTRACE((150,"im byte_order = %s\n",(xim->byte_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im bitmap_bit_order = %s\n",(xim->bitmap_bit_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_pad = %d\n",xim->bitmap_pad));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im depth = %d\n",xim->depth));
    FXTRACE((150,"im bytes_per_line = %d\n",xim->bytes_per_line));
    FXTRACE((150,"im bits_per_pixel = %d\n",xim->bits_per_pixel));
    
    // Determine what to do
    switch(vis->c_class){

      // True color
      case DirectColor:
      case TrueColor:
        render_true(xim,data,step);
        break;

      // Gray ramp
      case GrayScale:
      case StaticGray:
        render_gray(xim,data,step);
        break;
        
      // Pseudo color or gray ramp
      case StaticColor:
      case PseudoColor:
        render_index(xim,data,step);
        break;
      }

    // Transfer image with shared memory
#ifdef HAVE_XSHM
    if(shmi){
      XShmPutImage(getApp()->display,xid,gc,xim,0,0,0,0,width,height,False);
      XSync(getApp()->display,False);
      FXTRACE((150,"RGBPixmap XSHM detached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
      XShmDetach(getApp()->display,&shminfo);
      XDestroyImage(xim);
      shmdt(shminfo.shmaddr);
      shmctl(shminfo.shmid,IPC_RMID,0);
      }
#endif
   
    // Transfer the image old way
    if(!shmi){
      XPutImage(getApp()->display,xid,gc,xim,0,0,0,0,width,height);
      XDestroyImage(xim);
      }
    }
  
  // No data, fill with black
  else{
    XFillRectangle(getApp()->display,xid,gc,0,0,width,height);
    }
 
  // We're done
  XFreeGC(getApp()->display,gc);
  }


#else


void FXImage::render(){

  FXTRACE((100,"%s::render %0x8\n",getClassName(),this));
  
  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<2 || height<2){ fxerror("%s::render: illegal image size.\n",getClassName()); }

  // Just leave if black if no data
  if(data){
    
    // Have alpha?
    register FXuint step=3;
    if(options&IMAGE_ALPHA) step=4;
	
    // DIB format pads to multiples of 4 bytes...
    FXuint bytes_per_line=(width*3+3)/4*4;
    BYTE *pixels;
    FXMALLOC(&pixels,BYTE,bytes_per_line*height);
    register FXint h=height-1;
    FXuchar *img=data;
    do{
      register FXint w=width-1;
      register FXuint i=h*bytes_per_line;
      do{
        FXuint r,g,b;
        r=img[0]; g=img[1]; b=img[2]; img+=step;
        pixels[i]=b; pixels[i+1]=g; pixels[i+2]=r; i+=3;
        }
      while(--w>=0);
      } 
    while(--h>=0);
    
    // Set up the bitmap info
    BITMAPINFO bmi;
    bmi.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth=width;
    bmi.bmiHeader.biHeight=height;
    bmi.bmiHeader.biPlanes=1;
    bmi.bmiHeader.biBitCount=24;
    bmi.bmiHeader.biCompression=0;
    bmi.bmiHeader.biSizeImage=0;
    bmi.bmiHeader.biXPelsPerMeter=0;
    bmi.bmiHeader.biYPelsPerMeter=0;
    bmi.bmiHeader.biClrUsed=0;
    bmi.bmiHeader.biClrImportant=0;

    // The MSDN documentation for SetDIBits() states that "the device context
    // identified by the (first) parameter is used only if the DIB_PAL_COLORS
    // constant is set for the (last) parameter". This may be true, but under
    // Win95 you must pass in a non-NULL hdc for the first parameter; otherwise
    // this call to SetDIBits() will fail. (In contrast, it works fine under
    // Windows NT if you pass in a NULL hdc.)
    if(!SetDIBits((HDC)hdcmem,(HBITMAP)xid,0,height,pixels,&bmi,DIB_RGB_COLORS)){
      fxerror("%s::render: unable to render pixels\n",getClassName());
      }

    GdiFlush();
    SelectObject((HDC)hdcmem,(HBITMAP)xid);
    
    // Done with this stuff too
    FXFREE(&pixels);
    }

  // No data, fill with black
  else{
    SelectObject((HDC)hdcmem,(HBITMAP)xid);
    BitBlt((HDC)hdcmem,0,0,width,height,NULL,0,0,BLACKNESS);
    }
  }


// Return the device context
HDC FXImage::GetDC() const { return (HDC)hdcmem; }


// Release it (no-op)
int FXImage::ReleaseDC(HDC) const { return 1; }


#endif


// Save pixel data only
void FXImage::savePixels(FXStream& store) const {
  FXuint size=width*height;
  if(options&IMAGE_ALPHA) size*=4; else size*=3;
  store.save(data,size);
  }


// Load pixel data only
void FXImage::loadPixels(FXStream& store){
  FXuint size=width*height;
  if(options&IMAGE_ALPHA) size*=4; else size*=3;
  if(options&IMAGE_OWNED){FXFREE(&data);}
  FXMALLOC(&data,FXuchar,size);
  store.load(data,size);
  options|=IMAGE_OWNED;
  }


// Save data
void FXImage::save(FXStream& store) const {
  FXuchar haspixels=(data!=NULL);
  FXDrawable::save(store);
  store << options;
  store << haspixels;
  if(haspixels) savePixels(store);
  }


// Load data
void FXImage::load(FXStream& store){
  FXuchar haspixels;
  FXDrawable::load(store);
  store >> options;
  store >> haspixels;
  if(haspixels) loadPixels(store);
  }


// Clean up
FXImage::~FXImage(){
  FXTRACE((100,"FXImage::~FXImage %08x\n",this));
  if(xid){
#ifndef FX_NATIVE_WIN32
    XFreePixmap(getApp()->display,xid);
#else
    DeleteDC((HDC)hdcmem);
    DeleteObject(xid);
#endif
    }
  if(options&IMAGE_OWNED){FXFREE(&data);}
  data=(FXuchar*)-1;
  }
