/********************************************************************************
*                                                                               *
*                           O p e n G L   V i e w e r                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXGLViewer.cpp,v 1.14 1999/11/03 06:42:23 jeroen Exp $                   *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXVec.h"
#include "FXHVec.h"
#include "FXQuat.h"
#include "FXHMat.h"
#include "FXRange.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXObjectList.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXGLVisual.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXDCPrint.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXGIFIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXPacker.h"
#include "FXShell.h"
#include "FXTopWindow.h"
#include "FXDialogBox.h"
#include "FXMessageBox.h"
#include "FXTooltip.h"
#include "FXLabel.h"
#include "FXTextField.h"
#include "FXButton.h"
#include "FXRadioButton.h"
#include "FXArrowButton.h"
#include "FXMenuButton.h"
#include "FXSpinner.h"
#include "FXColorWell.h"
#include "FXCursor.h"
#include "FXGLCanvas.h"
#include "FXGLViewer.h"
#include "FXGLObject.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXList.h"
#include "FXComboBox.h"
#include "FXPrintDialog.h"
#include "jitter.h"


/*
  To Do:
  - Initialize GL to fastest of everything for drawing lines.
  - Group object needs current element.
  - use app->dragDelta for motion tolerance.
  - Default op to noop mode; all returns 0 in noop mode.
  - GLuint unfortunately not always big enough to store a pointer...
  - The selection seems to get lost with multiple viewers into.
    the same scene.  If you select a cube in one view, then select another
    cube in another view, both seem to get selected.  Can we push the
    "selection" member from the view to the scene object?
  - Instead of select/deselect, do focus gain/lost type deal.
  - Add methods for inquire of pick-ray.
  - Fix FXGLGroup to identify child or itself..
  - Need some way of updating ALL viewers.
  - Need a document/view type concept?
  - Load/Save need to save more...
  - Build mini display lists for offset/surface drawing.
  - Pass clicked/double-clicked/triple-clicked messages to object.
  - Distinguish between current object and selected ones.
    only one is current, but many may be selected.
  - When object(s) deleted, need to fix up selection...
  - GLViewer should source some messages to its target for important events.
  - Zoom-lasso feature.
  
  - Basic mouse actions:
 
    Button  Dir.   Modifier(s) Where              Moved? Action
    ----------------------------------------------------------------
    Left     Dn    None        Inside Selection   --     Start drag of selection
    Left     Dn    None        Outside Selection  --     Start rotate
    Left     Up    None        Inside Selection   No     End drag of selection
    Left     Up    None        Outside Selection  No     Deselect selection; select new selection; end rotate
    Left     Up    None        Anywhere           Yes    End rotate
    Left     Dn    Shift       Anywhere           --     Start lasso
    Left     Dn    Shift       Outside Selection  --     Start lasso
    Left     Up    Shift       Anywhere           No     End lasso; add new object(s) to selection
    Left     Up    Shift       Anywhere           Yes    End lasso; add lassoed to selection
    Left     Dn    Control     Anywhere           --     Start lasso
    Left     Dn    Control     Outside Selection  --     Start lasso
    Left     Up    Control     Anywhere           No     End lasso; toggle new object's selection status
    Left     Up    Control     Anywhere           Yes    End lasso; toggle lassoed objects selection status
    Left     Pr    X           Anywhere           --     Rotate about model X
    Left     Pr    Y           Anywhere           --     Rotate about model Y
    Left     Pr    Z           Anywhere           --     Rotate about model Z
    Left     Pr    U           Anywhere           --     Rotate about camera X
    Left     Pr    V           Anywhere           --     Rotate about camera Y
    Left     Pr    W           Anywhere           --     Rotate about camera Z
    Left   DblClk  None        Inside Selection   --     Edit object
    Left   DblClk  None        Outside Selection  --     Deselect selection
    -------------------------------------------------------------
    Middle   Dn    None        Anywhere           --     Start zoom
    Middle   Up    None        Anywhere           --     End zoom
    Middle   Dn    Shift       Anywhere           --     Start dolly
    Middle   Up    Shift       Anywhere           --     End dolly
    -------------------------------------------------------------
    Right    Dn    None        Selection          --     Popup menu for selection
    Right    Dn    None        Outside selection  --     Start translate
    Right    Up    None        Anywhere           --     End translate
    Right    Pr    X           Anywhere           --     Start translate in X=c (perhaps along X???)
    Right    Pr    Y           Anywhere           --     Start translate in Y=c
    Right    Pr    Z           Anywhere           --     Start translate in Z=c
    Right    Dn    Shift       Anywhere           --     Start FOV change
    Right    Up    Shift       Anywhere           --     End FOV change
 
  - Selection:
 
    o  Selection depth is 0 for top level.
    o  If selection depth is N for group object:
       - it is N+1 for current child.
       - it is N for other children.
    o  FXGLViewer maintains selection depth.

*/

/*******************************************************************************/


// Clamp value x to range [lo..hi]
#define CLAMP(lo,x,hi) ((x)<(lo)?(lo):((x)>(hi)?(hi):(x)))

// Size of pick buffer
#define MAX_PICKBUF    1024      

// Maximum length of selection path
#define MAX_SELPATH    64

// Rotation tolerance
#define EPS            1.0E-3

/*******************************************************************************/


// Map
FXDEFMAP(FXGLViewer) FXGLViewerMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXGLViewer::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXGLViewer::onMotion),
  FXMAPFUNC(SEL_UPDATE,0,FXGLViewer::onUpdate),
  FXMAPFUNC(SEL_TIMEOUT,FXGLViewer::ID_TIPTIMER,FXGLViewer::onTipTimer),
  FXMAPFUNC(SEL_DND_ENTER,0,FXGLViewer::onDNDEnter),
  FXMAPFUNC(SEL_DND_LEAVE,0,FXGLViewer::onDNDLeave),
  FXMAPFUNC(SEL_DND_DROP,0,FXGLViewer::onDNDDrop),
  FXMAPFUNC(SEL_DND_MOTION,0,FXGLViewer::onDNDMotion),
  FXMAPFUNC(SEL_ENTER,0,FXGLViewer::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXGLViewer::onLeave),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXGLViewer::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXGLViewer::onLeftBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,FXGLViewer::onMiddleBtnPress),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,FXGLViewer::onMiddleBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXGLViewer::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXGLViewer::onRightBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXGLViewer::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXGLViewer::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXGLViewer::onKeyRelease),
  FXMAPFUNC(SEL_FOCUSIN,0,FXGLViewer::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXGLViewer::onFocusOut),
  FXMAPFUNC(SEL_CHANGED,0,FXGLViewer::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXGLViewer::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXGLViewer::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXGLViewer::onTripleClicked),
  FXMAPFUNC(SEL_LASSOED,0,FXGLViewer::onLassoed),
  FXMAPFUNC(SEL_SELECTED,0,FXGLViewer::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXGLViewer::onDeselected),
  FXMAPFUNC(SEL_INSERTED,0,FXGLViewer::onInserted),
  FXMAPFUNC(SEL_DELETED,0,FXGLViewer::onDeleted),
  FXMAPFUNCS(SEL_UPDATE,FXGLViewer::ID_DIAL_X,FXGLViewer::ID_DIAL_Z,FXGLViewer::onUpdXYZDial),
  FXMAPFUNCS(SEL_CHANGED,FXGLViewer::ID_DIAL_X,FXGLViewer::ID_DIAL_Z,FXGLViewer::onCmdXYZDial),
  FXMAPFUNCS(SEL_COMMAND,FXGLViewer::ID_DIAL_X,FXGLViewer::ID_DIAL_Z,FXGLViewer::onCmdXYZDial),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXGLViewer::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXGLViewer::onQueryHelp),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_PERSPECTIVE,FXGLViewer::onUpdPerspective),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_PARALLEL,FXGLViewer::onUpdParallel),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_FRONT,FXGLViewer::onUpdFront),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_BACK,FXGLViewer::onUpdBack),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LEFT,FXGLViewer::onUpdLeft),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_RIGHT,FXGLViewer::onUpdRight),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_TOP,FXGLViewer::onUpdTop),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_BOTTOM,FXGLViewer::onUpdBottom),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_RESETVIEW,FXWindow::onUpdYes),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_FITVIEW,FXWindow::onUpdYes),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_CUT_SEL,FXGLViewer::onUpdDeleteSel),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_COPY_SEL,FXGLViewer::onUpdCurrent),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_PASTE_SEL,FXGLViewer::onUpdYes),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_DELETE_SEL,FXGLViewer::onUpdDeleteSel),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_BACK_COLOR,FXGLViewer::onUpdBackColor),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_AMBIENT_COLOR,FXGLViewer::onUpdAmbientColor),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LOCK,FXGLViewer::onUpdLock),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LIGHTING,FXGLViewer::onUpdLighting),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_FOG,FXGLViewer::onUpdFog),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_DITHER,FXGLViewer::onUpdDither),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_FOV,FXGLViewer::onUpdFov),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_ZOOM,FXGLViewer::onUpdZoom),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LIGHT_AMBIENT,FXGLViewer::onUpdLightAmbient),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LIGHT_DIFFUSE,FXGLViewer::onUpdLightDiffuse),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_LIGHT_SPECULAR,FXGLViewer::onUpdLightSpecular),
  FXMAPFUNC(SEL_UPDATE,FXGLViewer::ID_TURBO,FXGLViewer::onUpdTurbo),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_PERSPECTIVE,FXGLViewer::onCmdPerspective),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_PARALLEL,FXGLViewer::onCmdParallel),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_FRONT,FXGLViewer::onCmdFront),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_BACK,FXGLViewer::onCmdBack),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LEFT,FXGLViewer::onCmdLeft),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_RIGHT,FXGLViewer::onCmdRight),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_TOP,FXGLViewer::onCmdTop),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_BOTTOM,FXGLViewer::onCmdBottom),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_RESETVIEW,FXGLViewer::onCmdResetView),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_FITVIEW,FXGLViewer::onCmdFitView),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LOCK,FXGLViewer::onCmdLock),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_CUT_SEL,FXGLViewer::onCmdCutSel),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_COPY_SEL,FXGLViewer::onCmdCopySel),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_PASTE_SEL,FXGLViewer::onCmdPasteSel),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_DELETE_SEL,FXGLViewer::onCmdDeleteSel),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_BACK_COLOR,FXGLViewer::onCmdBackColor),
  FXMAPFUNC(SEL_CHANGED,FXGLViewer::ID_BACK_COLOR,FXGLViewer::onCmdBackColor),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_AMBIENT_COLOR,FXGLViewer::onCmdAmbientColor),
  FXMAPFUNC(SEL_CHANGED,FXGLViewer::ID_AMBIENT_COLOR,FXGLViewer::onCmdAmbientColor),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LIGHTING,FXGLViewer::onCmdLighting),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_FOG,FXGLViewer::onCmdFog),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_DITHER,FXGLViewer::onCmdDither),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_FOV,FXGLViewer::onCmdFov),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_ZOOM,FXGLViewer::onCmdZoom),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LIGHT_AMBIENT,FXGLViewer::onCmdLightAmbient),
  FXMAPFUNC(SEL_CHANGED,FXGLViewer::ID_LIGHT_AMBIENT,FXGLViewer::onCmdLightAmbient),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LIGHT_DIFFUSE,FXGLViewer::onCmdLightDiffuse),
  FXMAPFUNC(SEL_CHANGED,FXGLViewer::ID_LIGHT_DIFFUSE,FXGLViewer::onCmdLightDiffuse),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_LIGHT_SPECULAR,FXGLViewer::onCmdLightSpecular),
  FXMAPFUNC(SEL_CHANGED,FXGLViewer::ID_LIGHT_SPECULAR,FXGLViewer::onCmdLightSpecular),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_TURBO,FXGLViewer::onCmdTurbo),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_PRINT_IMAGE,FXGLViewer::onCmdPrintImage),
  FXMAPFUNC(SEL_COMMAND,FXGLViewer::ID_PRINT_VECTOR,FXGLViewer::onCmdPrintVector),
  };


// Object implementation
FXIMPLEMENT(FXGLViewer,FXGLCanvas,FXGLViewerMap,ARRAYNUMBER(FXGLViewerMap))


// Make a canvas
FXGLViewer::FXGLViewer(FXComposite* p,FXGLVisual *vis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXGLCanvas(p,vis,tgt,sel,opts,x,y,w,h){
  projection=PERSPECTIVE;                       // Projection
  zoom=1.0;                                     // Zoom factor
  offset=0.004;                                 // Offset for lines on surfaces
  fov=30.0;                                     // Field of view (1 to 90)
  aspect=1.0;                                   // Aspect ratio
  wvt.left=-1.0;                                // Init world box
  wvt.right=1.0;
  wvt.top=1.0;
  wvt.bottom=-1.0;
  wvt.hither=0.1;
  wvt.yon=1.0;
  wvt.w=100;                                    // Viewport width
  wvt.h=100;                                    // Viewport height
  screenmin=100.0;                              // Screen minimum
  screenmax=100.0;                              // Screen maximum
  diameter=2.0;                                 // Size of model
  distance=7.464116;                            // Distance of PRP to target
  rotation=FXQuat(0.0,0.0,0.0,1.0);             // Orientation
  center=FXVec(0.0,0.0,0.0);                    // Model center
  scale=FXVec(1.0,1.0,1.0);                     // Model scaling
  updateProjection();                           // Initial projection
  updateTransform();                            // Set transformation
  op=PICKING;                                   // Mouse operation
  maxhits=512;                                  // Maximum hit buffer size
  background[0]=0.003921568627f*FXREDVAL(getApp()->backColor);    // Background copied from GUI background
  background[1]=0.003921568627f*FXGREENVAL(getApp()->backColor);
  background[2]=0.003921568627f*FXBLUEVAL(getApp()->backColor);
  background[3]=1.0;
  ambient=FXHVec(0.2f,0.2f,0.2f,1.0);           // Scene ambient
  
  light.ambient=FXHVec(0.0,0.0,0.0,1.0);        // Light setup
  light.diffuse=FXHVec(1.0,1.0,1.0,1.0);
  light.specular=FXHVec(0.0,0.0,0.0,1.0);
  light.position=FXHVec(-2.0,2.0,3.0,0.0);
  light.direction=FXVec(0.0,0.0,-1.0);
  light.exponent=0.0;
  light.cutoff=180.0;
  light.c_attn=1.0;
  light.l_attn=0.0;
  light.q_attn=0.0;
  
  material.ambient=FXHVec(0.2f,0.2f,0.2f,1.0f); // Material setup
  material.diffuse=FXHVec(0.8f,0.8f,0.8f,1.0f);
  material.specular=FXHVec(1.0f,1.0f,1.0f,1.0f);
  material.emission=FXHVec(0.0f,0.0f,0.0f,1.0f);
  material.shininess=30.0;
  
  dial[0]=0;                                    // Old dial position
  dial[1]=0;                                    // Old dial position
  dial[2]=0;                                    // Old dial position
  
  doesturbo=FALSE;                              // In interaction
  turbomode=FALSE;                              // Turbo mode
  timer=NULL;                                   // Motion timer
  dropped=NULL;                                 // Nobody being dropped on
  selection=NULL;                               // No initial selection
  zsortfunc=NULL;                               // Routine to sort feedback buffer
  scene=NULL;                                   // Scene to look at
  }


// Create window
void FXGLViewer::create(){
  FXRange r(-1.0,1.0,-1.0,1.0,-1.0,1.0);
  
  // We would like to have this be true
#ifdef HAVE_OPENGL
  FXASSERT(sizeof(FXuint)==sizeof(GLuint));
#endif
  
  // Create Window
  FXGLCanvas::create();
  
  // Set up OpenGL environment
  glsetup();
  
  // Register drag type for color
  if(!colorType){colorType=getApp()->registerDragType(colorTypeName);}
  
  // Viewer accepts any drop type
  dropEnable();
  
  // If have scene already, get correct bounds
  if(scene) scene->bounds(r);
  
  // Set initial viewing volume
  setBounds(r);
  }


/*********************  V i e w i n g   S e t u p  ***************************/


// Set up GL context
void FXGLViewer::glsetup(){
#ifdef HAVE_OPENGL

  // Make GL context current
  if(makeCurrent()){

    // Initialize GL context
    glRenderMode(GL_RENDER);

    // Fast hints
    glHint(GL_POLYGON_SMOOTH_HINT,GL_FASTEST);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_FASTEST);
    glHint(GL_FOG_HINT,GL_FASTEST);
    glHint(GL_LINE_SMOOTH_HINT,GL_FASTEST);
    glHint(GL_POINT_SMOOTH_HINT,GL_FASTEST);

    // Z-buffer test on
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);
    glDepthRange(0.0,1.0);
    glClearDepth(1.0);
    glClearColor(background[0],background[1],background[2],1.0);

    // No face culling
    glDisable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    glFrontFace(GL_CCW);

    // Two sided lighting
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,TRUE);
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,FALSE);
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT,ambient);

    // Preferred blend over background
    glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

    // Light on
    glEnable(GL_LIGHT0);
    glLightfv(GL_LIGHT0,GL_AMBIENT,light.ambient);
    glLightfv(GL_LIGHT0,GL_DIFFUSE,light.diffuse);
    glLightfv(GL_LIGHT0,GL_SPECULAR,light.specular);
    glLightfv(GL_LIGHT0,GL_POSITION,light.position);
    glLightfv(GL_LIGHT0,GL_SPOT_DIRECTION,light.direction);
    glLightf(GL_LIGHT0,GL_SPOT_EXPONENT,light.exponent);
    glLightf(GL_LIGHT0,GL_SPOT_CUTOFF,light.cutoff);
    glLightf(GL_LIGHT0,GL_CONSTANT_ATTENUATION,light.c_attn);
    glLightf(GL_LIGHT0,GL_LINEAR_ATTENUATION,light.l_attn);
    glLightf(GL_LIGHT0,GL_QUADRATIC_ATTENUATION,light.q_attn);

    // Viewer is far away 
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,FALSE);

    // Material colors
    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,material.ambient);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,material.diffuse);
    glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,material.specular);
    glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,material.emission);
    glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,material.shininess);

    // Vertex colors change both diffuse and ambient
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    glDisable(GL_COLOR_MATERIAL);
    
    // Simplest and fastest drawing is default
    glShadeModel(GL_FLAT);
    glDisable(GL_BLEND);
    glDisable(GL_LINE_SMOOTH);
    glDisable(GL_POINT_SMOOTH);
    glDisable(GL_COLOR_MATERIAL);
    
    // Lighting
    glDisable(GL_LIGHTING);
    
    // No normalization of normals (it's broken on some machines anyway)
    glDisable(GL_NORMALIZE);
    
    // Dithering if needed
    glDisable(GL_DITHER);
    makeNonCurrent();
    }
#endif
  }


// Render all the graphics into a world box
void FXGLViewer::drawWorld(FXViewport& wv){
#ifdef HAVE_OPENGL

  // Set viewport
  glViewport(0,0,wv.w,wv.h);

  // Clear background
  glClearDepth(1.0);
  glClearColor(background[0],background[1],background[2],1.0);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

  // Set OFFSET Projection Matrix
  glNewList(FXGLViewer::OFFSETPROJECTION,GL_COMPILE);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glTranslatef(0.0,0.0,(GLfloat)(-offset/zoom));
  switch(projection){
    case PARALLEL: glOrtho(wv.left,wv.right,wv.bottom,wv.top,wv.hither,wv.yon); break;
    case PERSPECTIVE: glFrustum(wv.left,wv.right,wv.bottom,wv.top,wv.hither,wv.yon); break;
    }
  glMatrixMode(GL_MODELVIEW);
  glEndList();
  
  // Set SURFACE Projection Matrix
  glNewList(FXGLViewer::SURFACEPROJECTION,GL_COMPILE);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  switch(projection){
    case PARALLEL: glOrtho(wv.left,wv.right,wv.bottom,wv.top,wv.hither,wv.yon); break;
    case PERSPECTIVE: glFrustum(wv.left,wv.right,wv.bottom,wv.top,wv.hither,wv.yon); break;
    }
  glMatrixMode(GL_MODELVIEW);
  glEndList();
  
  // Set SURFACE Projection Matrix 
  glCallList(FXGLViewer::SURFACEPROJECTION);

  // Place the light
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  // Set light parameters
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0,GL_AMBIENT,light.ambient);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,light.diffuse);
  glLightfv(GL_LIGHT0,GL_SPECULAR,light.specular);
  glLightfv(GL_LIGHT0,GL_POSITION,light.position);
  glLightfv(GL_LIGHT0,GL_SPOT_DIRECTION,light.direction);
  glLightf(GL_LIGHT0,GL_SPOT_EXPONENT,light.exponent);
  glLightf(GL_LIGHT0,GL_SPOT_CUTOFF,light.cutoff);
  glLightf(GL_LIGHT0,GL_CONSTANT_ATTENUATION,light.c_attn);
  glLightf(GL_LIGHT0,GL_LINEAR_ATTENUATION,light.l_attn);
  glLightf(GL_LIGHT0,GL_QUADRATIC_ATTENUATION,light.q_attn);
    
  // Set model matrix
  glLoadMatrixf(transform);
 
  // Default material colors
  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,material.ambient);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,material.diffuse);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,material.specular);
  glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,material.emission);
  glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,material.shininess);
  
  // Color commands change both
  glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
  
  // Track material
  glDisable(GL_COLOR_MATERIAL);

  // Global ambient light
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT,ambient);

  // Enable lighting
  if(options&VIEWER_LIGHTING)
    glEnable(GL_LIGHTING);
  else
    glDisable(GL_LIGHTING);
  
  // Enable fog
  if(options&VIEWER_FOG){
    glEnable(GL_FOG);
    glFogfv(GL_FOG_COLOR,background);         // Disappear into the background
    //glFogf(GL_FOG_DENSITY,1.0);
    glFogf(GL_FOG_START,(GLfloat)(distance-diameter));   // Range tight around model position
    glFogf(GL_FOG_END,(GLfloat)(distance+diameter));     // Far place same as clip plane:- clipped stuff is in the mist!
    glFogi(GL_FOG_MODE,GL_LINEAR);	// Simple linear depth cueing
    }
  else{
    glDisable(GL_FOG);
    }

  // Dithering
  if(options&VIEWER_DITHER)
    glEnable(GL_DITHER);
  else
    glDisable(GL_DITHER);

  // Shade model
  //glShadeModel(GL_FLAT);
  glShadeModel(GL_SMOOTH);
  
  // Draw what's visible
  if(scene) scene->draw(this);
#endif
  }


// Render with anti-aliasing
void FXGLViewer::drawAnti(FXViewport& wv){
#ifdef HAVE_OPENGL
  FXViewport jt=wv;
  FXdouble d=0.5*worldpx;
  register FXuint i;
  glClearAccum(0.0,0.0,0.0,0.0);
  glClear(GL_ACCUM_BUFFER_BIT);
  for(i=0; i<ARRAYNUMBER(jitter); i++){
    jt.left = wv.left+jitter[i][0]*d;
    jt.right = wv.right+jitter[i][0]*d;
    jt.top = wv.top+jitter[i][1]*d;
    jt.bottom = wv.bottom+jitter[i][1]*d;
    drawWorld(jt);
    glAccum(GL_ACCUM,1.0/ARRAYNUMBER(jitter));
    }
  glAccum(GL_RETURN,1.0);
#endif
  }


// Fill select buffer with hits in rectangle
FXint FXGLViewer::selectHits(FXuint *hits,FXint x,FXint y,FXint w,FXint h,FXint mxhits){
  register float pickx,picky,pickw,pickh;
  register int nhits=0;
#ifdef HAVE_OPENGL
  if(makeCurrent()){

    // Where to pick
    pickx=(wvt.w-2.0f*x-w)/((float)w);
    picky=(2.0f*y+h-wvt.h)/((float)h);
    pickw=wvt.w/((float)w);
    pickh=wvt.h/((float)h);

    // Set OFFSET Pick Matrix
    glNewList(FXGLViewer::OFFSETPROJECTION,GL_COMPILE);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glTranslatef(pickx,picky,(GLfloat)(-offset/zoom));   // Slightly toward eye; one wants to pick lines first!
    glScalef(pickw,pickh,1.0);
    switch(projection){
      case PARALLEL: glOrtho(wvt.left,wvt.right,wvt.bottom,wvt.top,wvt.hither,wvt.yon); break;
      case PERSPECTIVE: glFrustum(wvt.left,wvt.right,wvt.bottom,wvt.top,wvt.hither,wvt.yon); break;
      }
    glMatrixMode(GL_MODELVIEW);
    glEndList();

    // Set SURFACE Pick Matrix
    glNewList(FXGLViewer::SURFACEPROJECTION,GL_COMPILE);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glTranslatef(pickx,picky,0.0);
    glScalef(pickw,pickh,1.0);
    switch(projection){
      case PARALLEL: glOrtho(wvt.left,wvt.right,wvt.bottom,wvt.top,wvt.hither,wvt.yon); break;
      case PERSPECTIVE: glFrustum(wvt.left,wvt.right,wvt.bottom,wvt.top,wvt.hither,wvt.yon); break;
      }
    glMatrixMode(GL_MODELVIEW);
    glEndList();

    // Set SURFACE Pick Matrix
    glCallList(FXGLViewer::SURFACEPROJECTION);

    // Model Matrix
    glMatrixMode(GL_MODELVIEW);
    glLoadMatrixf(transform);
    glSelectBuffer(mxhits,(GLuint*)hits);
    glRenderMode(GL_SELECT);
    glInitNames();
    glPushName(0);
    if(scene) scene->hit(this);
    glPopName();
    nhits=glRenderMode(GL_RENDER);
    makeNonCurrent();
    }
#endif
  return nhits;
  }


// Process picks
FXGLObject* FXGLViewer::processHits(FXuint *pickbuffer,FXint nhits){
  register FXuint d1,d2,n,zmin,zmax;
  register FXint i,sel;
  sel=0;
  for(i=0,zmin=zmax=4294967295U; nhits>0; i+=n+3,nhits--){
    n=pickbuffer[i];
    d1=pickbuffer[1+i];
    d2=pickbuffer[2+i];
    if(d1<zmin || (d1==zmin && d2<=zmax)){
      zmin=d1;
      zmax=d2;
      sel=i;
      }
    }
  FXASSERT(0<=sel);
  return scene->identify((FXuint*)&pickbuffer[4+sel]);
  }


// Build NULL-terminated list of ALL picked objects overlapping rectangle
FXGLObject** FXGLViewer::select(FXint x,FXint y,FXint w,FXint h){
  FXGLObject **objects=NULL;
  register FXGLObject *obj;
  register int nhits,i,j;
  register FXuint n;
  FXuint *hits;
  if(maxhits<=0) return NULL;
  if(!scene) return NULL;
  FXMALLOC(&hits,FXuint,maxhits);
  if(!hits){ fxerror("%s::select: unable to allocate hit buffer.\n",getClassName()); }
  nhits=selectHits(hits,x,y,w,h,maxhits);
  if(nhits<0){ fxwarning("%s::select: hit buffer overflowed\n",getClassName()); }
  if(nhits>0){
    FXMALLOC(&objects,FXGLObject*,nhits+1);
    for(i=j=0; nhits>0; i+=n+3,nhits--){
      n=hits[i];
      obj=scene->identify(&hits[4+i]);
      if(obj) objects[j++]=obj;
      }
    objects[j]=NULL;
    }
  FXFREE(&hits);
  return objects;
  }


// Lasso objects
FXGLObject** FXGLViewer::lasso(FXint x1,FXint y1,FXint x2,FXint y2){
  FXint xlo,xhi,ylo,yhi;
  FXMINMAX(xlo,xhi,x1,x2);
  FXMINMAX(ylo,yhi,y1,y2);
  return select(xlo,ylo,xhi-xlo+1,yhi-ylo+1);
  }
        

// Pick ONE object at x,y
FXGLObject* FXGLViewer::pick(FXint x,FXint y){
  register FXGLObject *obj=NULL;
  register FXint nhits;
  FXuint *hits;
  if(maxhits<=0) return NULL;
  if(!scene) return NULL;
  FXMALLOC(&hits,FXuint,maxhits);
  if(!hits){ fxerror("%s::pick: unable to allocate hit buffer.\n",getClassName()); }
  nhits=selectHits(hits,x-PICK_TOL,y-PICK_TOL,PICK_TOL*2,PICK_TOL*2,maxhits);
  if(nhits<0){ fxwarning("%s::pick: hit buffer overflowed\n",getClassName()); } 
  if(nhits>0){
    obj=processHits(hits,nhits);
    }
  FXFREE(&hits);
  return obj;
  }


// Handle expose event; if we're double buffering and using MESA
// we just blit the exposed section again from the back buffer;
// If its a synthesized expose event, or we're using true OpenGL,
// then we perform the regular OpenGL drawing to regenerate the
// whole window (this could be expensive!)
long FXGLViewer::onPaint(FXObject*,FXSelector,void* ptr){
#ifdef HAVE_OPENGL
  FXGLVisual *vis=(FXGLVisual*)getVisual();
#ifdef HAVE_MESA
  FXEvent *event=(FXEvent*)ptr;
#endif
  FXASSERT(xid);
#ifdef HAVE_MESA
  if(vis->isDoubleBuffer() && !event->synthetic){
    if(makeCurrent()){
      swapSubBuffers(event->rect.x,event->rect.y,event->rect.w,event->rect.h);
      makeNonCurrent();
      }
    return 1;
    }
#endif
  if(makeCurrent()){
    drawWorld(wvt);
    if(vis->isDoubleBuffer()) swapBuffers();
    makeNonCurrent();
    }
#endif
  return 1;
  }


// Move and resize; note we need to force a synthetic repaint
// event because we need to regenerate the OpenGL
void FXGLViewer::position(FXint x,FXint y,FXint w,FXint h){
  FXbool repaint=((w!=width)||(h!=height));
  FXGLCanvas::position(x,y,w,h);
  if(repaint) update();
  }

      
// Resize; note we need to force a synthetic repaint
// event because we need to regenerate the OpenGL
void FXGLViewer::resize(int w,int h) {
  FXGLCanvas::resize(w,h);
  update();
  }


// Start motion timer while in this window
long FXGLViewer::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXGLCanvas::onEnter(sender,sel,ptr);
  if(isEnabled()){
    if(!timer){timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);}
    }
  return 1;
  }


// Stop motion timer when leaving window
long FXGLViewer::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXGLCanvas::onLeave(sender,sel,ptr);
  if(isEnabled()){
    if(timer){getApp()->removeTimeout(timer);timer=NULL;}
    }
  return 1;
  }


// Gained focus
long FXGLViewer::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXGLCanvas::onFocusIn(sender,sel,ptr);
  if(selection && selection->handle(this,MKUINT(0,SEL_FOCUSIN),ptr)){
    update();
    }
  return 1;
  }

  
// Lost focus
long FXGLViewer::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXGLCanvas::onFocusOut(sender,sel,ptr);
  if(selection && selection->handle(this,MKUINT(0,SEL_FOCUSOUT),ptr)){
    update();
    }
  return 1;
  }


// Handle configure notify
void FXGLViewer::layout(){
  wvt.w=width;
  wvt.h=height;
  if(wvt.w>0 && wvt.h>0){
    aspect = (double)wvt.h / (double)wvt.w;
    if(wvt.w>wvt.h){
      screenmax=wvt.w;
      screenmin=wvt.h;
      }
    else{
      screenmax=wvt.h;
      screenmin=wvt.w;
      }
    updateProjection();
    }
  flags&=~FLAG_DIRTY;
  }


// Change scene
void FXGLViewer::setScene(FXGLObject* sc){
  scene=sc;
  update();
  }


// Change field of view
void FXGLViewer::setFieldOfView(FXdouble fv){
  FXdouble tn;
  fov=CLAMP(2.0,fv,90.0);
  tn=tan(0.5*DTOR*fov);
  FXASSERT(tn>0.0);
  distance=diameter/tn;
  FXASSERT(distance>0.0);
  updateProjection();
  updateTransform();
  update();
  }


// Change eye distance
void FXGLViewer::setDistance(FXdouble ed){
  if(ed<diameter) ed=diameter;
  if(ed>114.0*diameter) ed=114.0*diameter;
  distance=ed;
  FXASSERT(distance>0.0);
  fov=2.0*RTOD*atan2(diameter,distance);
  updateProjection();
  updateTransform();
  update();
  }


// Change zoom factor
void FXGLViewer::setZoom(FXdouble zm){
  if(1.0E-30<zm){
   zoom=zm;
   updateProjection();
   update();
   }
 }


// Change orientation to new quaternion
void FXGLViewer::setOrientation(const FXQuat& rot){
  rotation=rot;
  rotation.adjust();
  updateTransform();
  update();
  }


// Change world projection
void FXGLViewer::updateProjection(){
  double hither_fac,r;

  // Should be non-0 size viewport
  FXASSERT(wvt.w>0 && wvt.h>0);
  
  // Get world box
  r=0.5*diameter/zoom;
  if(wvt.w<=wvt.h){
    wvt.left=-r;
    wvt.right=r;
    wvt.bottom=-r*aspect;
    wvt.top=r*aspect;
    }
  else{
    wvt.left=-r/aspect;
    wvt.right=r/aspect;
    wvt.bottom=-r;
    wvt.top=r;
    }
  FXASSERT(distance>0.0);
  FXASSERT(diameter>0.0);
  
  // A more intelligent method for setting the
  // clip planes might be interesting...
  wvt.yon=distance+diameter;
  wvt.hither=0.1*wvt.yon;
  //wvt.hither=distance-diameter;
//  if(wvt.hither<distance-diameter) wvt.hither=distance-diameter;
  
  // New window
  FXTRACE((100,"wvt.left=%g wvt.right=%g wvt.top=%g wvt.bottom=%g wvt.hither=%g wvt.yon=%g\n",wvt.left,wvt.right,wvt.top,wvt.bottom,wvt.hither,wvt.yon));

  // Size of a pixel in world and model
  worldpx=(wvt.right-wvt.left)/wvt.w;
  modelpx=worldpx*diameter;

  // Precalc stuff for view->world backmapping
  ax=wvt.left;
  ay=wvt.top-worldpx;
  
  // Report factors
  FXTRACE((100,"worldpx=%g modelpx=%g\n",worldpx,modelpx));

  // Correction for perspective
  if(projection==PERSPECTIVE){
    FXASSERT(distance>0.0);
    hither_fac=wvt.hither/distance;
    wvt.left*=hither_fac;
    wvt.right*=hither_fac;
    wvt.top*=hither_fac;
    wvt.bottom*=hither_fac;
    }
  }


// Change transformation matrix
void FXGLViewer::updateTransform(){
  transform.eye();
  transform.trans(0.0,0.0,(FXfloat)-distance);
  transform.rot(rotation);
  transform.scale(scale);
  transform.trans(-center);
  itransform=invert(transform);
//   FXTRACE((150,"itrans=%11.8f %11.8f %11.8f %11.8f\n",itransform[0][0],itransform[0][1],itransform[0][2],itransform[0][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",itransform[1][0],itransform[1][1],itransform[1][2],itransform[1][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",itransform[2][0],itransform[2][1],itransform[2][2],itransform[2][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",itransform[3][0],itransform[3][1],itransform[3][2],itransform[3][3]));
//   FXTRACE((150,"\n"));
//   FXTRACE((150," trans=%11.8f %11.8f %11.8f %11.8f\n",transform[0][0],transform[0][1],transform[0][2],transform[0][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",transform[1][0],transform[1][1],transform[1][2],transform[1][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",transform[2][0],transform[2][1],transform[2][2],transform[2][3]));
//   FXTRACE((150,"       %11.8f %11.8f %11.8f %11.8f\n",transform[3][0],transform[3][1],transform[3][2],transform[3][3]));
//   FXTRACE((150,"\n"));
  }


// Set model bounding box
FXbool FXGLViewer::setBounds(const FXRange& r){
//   FXTRACE((100,"xlo=%g xhi=%g ylo=%g yhi=%g zlo=%g zhi=%g\n",r[0][0],r[0][1],r[1][0],r[1][1],r[2][0],r[2][1]));
  
  // Model center
  center=boxCenter(r);

  // Model size
  diameter=r.longest();

  // Fix zero size model
  if(diameter<1.0E-30) diameter=1.0;

  // Set equal scaling initially
  scale=FXVec(1.0,1.0,1.0);

  // Reset distance (and thus field of view)
  setDistance(1.1*diameter);

//   FXTRACE((100,"center=%g %g %g\n",center[0],center[1],center[2]));
//   FXTRACE((100,"diameter=%g\n",diameter));
  return TRUE;
  }


// Fit view to new bounds
FXbool FXGLViewer::fitToBounds(const FXRange& box){
  FXRange r(FLT_MAX,-FLT_MAX,FLT_MAX,-FLT_MAX,FLT_MAX,-FLT_MAX);
  FXVec corner[8],v;
  FXHMat m;
  register int i;

  // Get corners
  boxCorners(corner,box);

  // Get rotation of model
  m.eye();
  m.rot(rotation);
  m.trans(-boxCenter(box));

  // Transform box
  for(i=0; i<8; i++){
    v=corner[i]*m;
    r.include(v[0],v[1],v[2]);
    }

  setBounds(r);
  return TRUE;
  }


// Obtain viewport
void FXGLViewer::getViewport(FXViewport& v) const {
  v=wvt;
  }


// Set material
void FXGLViewer::setMaterial(const FXMaterial &mtl){
  material=mtl;
  update();
  }


// Get material
void FXGLViewer::getMaterial(FXMaterial &mtl) const {
  mtl=material;
  }


// Get screen point from eye coordinate
void FXGLViewer::eyeToScreen(FXint& sx,FXint& sy,const FXVec& e){
  register double xp,yp;
  if(projection==PERSPECTIVE){
    if(e[2]==0.0){ fxerror("%s::eyeToScreen: cannot transform point.\n",getClassName()); }
    xp=-distance*e[0]/e[2];
    yp=-distance*e[1]/e[2];
    }
  else{
    xp=e[0];
    yp=e[1];
    }
  sx=(int)((xp-ax)/worldpx);
  sy=(int)((ay-yp)/worldpx);
  }


// Convert screen point to eye coords
FXVec FXGLViewer::screenToEye(FXint sx,FXint sy,FXfloat eyez){
  register float xp,yp;
  FXVec e;
  xp=(float)(worldpx*sx+ax);
  yp=(float)(ay-worldpx*sy);
  if(projection==PERSPECTIVE){
    FXASSERT(distance>0.0);
    e[0]=(FXfloat)(-eyez*xp/distance);
    e[1]=(FXfloat)(-eyez*yp/distance);
    e[2]=eyez;
    }
  else{
    e[0]=xp;
    e[1]=yp;
    e[2]=eyez;
    }
  return e;
  }


// Convert screen to eye, on projection plane
FXVec FXGLViewer::screenToTarget(FXint sx,FXint sy){
  return FXVec((FXfloat)(worldpx*sx+ax), (FXfloat)(ay-worldpx*sy), (FXfloat)-distance);
  } 


// Convert world to eye coords
FXVec FXGLViewer::worldToEye(const FXVec& w){ 
  return w*transform;
  }


// Get eye Z-coordinate of world point
FXfloat FXGLViewer::worldToEyeZ(const FXVec& w){
  return w[0]*transform[0][2]+w[1]*transform[1][2]+w[2]*transform[2][2]+transform[3][2];
  }


// Convert eye to world coords
FXVec FXGLViewer::eyeToWorld(const FXVec& e){ 
  return e*itransform; 
  }


// Get world vector
FXVec FXGLViewer::worldVector(FXint fx,FXint fy,FXint tx,FXint ty){
  FXVec wfm,wto,vec;
  wfm=screenToTarget(fx,fy);
  wto=screenToTarget(tx,ty);
  vec=wto*itransform-wfm*itransform;
  return vec;
  }


// Get a bore vector
FXbool FXGLViewer::getBoreVector(FXint sx,FXint sy,FXVec& point,FXVec& dir){
  FXVec p;
  FXdouble d;
  p=eyeToWorld(screenToEye(sx,sy,(FXfloat)(diameter-distance)));
  if(PARALLEL==projection)
    point=eyeToWorld(screenToEye(sx,sy,0));
  else
    point=eyeToWorld(FXVec(0,0,0));
  dir=p-point;
  d=len(dir);
  if(0.0<d) dir/=(FXfloat)d;         // normalize
  return TRUE;
  }


// Get eye viewing direction (non-normalized)
FXVec FXGLViewer::getEyeVector() const {
  return FXVec(-itransform[2][0],-itransform[2][1],-itransform[2][2]);
  }


// Get eye position
FXVec FXGLViewer::getEyePosition() const{
  return FXVec(itransform[3][0],itransform[3][1],itransform[3][2]);
  }


// Change model center
void FXGLViewer::setCenter(const FXVec& cntr){
  center=cntr;
  updateTransform();
  update();
  }


// Translate in world 
void FXGLViewer::translate(const FXVec& vec){
  center+=vec;
  updateTransform();
  update();
  }


// Change selection
void FXGLViewer::setSelection(FXGLObject* sel){
  if(selection!=sel){
    selection=sel;
    update();
    }
  }


// Change help text
void FXGLViewer::setHelpText(const FXString& text){
  help=text;
  }


// Change tip text
void FXGLViewer::setTipText(const FXString& text){
  tip=text;
  }


// Translate point into unit-sphere coordinate
FXVec FXGLViewer::spherePoint(FXint px,FXint py){
  FXdouble d,t;
  FXVec v;
  FXASSERT(screenmin>0);
  v[0]=(FXfloat)(2.0*(px-0.5*wvt.w)/screenmin);
  v[1]=(FXfloat)(2.0f*(0.5*wvt.h-py)/screenmin);
  d=v[0]*v[0]+v[1]*v[1];
  if(d<0.75){
    v[2]=(FXfloat)sqrt(1.0-d);
    }
  else if(d<3.0){ 
    d=1.7320508008-sqrt(d); 
    t=1.0-d*d;
    if(t<0.0) t=0.0;
    v[2]=(FXfloat)(1.0-sqrt(t));
    }
  else{ 
    v[2]=0.0;
    }
  return normalize(v);
  }


// Turn camera
FXQuat FXGLViewer::turn(FXint fx,FXint fy,FXint tx,FXint ty){
  FXVec oldsp,newsp;
  FXQuat q;
  FXdouble t;
  oldsp=spherePoint(fx,fy);
  newsp=spherePoint(tx,ty);
  q=arc(oldsp,newsp);
  q[0]*=0.5; 
  q[1]*=0.5; 
  q[2]*=0.5; 
  t=1.0-(q[0]*q[0]+q[1]*q[1]+q[2]*q[2]);
  if(t<0.0) t=0.0;
  q[3]=(FXfloat)sqrt(t);
  return q;
  }


// It can be focused on
FXbool FXGLViewer::canFocus() const { return TRUE; }


// Draw non-destructive lasso box; drawing twice will erase it again
void FXGLViewer::drawLasso(FXint x0,FXint y0,FXint x1,FXint y1){
#ifdef HAVE_OPENGL
  FXGLVisual *vis=(FXGLVisual*)getVisual();

#ifdef HAVE_MESA
  
  // With Mesa, this is quite simple
  FXDCWindow dc(this);
  dc.setFunction(BLT_NOT_DST);
  dc.drawLine(x0,y0,x1,y0);
  dc.drawLine(x1,y0,x1,y1);
  dc.drawLine(x1,y1,x0,y1);
  dc.drawLine(x0,y1,x0,y0);
    
#else
    
  // With OpenGL, first change back to 1:1 projection mode
  if(makeCurrent()){
    glPushAttrib(GL_COLOR_BUFFER_BIT|GL_ENABLE_BIT);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0.0,width-1.0,0.0,height-1.0,0.0,1.0); 
    if(vis->isDoubleBuffer()) glDrawBuffer(GL_FRONT);
    glLineWidth(1.0);
    glEnable(GL_BLEND);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_COLOR_MATERIAL);
    glDisable(GL_LIGHTING);
    glShadeModel(GL_FLAT);
    glDepthMask(FALSE);
    
    // On SGI's we will probably have logic ops; 
    // certainly, on DEC's we don't even if it sayz we have OpenGL 1.1
#if defined(GL_VERSION_1_1) && defined(GL_EXT_blend_logic_op)

#ifndef WIN32
    glBlendEquationEXT(GL_LOGIC_OP);
    //glEnable(GL_LOGIC_OP);
    glLogicOp(GL_XOR);
#else
    glEnable(GL_LOGIC_OP);
    glLogicOp(GL_XOR);
#endif

    // On other machines we may not
#else
    //glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
    glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ZERO);
#endif

    glBegin(GL_LINE_LOOP);
    glColor3f(1.0,1.0,1.0);
    glVertex2i(x0,wvt.h-y0-1);
    glVertex2i(x0,wvt.h-y1-1);
    glVertex2i(x1,wvt.h-y1-1);
    glVertex2i(x1,wvt.h-y0-1);
    glEnd();
    glFinish();
    if(vis->isDoubleBuffer()) glDrawBuffer(GL_BACK);
    glDepthMask(TRUE);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
    glPopAttrib();
    makeNonCurrent();
    }
#endif
#endif
  }
  
/*************************  Mouse Actions in Viewer  ***************************/


// Current object changed
long FXGLViewer::onChanged(FXObject*,FXSelector,void* ptr){
  setSelection((FXGLObject*)ptr);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Clicked in widget
long FXGLViewer::onClicked(FXObject*,FXSelector,void* ptr){
  if(target){
    if(target->handle(this,MKUINT(message,SEL_CLICKED),ptr)) return 1;
    if(ptr && target->handle(this,MKUINT(message,SEL_COMMAND),ptr)) return 1;
    }
  return 1;
  }


// Double clicked in widget; ptr may or may not point to an object
long FXGLViewer::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple clicked in widget; ptr may or may not point to an object
long FXGLViewer::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Lassoed object(s)
long FXGLViewer::onLassoed(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXGLObject **objlist;  

  // Notify target of lasso
  if(target && target->handle(this,MKUINT(message,SEL_LASSOED),ptr)) return 1;

  // Grab all objects in lasso
  objlist=lasso(event->click_x,event->click_y,event->win_x,event->win_y);

  // Add selection mode
  if(event->state&SHIFTMASK){
    handle(this,MKUINT(0,SEL_SELECTED),(void*)objlist);
    }

  // Toggle selection mode
  else if(event->state&CONTROLMASK){
    handle(this,MKUINT(0,SEL_DESELECTED),(void*)objlist);
    }

  // Free list
  FXFREE(&objlist);

  return 1;
  }


// Selected object(s)
long FXGLViewer::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_SELECTED),ptr);
  }


// Deselected object(s)
long FXGLViewer::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DESELECTED),ptr);
  }


// Inserted object(s)
long FXGLViewer::onInserted(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_INSERTED),ptr);
  }


// Deleted object(s)
long FXGLViewer::onDeleted(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DELETED),ptr);
  }


// Perform the usual mouse manipulation
long FXGLViewer::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(event->click_count==1){
      
      // Start a lasso if shift or control
      if(event->state&(SHIFTMASK|CONTROLMASK)){
        drawLasso(event->click_x,event->click_y,event->win_x,event->win_y);
        op=LASSOING;
        flags&=~FLAG_UPDATE;
        }
        
      // Try object pick if view locked
      else if(options&VIEWER_LOCKED){
        FXGLObject *objects[2];
        
        // Determine if a hit
        objects[0]=pick(event->click_x,event->click_y);
        objects[1]=NULL;
        
        // Change selected object
        handle(this,MKUINT(0,SEL_CHANGED),(void*)objects[0]);
        
        // Notify target
        handle(this,MKUINT(0,SEL_SELECTED),(void*)objects);

        // Hit an object
        if(objects[0]){

          // Now we may want to drag it
          if(objects[0]->canDrag()){
            op=DRAGGING;
            doesturbo=turbomode;
            flags&=~FLAG_UPDATE;
            }
          }
        }
 
      // In non-locked mode, hit current object
      else if(selection && selection->canDrag() && selection==pick(event->click_x,event->click_y)){
        op=DRAGGING;
        doesturbo=turbomode;
        flags&=~FLAG_UPDATE;
        }
      
      // Else start rotation
      else{
        op=ROTATING;
        doesturbo=turbomode;
        flags&=~FLAG_UPDATE;
        }
      }
    }
  return 1;
  }


// Left mouse button released
long FXGLViewer::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  register FXuint operation=op;
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    op=PICKING;
    if(doesturbo){
      update();
      doesturbo=FALSE;
      }
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    
    // Single clicked
    if(event->click_count==1){
 
      // End lasso
      if(operation==LASSOING){
        
        // Hide the lasso
        drawLasso(event->click_x,event->click_y,event->win_x,event->win_y);
        
        // Lassoed
        handle(this,MKUINT(0,SEL_LASSOED),ptr);        
        }

      // End object drag
      else if(operation==DRAGGING){
        if(target) target->handle(this,MKUINT(message,SEL_DRAGGED),selection);
        }

      // We didn't move much during a rotate, try a pick
      else if(operation==ROTATING && !event->moved){
        FXGLObject *objects[2];
        objects[0]=pick(event->click_x,event->click_y);
        objects[1]=NULL;
        handle(this,MKUINT(0,SEL_CHANGED),(void*)objects[0]);
        handle(this,MKUINT(0,SEL_SELECTED),(void*)objects);
        }
      handle(this,MKUINT(0,SEL_CLICKED),(void*)selection);
      }

    // Double clicked
    else if(event->click_count==2){
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)selection);
      }

    // Triple clicked
    else if(event->click_count==3){
      handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)selection);
      }
    }
  return 1;
  }


// Pressed middle mouse button
long FXGLViewer::onMiddleBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONPRESS),ptr)) return 1;
    if(event->click_count==1){
      
      // Allow viewing manipulations
      if(!(options&VIEWER_LOCKED)){

        // Truck camera into/out of scene
        if(event->state&(SHIFTMASK|CONTROLMASK)){
          op=TRUCKING;
          doesturbo=turbomode;
          flags&=~FLAG_UPDATE;
          }

        // Zoom in
        else{
          op=ZOOMING;
          doesturbo=turbomode;
          flags&=~FLAG_UPDATE;
          }
        }
      }
    }
  return 1;
  }


// Released middle mouse button
long FXGLViewer::onMiddleBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    op=PICKING;
    if(doesturbo){
      update();
      doesturbo=FALSE;
      }
    if(target && target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONRELEASE),ptr)) return 1;
    }
  return 1;
  }


// Pressed right button
long FXGLViewer::onRightBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONPRESS),ptr)) return 1;
    if(event->click_count==1){
      
      // Allow viewing manipulations
      if(!(options&VIEWER_LOCKED)){

        // Rotation about camera
        if(event->state&SHIFTMASK){
          op=GYRATING;
          doesturbo=turbomode;
          flags&=~FLAG_UPDATE;
          }

        // Field of view change
        else if(event->state&CONTROLMASK){
          op=FOVING;
          doesturbo=turbomode;
          flags&=~FLAG_UPDATE;
          }

        // Translate
        else{
          op=TRANSLATING;
          doesturbo=turbomode;
          flags&=~FLAG_UPDATE;
          }
        }
      }
    }
  return 1;
  }


// Released right button
long FXGLViewer::onRightBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXGLObject *hit;
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    op=PICKING;
    if(doesturbo){
      update();
      doesturbo=FALSE;
      }
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONRELEASE),ptr)) return 1;
    if(event->click_count==1){
      if(!event->moved){

        // Ask object for its menu if we hit an object
        if((hit=pick(event->click_x,event->click_y))!=NULL){
          if(hit->handle(this,MKUINT(ID_QUERY_MENU,SEL_COMMAND),ptr)) return 1;
          }

        // Otherwide, ask the target
        if(target){
          if(target->handle(this,MKUINT(ID_QUERY_MENU,SEL_COMMAND),ptr)) return 1;
          }
        }
      }
    }
  return 1;
  }


// The widget lost the grab for some reason
long FXGLViewer::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXGLCanvas::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  op=PICKING;
  doesturbo=FALSE;
  return 1;
  }


// We timed out, i.e. the user didn't move for a while
long FXGLViewer::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((250,"%s::onTipTimer %08x\n",getClassName(),this));
  timer=NULL;
  flags|=FLAG_TIP;
  return 1;
  }


// We were asked about status text
long FXGLViewer::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){
    FXTRACE((250,"%s::onQueryHelp %08x\n",getClassName(),this));
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// We were asked about tip text
long FXGLViewer::onQueryTip(FXObject* sender,FXSelector sel,void* ptr){
  FXGLObject *hit;
  FXuint state;
  FXint x,y; 
  if(flags&FLAG_TIP){
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %08x (%d,%d)\n",getClassName(),this,x,y));
    hit=pick(x,y);
    if(hit && hit->handle(sender,sel,ptr)) return 1;
    if(!tip.empty()){
      sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&tip);
      return 1;
      }
    }
  return 0;
  }


// Mouse moved
long FXGLViewer::onMotion(FXObject*,FXSelector,void* ptr){
  register long changed=(flags&FLAG_TIP)!=0;
  FXEvent* event=(FXEvent*)ptr;
  register FXdouble delta;
  register FXfloat tmp;
  FXVec vec;
  FXQuat q;
  //flags&=~FLAG_TIP;
  if(isEnabled()){
    if(target && target->handle(this,MKUINT(message,SEL_MOTION),ptr)) return 1;
    if(event->last_x!=event->win_x || event->last_y!=event->win_y){
      flags&=~FLAG_TIP;
      if(timer){ getApp()->removeTimeout(timer); timer=NULL; }
      switch(op){
        case PICKING:           // Reset the timer each time we moved the cursor
          timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);
          break;
        case TRUCKING:          // Trucking camera forward or backward
          tmp=(FXfloat)(worldpx*(event->win_y-event->last_y));
          vec=normalize(getEyeVector());
          translate(tmp*vec);
          changed=1;
          break;
        case TRANSLATING:       // Translating camera
          vec=worldVector(event->last_x,event->last_y,event->win_x,event->win_y);
          translate(-vec);
          //qqqx+=(FXfloat)(worldpx*(event->win_x-event->last_x));
          //qqqy-=(FXfloat)(worldpx*(event->win_y-event->last_y));
  //updateTransform();
  //update();
          changed=1;
          break;
        case ZOOMING:           // Zooming camera
          delta=0.005*(event->win_y-event->last_y);
          setZoom(getZoom()*pow(2.0,delta));
          changed=1;
          break;
        case ROTATING:          // Rotating camera around target
          q=turn(event->last_x,event->last_y,event->win_x,event->win_y) * getOrientation();
          setOrientation(q);
          changed=1;
          break;
        case GYRATING:          // Rotating camera around eye
          update();
          changed=1;
          break;
        case LASSOING:          // Lasso an bunch of objects
          drawLasso(event->click_x,event->click_y,event->last_x,event->last_y);
          drawLasso(event->click_x,event->click_y,event->win_x,event->win_y);
          changed=1;
          break;
        case FOVING:            // Change FOV
          setFieldOfView(getFieldOfView()+90.0*(event->win_y-event->last_y)/(double)wvt.h);
          changed=1;
          break;
        case DRAGGING:          // Dragging a shape
          FXASSERT(selection);
          if(selection->drag(this,event->last_x,event->last_y,event->win_x,event->win_y)){
            update();
            }
          changed=1;
          break;
        }
      }
    }
  return changed;
  }


// Update this widgets state, when it is not active
long FXGLViewer::onUpdate(FXObject* sender,FXSelector sel,void* ptr){
  FXWindow::onUpdate(sender,sel,ptr);
  return 1;
  }


/*****************************  Keyboard Input  ********************************/


// Handle keyboard press/release 
long FXGLViewer::onKeyPress(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
    }
  return 0;
  }


// Key release
long FXGLViewer::onKeyRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
    }
  return 0;
  }


/*****************************  Switch Projection  *****************************/


// Switch to perspective mode
long FXGLViewer::onCmdPerspective(FXObject*,FXSelector,void*){
  setProjection(PERSPECTIVE);
  return 1;
  }


// Update sender  
long FXGLViewer::onUpdPerspective(FXObject* sender,FXSelector,void*){
  FXuint msg=projection==PERSPECTIVE ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Switch to parallel mode
long FXGLViewer::onCmdParallel(FXObject*,FXSelector,void*){
  setProjection(PARALLEL);
  return 1;
  }


// Update sender
long FXGLViewer::onUpdParallel(FXObject* sender,FXSelector,void*){
  FXuint msg=projection==PARALLEL ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


/*****************************  Switch Viewpoints  *****************************/

// View front
long FXGLViewer::onCmdFront(FXObject*,FXSelector,void*){
  rotation=FXQuat(0.0,0.0,0.0,1.0);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdFront(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]) && 
     EPS>fabs(rotation[1]) && 
     EPS>fabs(rotation[2]) && 
     EPS>fabs(rotation[3]-1.0)) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }

// View back
long FXGLViewer::onCmdBack(FXObject*,FXSelector,void*){
  rotation=FXQuat(0.0,-1.0,0.0,0.0);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdBack(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]) &&
     EPS>fabs(rotation[1]+1.0) &&
     EPS>fabs(rotation[2]) &&
     EPS>fabs(rotation[3])) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }

// View left
long FXGLViewer::onCmdLeft(FXObject*,FXSelector,void*){
  rotation=FXQuat(0.0,0.7071067811865f,0.0,0.7071067811865f);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdLeft(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]) &&
     EPS>fabs(rotation[1]-0.7071067811865) &&
     EPS>fabs(rotation[2]) &&
     EPS>fabs(rotation[3]-0.7071067811865)) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }

// View right
long FXGLViewer::onCmdRight(FXObject*,FXSelector,void*){
  rotation=FXQuat(0.0,-0.7071067811865f,0.0,0.7071067811865f);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdRight(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]) &&
     EPS>fabs(rotation[1]+0.7071067811865) &&
     EPS>fabs(rotation[2]) &&
     EPS>fabs(rotation[3]-0.7071067811865)) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }

// View top
long FXGLViewer::onCmdTop(FXObject*,FXSelector,void*){
  rotation=FXQuat(0.7071067811865f,0.0,0.0,0.7071067811865f);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdTop(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]-0.7071067811865) &&
     EPS>fabs(rotation[1]) &&
     EPS>fabs(rotation[2]) &&
     EPS>fabs(rotation[3]-0.7071067811865)) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }

// View bottom
long FXGLViewer::onCmdBottom(FXObject*,FXSelector,void*){
  rotation=FXQuat(-0.7071067811865f,0.0,0.0,0.7071067811865f);
  updateTransform();
  update();
  return 1;
  }


// Update sender
long FXGLViewer::onUpdBottom(FXObject* sender,FXSelector,void*){
  FXuint msg=ID_UNCHECK;
  if(EPS>fabs(rotation[0]+0.7071067811865) &&
     EPS>fabs(rotation[1]) &&
     EPS>fabs(rotation[2]) &&
     EPS>fabs(rotation[3]-0.7071067811865)) msg=ID_CHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Reset view
long FXGLViewer::onCmdResetView(FXObject*,FXSelector,void*){
  FXRange r(-1.0,1.0,-1.0,1.0,-1.0,1.0);
  rotation=FXQuat(0.0,0.0,0.0,1.0);
  zoom=1.0;
  scale=FXVec(1.0,1.0,1.0);
  if(scene) scene->bounds(r);
  setBounds(r);
  updateTransform();
  update();
  return 1;
  }


// Fit view
long FXGLViewer::onCmdFitView(FXObject*,FXSelector,void*){
  FXRange r(-1.0,1.0,-1.0,1.0,-1.0,1.0);
  if(scene) scene->bounds(r);
  setBounds(r);
  update();
  return 1;
  }


// Update zoom
long FXGLViewer::onUpdZoom(FXObject* sender,FXSelector,void*){
  sender->handle(this,MKUINT(FXWindow::ID_SETREALVALUE,SEL_COMMAND),(void*)&zoom);
  return 1;
  }


// Change zoom
long FXGLViewer::onCmdZoom(FXObject* sender,FXSelector,void*){
  FXdouble z=0.0;
  if(sender->handle(this,MKUINT(FXWindow::ID_GETREALVALUE,SEL_COMMAND),(void*)&z)){
    setZoom(z);
    }
  return 1;
  }


// Update field of view
long FXGLViewer::onUpdFov(FXObject* sender,FXSelector,void*){
  sender->handle(this,MKUINT(FXWindow::ID_SETREALVALUE,SEL_COMMAND),(void*)&fov);
  return 1;
  }


// Change field of view
long FXGLViewer::onCmdFov(FXObject* sender,FXSelector,void*){
  FXdouble f;
  if(sender->handle(this,MKUINT(FXWindow::ID_GETREALVALUE,SEL_COMMAND),(void*)&f)){
    setFieldOfView(f);
    }
  return 1;
  }


  
// Rotate camera about model by means of dials
long FXGLViewer::onCmdXYZDial(FXObject*,FXSelector sel,void* ptr){
  const FXVec xaxis(1.0,0.0,0.0);
  const FXVec yaxis(0.0,1.0,0.0);
  const FXVec zaxis(0.0,0.0,1.0);
  FXint dialnew=(FXint)(long)ptr;
  FXfloat ang;
  FXQuat q;
  doesturbo=(SELTYPE(sel)==SEL_CHANGED)?turbomode:FALSE;
  FXuint sid=SELID(sel);
  switch(sid){
    case ID_DIAL_X: 
      ang=(FXfloat)(DTOR*(dialnew-dial[0]));
      q=FXQuat(xaxis,-ang); 
      dial[0]=dialnew;
      break;
    case ID_DIAL_Y:
      ang=(FXfloat)(DTOR*(dialnew-dial[1]));
      q=FXQuat(yaxis, ang);
      dial[1]=dialnew;
      break;
    case ID_DIAL_Z: 
      ang=(FXfloat)(DTOR*(dialnew-dial[2]));
      q=FXQuat(zaxis, ang); 
      dial[2]=dialnew; 
      break;
    }
  FXQuat p=q*getOrientation();
  setOrientation(p);
  return 1;
  }


// Update dial value
long FXGLViewer::onUpdXYZDial(FXObject* sender,FXSelector sel,void*){
  FXASSERT(ID_DIAL_X<=SELID(sel) && SELID(sel)<=ID_DIAL_Z);
  sender->handle(this,MKUINT(FXWindow::ID_SETINTVALUE,SEL_COMMAND),(void*)&dial[SELID(sel)-ID_DIAL_X]);
  return 1;
  }


/******************************  Printing Support  *****************************/


// Read back pixels
// Derived from code contributed by <sancelot@crosswinds.net>
FXbool FXGLViewer::readPixels(FXuchar*& buffer,FXint x,FXint y,FXint w,FXint h){
#ifdef HAVE_OPENGL
  GLint swapbytes,lsbfirst,rowlength,oldbuf;
  GLint skiprows,skippixels,alignment;
  GLuint size=w*h*3;

  // Try allocate buffer
  if(!FXMALLOC(&buffer,FXuchar,size)) return FALSE;
  
  makeCurrent();
  
  // Save old pixel formats
  glGetIntegerv(GL_PACK_SWAP_BYTES,&swapbytes);
  glGetIntegerv(GL_PACK_LSB_FIRST,&lsbfirst);
  glGetIntegerv(GL_PACK_ROW_LENGTH,&rowlength);
  glGetIntegerv(GL_PACK_SKIP_ROWS,&skiprows);
  glGetIntegerv(GL_PACK_SKIP_PIXELS,&skippixels);
  glGetIntegerv(GL_PACK_ALIGNMENT,&alignment);
  glGetIntegerv(GL_READ_BUFFER,&oldbuf);

  // Set pixel readback formats
  glPixelStorei(GL_PACK_SWAP_BYTES,GL_FALSE);
  glPixelStorei(GL_PACK_LSB_FIRST,GL_FALSE);
  glPixelStorei(GL_PACK_ROW_LENGTH,0);
  glPixelStorei(GL_PACK_SKIP_ROWS,0);
  glPixelStorei(GL_PACK_SKIP_PIXELS,0);
  glPixelStorei(GL_PACK_ALIGNMENT,1);
  
  // Read from the right buffer
  glReadBuffer((GLenum)GL_FRONT);

  // Read the pixels
  glReadPixels(x,y,w,h,GL_RGB,GL_UNSIGNED_BYTE,(GLvoid*)buffer);
   
  // Restore old formats
  glPixelStorei(GL_PACK_SWAP_BYTES,swapbytes);
  glPixelStorei(GL_PACK_LSB_FIRST,lsbfirst);
  glPixelStorei(GL_PACK_ROW_LENGTH,rowlength);
  glPixelStorei(GL_PACK_SKIP_ROWS,skiprows);
  glPixelStorei(GL_PACK_SKIP_PIXELS,skippixels);
  glPixelStorei(GL_PACK_ALIGNMENT,alignment);
  glReadBuffer((GLenum)oldbuf);
  
  makeNonCurrent();
  
  return TRUE;
#else
  return FALSE;
#endif
  }


// Print the window by grabbing pixels
long FXGLViewer::onCmdPrintImage(FXObject*,FXSelector,void*){
  FXPrintDialog dlg(this,"Print Scene");
  FXuint red,green,blue,size;
  FXPrinter printer;
  FXuchar *buffer;
  FXint i;
  
  // Run dialog
  if(dlg.execute()){
    dlg.getPrinter(printer);
    FXDCPrint pdc(getApp());
    if(!pdc.beginPrint(printer)){
      FXMessageBox::error(this,MBOX_OK,"Printer Error","Unable to print");
      return 1;
      }
    
    // Page header
    pdc.beginPage(1);
    
    // Grab pixels
    if(readPixels(buffer,0,0,width,height)){

      size=width*height*3;

      pdc.outf("/picstr %d string def\n",width*3);
      pdc.outf("%d %d translate\n",0,0+height);
      pdc.outf("%d %d scale\n",width,-height);
      pdc.outf("%d %d %d\n",width,height,8);
      pdc.outf("[%d 0 0 -%d 0 %d]\n",width,height,height);
      pdc.outf("{currentfile picstr readhexstring pop}\n");
      pdc.outf("false %d\n",3);
      pdc.outf("colorimage\n");

      for(i=0; i<width*height; i++){
        red=buffer[3*i];
        green=buffer[3*i+1];
        blue=buffer[3*i+2];
        pdc.outhex(red);
        pdc.outhex(green);
        pdc.outhex(blue);
        }
      pdc.outf("\n");

      FXFREE(&buffer);
      }
  
    // Page trailer
    pdc.endPage();
    pdc.endPrint();
    }
  return 1;
  }


// Render 
FXint FXGLViewer::renderFeedback(FXfloat *buffer,FXint x,FXint y,FXint w,FXint h,FXint maxbuffer){
#ifdef HAVE_OPENGL  
  FXint used;
  makeCurrent();
  glFeedbackBuffer(maxbuffer,GL_3D_COLOR,buffer);
  glRenderMode(GL_FEEDBACK);
  drawWorld(wvt);
  used=glRenderMode(GL_RENDER);
  makeNonCurrent();
  return used;
#else
  return -1;
#endif
  }


// Read feedback buffer
FXbool FXGLViewer::readFeedback(FXfloat*& buffer,FXint& used,FXint& size,FXint x,FXint y,FXint w,FXint h){
  FXbool ok=FALSE;  
  buffer=NULL;
  used=0;
  size=10000;
  while(1){
    
    // Allocate buffer
    FXMALLOC(&buffer,FXfloat,size);
    
    // It got too big, give up
    if(!buffer) break;
    
    // Try to render scene into it
    used=renderFeedback(buffer,x,y,w,h,size);
    
    // No errors, got our stuff
    if(0<used){
      ok=TRUE;
      break;
      }
    
    // It didn't fit, lets double the buffer and try again
    FXFREE(&buffer);
    size*=2;
    continue;
    }
  return ok;
  }


// Draw feedback buffer into dc
void FXGLViewer::drawFeedback(FXDCPrint& pdc,const FXfloat* buffer,FXint used){
#ifdef HAVE_OPENGL
  FXint nvertices,smooth,token,i,p;
  
  // Draw background
  pdc.outf("%g %g %g C\n",background[0],background[1],background[2]);
  pdc.outf("newpath\n");
  pdc.outf("%g %g moveto\n",0.0,0.0);
  pdc.outf("%g %g lineto\n",0.0,(double)height);
  pdc.outf("%g %g lineto\n",(double)width,(double)height);
  pdc.outf("%g %g lineto\n",(double)width,0.0);
  pdc.outf("closepath fill\n");
  
  pdc.outf("1 setlinewidth\n");

  // Crank out primitives
  p=0;
  while(p<used){
    token=(FXint)buffer[p++];
    switch(token){
      
      // Point primitive
      case GL_POINT_TOKEN:
        pdc.outf("%g %g %g %g %g P\n",buffer[p+0],buffer[p+1],buffer[p+3],buffer[p+4],buffer[p+5]);
        p+=7;             // Each vertex element in the feedback buffer is 7 floats
        break;
        
      // Line primitive
      case GL_LINE_RESET_TOKEN:
      case GL_LINE_TOKEN:
        if(fabs(buffer[p+3]-buffer[p+7+3])<1E-4 || fabs(buffer[p+4]-buffer[p+7+4])<1E-4 || fabs(buffer[p+5]-buffer[p+7+5])<1E-4){
          pdc.outf("%g %g %g %g %g %g %g %g %g %g SL\n",buffer[p+0],buffer[p+1],buffer[p+3],buffer[p+4],buffer[p+5], buffer[p+7+0],buffer[p+7+1],buffer[p+7+3],buffer[p+7+4],buffer[p+7+5]);
          }
        else{
          pdc.outf("%g %g %g %g %g %g %g L\n",buffer[p+0],buffer[p+1],buffer[p+7+0],buffer[p+7+1],buffer[p+3],buffer[p+4],buffer[p+5]);
          }
        p+=14;            // Each vertex element in the feedback buffer is 7 GLfloats
        break;
        
      // Polygon primitive
      case GL_POLYGON_TOKEN:
        nvertices = (FXint)buffer[p++];
        if(nvertices==3){ // We assume polybusting has taken place already!
          smooth=0;
          for(i=1; i<nvertices; i++){
            if(fabs(buffer[p+3]-buffer[p+i*7+3])<1E-4 || fabs(buffer[p+4]-buffer[p+i*7+4])<1E-4 || fabs(buffer[p+5]-buffer[p+i*7+5])<1E-4){ smooth=1; break; }
            }
          if(smooth){
            pdc.outf("%g %g %g %g %g %g %g %g %g %g %g %g %g %g %g ST\n",buffer[p+0],buffer[p+1],buffer[p+3],buffer[p+4],buffer[p+5], buffer[p+7+0],buffer[p+7+1],buffer[p+7+3],buffer[p+7+4],buffer[p+7+5], buffer[p+14+0],buffer[p+14+1],buffer[p+14+3],buffer[p+14+4],buffer[p+14+5]);
            }
          else{
            pdc.outf("%g %g %g %g %g %g %g %g %g T\n",buffer[p+0],buffer[p+1], buffer[p+7+0],buffer[p+7+1], buffer[p+14+0],buffer[p+14+1], buffer[p+3],buffer[p+4],buffer[p+5]);
            }
          }
        p+=nvertices*7;   // Each vertex element in the feedback buffer is 7 GLfloats
        break;
        
      // Skip these, don't deal with it here
      case GL_BITMAP_TOKEN:
      case GL_DRAW_PIXEL_TOKEN:
      case GL_COPY_PIXEL_TOKEN:
        p+=7;
        break;
        
      // Skip passthrough tokens
      case GL_PASS_THROUGH_TOKEN:
        p++;
        break;
        
      // Bad token, this is the end
      default:
        return;
      }
    }
#endif
  }





// Print the window by means of feedback buffer
long FXGLViewer::onCmdPrintVector(FXObject*,FXSelector,void*){
  FXPrintDialog dlg(this,"Print Scene");
  FXPrinter printer;
  FXfloat *buffer;
  FXint used,size;
  
  // Run dialog
  if(dlg.execute()){
    dlg.getPrinter(printer);
    FXDCPrint pdc(getApp());
    if(!pdc.beginPrint(printer)){
      FXMessageBox::error(this,MBOX_OK,"Printer Error","Unable to print");
      return 1;
      }
    
    // Page header
    pdc.beginPage(1);
    
    // Read feedback
    if(readFeedback(buffer,used,size,0,0,width,height)){
      if(zsortfunc) (*zsortfunc)(buffer,used,size);
      drawFeedback(pdc,buffer,used);
      }
  
    // Page trailer
    pdc.endPage();
    pdc.endPrint();
    }
  return 1;
  }

/*****************************  Selection Support  *****************************/


// Cut selected object
long FXGLViewer::onCmdCutSel(FXObject*,FXSelector,void*){
  // Serialize object into temp buffer
  // Delete object, tell target it was deleted
  fxwarning("%s::onCmdCutSel: unimplemented.\n",getClassName());
  return 1;
  }


// Copy selected object
long FXGLViewer::onCmdCopySel(FXObject*,FXSelector,void*){
  // Serialize object into buffer
  fxwarning("%s::onCmdCopySel: unimplemented.\n",getClassName());
  return 1;
  }


// Paste object
long FXGLViewer::onCmdPasteSel(FXObject*,FXSelector,void*){
  // Ask clipboard for object data [What type?]
  // Deserialize data [type?]
  // Tell target about the data?
  fxwarning("%s::onCmdPasteSel: unimplemented.\n",getClassName());
  return 1;
  }


// Delete selected object
long FXGLViewer::onCmdDeleteSel(FXObject*,FXSelector,void*){
  FXGLObject *obj[2];
  obj[0]=selection;
  obj[1]=NULL;
  if(obj[0] && obj[0]->canDelete()){
    handle(this,MKUINT(0,SEL_CHANGED),NULL);
    handle(this,MKUINT(0,SEL_DELETED),(void*)obj);
    //delete obj[0];
    }
  else{
    getApp()->beep();
    }
  return 1;
  }


// Update delete object
long FXGLViewer::onUpdDeleteSel(FXObject* sender,FXSelector,void*){
  if(selection && selection->canDelete()){
    sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
    return 1;
    }
  return 0;
  }


// Update for current object
long FXGLViewer::onUpdCurrent(FXObject* sender,FXSelector,void*){
  if(selection){
    sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
    return 1;
    }
  return 0;
  }


// Set background color
long FXGLViewer::onCmdBackColor(FXObject*,FXSelector sel,void* ptr){
  background[0]=0.003921568627f*FXREDVAL((long)ptr);
  background[1]=0.003921568627f*FXGREENVAL((long)ptr);
  background[2]=0.003921568627f*FXBLUEVAL((long)ptr);
  background[3]=1.0;
  if(SELTYPE(sel)==SEL_COMMAND || !turbomode){
    update();
    }
  return 1;
  }


// Update background color
long FXGLViewer::onUpdBackColor(FXObject* sender,FXSelector,void*){
  FXColor clr=FXRGBA((255.0*background[0]),(255.0*background[1]),(255.0*background[2]),(255.0*background[3]));
  sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)clr);
  return 1;
  }


// Set ambient light color
long FXGLViewer::onCmdAmbientColor(FXObject*,FXSelector sel,void* ptr){
  ambient[0]=0.003921568627f*FXREDVAL((long)ptr);
  ambient[1]=0.003921568627f*FXGREENVAL((long)ptr);
  ambient[2]=0.003921568627f*FXBLUEVAL((long)ptr);
  ambient[3]=1.0;
  if(SELTYPE(sel)==SEL_COMMAND || !turbomode){
    update();
    }
  return 1;
  }


// Update ambient light color
long FXGLViewer::onUpdAmbientColor(FXObject* sender,FXSelector,void*){
  FXColor clr=FXRGBA((255.0*ambient[0]),(255.0*ambient[1]),(255.0*ambient[2]),(255.0*ambient[3]));
  sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)clr);
  return 1;
  }


// Set ambient light color
long FXGLViewer::onCmdLightAmbient(FXObject*,FXSelector sel,void* ptr){
  light.ambient[0]=0.003921568627f*FXREDVAL((long)ptr);
  light.ambient[1]=0.003921568627f*FXGREENVAL((long)ptr);
  light.ambient[2]=0.003921568627f*FXBLUEVAL((long)ptr);
  light.ambient[3]=0.003921568627f*FXALPHAVAL((long)ptr);
  if(SELTYPE(sel)==SEL_COMMAND || !turbomode){
    update();
    }
  return 1;
  }


// Update ambient light color
long FXGLViewer::onUpdLightAmbient(FXObject* sender,FXSelector,void*){
  FXColor clr=FXRGBA((255.0*light.ambient[0]),
                     (255.0*light.ambient[1]),
                     (255.0*light.ambient[2]),
                     (255.0*light.ambient[3]));
  sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)clr);
  return 1;
  }


// Set diffuse light color
long FXGLViewer::onCmdLightDiffuse(FXObject*,FXSelector sel,void* ptr){
  light.diffuse[0]=0.003921568627f*FXREDVAL((long)ptr);
  light.diffuse[1]=0.003921568627f*FXGREENVAL((long)ptr);
  light.diffuse[2]=0.003921568627f*FXBLUEVAL((long)ptr);
  light.diffuse[3]=0.003921568627f*FXALPHAVAL((long)ptr);
  if(SELTYPE(sel)==SEL_COMMAND || !turbomode){
    update();
    }
  return 1;
  }


// Update diffuse light color
long FXGLViewer::onUpdLightDiffuse(FXObject* sender,FXSelector,void*){
  FXColor clr=FXRGBA((255.0*light.diffuse[0]),
                     (255.0*light.diffuse[1]),
                     (255.0*light.diffuse[2]),
                     (255.0*light.diffuse[3]));
  sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)clr);
  return 1;
  }


// Set specular light color
long FXGLViewer::onCmdLightSpecular(FXObject*,FXSelector sel,void* ptr){
  light.specular[0]=0.003921568627f*FXREDVAL((long)ptr);
  light.specular[1]=0.003921568627f*FXGREENVAL((long)ptr);
  light.specular[2]=0.003921568627f*FXBLUEVAL((long)ptr);
  light.specular[3]=0.003921568627f*FXALPHAVAL((long)ptr);
  if(SELTYPE(sel)==SEL_COMMAND || !turbomode){
    update();
    }
  return 1;
  }


// Update specular light color
long FXGLViewer::onUpdLightSpecular(FXObject* sender,FXSelector,void*){
  FXColor clr=FXRGBA((255.0*light.specular[0]),
                     (255.0*light.specular[1]),
                     (255.0*light.specular[2]),
                     (255.0*light.specular[3]));
  sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)clr);
  return 1;
  }


// Toggle Lock view
long FXGLViewer::onCmdLock(FXObject*,FXSelector,void*){
  setViewLock(!getViewLock());
  return 1;
  }



// Update lock view
long FXGLViewer::onUpdLock(FXObject* sender,FXSelector,void*){
  FXuint msg=getViewLock() ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Toggle Turbo Mode
long FXGLViewer::onCmdTurbo(FXObject*,FXSelector,void*){
  setTurboMode(!getTurboMode());
  return 1;
  }



// Update Turbo Mode
long FXGLViewer::onUpdTurbo(FXObject* sender,FXSelector,void*){
  FXuint msg=getTurboMode() ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Toggle lighting
long FXGLViewer::onCmdLighting(FXObject*,FXSelector,void*){
  options^=VIEWER_LIGHTING;
  update();
  return 1;
  }


// Update lighting
long FXGLViewer::onUpdLighting(FXObject* sender,FXSelector,void*){
  FXuint msg=(options&VIEWER_LIGHTING) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Toggle fog
long FXGLViewer::onCmdFog(FXObject*,FXSelector,void*){
  options^=VIEWER_FOG;
  update();
  return 1;
  }


// Update fog
long FXGLViewer::onUpdFog(FXObject* sender,FXSelector,void*){
  FXuint msg=(options&VIEWER_FOG) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Toggle dithering
long FXGLViewer::onCmdDither(FXObject*,FXSelector,void*){
  options^=VIEWER_DITHER;
  update();
  return 1;
  }


// Update dithering
long FXGLViewer::onUpdDither(FXObject* sender,FXSelector,void*){
  FXuint msg=(options&VIEWER_DITHER) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


/*******************************  Drag and Drop  *******************************/


// Handle drag-and-drop enter
long FXGLViewer::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr){
  if(FXGLCanvas::onDNDEnter(sender,sel,ptr)) return 1;
  dropped=NULL;
  return 1;
  }

// Handle drag-and-drop leave
long FXGLViewer::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr){
  if(FXGLCanvas::onDNDLeave(sender,sel,ptr)) return 1;
  dropped=NULL;
  return 1;
  }


// Handle drag-and-drop motion
long FXGLViewer::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  
  // Handled elsewhere
  if(FXGLCanvas::onDNDMotion(sender,sel,ptr)) return 1;
  
  // Dropped on some object
  if((dropped=pick(event->win_x,event->win_y))!=NULL){
    
    // Object agrees with drop type
    if(dropped->handle(this,sel,ptr)){
      acceptDrop(DRAG_COPY);
      return 1;
      }
    
    // Forget about the whole thing
    dropped=NULL;
    return 0;
    }
  
  // Dropped in viewer background; hope its a color
  if(offeredDNDType(FROM_DRAGNDROP,colorType)){
    acceptDrop(DRAG_COPY);
    return 1;
    }
  
  // Won't accept drop, dont know what it is
  return 0;
  }


// Handle drag-and-drop drop
long FXGLViewer::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr){
  FXuint len,r,g,b,a;
  FXuchar *data; 
  
  // Try base class first
  if(FXGLCanvas::onDNDDrop(sender,sel,ptr)) return 1;
  
  // Dropped on object?
  if(dropped){
    
    // Object handled drop; so probably want to repaint
    if(dropped->handle(this,sel,ptr)){
      update();
      return 1;
      }
    
    // We're done
    return 0;
    }

  // Dropped on viewer
  if(offeredDNDType(FROM_DRAGNDROP,colorType)){
    if(getDNDData(FROM_DRAGNDROP,FXGLViewer::colorType,data,len)){
      sscanf((char*)data,"#%02x%02x%02x%02x",&r,&g,&b,&a);
      background[0]=r*0.003921568627f;
      background[1]=g*0.003921568627f;
      background[2]=b*0.003921568627f;
      FXFREE(&data);
      update();
      return 1;
      }
    }
  return 0;
  }


// Change projection
void FXGLViewer::setProjection(FXuint proj){
  projection=proj;
  updateProjection();
  update();
  }


// Lock viewpoint
void FXGLViewer::setViewLock(FXbool lock){
  options=lock?(options|VIEWER_LOCKED):(options&~VIEWER_LOCKED);
  }


// Get locked
FXbool FXGLViewer::getViewLock() const {
  return (options&VIEWER_LOCKED)!=0;
  }


// Set background
void FXGLViewer::setBackgroundColor(const FXHVec& clr){
  background=clr;
  update();
  }


// Set ambient color
void FXGLViewer::setAmbientColor(const FXHVec& clr){
  ambient=clr;
  update();
  }


// Delegate some messages to the GL Object
long FXGLViewer::handle(FXObject* sender,FXSelector key,void* data){
  FXSelFunction func;
  
  // Filter out messages for the GL Viewer, regardless of return value
  if((func=assoc(key))!=FXObject::null){ return (this->*func)(sender,key,data); }
  
  // Unknown messages are passed to the selected GL Object
  return selection && selection->handle(sender,key,data);
  }


// Change surface/lines offset
void FXGLViewer::setOffset(FXdouble offs){
  if(offset!=offs){
    offset=offs;
    update();
    }
  }


// Change turbo mode  
void FXGLViewer::setTurboMode(FXbool turbo){
  if(!turbo) doesturbo=FALSE;
  turbomode=turbo;
  }


// Return light settings
void FXGLViewer::getLight(FXLight& lite) const {
  lite=light;
  }


// Change light settings
void FXGLViewer::setLight(const FXLight& lite) {
  light=lite;
  update();
  }


// Save object to stream
void FXGLViewer::save(FXStream& store) const {
  FXGLCanvas::save(store);
  store << wvt.w;
  store << wvt.h;
  store << wvt.left;
  store << wvt.right;
  store << wvt.bottom;
  store << wvt.top;
  store << wvt.hither;
  store << wvt.yon;
  store << transform;
  store << itransform;
  store << screenmin;
  store << screenmax;
  store << projection;
  store << rotation;
  store << aspect;
  store << fov;
  store << zoom;
  store << offset;
  store << center;
  store << scale;
  store << worldpx;
  store << modelpx;
  store << diameter;
  store << distance;
  store << background;
  store << ambient;
  store << help;
  }

      

// Load object from stream
void FXGLViewer::load(FXStream& store){
  FXGLCanvas::load(store);
  store >> wvt.w;
  store >> wvt.h;
  store >> wvt.left;
  store >> wvt.right;
  store >> wvt.bottom;
  store >> wvt.top;
  store >> wvt.hither;
  store >> wvt.yon;
  store >> transform;
  store >> itransform;
  store >> screenmin;
  store >> screenmax;
  store >> projection;
  store >> rotation;
  store >> aspect;
  store >> fov;
  store >> zoom;
  store >> offset;
  store >> center;
  store >> scale;
  store >> worldpx;
  store >> modelpx;
  store >> diameter;
  store >> distance;
  store >> background;
  store >> ambient;
  store >> help;
  }  


// Close and release any resources
FXGLViewer::~FXGLViewer(){
  if(timer) getApp()->removeTimeout(timer);
  timer=(FXTimer*)-1;
  dropped=(FXGLObject*)-1;
  selection=(FXGLObject*)-1;
  scene=(FXGLObject*)-1;
  }
