;;; "build.scm" Build database and program	-*-scheme-*-
;;; Copyright (C) 1994-1999 Aubrey Jaffer.
;;; See the file `COPYING' for terms applying to this program.

(require 'parameters)
(require 'database-utilities)
(set! OPEN_WRITE "w")			; Because MS-DOS scripts need ^M

;;;(define build (create-database "buildscm.scm" 'alist-table))
(define build (create-database #f 'alist-table))

(require 'glob)
(require 'batch)
(batch:initialize! build)
((((build 'open-table) 'batch-dialect #t) 'row:insert)
 '(default-for-platform 0))

;;;; This first part is about SCM files and features.

(define-tables build

  '(file-formats
    ((format symbol))
    ()
    ((plaintext)
     (c-source)
     (c-header)
     (scheme)
     (vax-asm)
     (gnu-as)
     (gdb-init)
     (cray-asm)
     (makefile)
     (MS-DOS-batch)
     (nroff)
     (texinfo)))

  '(file-categories
    ((category symbol))
    ((documentation string))
    ((documentation "documentation")
     (platform-specific "required for certain platforms")
     (required "required for building executable SCM")
     (optional "required for some feature")
     (linkable "required and can be dynamically linked")
     (test "test SCM")
     (none "no files")))

  '(manifest
    ((file string)
     (format file-formats)
     (category file-categories))
    ((documentation string))
    (("README"	plaintext	documentation	"contains a MANIFEST, INSTALLATION INSTRUCTIONS, hints for EDITING SCHEME CODE, and a TROUBLE SHOOTING GUIDE.")
     ("COPYING"	plaintext	documentation	"details the LACK OF WARRANTY for SCM and the conditions for distributing SCM.")
     ("scm.1"	nroff	documentation	"unix style man page.")
     ("scm.doc"	plaintext	documentation	"man page generated from scm.1.")
     ("QUICKREF"	plaintext	documentation	"Quick Reference card for R4RS and IEEE Scheme.")
     ("scm.texi"	Texinfo	documentation	"SCM installation and use.")
     ("ChangeLog"	plaintext	documentation	"changes to SCM.")
     ("r4rstest.scm"	Scheme	test	"tests conformance with Scheme specifications.")
     ("example.scm"	Scheme	test	"example from R4RS which uses inexact numbers.")
     ("pi.scm"	Scheme	test	"computes digits of pi [type (pi 100 5)].  Test performance against pi.c.")
     ("pi.c"	c-source	test	"computes digits of pi [cc -o pi pi.c;time pi 100 5].")
     ("bench.scm"	Scheme	test	"computes and records performance statistics of pi.scm.")
     ("Makefile"	Makefile	required	"builds SCMLIT using the `make' program.")
     ("build.scm"	Scheme	required	"database for compiling and linking new SCM programs.")
     ("build.bat"	MS-DOS-batch	platform-specific	"invokes build.scm for MS-DOS")
     ("mkimpcat.scm"	Scheme	required	"build SCM-specific catalog for SLIB.")
     (".gdbinit"	gdb-init	optional "provides commands for debugging SCM with GDB")
     ("setjump.mar"	Vax-asm	platform-specific	"provides setjump and longjump which do not use $unwind utility on VMS.")
     ("ugsetjump.s"	gnu-as	platform-specific	"provides setjump and longjump which work on Ultrix VAX.")
     ("setjump.s"	Cray-asm	platform-specific	"provides setjump and longjump for the Cray YMP.")
     ("Init.scm"	Scheme	required	"Scheme initialization.")
     ("Transcen.scm"	Scheme	required	"inexact builtin procedures.")
     ("Link.scm"	Scheme	required	"compiles and dynamically links.")
     ("Macro.scm"	Scheme	required	"Supports Syntax-Rules Macros.")
     ("scmfig.h"	c-header	required	"contains system dependent definitions.")
     ("patchlvl.h"	c-header	required	"patchlevel of this release.")
     ("setjump.h"	c-header	required	"continuations, stacks, and memory allocation.")
     ("continue.h"	c-header	required	"continuations.")
     ("continue.c"	c-source	required	"continuations.")
     ("scm.h"	c-header	required	"data type and external definitions of SCM.")
     ("scm.c"	c-source	required	"initialization, interrupts, and non-IEEE utility functions.")
     ("scmmain.c"	c-source	required	"initialization, interrupts, and non-IEEE utility functions.")
     ("findexec.c"	c-source	required	"find the executable file function.")
     ("script.c"	c-source	required	"utilities for running as `#!' script.")
     ("time.c"	c-source	required	"functions dealing with time.")
     ("repl.c"	c-source	required	"error, read-eval-print loop, read, write and load.")
     ("scl.c"	c-source	required	"inexact arithmetic")
     ("eval.c"	c-source	required	"evaluator, apply, map, and foreach.")
     ("sys.c"	c-source	required	"call-with-current-continuation, opening and closing files, storage allocation and garbage collection.")
     ("subr.c"	c-source	required	"the rest of IEEE functions.")
     ("unif.c"	c-source	required	"uniform vectors.")
     ("rope.c"	c-source	required	"C interface functions.")
     ("ramap.c"	c-source	optional	"array mapping")
     ("dynl.c"	c-source	optional	"dynamically load object files.")
     ("sc2.c"	c-source	linkable	"procedures from R2RS and R3RS not in R4RS.")
     ("rgx.c"	c-source	linkable	"string regular expression match.")
     ("crs.c"	c-source	linkable	"interactive terminal control.")
     ("split.scm"	Scheme	test	"example use of crs.c.  Input, output, and diagnostic output directed to separate windows.")
     ("edline.c"	c-source	linkable	"Gnu readline input editing (get ftp.sys.toronto.edu:/pub/rc/editline.shar).")
     ("Iedline.scm"	Scheme	optional	"Gnu readline input editing.")
     ("record.c"	c-source	linkable	"proposed `Record' user definable datatypes.")
     ("gsubr.c"	c-source	linkable	"make_gsubr for arbitrary (< 11) arguments to C functions.")
     ("ioext.c"	c-source	linkable	"system calls in common between PC compilers and unix.")
     ("posix.c"	c-source	linkable	"posix library interface.")
     ("unix.c"	c-source	linkable	"non-posix system calls on unix systems.")
     ("socket.c"	c-source	linkable	"BSD socket interface.")
     ("pre-crt0.c"	c-source	platform-specific	"loaded before crt0.o on machines which do not remap part of the data space into text space in unexec.")
     ("ecrt0.c"	c-source	platform-specific	"discover the start of initialized data space dynamically at runtime.")
     ("gmalloc.c"	c-source	platform-specific	"Gnu malloc(); used for unexec.")
     ("unexec.c"	c-source	platform-specific	"Convert a running program into an executable file.")
     ("unexhp9k800.c"	c-source	platform-specific	"Convert a running HP-UX program into an executable file.")
     ("unexelf.c"	c-source	platform-specific	"Convert a running ELF program into an executable file.")
     ("unexalpha.c"	c-source	platform-specific	"Convert a running program into an Alpha executable file.")
     ("unexsgi.c"	c-source	platform-specific	"Convert a running program into an IRIX executable file.")
     ("unexsunos4.c"	c-source	platform-specific	"Convert a running program into an executable file.")
     ))

  '(build-whats
    ((name symbol))
    ((class file-categories)
     (c-proc symbol)
     (o-proc symbol)
     (spec expression)
     (documentation string))
    ((exe required compile-c-files link-c-program #f
	  "executable program")
     (lib required compile-c-files make-archive ((c-lib lib))
	  "library module")
     (dlls linkable compile-dll-c-files make-dll-archive ((define "DLL"))
	   "archived dynamically linked library object files")
     (dll none compile-dll-c-files update-catalog ((define "DLL"))
	  "dynamically linked library object file")))

  '(features
    ((name symbol))
    ((spec expression)
     (documentation string))
    ((none () "No features"))))

(for-each (build 'add-domain)
	  '((optstring #f (lambda (x) (or (not x) (string? x))) string #f)
	    (filename #f #f string #f)
	    (features features #f symbol #f)
	    (build-whats build-whats #f symbol #f)))

(define define-build-feature
  (let ((defeature (((build 'open-table) 'features #t) 'row:insert)))
    (lambda args
      (defeature (append args (list (comment)))))))

#;Lightweight -- no features
(define-build-feature
  'lit
  '())

#;Normally, the number of arguments arguments to interpreted closures
#;(from LAMBDA) are checked if the function part of a form is not a
#;symbol or only the first time the form is executed if the function
#;part is a symbol.  defining @samp{reckless} disables any checking.
#;If you want to have SCM always check the number of arguments to
#;interpreted closures define feature @samp{cautious}.
(define-build-feature
 'cautious
 '((define "CAUTIOUS")))

#;Define this for extra checking of interrupt masking and some simple
#;checks for proper use of malloc and free.  This is for debugging C
#;code in @file{sys.c}, @file{eval.c}, @file{repl.c} and makes the
#;interpreter several times slower than usual.
(define-build-feature
 'careful-interrupt-masking
 '((define "CAREFUL_INTS")))

#;Turns on the features @samp{cautious},
#;@samp{careful-interrupt-masking}, and @samp{stack-limit}; uses
#;@code{-g} flags for debugging SCM source code.
(define-build-feature
 'debug
 '((c-lib debug) (features cautious careful-interrupt-masking stack-limit)))

#;If your scheme code runs without any errors you can disable almost
#;all error checking by compiling all files with @samp{reckless}.
(define-build-feature
 'reckless
 '((define "RECKLESS")))

#;Use to enable checking for stack overflow.  Define value of the C
#;preprocessor variable @var{STACK_LIMIT} to be the size to which SCM
#;should allow the stack to grow.  STACK_LIMIT should be less than the
#;maximum size the hardware can support, as not every routine checks the
#;stack.
(define-build-feature
 'stack-limit
 '((define ("STACK_LIMIT" "(HEAP_SEG_SIZE/2)"))))

#;C level support for hygienic and referentially transparent macros
#;(syntax-rules macros).
(define-build-feature
 'macro
 '((define "MACRO") (features rev2-procedures record)))

#;Large precision integers.
(define-build-feature
 'bignums
 '((define "BIGNUMS")))

#;Use if you want arrays, uniform-arrays and uniform-vectors.
(define-build-feature
 'arrays
 '((define "ARRAYS")))

#;Alias for ARRAYS
(define-build-feature
 'array
 '((define "ARRAYS")))

#;array-map! and array-for-each (arrays must also be featured).
(define-build-feature
 'array-for-each
 '((c-file "ramap.c") (init "init_ramap")))

#;Use if you want floating point numbers.
(define-build-feature
 'inexact
 '((define "FLOATS") (c-lib m)))

#;Use if you want floats to display in engineering notation (exponents
#;always multiples of 3) instead of scientific notation.
(define-build-feature
 'engineering-notation
 '((define "ENGNOT")))

#;Use if you want all inexact real numbers to be single precision.  This
#;only has an effect if SINGLES is also defined (which is the default).
#;This does not affect complex numbers.
(define-build-feature
 'single-precision-only
 '((define "SINGLESONLY")))

#;Use if you want to run code from:
#;
#;@cindex SICP
#;Harold Abelson and Gerald Jay Sussman with Julie Sussman.
#;@cite{Structure and Interpretation of Computer Programs.}
#;The MIT Press, Cambridge, Massachusetts, USA, 1985.
#;
#;Differences from R5RS are:
#;@itemize @bullet
#;@item
#;(eq? '() '#f)
#;@item
#;(define a 25) returns the symbol a.
#;@item
#;(set! a 36) returns 36.
#;@end itemize
(define-build-feature
 'sicp
 '((define "SICP")))

#;These procedures were specified in the @cite{Revised^2 Report on Scheme}
#;but not in @cite{R4RS}.
(define-build-feature
 'rev2-procedures
 '((c-file "sc2.c") (init "init_sc2")))

#;The Record package provides a facility for user to define their own
#;record data types.  See SLIB for documentation.
(define-build-feature
 'record
 '((define "CCLO") (c-file "record.c") (init "init_record")))

#;Use if you want to use compiled closures.
(define-build-feature
 'compiled-closure
 '((define "CCLO")))

#;@code{make_gsubr} for arbitrary (< 11) arguments to C functions.
(define-build-feature
 'generalized-c-arguments
 '((c-file "gsubr.c") (init "init_gsubr")))

#;Use if you want the ticks and ticks-interrupt functions.
(define-build-feature
 'tick-interrupts
 '((define "TICKS")))

#;Commonly available I/O extensions: @dfn{exec}, line I/O, file
#;positioning, file delete and rename, and directory functions.
(define-build-feature
 'i/o-extensions
 '((c-file "ioext.c") (init "init_ioext")))

#;@dfn{Turtle} graphics calls for both Borland-C and X11 from
#;sjm@@ee.tut.fi.
(define-build-feature
 'turtlegr
 '((c-file "turtlegr.c") (c-lib graphics) (features inexact)
   (init "init_turtlegr")))

#;Interface to Xlib graphics routines.
(define-build-feature
 'Xlib
 '((c-file "x.c") (c-lib graphics) (compiled-init "init_x") (features arrays)))

#;Alias for Xlib feature.
(define-build-feature
 'X
 '((features Xlib)))

#;For the @dfn{curses} screen management package.
(define-build-feature
 'curses
 '((c-file "crs.c") (c-lib curses) (init "init_crs")))

#;interface to the editline or GNU readline library.
(define-build-feature
 'edit-line
 '((c-file "edline.c") (c-lib termcap editline) (compiled-init "init_edline")))

#;Client connections to the mysql databases.
(define-build-feature
 'mysql
 '((c-file "database.c") (c-lib mysql) (init "init_database")))

#;String regular expression matching.
(define-build-feature
 'regex
 '((c-file "rgx.c") (c-lib regex) (init "init_rgx")))

#;BSD @dfn{socket} interface.
(define-build-feature
 'socket
 '((c-lib socket) (c-file "socket.c") (init "init_socket")))

#;Posix functions available on all @dfn{Unix-like} systems.  fork and
#;process functions, user and group IDs, file permissions, and
#;@dfn{link}.
(define-build-feature
 'posix
 '((c-file "posix.c") (init "init_posix")))

#;Those unix features which have not made it into the Posix specs:
#;nice, acct, lstat, readlink, symlink, mknod and sync.
(define-build-feature
 'unix
 '((c-file "unix.c") (init "init_unix")))

#;Microsoft Windows executable.
(define-build-feature
 'windows
 '((c-lib windows)))			; (define "NON_PREEMPTIVE")

#;Be able to load compiled files while running.
(define-build-feature
 'dynamic-linking
 '((c-file "dynl.c") (c-lib dlll)))

#;Convert a running scheme program into an executable file.
(define-build-feature
 'dump
 '((define "CAN_DUMP") (c-lib dump) (c-lib nostart)))

;;; Descriptions of these parameters is in "setjump.h".
;;	(initial-heap-size ((define "INIT_HEAP_SIZE" (* 25000 sizeof-cell))))
;;	(heap-segment-size ((define "HEAP_SEG_SIZE" (* 8100 sizeof-cell))))
;;	(short-aligned-stack ((define "SHORT_ALIGN")))
;;	(initial-malloc-limit ((define "INIT_MALLOC_LIMIT" 100000)))
;;	(number-of-hash-buckets ((define "NUM_HASH_BUCKETS" 137)))
;;	(minimum-gc-yield ((define "MIN_GC_YIELD" "(heap_cells/4)")))

#;Use if you want segments of unused heap to not be freed up after
#;garbage collection.  This may increase time in GC for *very* large
#;working sets.
(define-build-feature
 'no-heap-shrink
 '((define "DONT_GC_FREE_SEGMENTS")))

#;If you only need straight stack continuations, executables compile with
#;this feature will run faster and use less storage than not having it.
#;Machines with unusual stacks @emph{need} this.  Also, if you incorporate
#;new C code into scm which uses VMS system services or library routines
#;(which need to unwind the stack in an ordrly manner) you may need to
#;use this feature.
(define-build-feature
 'cheap-continuations
 '((define "CHEAP_CONTINUATIONS")))


;;;; The rest is about building on specific platforms.

(define-tables build

  '(processor-family
    ((family atom))
    ((also-runs processor-family))
    ((*unknown* #f)
     (8086 #f)
     (acorn #f)
     (alpha #f)
     (cray #f)
     (hp-risc #f)
     (i386 8086)
     (m68000 #f)
     (m68030 m68000)
     (mips #f)
     (nos/ve #f)
     (pdp-10 #f)
     (pdp-11 #f)
     (pdp-8 #f)
     (powerpc #f)
     (pyramid #f)
     (sequent #f)
     (sparc #f)
     (tahoe #f)
     (vax pdp-11)
     ))

  '(platform
    ((name symbol))
    ((processor processor-family)
     (operating-system operating-system)
     (compiler symbol)
     ;;(linker symbol)
     )
    ((*unknown*		 *unknown* unix	     cc	       ) ;ld
     (acorn-unixlib	 acorn	   *unknown* cc	       ) ;link
     (aix		 powerpc   aix	     cc	       ) ;cc
     (alpha		 alpha	   osf1	     cc	       ) ;cc
     (alpha-elf		 alpha	   unix	     cc	       ) ;cc
     (alpha-linux	 alpha	   linux     gcc       ) ;gcc
     (amiga-aztec	 m68000	   amiga     cc	       ) ;cc
     (amiga-dice-c	 m68000	   amiga     dcc       ) ;dcc
     (amiga-gcc		 m68000	   amiga     gcc       ) ;gcc
     (amiga-sas		 m68000	   amiga     lc	       ) ;link
     (atari-st-gcc	 m68000	   atari.st  gcc       ) ;gcc
     (atari-st-turbo-c	 m68000	   atari.st  tcc       ) ;tlink
     (borland-c		 8086	   ms-dos    bcc       ) ;bcc
     (cygwin32		 i386	   unix	     gcc       ) ;gcc
     (djgpp		 i386	   ms-dos    gcc       ) ;gcc
     (freebsd		 i386	   unix	     cc	       ) ;cc
     (gcc		 *unknown* unix	     gcc       ) ;gcc
     (highc		 i386	   ms-dos    hc386     ) ;bind386
     (hp-ux		 hp-risc   hp-ux     cc	       ) ;cc
     (irix		 mips	   irix	     gcc       ) ;gcc
     (linux		 i386	   linux     gcc       ) ;gcc
     (linux-aout	 i386	   linux     gcc       ) ;gcc
     (microsoft-c	 8086	   ms-dos    cl	       ) ;link
     (microsoft-c-nt	 i386	   ms-dos    cl	       ) ;link
     (microsoft-quick-c	 8086	   ms-dos    qcl       ) ;qlink
     (ms-dos		 8086	   ms-dos    cc	       ) ;link
     (os/2-cset		 i386	   os/2	     icc       ) ;link386
     (os/2-emx		 i386	   os/2	     gcc       ) ;gcc
     (svr4-gcc-sun-ld	 sparc	   sunos     gcc       ) ;ld
     (sunos		 sparc	   sunos     cc	       ) ;ld
     (svr4		 *unknown* unix	     cc	       ) ;ld
     (turbo-c		 8086	   ms-dos    tcc       ) ;tcc
     (unicos		 cray	   unicos    cc	       ) ;cc
     (unix		 *unknown* unix	     cc	       ) ;cc
     (vms		 vax	   vms	     cc	       ) ;link
     (vms-gcc		 vax	   vms	     gcc       ) ;link
     (watcom-9.0	 i386	   ms-dos    wcc386p   ) ;wlinkp
     ))

  '(C-libraries
    ((library symbol)
     (platform platform))
    ((compiler-flags string)
     (link-lib-flag string)
     (lib-path optstring)
     (lib-support expression)
     (suppress-files expression))

    ((m *unknown* "" "-lm" "/usr/lib/libm.a" () ())
     (c *unknown* "" "-lc" "/usr/lib/libc.a" () ())
     (regex *unknown* "" "-lregex" "/usr/lib/libregex.a" () ())
     (curses *unknown* "" "-lcurses" "/usr/lib/libcurses.a" () ())
     (graphics *unknown* "-I/usr/X11/include -DX11" "-lX11"
	       "/usr/X11/lib/libX11.sa" () ())
     (editline *unknown* "" "-lreadline" "/usr/lib/libreadline.a" () ())
     (termcap *unknown* "" "-ltermcap" "/usr/lib/libtermcap.a" () ())
     (debug *unknown* "-g" "-g" #f () ())
     (socket *unknown* "" "" #f () ())
     (lib *unknown* "" "" #f () ("scmmain.c"))
     (mysql *unknown* ""
	    "-lmysqlclient" "/usr/lib/mysql/libmysqlclient.la" () ())

     (c cygwin32 "" "" "" () ())
     (m linux-aout "" "-lm" "/usr/lib/libm.sa" () ())
     (c linux-aout "" "-lc" "/usr/lib/libc.sa" () ())
     (dlll linux-aout "-DDLD -DDLD_DYNCM" "-ldld" #f () ("findexec.c"))
     (regex linux-aout "" "" "" () ())
     (curses linux-aout "-I/usr/include/ncurses" "-lncurses"
	     "/usr/lib/libncurses.a" () ())
     (nostart linux-aout "" "-nostartfiles" #f ("pre-crt0.c") ())
     (dump linux-aout "" "/usr/lib/crt0.o" #f ("unexec.c" "gmalloc.c") ())

     (m linux "" "-lm" "/lib/libm.so" () ())
     (c linux "" "-lc" "/lib/libc.so" () ())
     (dlll linux "-DSUN_DL" "-ldl" #f () ())
     (graphics linux "-I/usr/include/X11 -DX11" "-L/usr/X11R6/lib -lX11"
	       "/usr/X11R6/lib/libX11.so" () ())
     (curses linux "" "-lcurses" "/lib/libncurses.so" () ())
     (nostart linux "" "" #f () ())
     (dump linux "" "" #f ("unexelf.c" "gmalloc.c") ())

     (dump irix "" "-G 0" #f () ())

     (m acorn-unixlib "" "" #f () ())

     (nostart alpha "" "-non_shared" #f ("pre-crt0.c") ())
     (dump alpha "" "" #f ("unexalpha.c") ())

     (m amiga-dice-c "" "-lm" #f () ())
     (m amiga-sas "" "lcmieee.lib" #f () ())
     (c amiga-sas "" "lc.lib" #f () ())

     (m vms-gcc "" "" #f () ())
     (m vms "" "" #f () ())

     (m atari-st-gcc "" "-lpml" #f () ())
     (m atari-st-turbo-c "" "" #f () ())

     (m sunos "" "-lm" #f () ())
     (dlll sunos "-DSUN_DL" "-ldl" #f () ())
     (nostart sunos "" "-e __start -nostartfiles -static" #f ("ecrt0.c") ())
     (dump sunos "" "" #f ("unexelf.c" "gmalloc.c") ())

     (m svr4-gcc-sun-ld "" "-lm" #f () ())
     (dlll svr4-gcc-sun-ld "-DSUN_DL" "-Wl,-ldl" #f () ())
     (nostart svr4-gcc-sun-ld "" "-e __start -nostartfiles" #f ("ecrt0.c") ())
     (dump svr4-gcc-sun-ld "" "" #f ("unexelf.c" "gmalloc.c") ())
     (socket svr4-gcc-sun-ld "" "-lsocket -lnsl" #f () ())
     (regex svr4-gcc-sun-ld "" "" #f () ())

     (nostart gcc "" "-e __start -nostartfiles" #f ("ecrt0.c") ())
     (dump gcc "" "" #f ("unexelf.c" "gmalloc.c") ())

     (m hp-ux "" "-lm" #f () ())
     (dlll hp-ux "-DHAVE_DYNL" "-Wl,-E -ldld" #f () ())
     (graphics hp-ux "-DX11" "-lX" "/usr/lib/X11R5/libX11.sl" () ())
     (nostart hp-ux "" "" #f ("ecrt0.c") ())
     (dump hp-ux "" "" #f ("unexhp9k800.c" "gmalloc.c") ())

     (c djgpp "" "-lc" #f () ("findexec.c"))
     (curses djgpp "-I/djgpp/contrib/pdcurses/include/"
	     "-L/djgpp/contrib/pdcurses/lib/ -lcurses"
	     "\\djgpp\\contrib\\pdcurses\\lib\\libcurse.a" () ())
     (nostart djgpp "" "-nostartfiles" #f ("pre-crt0.c") ())
     (dump djgpp "" "c:/djgpp/lib/crt0.o" #f ("unexec.c" "gmalloc.c") ())
;;;     (nostart djgpp "" "" #f ("ecrt0.c") ())
;;;     (dump djgpp "" "" #f ("unexelf.c" "gmalloc.c") ())
;;;     (nostart djgpp "" "-e __start -nostartfiles -static" #f ("ecrt0.c") ())
;;;     (dump djgpp "" "" #f ("unexelf.c" "gmalloc.c") ())

     (c Microsoft-C "" "" #f () ("findexec.c"))
     (m Microsoft-C "" "" #f () ())
     (c Microsoft-C-nt "" "" #f () ("findexec.c"))
     (m Microsoft-C-nt "" "" #f () ())
     (c Microsoft-Quick-C "" "" #f () ("findexec.c"))
     (m Microsoft-Quick-C "" "" #f () ())

     (c turbo-c "" "" #f () ("findexec.c"))
     (m turbo-c "" "" #f () ())
     (graphics turbo-c "" "graphics.lib" #f () ())

     (c borland-c "" "" #f () ("findexec.c"))
     (m borland-c "" "" #f () ())
     (graphics borland-c "" "graphics.lib" #f () ())
     (windows borland-c "-N -W" "-W" #f () ())

     (c highc "" "" #f () ("findexec.c"))
     (m highc "" "" #f () ())
     (windows highc "-Hwin" "-Hwin" #f () ())

     (m freebsd "" "-lm" #f () ())
     (regex freebsd "" "-lgnuregex" "" () ())
     (editline freebsd "" "-lreadline" "" () ())
     (dlll freebsd "-DSUN_DL" "" "" () ())
     (nostart freebsd "" "-e start -dc -dp -Bstatic -lgnumalloc" #f ("pre-crt0.c") ())
     (dump freebsd "" "/usr/lib/crt0.o" "" ("unexsunos4.c") ())
     ))

  '(compile-commands
    ((name symbol)
     (platform platform))
    ((procedure expression))
    ((update-catalog *unknown*
		     (lambda (oname objects libs parms)
		       (batch:rebuild-catalog parms)
		       (if (= 1 (length objects)) (car objects)
			   objects))))))

(define define-compile-commands
  (let ((defcomms (((build 'open-table) 'compile-commands #t) 'row:insert)))
    (lambda args
      (defcomms args))))		;(append args (list (comment)))
(defmacro defcommand (name platform procedure)
  `(define-compile-commands ',name ',platform ',procedure))

(defcommand compile-c-files borland-c
  (lambda (files parms)
    (define rsp-name "temp.rsp")
    (apply batch:lines->file parms rsp-name files)
    (and (batch:try-command
	  parms
	  "bcc" "-d" "-O" "-Z" "-G" "-w-pro" "-ml" "-c"
	  (if (member '(define "FLOATS" #t)
		      (c-defines parms))
	      "" "-f-")
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  (string-append "@" rsp-name))
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program borland-c
  (lambda (oname objects libs parms)
    (define lnk-name (string-append oname ".lnk"))
    (apply batch:lines->file parms
	   lnk-name
	   (append libs objects))
    (and (batch:try-command
	  parms "bcc" (string-append "-e" oname)
	  "-ml" (string-append "@" lnk-name))
	 (string-append oname ".exe"))))

(defcommand compile-c-files turbo-c
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "tcc" "-c" "-d" "-O" "-Z" "-G" "-ml" "-c"
	  "-Ic:\\turboc\\include"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program turbo-c
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) #\\))
	  (oexe (string-append oname ".exe")))
      (and (or (string-ci=? exe oexe)
	       (batch:delete-file parms oexe))
	   (batch:try-command
	    parms "tcc" "-Lc:\\turboc\\lib" libs objects)
	   (or (string-ci=? exe oexe)
	       (batch:rename-file parms exe oexe))
	   oexe))))

(defcommand compile-c-files Microsoft-C
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms "cl" "-c" "Oxp" "-AH"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program Microsoft-C
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) #\\))
	  (oexe (string-append oname ".exe")))
      (and (or (string-ci=? exe oexe)
	       (batch:delete-file parms oexe))
	   (batch:try-command
	    parms "link" "/noe" "/ST:40000"
	    (apply string-join "+" (map obj-> objects))
	    libs)
	   (or (string-ci=? exe oexe)
	       (batch:rename-file parms exe oexe))
	   oexe))))
(defcommand compile-c-files Microsoft-C-nt
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cl" "-c" "-nologo" "-O2"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program Microsoft-C-nt
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) #\\))
	  (oexe (string-append oname ".exe")))
      (and (batch:try-command
	    parms "link" "/nologo"
	    (string-append "/out:" oexe)
	    (apply string-join " " (map obj-> objects))
	    libs)
	   oexe))))

(defcommand compile-c-files Microsoft-Quick-C
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "qcl" "/AH" "/W1" "/Ze" "/O" "/Ot" "/DNDEBUG"
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program Microsoft-Quick-C
  (lambda (oname objects libs parms)
    (define crf-name (string-append oname ".crf"))
    (apply batch:lines->file parms
	   crf-name
	   `(,@(map (lambda (f) (string-append f " +"))
		    objects)
	     ""
	     ,(string-append oname ".exe")
	     ,(apply string-join " " libs)
	     ";"))
    (and (batch:try-command
	  parms "qlink"
	  "/CP:0xffff" "/NOI" "/SE:0x80" "/ST:0x9c40"
	  crf-name)
	 (string-append oname ".exe"))))

(defcommand compile-c-files Watcom-9.0
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "wcc386p" "/mf" "/d2" "/ze" "/oxt" "/3s"
	  "/zq" "/w3"
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program Watcom-9.0
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) #\\))
	  (oexe (string-append oname ".exe")))
      (and (or (string-ci=? exe oexe)
	       (batch:delete-file parms oexe))
	   (batch:try-command
	    parms
	    "wlinkp" "option" "quiet" "option"
	    "stack=40000" "FILE"
	    (apply string-join "," (map obj-> objects))
	    libs)
	   (if (not (string-ci=? exe oexe))
	       (batch:rename-file parms exe oexe))
	   oexe))))
(defcommand compile-c-files highc
  (lambda (files parms)
    (define hcc-name "temp.hcc")
    (apply batch:lines->file parms hcc-name files)
    (and (batch:try-command
	  parms
	  "d:\\hi_c\\hc386.31\\bin\\hc386"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  "-c" (string-append "@" hcc-name))
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program highc
  (lambda (oname objects libs parms)
    (let ((oexe (string-append oname ".exe")))
      (define lnk-name (string-append oname ".lnk"))
      (apply batch:lines->file parms
	     lnk-name (append libs objects))
      (and (batch:try-command
	    parms
	    "d:\\hi_c\\hc386.31\\bin\\hc386" "-o" oname
	    "-stack 65000"
	    (string-append "@" lnk-name))
	   (batch:try-command
	    parms
	    "bind386" "d:/hi_c/pharlap.51/run386b.exe" oname
	    "-exe" oexe)
	   oexe))))

(defcommand compile-c-files djgpp
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "gcc" "-Wall" "-O2" "-c"
	  (include-spec "-I" parms)
	  (c-includes parms) (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) "\\/"))))
(defcommand link-c-program djgpp
  (lambda (oname objects libs parms)
    (let ((exe (string-append oname ".exe")))
      (and (or (batch:try-command parms
				  "gcc" "-o" oname
				  (must-be-first
				   '("-nostartfiles"
				     "pre-crt0.o" "ecrt0.o"
				     "c:/djgpp/lib/crt0.o")
				   (append objects libs)))
	       (let ((arname (string-append oname ".a")))
		 (batch:delete-file parms arname)
		 (and (batch:try-chopped-command
		       parms
		       "ar" "r" arname objects)
		      (batch:try-command
		       parms "gcc" "-o" oname
		       (must-be-first
			'("-nostartfiles"
			  "pre-crt0.o" "ecrt0.o"
			  "c:/djgpp/lib/crt0.o")
			(cons arname libs)))
		      (batch:delete-file parms arname)))
	       ;;(build:error 'build "couldn't build archive")
	       )
	   (batch:try-command parms "strip" exe)
	   (batch:delete-file parms oname)
	   ;;(batch:delete-file parms exe)
	   ;;(batch:try-command parms "coff2exe" "-s" "c:\\djgpp\\bin\\go32.exe" oname)
	   exe))))

(defcommand compile-c-files os/2-emx
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-O" "-m386" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\\))))
(defcommand link-c-program os/2-emx
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "gcc" "-o" (string-append oname ".exe")
	  objects libs)
	 (string-append oname ".exe"))))

(defcommand compile-c-files os/2-cset
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms "icc" "/Gd-" "/Ge+" "/Gm+" "/Q" "-c"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->obj files) #\\))))
(defcommand link-c-program os/2-cset
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "link386" objects libs
	  (string-append "," oname ".exe,,,;"))
	 (string-append oname ".exe"))))

(defcommand compile-c-files HP-UX
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cc" "+O1" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand compile-dll-c-files HP-UX
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms "cc" "+O1" "-Wl,-E" "+z" "-c"
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (let ((results
		(map
		 (lambda (fname)
		   (batch:rename-file
		    parms
		    (string-append fname ".sl")
		    (string-append fname ".sl~"))
		   (and (batch:try-command
			 parms "ld" "-b" "-o"
			 (string-append fname ".sl")
			 (string-append fname ".o"))
			(string-append fname ".sl")))
		 (truncate-up-to (map c-> files) #\/))))
	   (and (apply and? results) results)))))
;    (make-dll-archive HP-UX
;		       (lambda (oname objects libs parms)
;			 (and (batch:try-command
;			       parms "ld" "-b" "-o" (string-append oname ".sl")
;			       objects)
;			      (batch:rebuild-catalog parms)
;			      (string-append oname ".sl"))))
(defcommand make-dll-archive sunos
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms
	  "ld" "-assert" "pure-text" "-o"
	  (string-append
	   (car (parameter-list-ref parms 'implvic))
	   oname ".so.1.0")
	  objects)
	 (batch:rebuild-catalog parms)
	 (string-append
	  (car (parameter-list-ref parms 'implvic))
	  oname ".so.1.0"))))

(defcommand compile-c-files linux-aout
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-Wall" "-O2" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand compile-dll-c-files linux-aout
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "gcc" "-Wall" "-O2" "-c"
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) #\/))))
;;;     (make-dll-archive linux-aout
;;;		       (lambda (oname objects libs parms) #t
;;;			       (batch:rebuild-catalog parms)
;;;			       oname))

(defcommand compile-c-files linux
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "gcc" "-O2"
	  ;;(if (member "-g" (c-includes parms)) "" "-O2")
	  "-c" (c-includes parms)
	  (include-spec "-I" parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand compile-dll-c-files linux
  (lambda (files parms)
    (and
     (batch:try-chopped-command
      parms
      "gcc" "-O2"
      "-fpic" "-c" (c-includes parms)
      (c-flags parms)
      files)
     (let* ((platform (car (parameter-list-ref
			    parms 'platform)))
	    (ld-opts
	     (map (lambda (l)
		    (build:lib-ld-flag l platform))
		  (parameter-list-ref parms 'c-lib)))
	    (results
	     (map
	      (lambda (fname)
		(and (batch:try-command
		      parms
		      "gcc" "-shared" "-o"
		      (string-append fname ".so")
		      (string-append fname ".o")
		      ld-opts)
		     (batch:delete-file
		      parms (string-append fname ".o"))
		     (string-append fname ".so")))
	      (truncate-up-to (map c-> files) #\/))))
       (and (apply and? results) results)))))
(defcommand make-dll-archive linux
  (lambda (oname objects libs parms)
    (let ((platform (car (parameter-list-ref
			  parms 'platform))))
      (and (batch:try-command
	    parms
	    "gcc" "-shared" "-o"
	    (string-append
	     (car (parameter-list-ref parms 'implvic))
	     oname ".so")
	    objects
	    (map (lambda (l)
		   (build:lib-ld-flag l platform))
		 (parameter-list-ref parms 'c-lib)))
	   (batch:rebuild-catalog parms)
	   (string-append
	    (car (parameter-list-ref parms 'implvic))
	    oname ".so")))))
(defcommand link-c-program linux
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "gcc" "-rdynamic" "-o" oname
	  (must-be-first
	   '("pre-crt0.o" "ecrt0.o" "/usr/lib/crt0.o")
	   (append objects libs)))
	 oname)))

(defcommand compile-c-files Unicos
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "cc" "-hvector2" "-hscalar2" "-c"
	  (include-spec "-i" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program unicos
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "cc" "setjump.o" "-o" oname objects libs)
	 oname)))

(defcommand compile-c-files gcc
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-O2" "-c" ; "-Wall"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program gcc
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       oname (string-append oname "~"))
    (and (batch:try-command parms
			    "gcc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "ecrt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))
(defcommand compile-dll-c-files gcc
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-O" "-c"
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) "\\/]"))))

(defcommand compile-c-files cygwin32
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-Wall" "-O2" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program cygwin32
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       (string-append oname ".exe")
		       (string-append oname "~"))
    (and (batch:try-command parms
			    "gcc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "ecrt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))

(defcommand compile-c-files svr4-gcc-sun-ld
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-O2" "-c" ; "-Wall"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program svr4-gcc-sun-ld
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       oname (string-append oname "~"))
    (and (batch:try-command parms
			    "gcc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "ecrt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))

(defcommand compile-c-files svr4
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cc" "-O" "-DSVR4" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))

(defcommand compile-c-files aix
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cc" "-O" "-Dunix" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program aix
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "cc" "-lansi" "-o" oname objects libs)
	 oname)))

(defcommand compile-c-files amiga-aztec
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cc" "-dAMIGA"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program amiga-aztec
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "cc" "-o" oname objects libs "-lma")
	 oname)))

(defcommand compile-c-files amiga-sas
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "lc" "-d3" "-M" "-fi" "-O"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (batch:try-command
	  parms "blink with link.amiga NODEBUG")
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program amiga-sas
  (lambda (oname objects libs parms)
    (define lnk-name "link.amiga")
    (apply batch:lines->file parms
	   lnk-name
	   (apply string-join "+" ">FROM LIB:c.o"
		  (map object->string objects))
	   (string-append
	    "TO " (object->string (string-append "/" oname)))
	   (append
	    (cond
	     ((pair? libs)
	      (cons (string-append "LIB LIB:" (car libs))
		    (map (lambda (s)
			   (string-append "    LIB:" s))
			 (cdr libs))))
	     (else '()))
	    '("VERBOSE" "SC" "SD")))
    oname))

(defcommand compile-c-files amiga-dice-c
  (lambda (files parms)
    (and (batch:try-command
	  parms
	  "dcc" "-r" "-gs" "-c"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files "-o" (truncate-up-to (map c->o files) #\/))
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program amiga-dice-c
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "dcc" "-r" "-gs" "-o" oname objects libs)
	 oname)))

(defcommand compile-c-files amiga-gcc
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-Wall" "-O2" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program amiga-gcc
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       oname (string-append oname "~"))
    (and (batch:try-command parms
			    "gcc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "ecrt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))

(defcommand compile-c-files atari-st-gcc
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "gcc" "-v" "-O" "-c"
				    (include-spec "-I" parms)
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program atari-st-gcc
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "gcc" "-v" "-o" (string-append oname ".ttp")
	  objects libs)
	 (string-append oname ".ttp"))))

(defcommand compile-c-files atari-st-turbo-c
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "tcc" "-P" "-W-" "-Datarist"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program atari-st-turbo-c
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "tlink" "-o" (string-append oname ".ttp")
	  objects libs "mintlib.lib" "osbind.lib"
	  "pcstdlib.lib" "pcfltlib.lib")
	 (string-append oname ".ttp"))))

(defcommand compile-c-files acorn-unixlib
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "cc" "-c" "-depend" "!Depend" "-IUnixLib:"
	  "-pcc" "-Dunix" "-DSVR3" "-DARM_ULIB"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (truncate-up-to (map c->o files) #\/))))
(defcommand link-c-program acorn-unixlib
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms "link" "-o" oname objects libs
	  ":5.$.dev.gcc.unixlib36d.clib.o.unixlib")
	 (batch:try-command parms "squeeze" oname)
	 oname)))

(defcommand compile-c-files vms
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "cc"
	  (c-includes parms)
	  (c-flags parms)
	  (map c-> files))
	 (truncate-up-to (map c->obj files) "/]"))))
(defcommand link-c-program vms
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) "/]"))
	  (oexe (string-append oname ".exe")))
      (and (batch:try-command parms "macro" "setjump")
	   (batch:try-command
	    parms
	    "link"
	    (apply string-join ","
		   (append (map obj-> objects)
			   '("setjump" "sys$input/opt\n   ")))
	    (apply string-join
		   "," (append (remove "" libs)
			       '("sys$share:vaxcrtl/share"))))
	   (or (string-ci=? exe oexe)
	       (batch:rename-file parms exe oexe))
	   oexe))))

(defcommand compile-c-files vms-gcc
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "gcc"
	  (include-spec "-I" parms)
	  (c-includes parms)
	  (c-flags parms)
	  (map c-> files))
	 (truncate-up-to (map c->obj files) "/]"))))
(defcommand link-c-program vms-gcc
  (lambda (oname objects libs parms)
    (let ((exe (truncate-up-to (obj->exe (car objects)) "/]"))
	  (oexe (string-append oname ".exe")))
      (and (batch:try-command parms "macro" "setjump")
	   (batch:try-command
	    parms
	    "link"
	    (apply string-join ","
		   (append objects
			   '("setjump.obj"
			     "sys$input/opt\n   ")))
	    (apply string-join
		   "," (append (remove "" libs)
			       '("gnu_cc:[000000]gcclib/lib"
				 "sys$share:vaxcrtl/share"))))
	   (or (string-ci=? exe oexe)
	       (batch:rename-file parms exe oexe))
	   oexe))))

(defcommand compile-c-files *unknown*
  (lambda (files parms)
    (batch:try-chopped-command
     parms
     "cc" "-O" "-c"
     (include-spec "-I" parms)
     (c-includes parms)
     (c-flags parms)
     files)
    (truncate-up-to (map c->o files) "\\/]")))
(defcommand link-c-program *unknown*
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       oname (string-append oname "~"))
    (and (batch:try-command parms
			    "cc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "ecrt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))
(defcommand make-archive *unknown*
  (lambda (oname objects libs parms)
    (let ((aname (string-append "lib" oname ".a")))
      (and (batch:try-command parms "ar rc" aname objects)
	   (batch:try-command parms "ranlib" aname)
	   aname))))
(defcommand compile-dll-c-files *unknown*
  (lambda (files parms)
    (and (batch:try-chopped-command parms
				    "cc" "-O" "-c"
				    (c-includes parms)
				    (c-flags parms)
				    files)
	 (truncate-up-to (map c->o files) "\\/]"))))
(defcommand make-dll-archive *unknown*
  (lambda (oname objects libs parms)
    (let ((aname
	   (string-append
	    (car (parameter-list-ref parms 'implvic))
	    oname ".a")))
      (and (batch:try-command parms "ar rc" aname objects)
	   (batch:try-command parms "ranlib" aname)
	   (batch:rebuild-catalog parms)
	   aname))))

(defcommand compile-c-files freebsd
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "cc" "-O" "-c"
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (map c->o files))))
(defcommand link-c-program freebsd
  (lambda (oname objects libs parms)
    (batch:rename-file parms
		       oname (string-append oname "~"))
    (and (batch:try-command parms
			    "cc" "-o" oname
			    (must-be-first
			     '("-nostartfiles"
			       "pre-crt0.o" "crt0.o"
			       "/usr/lib/crt0.o")
			     (append objects libs)))
	 oname)))
(defcommand compile-dll-c-files freebsd
  (lambda (files parms)
    (and (batch:try-chopped-command
	  parms
	  "cc" "-O" "-fpic" "-c"
	  (string-append
	   "-I" (parameter-list-ref parms 'scm-srcdir))
	  (c-includes parms)
	  (c-flags parms)
	  files)
	 (let ((objs (map c->o files)))
	   (every
	    (lambda (f)
	      (and (batch:try-command
		    parms "ld" "-Bshareable" f)
		   (batch:try-command
		    parms "mv" "a.out" f)))
	    objs)
	   objs))))

(defcommand make-dll-archive freebsd
  (lambda (oname objects libs parms)
    (and (batch:try-command
	  parms
	  "ld" "-Bshareable" "-o"
	  (string-append
	   (car (parameter-list-ref parms 'implvic))
	   oname ".so")
	  objects)
	 (batch:rebuild-catalog parms)
	 (string-append
	  (car (parameter-list-ref parms 'implvic))
	  oname ".so"))))

(for-each (build 'add-domain)
	  '((C-libraries C-libraries #f symbol #f)))

(define-tables build

  '(build-params
    *parameter-columns*
    *parameter-columns*
    ((1 platform single platform
	(lambda (pl) (list batch:platform))
	#f
	"what to build it for")
     (2 target-name single string (lambda (pl) '("scm")) #f
	"base name of target")
     (3 c-lib nary C-libraries (lambda (pl) '(c)) #f
	"C library (and include files)")
     (4 define nary string #f #f "#define FLAG")
     (5 implvic single string (lambda (pl) (list ""))
	#f "implementation vicinity")
     (6 c-file nary filename #f #f "C source files")
     (7 o-file nary filename #f #f "other object files")
     (8 init nary string #f #f "initialization calls")
     (9 compiled-init nary string #f #f "later initialization calls")
     (10 features nary features
	 (lambda (pl) '(arrays inexact bignums))
	 (lambda (rdb) (((rdb 'open-table) 'features #f) 'get 'spec))
	 "features to include")
     (11 what single build-whats
	 (lambda (pl) '(exe))
	 (lambda (rdb)
	   (let* ((bwt ((rdb 'open-table) 'build-whats #f))
		  (getclass (bwt 'get 'class))
		  (getspec (bwt 'get 'spec))
		  (getfile (((rdb 'open-table) 'manifest #f) 'get* 'file)))
	     (lambda (what)
	       `((c-file ,@(getfile #f 'c-source (getclass what)))
		 ,@(or (getspec what) '())))))
	 "what to build")
     (12 batch-dialect single batch-dialect
	 (lambda (pl) '(default-for-platform)) ;;guess-how
	 #f
	 "scripting language")
     (13 who optional expression #f #f "name of buildfile")
     (14 compiler-options nary string #f #f "command-line compiler options")
     (15 linker-options nary string #f #f "command-line linker options")

     (16 scm-srcdir single filename
	 (lambda (pl) (list (user-vicinity))) #f
	 "directory path for files in the manifest")
     (17 scm-libdir single filename
	 (lambda (pl) (list (implementation-vicinity))) #f
	 "directory path for files in the manifest")
     (18 c-defines nary expression #f #f "#defines for C")
     (19 c-includes nary expression #f #f "library induced defines for C")
     (20 batch-port nary expression #f #f
	 "port batch file will be written to.")
     ))
  '(build-pnames
    ((name string))
    ((parameter-index uint))		;should be build-params
    (
     ("p" 1) ("platform" 1)
     ("o" 2) ("outname" 2)
     ("l" 3) ("libraries" 3)
     ("D" 4) ("defines" 4)
     ("s" 5) ("scheme initialization file" 5)
     ("c" 6) ("c source files" 6)
     ("j" 7) ("object files" 7)
     ("i" 9) ("initialization calls" 9)
     ("F" 10) ("features" 10)
     ("t" 11) ("type" 11)
     ("h" 12) ("batch dialect" 12)
     ("w" 13) ("script name" 13)
     ("compiler options" 14)
     ("linker options" 15)
     ("scm srcdir" 16)
     ("scm libdir" 17)
     ))

  '(*commands*
    ((name symbol))			;or just desc:*commands*
    ((parameters parameter-list)
     (parameter-names parameter-name-translation)
     (procedure expression)
     (documentation string))
    ((build
      build-params
      build-pnames
      build:build
      "compile and link SCM programs.")
     (*initialize*
      no-parameters
      no-parameters
      #f
      "SCM Build Database"))))

;;;((build 'close-database))
;;;(define build (open-database! "buildscm.scm" 'alist-table))

(define build:error slib:error)
(define build:c-libraries #f)
(define build:lib-cc-flag #f)
(define build:lib-ld-flag #f)
(define build:c-lib-support #f)
(define build:c-suppress #f)
(define plan-command #f)

;;; Look up command on a platform, but default to '*unknown* if not
;;; initially found.

(define (make-defaulting-platform-lookup getter)
  (lambda (thing plat)
    (define (look platform)
      (let ((ans (getter thing platform)))
	(cond (ans ans)
	      ((eq? '*unknown* platform)
	       ;;(slib:warn "Couldn't find: " plat thing)
	       '())
	      (else (look '*unknown*)))))
    (look plat)))

(require 'alist)
(require 'common-list-functions)
(require 'object->string)

(define (build:build rdb)
  (lambda (parms)
    (let ((expanders
	   (map (lambda (e) (and e (lambda (s) (e s))))
		(map (lambda (f) (if f ((slib:eval f) rdb) f))
		     ((((rdb 'open-table) 'build-params #f)
		       'get* 'expander))))))
      (parameter-list-expand expanders parms)
      (set! parms
	    (fill-empty-parameters
	     (map slib:eval
		  ((((rdb 'open-table) 'build-params #f)
		    'get* 'defaulter)))
	     parms))
      (parameter-list-expand expanders parms))
    (let* ((platform (car (parameter-list-ref parms 'platform)))
	   (init= (apply string-append
			 (map (lambda (c)
				(string-append c "();"))
			      (parameter-list-ref parms 'init))))
	   (compiled-init=
	    (apply string-append
		   (map (lambda (c)
			  (string-append c "();"))
			(parameter-list-ref parms 'compiled-init))))
	   (implvic (let ((impl (car (parameter-list-ref parms 'implvic))))
		      (if (equal? "" impl)
			  (car (parameter-list-ref parms 'scm-srcdir))
			  impl)))
	   (c-defines
	    `((define "IMPLINIT"
		,(object->string
		  (string-append
		   implvic "Init"
		   (read-version
		    (in-vicinity (car (parameter-list-ref parms 'scm-srcdir))
				 "patchlvl.h"))
		   ".scm")))
	      ;;,@`(if (equal? "" implvic) '() (...))
	      ,@(if (string=? "" init=) '()
		    `((define "INITS" ,init=)))
	      ,@(if (string=? "" compiled-init=) '()
		    `((define "COMPILED_INITS" ,compiled-init=)))
	      ,@(map (lambda (d) (if (pair? d)
				     `(define ,@d)
				     `(define ,d #t)))
		     (parameter-list-ref parms 'define))))
	   (c-includes
	    (map (lambda (l) (build:lib-cc-flag l platform))
		 (parameter-list-ref parms 'c-lib)))
	   (what (car (parameter-list-ref parms 'what)))
	   (c-proc (plan-command ((((rdb 'open-table) 'build-whats #f)
				   'get 'c-proc)
				  what)
				 platform)))

      (case (car (parameter-list-ref parms 'batch-dialect))
	((default-for-platform)
	 (let ((os ((((build 'open-table) 'platform #f)
		     'get 'operating-system) platform)))
	   (if (not os)
	       (build:error "OS corresponding to " platform " unknown"))
	   (adjoin-parameters!
	    parms (cons 'batch-dialect (list (os->batch-dialect os)))))))

      (adjoin-parameters!
       parms
       (cons 'c-defines c-defines)
       (cons 'c-includes c-includes))

      (let ((name (parameter-list-ref parms 'who)))
	(set! name (if (null? name) (current-output-port) (car name)))
	(batch:call-with-output-script
	 parms
	 name
	 (lambda (batch-port)
	   (define o-files #f)
	   (adjoin-parameters!
	    parms
	    (list 'batch-port batch-port))

	   (batch:comment parms "================ Write file with C defines")
	   (cond
	    ((not (apply batch:lines->file parms
			 "scmflags.h"
			 (defines->c-defines c-defines)))
	     (batch:comment parms "================ Write failed!") #f)
	    (else
	     (batch:comment parms "================ Compile C source files")
	     (set! o-files
		   (let ((suppressors
			  (apply append
				 (map (lambda (l) (build:c-suppress l platform))
				      (parameter-list-ref parms 'c-lib))))
			 (ssdir (car (parameter-list-ref parms 'scm-srcdir))))
		     (c-proc
		      (map (lambda (file) (in-vicinity ssdir file))
			   (apply
			    append
			    (remove-if (lambda (file) (member file suppressors))
				       (parameter-list-ref parms 'c-file))
			    (map
			     (lambda (l) (build:c-lib-support l platform))
			     (parameter-list-ref parms 'c-lib))))
		      parms)))
	     (cond
	      ((not o-files)
	       (batch:comment parms "================ Compilation failed!") #f)
	      (else

	       (batch:comment parms "================ Link C object files")
	       (let ((ans
		      ((plan-command
			((((rdb 'open-table) 'build-whats #f) 'get 'o-proc) what)
			platform)
		       (car (parameter-list-ref parms 'target-name))
		       (append o-files (parameter-list-ref parms 'o-file))
		       (append
			(parameter-list-ref parms 'linker-options)
			(map (lambda (l) (build:lib-ld-flag l platform))
			     (parameter-list-ref parms 'c-lib)))
		       parms)))
		 (cond ((not ans)
			(batch:comment parms "================ Link failed!") #f)
		       (else ans)))))))))))))

(define (include-spec str parms)
  (let ((path (car (parameter-list-ref parms 'scm-srcdir))))
    (if (eqv? "" path) () (list str path))))
(define (c-defines parms)
  (parameter-list-ref parms 'c-defines))
(define (c-includes parms)
  (parameter-list-ref parms 'c-includes))
(define (c-flags parms)
  (parameter-list-ref parms 'compiler-options))

(define (defines->c-defines defines)
  (map
   (lambda (d)
     (case (caddr d)
       ((#t) (string-join " " "#define" (cadr d)))
       ((#f) (string-join " " "#undef" (cadr d)))
       (else (apply string-join " " "#define" (cdr d)))))
   defines))

(define (defines->flags defines)
  (map
   (lambda (d)
     (case (caddr d)
       ((#t) (string-append "-D" (cadr d)))
       ((#f) (string-append "-U" (cadr d)))
       (else (string-append "-D" (cadr d) "=" (object->string (caddr d))))))
   defines))

(define c-> (filename:substitute?? "*.c" "*"))
(define c->o (filename:substitute?? "*.c" "*.o"))
(define c->obj (filename:substitute?? "*.c" "*.obj"))
(define obj-> (filename:substitute?? "*.obj" "*"))
(define obj->exe (filename:substitute?? "*.obj" "*.exe"))

(define (read-version revfile)
  (call-with-input-file
      (if (file-exists? revfile)
	  revfile
	  (in-vicinity (implementation-vicinity) "patchlvl.h"))
    (lambda (port)
      (do ((c (read-char port) (read-char port)))
	  ((or (eof-object? c) (eqv? #\= c))
	   (do ((c (read-char port) (read-char port))
		(lst '() (cons c lst)))
	       ((or (eof-object? c) (char-whitespace? c))
		(list->string (reverse lst)))))))))

(define (batch:rebuild-catalog parms)
  (batch:delete-file parms
		     (in-vicinity (car (parameter-list-ref parms 'implvic))
				  "slibcat"))
  #t)

(define build:initializer
  (lambda (rdb)
    (set! build:c-libraries ((rdb 'open-table) 'c-libraries #f))
    (set! build:lib-cc-flag
	  (make-defaulting-platform-lookup
	   (build:c-libraries 'get 'compiler-flags)))
    (set! build:lib-ld-flag
	  (make-defaulting-platform-lookup
	   (build:c-libraries 'get 'link-lib-flag)))
    (set! build:c-lib-support
	  (make-defaulting-platform-lookup
	   (build:c-libraries 'get 'lib-support)))
    (set! build:c-suppress
	  (make-defaulting-platform-lookup
	   (build:c-libraries 'get 'suppress-files)))
    (set! plan-command
	  (let ((lookup (make-defaulting-platform-lookup
			 (((rdb 'open-table) 'compile-commands #f)
			  'get 'procedure))))
	    (lambda (thing plat)
	      (slib:eval (lookup thing plat)))))))
(build:initializer build)
