/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "TFGui.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "agename.h"


FlexArray 	*TFGui::s_windowList = NULL;
FlexArray 	*TFGui::s_deleteList = NULL;
TFWindowMain 	*TFGui::s_winMain = NULL;
TFDialogMerge	*TFGui::s_dlgMerge = NULL;


/*
 *  constructor: initialize everything 
 */
TFGui::TFGui ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFGui\n");

	SanityCheck::bailout ((s_windowList), "s_windowList!=NULL (Gui already running?)", "TFGui::TFGui");
	SanityCheck::bailout ((s_deleteList), "s_deleteList!=NULL (Gui already running?)", "TFGui::TFGui");
	SanityCheck::bailout ((s_winMain), "s_winMain!=NULL (Gui already running?)", "TFGui::TFGui");

	TFGui::s_windowList = new FlexArray ();
	TFGui::s_deleteList = new FlexArray ();
	TFGui::s_winMain = new TFWindowMain (TFGui::s_windowList);
}


/*
 *  destructor: clean up 
 */
TFGui::~TFGui ()
{
	delete TFGui::s_windowList;
	delete TFGui::s_deleteList;
	delete TFGui::s_winMain;
	
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFGui\n");
}


/*
 *  addWindow: add an already created HF window to the window list. This 
 * 	also adds a window pointer to the CList data field for the HF's row. 
 */
void TFGui::addWindow (TFWindowHandler *TFWH)
{
	SanityCheck::bailout ((!TFWH), "TFWH==NULL", "TFGui::addWindow");

	gchar			*name;
	int			row;
	
	name = TFWH->getHeightField()->getName ();
	TFGui::s_windowList->append (TFWH);
	row = s_winMain->getCList()->append (const_cast<const char**>(&name));
	s_winMain->getCList()->set_row_data (row, TFWH);
	s_winMain->checkMenus ();

	if (GlobalTrace::isSet (GlobalTrace::TRACE_FLOW))
		{
		char	buf[256];

		sprintf (buf, "Added HF %s to Window List now holding %d items\n", 
			name, TFGui::s_windowList->getSize());
		GlobalTrace::trace (GlobalTrace::TRACE_FLOW, buf);
		}

	// update menu sensitivity
	s_winMain->checkMenus ();
	int limWin = TFGui::s_windowList->getSize ();
	for (int i=0; i<limWin; i++)
		{
		TFWH = static_cast<TFWindowHandler*>(TFGui::s_windowList->El(i));
		TFWH->checkMenus ();
		}

	//connect_to_method (Gtk_Main::timeout(50), TFWH, &TFWindowHandler::idleRedrawCallback);
}


/*
 *  addHeightField: set the Height Field we draw and create a window for it
 */
TFWindowHandler *TFGui::addHeightField (HeightField *HF)
{
	SanityCheck::bailout ((!HF), "HF==NULL", "TFGui::addHeightField");

	TFWindowHandler 	*tfWin;

	tfWin = new TFWindowHandler (TFGui::s_windowList);
	tfWin->setHeightField (HF);
	addWindow (tfWin);

	return tfWin;
}


/*
 *  getNewName: check if a name is not already in use 
 */
char *TFGui::getNewName (char *name)
{
	SanityCheck::bailout ((!name), "name==NULL", "TFGui::getNewName");
	SanityCheck::bailout ((!strlen(name)), "strlen(name)==0", "TFGui::getNewName");

	char *newName; 

	if (strstr(name, TF_DEFAULT_FILENAME))
		newName = HeightField::getNewName ();
	else
		newName = agename (name);

	while (isDuplicateName(newName))
		{
		free (newName);
		newName = HeightField::getNewName ();
		}
	return newName;
}


/*
 *  isDuplicateName: check if a name is not already in use 
 */
bool TFGui::isDuplicateName (char *name)
{
	SanityCheck::bailout ((!name), "name==NULL", "TFGui::isDuplicateName");
	SanityCheck::bailout ((!strlen(name)), "strlen(name)==0", "TFGui::isDuplicateName");

	string		S;
	int		i=0, 
			lim;

	SanityCheck::bailout ((!s_winMain->getCList()), "cList==NULL", "TFGui::isDuplicateName");

	lim = s_winMain->getCList()->rows();

	while (i < lim)
		{
		s_winMain->getCList()->get_text (i, 0, S);
		if (!strcmp (S.c_str(), name))
			break;
		i++;
		}

	if (i < lim)
		return TRUE;
		
	return FALSE;
}

/*
 *  getMergeDialog: return an existing or a newly created merge dialog
 */
TFDialogMerge *TFGui::getMergeDialog ()
{
	if (!TFGui::s_dlgMerge)
		TFGui::s_dlgMerge = new TFDialogMerge (TFGui::s_windowList);
	return s_dlgMerge;
}


/*
 *  find TFWin: loop through TFWindowBase::s_winList and find the desired HF 
 * 	by HF pointer. 
 */
TFWindowHandler *TFGui::findTFWin (HeightField *HF)
{
	TFWindowHandler		*TFWH;
	int			i = 0,
				lim = s_windowList->getSize ();

	if (!HF)
		return NULL;

	while (i < lim)
		{
		TFWH = static_cast<TFWindowHandler*>(s_windowList->El(i));
		if (HF == TFWH->getHeightField())
			break;
		i++;
		}

	if (i < lim)
		return TFWH;

	return NULL;
}


/*
 *  allWindowsSaved: check if there are any unsaved windows in the window list
 */
bool TFGui::allWindowsSaved ()
{
	TFWindowHandler		*TFWH;
	HeightField		*HF;
	int			i = 0,
				lim = s_windowList->getSize ();

	if (!lim)
		return TRUE;

	while (i < lim)
		{
		TFWH = static_cast<TFWindowHandler*>(s_windowList->El(i));
		HF = TFWH->getHeightField();
		SanityCheck::bailout ((!HF), "HF==NULL", "TFWindowHandler::allWindowsSaved");
		if (!(HF->isSaved()))
			break;
		i++;
		}

	if (i < lim)
		return FALSE;

	return TRUE;
}


/*
 *  scheduleDelete : add a height field to the s_deleteList so that 
 * 	cleanup() can dispose of it when it is called. 
 */
void TFGui::scheduleDelete (HeightField *HF)
{
	SanityCheck::bailout ((!HF), "HF==NULL", "TFGui::scheduleForCleanup");

	TFGui::s_deleteList->append (HF);
}


/*
 *  cleanup: delete all the windows pointed at by the HFs in s_deleteList from 
 * 	s_windowList. Also deletes the CList row for the HF. 
 * 	Always returns 0 so that the gtk timeout will terminate. 
 */
int TFGui::cleanup ()
{
	TFWindowHandler		*TFWH;
	HeightField		*HF;
	int			i = 0, 
				j = 0,
				limDel = TFGui::s_deleteList->getSize (),
				limWin;

	SanityCheck::bailout ((!limDel), "limDel==0", "TFGui::cleanup");

	while (i < limDel)
		{
		HF = static_cast<HeightField*>(TFGui::s_deleteList->El(i));
		limWin = TFGui::s_windowList->getSize ();

		SanityCheck::bailout ((!HF), "HF==NULL", "TFGui::cleanup");
		SanityCheck::bailout ((!limWin), "limWin==0", "TFGui::cleanup");

		j = 0; 
		while (j < limWin)
			{
			TFWH = static_cast<TFWindowHandler*>(TFGui::s_windowList->El(j));
			if (HF == (TFWH->getHeightField()))
				break;
			j++;
			}

		if (j < limWin)
			{
			TFGui::s_deleteList->remove (i);
			TFGui::s_windowList->remove (j);

			string		s;
			bool		rc;
			rc = s_winMain->getCList()->get_text (j, 0, s);
			SanityCheck::bailout ((!rc), "can't get row text", "TFGui::cleanup");
			SanityCheck::bailout ((s != HF->getName()), "CList entry != HF->getName()", "TFGui::cleanup");

			// here we assume that s_windowList[j]==CList[j]
			s_winMain->getCList()->remove_row (j);
			delete TFWH;
			limDel--;
			}
		else
			i++;
		}

	// update menu sensitivity
	s_winMain->checkMenus ();
	limWin = TFGui::s_windowList->getSize ();
	for (i=0; i<limWin; i++)
		{
		TFWH = static_cast<TFWindowHandler*>(TFGui::s_windowList->El(i));
		TFWH->checkMenus ();
		}

	return 0;			// force timeout connection to stop
}


