# Sketch - A Python-based interactive drawing program
# Copyright (C) 1997, 1998, 1999 by Bernhard Herzog
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

###Sketch Config
#type = Export
#tk_file_type = ("Scalable Vector Graphics (SVG)", '.svg')
#extensions = '.svg'
format_name = 'SVG'
#unload = 1
###End

from string import join

from Sketch import const, Bezier, EmptyPattern, Trafo

def csscolor(color):
    return "#%02x%02x%02x" \
           % (255 * color.red, 255 * color.green, 255 * color.blue)

svg_joins = ('miter', 'round', 'bevel')
svg_caps = (None, 'butt', 'round', 'square')

class SVGSaver:

    def __init__(self, file, filename, document, options):
        self.file = file
        self.filename = filename
        self.document = document
        self.options = options

    def close(self):
        self.file.close()

    def make_style(self, properties, bounding_rect = None):
        style = []
        if properties.line_pattern is not EmptyPattern:
            if properties.line_pattern.is_Solid:
                color = properties.line_pattern.Color()
                style.append("stroke:" + csscolor(color))
            if properties.line_dashes != ():
                pass
            style.append('stroke-width:' + `properties.line_width`)
            if properties.line_join != const.JoinMiter:
                style.append('stroke-linejoin:'
                             + svg_joins[properties.line_join])
            if properties.line_cap != const.CapButt:
                style.append('stroke-linecap:' + svg_caps[properties.line_cap])
        else:
            pass
            #style.append("stroke:none")
        if properties.fill_pattern is not EmptyPattern:
            pattern = properties.fill_pattern
            if pattern.is_Solid:
                style.append("fill:" + csscolor(pattern.Color()))
            elif pattern.is_Gradient and bounding_rect:
                style.append("fill:black")
                #if pattern.is_AxialGradient or pattern.is_RadialGradient:
                #    self.write_gradient((pattern, bounding_rect), style)
                #    self.fill_color = None
        else:
            pass
            #style.append("fill:none")

        return join(style, '; ')

    def PolyBezier(self, paths, properties, bounding_rect):
        write = self.file.write
        # XXX put path group to make gill recognise styles
        #write('<g style="%s">' % self.make_style(properties, bounding_rect))
        write('<path ') 
        write('style="%s" ' % self.make_style(properties, bounding_rect))
        data = []
        for path in paths:
            for i in range(path.len):
                type, control, p, cont = path.Segment(i)
                p = self.trafo(p)
                if type == Bezier:
                    p1, p2 = control
                    p1 = self.trafo(p1)
                    p2 = self.trafo(p2)
                    data.append('C %g %g %g %g %g %g' % (p1.x, p1.y,
                                                         p2.x, p2.y,
                                                         p.x, p.y))
                else:
                    if i > 0:
                        data.append('L %g %g' % tuple(p))
                    else:
                        data.append('M %g %g' % tuple(p))
            if path.closed or properties.HasFill():
                data.append('z')
        write('d="%s"/>\n' % join(data, ''))
        #write('</g>\n')

    def BeginGroup(self):
        self.file.write('<g>\n')

    def EndGroup(self):
        self.file.write('</g>\n')

    def Save(self):
        self.file.write('<?xml version="1.0" standalone="yes"?>\n')

        left, bottom, right, top = self.document.BoundingRect()
        width = right - left
        height = top - bottom
        self.trafo = Trafo(1, 0, 0, -1, -left, top)
        self.file.write('<svg width="%g" height="%g"' % (width, height))
        #self.file.write(' transform="matrix(%g,%g,%g,%g,%g,%g)">\n' % trafo)
        self.file.write('>\n')
        for layer in self.document.Layers():
            if not layer.is_SpecialLayer and layer.Printable():
                self.BeginGroup()
                self.save_objects(layer.GetObjects())
                self.EndGroup()
        self.file.write('</svg>')

    def save_objects(self, objects):
        for object in objects:
            if object.is_Compound:
                self.BeginGroup()
                self.save_objects(object.GetObjects())
                self.EndGroup()
            #elif object.is_SimpleText:
            #    self.SimpleText(object)
            #elif object.is_Image:
            #    self.Image(object)
            elif object.is_Bezier or object.is_Rectangle or object.is_Ellipse:
                self.PolyBezier(object.Paths(), object.Properties(),
                                object.bounding_rect)



def save(document, file, filename, options = {}):
    saver = SVGSaver(file, filename, document, options)
    saver.Save()
    saver.close()
