/***************************************************************************
                          rstring.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rstring.cpp 1998/08/29 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include "rstring.h"

#include <ctype.h>
#include <string.h>

#include <qfileinfo.h>

#include "rlog.h"


// Get nick name of a file (short file name)
//
QString
strGetNickName(QString _fileName)
{
  QFileInfo fileInfo(_fileName);

  if(!fileInfo.fileName().isNull()) {
    return fileInfo.fileName();
  }
  else {
    return 0;
  }
}




// Get the int value from a line from a INI-File:
//   e.g.: "ArrowSize = 5"  returns 5
//
// _iniLine: Line from ini-file
//
// return: read int value
//         0: failure or 0-Value
//
int     
strGetIntFromIniLine(QString _iniLine)
{
  int     eqPos;      // position of equal symbol (=) in string
  QString numStr;     // 
  
  eqPos = _iniLine.find('=', 0);
  if(eqPos>0) {
    ++eqPos;
    numStr = _iniLine.mid(eqPos, 16);
    numStr = numStr.stripWhiteSpace();
    return numStr.toInt();
  }
  else {
    return 0;
  }
}



// Get the float value from a line from a INI-File:
//   e.g.: "Tolerance = 0.001"  returns 0.001
//
// _iniLine: Line from ini-file
//
// return: read float value
//         0.0: failure or 0.0-Value
//
float
strGetFloatFromIniLine(QString _iniLine)
{
  int     eqPos;      // position of equal symbol (=) in string
  QString numStr;     // 
  
  eqPos = _iniLine.find('=', 0);
  if(eqPos>0) {
    ++eqPos;
    numStr = _iniLine.mid(eqPos, 16);
    numStr = numStr.stripWhiteSpace();
    return numStr.toFloat();
  }
  else {
    return 0;
  }
}



// Get the character from a line from a INI-File:
//   e.g.: "QuotationMark1 = <"  returns '<'
//
// _iniLine: Line from ini-file
//
// return: read char value
//         '\0': failure or no value
//
char
strGetCharFromIniLine(QString _iniLine)
{
  int     eqPos;      // position of equal symbol (=) in string
  QString numStr;     // 
  
  eqPos = _iniLine.find('=', 0);
  if(eqPos>0) {
    ++eqPos;
    numStr = _iniLine.mid(eqPos, 64);
    numStr = numStr.stripWhiteSpace();
    return numStr.at(0).latin1();
  }
  else {
    return '\0';
  }
}



// Get the string from a line from a INI-File:
//   e.g.: "DxfPath = C:\data\dxf"  returns C:\data\dxf
//
// _iniLine: Line from ini-file
//
// return: read string value
//         "": failure or no value
//
QString
strGetStringFromIniLine(QString _iniLine)
{
  int     eqPos;      // position of equal symbol (=) in string
  QString numStr;     // 
  
  eqPos = _iniLine.find('=', 0);
  if(eqPos>0) {
    ++eqPos;
    numStr = _iniLine.mid(eqPos, 1024);
    numStr = numStr.stripWhiteSpace();
    return numStr;
  }
  else {
    return "";
  }
}



// Is the character _ch a valid character for an order?
//
bool
strIsValidNcOrderChar(char _ch) 
{
  if(isdigit(_ch) || _ch=='.' || _ch=='-' || _ch==':') {
    return true;
  }
  else {
    return false;
  }
}



// Test if the string _str is an integer value:
//   (like "37" / NOT "37.5")
//
bool     
strIsInteger(const char* _str)
{
  bool ret=true;
  
  if(_str) {
    int i;
    for(i=0; i<(int)strlen(_str); ++i) {
      if(!isdigit(_str[i])) {
        ret=false;
        break;
      }
    }
  }
  else {
    ret=false;
  }
  
  return ret;
}



// Test if the string _str contains only spaces:
//   (like "  " / NOT "" / NOT "   \n")
//
bool     
strIsSpace(const char* _str)
{
  bool ret=true;
  
  if(_str) {
    int i;
    for(i=0; i<(int)strlen(_str); ++i) {
      if(_str[i]!=' ') {
        ret=false;
        break;
      }
    }
  }
  else {
    ret=false;
  }
  
  return ret;
}



// Converts a float into a String which is as short as possible
//   (2.10000 -> 2.1, 2.00000 -> 2)
//
QString
strFloatToString(float _value)
{
  QString valStr;

  valStr.setNum(_value, 'f', 6);

  if(valStr.contains('.')) {
    // Remove zeros at the end:
    //
    while(valStr.at(valStr.length()-1)=='0') {
      valStr.truncate(valStr.length()-1);  /* ### UNSURE ### OK: 19990804 ###*/
    }

    if(valStr.at(valStr.length()-1)=='.') {
      valStr.truncate(valStr.length()-1);
    }
  }

  return valStr;
}



// Replace all char _old with char _new:
//   (e.g.: '_' with ' ')
//
// return: true: _old found and replaced
//         false: otherwise
//
bool     
strReplaceChar(char* _str, char _old, char _new)
{
  bool ret=false;
  
  if(_str) {
    int i;
    for(i=0; i<(int)strlen(_str); ++i) {
      if(_str[i]==_old) {
        _str[i]=_new;
        ret=true;
      }
    }
  }
  
  return ret;
}



// Get values from a parameter line (usually 
//   read from a machine generator)
//
// Line must have the following format:
//   Parameter free $e Extension NC   CNC D
//   --------- ---- -- --------- --   -----
//       \       \   \      \     \      \_____ List with possible values
//        \       \   \      \     \___________ Standard value
//         \       \   \      \________________ Description
//          \       \   \______________________ Parameter code
//           \       \_________________________ free or fixed
//            \________________________________ Parameter or LayerParameter
//
// values are returned in args
//
void
strGetValuesFromParameterLine(const char* _paraLine, 
                              bool* _layer, 
                              bool* _rw,
                              char* _code,
                              char* _label,
                              char* _std,
                              char* _list)
{
  if(_paraLine) {
    char  scode;            // Code for parameter
    char  slayer[128];      // "LayerParameter" / "Parameter"
    char  slabel[128];      // Label for parameter
    char  smode[16];        // Mode: free or fixed
    char  sstd[64];         // Standard value
    char  slist[256];       // List of other values seperated with spaces

    scode=0;
    slayer[0]='\0';
    slabel[0]='\0';
    smode[0]='\0';
    sstd[0]='\0';
    slist[0]='\0';

    RLOG("\n_paraLine: ");
    RLOG(_paraLine);
  
    sscanf(_paraLine, 
           // ."LayerParameter" ."free " ."$x "         .Label       .Std     .List
            //"%*[ ]%[^ \n]%*[ ]%15s%*[ ]%*[$]%c%*[^ \n]%127s%*[^ \n]%63s%*[ ]%255[^\n]", 
            "%[^ \n]%*[ ]%15s%*[ ]%*[$]%c%127s%63s%*[ ]%255[^\n]",
            slayer, smode, &scode, slabel, sstd, slist);

    RLOG("\nscode: ");
    RLOG(scode);
    RLOG("\nslayer: ");
    RLOG(slayer);
    RLOG("\nslabel: ");
    RLOG(slabel);
    RLOG("\nsmode: ");
    RLOG(smode);
    RLOG("\nsstd: ");
    RLOG(sstd);
    RLOG("\nslist: ");
    RLOG(slist);
  
    if(_layer) {
      if(!strcmp(slayer, "LayerParameter")) *_layer=true;
      else                                  *_layer=false;
    }
    if(_rw) {
      if(!strcmp(smode, "free")) *_rw=true;
      else                       *_rw=false;
    }
    if(_code) *_code=scode;
    if(_label) strcpy(_label, slabel);
    if(_std)   strcpy(_std, sstd);
    if(_list)  strcpy(_list, slist);
  }
}



// Get read write flag (fixed/free) from a parameter line (usually 
//   read from a machine generator)
//
// return: true:  read/write
//         false: read only
//
bool 
strGetReadWriteFromParameterLine(QString _paraLine)
{
  char readWrite[128];
  
  sscanf(_paraLine.data(), 
         //      ."Parameter" ."free"
           "%*[ ]%*[^ \n]%*[ ]%15s", 
           readWrite);

  if(!strcmp(readWrite, "free")) {
    return true;
  }
  else {
    return false;
  }
}



// Step to order item _index:
//
//  "OrderLine { <NewLine>N <Number>... }"
//               0       1^ ^2
//   012345.............21^
//
//   _overReadEnd: true: We don't count end items (delete item)
//
//   return: char index of position 1->21
//           -1: failure
//               
int
strStepToOrderItem(const char* _order, int _index, bool _overReadEnd)
{
  const char* pos;    // Ptr (walks through order)
  int         ret=-1; // Return value (integer pos)
  int         i;      // Counter
  
  pos = _order;

  if(pos && strlen(pos)>0) {
    ret=0;
          
    // Step to 1st char after '{':
    //
    while(pos[0]!='\0' && pos[0]!='{') { ++pos; ++ret; }
    if(pos[0]!='\0') { ++pos; ++ret; }
    
    i=0;
    if(i<_index) {
      do {      
        if(pos[0]=='<') {
          // Don't count end items:
          //
          if(!_overReadEnd || strncmp(pos, "</", 2)) {
            ++i;
          }
          
          while(pos[0]!='\0' && pos[0]!='>') { ++pos; ++ret; }
          if(pos[0]!='\0') { ++pos; ++ret; }
        }
        else if(pos[0]=='$') {
          if(pos[0]!='\0') { ++pos; ++ret; }
          if(pos[0]!='\0') { ++pos; ++ret; }
          ++i;
        }
        else {
          while(pos[0]!='\0' && pos[0]!='<' && pos[0]!='$' && pos[0]!='}') { ++pos; ++ret; }
          ++i;
        }

        RLOG("\nWe're at item: ");
        RLOG(i);
        RLOG(": ");
        RLOG(pos);
          
      }while(i<_index || (_overReadEnd && !strncmp(pos, "</", 2)));
    }
  }
  return ret;
}



// Get order item length:
//
//  "OrderLine { <NewLine>N <Number>... }"
//               0       1^ ^2
//   012345.............21^
//
//  _orderItem points to item
//
// return: char index of position 1->21
//         -1: failure
//               
int
strOrderItemLength(const char* _orderItem)
{
  int ret=-1; // Return value (integer length)
  
  if(_orderItem && strlen(_orderItem)>0) {
    ret=0;
    
    // Special Item (<...>):
    //      
    if(_orderItem[0]=='<') {
      while(_orderItem[0]!='\0' && 
            _orderItem[0]!='>'  &&
            _orderItem[0]!='}'     ) { ++_orderItem; ++ret; }
      if   (_orderItem[0]!='\0' &&
            _orderItem[0]!='}'     ) { ++_orderItem; ++ret; }
    }

    // Special Item ($x):
    //
    else if(_orderItem[0]=='$') {
      return 2;
    }

    // Text-Item:
    //
    else {
      while(_orderItem[0]!='\0' && 
            _orderItem[0]!='<'  && 
            _orderItem[0]!='$'  &&
            _orderItem[0]!='}'     ) { ++_orderItem; ++ret; }
    }
  }
  return ret;
}


// EOF


