/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

#include <stdlib.h>
#include <curses.h>
#include <assert.h>

#include "snipes.h"
#include "coords.h"
#include "collision.h"
#include "chars.h"
#include "weapons.h"
#include "player.h"
#include "sound.h"
#include "screen.h"

#define MAX_ENEMIES		200
#define MOVE_DELAY_COUNT	24
#define SNIPE_VISION_RANGE	20
#define SNIPE_VISION_DISTANCE_SQUARE	200
#define GHOST_VISION_DISTANCE_SQUARE	80

#define ABS(x) (((x)<0)?(-(x)):(x))
#define SIGN(x) (((x)<0)?-1:1)
#define MIN(x,y) (((x)<(y))?(x):(y))

int max_allowed_enemies;
int num_enemies, num_ghost_enemies, num_non_ghost_enemies;
int num_dead_ghost_enemies, num_dead_non_ghost_enemies;

int firing_probability;
int enable_ghost_enemies=0;

int enemy_color=0;

enum enemy_movement_mode {
  TRACKING,
  FORCED_WANDERING,
  WANDERING
};

struct enemy_struct {
  struct enemy_struct *next;

  coordinate position;
  int last_dir;
  int step_count;

  char has_collided;
  char has_a_shot;
  enum enemy_movement_mode movement_mode;
  char ghost;
  char turns_to_ghost;
} enemies[MAX_ENEMIES];


struct enemy_struct *enemy_update_list[MOVE_DELAY_COUNT];
struct enemy_struct *unused_enemies;


void init_enemies( int color, int probability, int enable_ghosts,
		   int max_enemies )
{
  int i;

  enemy_color = color;
  firing_probability = probability;
  enable_ghost_enemies = enable_ghosts;
  max_allowed_enemies = max_enemies;
  
  if( max_allowed_enemies < 0 || max_allowed_enemies > MAX_ENEMIES )
    max_allowed_enemies = MAX_ENEMIES;
  
  num_enemies = num_ghost_enemies = num_non_ghost_enemies = 0;
  num_dead_ghost_enemies = num_dead_non_ghost_enemies = 0;
  
  for( i=0; i<MOVE_DELAY_COUNT; i++ )
    enemy_update_list[i] = NULL;

  unused_enemies = NULL;
  for( i=0; i<MAX_ENEMIES; i++ ) {
    enemies[i].next = unused_enemies;
    unused_enemies = &enemies[i];
  }
}


int get_num_live_enemies( int type )
{
  return type?num_ghost_enemies:num_non_ghost_enemies;
}


int get_num_dead_enemies( int type )
{
  return type?num_dead_ghost_enemies:num_dead_non_ghost_enemies;
}


void add_enemy( coordinate p, char type )
{
  struct enemy_struct *enemy;
  int index;
  
  if( num_enemies == max_allowed_enemies )
    return;

  if( unused_enemies ) {
    enemy = unused_enemies;
    unused_enemies = unused_enemies->next;
    
    index = random()%MOVE_DELAY_COUNT;

    enemy->next = enemy_update_list[index];
    enemy_update_list[index] = enemy;

    enemy->position = p;
    enemy->last_dir = random()%8;
    enemy->step_count = 2 + random()%8;

    enemy->ghost = type;
    enemy->turns_to_ghost = 0;
    enemy->has_collided = 0;
    enemy->has_a_shot = 0;
    enemy->movement_mode = WANDERING;

    collision_map_place_obj( enemy->position.x,
			     enemy->position.y, enemy->ghost?1:2, 1,
			     MAZE_OBJECT_SNIPE );
    num_enemies++;
    if( type )
      num_ghost_enemies++;
    else
      num_non_ghost_enemies++;
  }
}


void enemy_remove_from_collision_map( struct enemy_struct *enemy )
{
  collision_map_remove_obj( enemy->position.x,
			    enemy->position.y,
			    enemy->ghost?1:2, 1,
			    enemy->ghost?MAZE_OBJECT_SNIPE_GHOST:
				MAZE_OBJECT_SNIPE );
}


void enemy_place_on_collision_map( struct enemy_struct *enemy )
{
  collision_map_place_obj( enemy->position.x,
			   enemy->position.y,
			   enemy->ghost?1:2, 1,
			   enemy->ghost?MAZE_OBJECT_SNIPE_GHOST:
			       MAZE_OBJECT_SNIPE );
}


void delete_enemy( struct enemy_struct *enemy )
{
  enemy_remove_from_collision_map( enemy );
  num_enemies--;
  if( enemy->ghost ) {
    num_ghost_enemies--;
    num_dead_ghost_enemies++;
  }
  else {
    num_non_ghost_enemies--;
    num_dead_non_ghost_enemies++;
  }
}


void convert_enemy_to_ghost( struct enemy_struct *enemy )
{
  enemy_remove_from_collision_map( enemy );
  enemy->ghost = 1;
  enemy->has_collided = 0;
  num_non_ghost_enemies--;
  num_dead_non_ghost_enemies++;
  num_ghost_enemies++;
  enemy_place_on_collision_map( enemy );
}


void show_enemies( screen_coords *sc )
{
  int i;
  struct enemy_struct *enemy;
  coordinate screen_pos;
  
  for( i=0; i<MOVE_DELAY_COUNT; i++ ) {
    for( enemy=enemy_update_list[i]; enemy; enemy=enemy->next ) {
      if( !enemy->has_collided &&
	  coords_check_on_screen( sc, enemy->position, 0 ) ) {
	coords_to_screen_pos( sc, enemy->position, &screen_pos );

#if 0
	if( enemy->movement_mode == TRACKING )
	  screen_setcolorpair( enemy_color-1 );
	else
#endif
	  screen_setcolorpair( enemy_color );

	screen_move( screen_pos.y, screen_pos.x );
	if( enemy->ghost )
	  screen_addch( GHOSTFACE );
	else
	  switch( enemy->last_dir )
	  {
	    case 1:
	      screen_addch( EVILFACE );
	      screen_addch( WEAPON_UP );
	      break;
	    case 6:
	      screen_addch( EVILFACE );
	      screen_addch( WEAPON_DOWN );
	      break;
	    case 0:
	    case 3:
	    case 5:
	      screen_addch( WEAPON_LEFT );
	      screen_addch( EVILFACE );
	      break;
	    case 2:
	    case 4:
	    case 7:
	      screen_addch( EVILFACE );
	      screen_addch( WEAPON_RIGHT );
	      break;
	  }
      }
    }
  }
}


void enemy_update_has_collided( struct enemy_struct *enemy, screen_coords *sc )
{
  char right_collided;

  enemy->has_collided = check_collisions( enemy->position.x,
					  enemy->position.y, 1, 1,
					  MAZE_OBJECT_PLAYER_BULLET |
					    MAZE_OBJECT_SNIPE_BULLET);

  if( !enemy->ghost ) {
    right_collided = check_collisions( (enemy->position.x+1)%(sc->max_x+1),
				       enemy->position.y, 1, 1,
				       MAZE_OBJECT_PLAYER_BULLET |
					 MAZE_OBJECT_SNIPE_BULLET);

    /* If weapon is on the left side of the enemy. */
    if( enemy->last_dir == 0 || enemy->last_dir == 3 || enemy->last_dir == 5 ) {
      /* If weapon got hit but not the enemy itself, the enemy ghostifies. */
      if( !right_collided && enemy->has_collided )
	enemy->turns_to_ghost = 1;
    }
    /* If weapon is on the right side of the enemy. */
    else {
      /* If weapon got hit but not the enemy itself, the enemy ghostifies. */
      if( right_collided && !enemy->has_collided )
	enemy->turns_to_ghost = 1;
    }
    enemy->has_collided |= right_collided;
  }
  
  if( enemy->has_collided )
    sound_post_enemy_death();
}


void move_enemy( struct enemy_struct *enemy, coordinate delta,
		 screen_coords *sc )
{
  enemy_remove_from_collision_map( enemy );

  coords_add_delta( sc, &enemy->position, delta );

  enemy_place_on_collision_map( enemy );
}


void enemy_fire( struct enemy_struct *enemy, coordinate direction,
		 screen_coords *sc )
{
  coordinate tc, tp;
  
  if( (direction.x != 0 || direction.y != 0) ) {

    tc = direction;
    
    /* Enemies are two spaces wide, so if they are firing right
     * we have to compensate for that so that the bullet starts
     * beside the enemy.
     */
    if( tc.x == 1 )
      tc.x = 2;
  
    tp = enemy->position;
    coords_add_delta( sc, &tp, tc );

    add_weapon( tp, direction, 0 );
    if( coords_check_on_screen( sc, tp, 0 ) )
      sound_post_snipe_shot();
  }
}


void process_enemies_collisions( screen_coords *sc )
{
  struct enemy_struct *enemy, *prev_enemy;
  int i;

  for( i=0; i<MOVE_DELAY_COUNT; i++ ) {
    prev_enemy = NULL;
    enemy=enemy_update_list[i];
    while( enemy ) {
      if( enemy->has_collided ) {
	if( enable_ghost_enemies && !enemy->ghost && enemy->turns_to_ghost )
	  convert_enemy_to_ghost( enemy );
	else {
	  /* unlink */
	  if( prev_enemy )
	    prev_enemy->next = enemy->next;
	  else
	    enemy_update_list[i] = enemy->next;
	  
	  /* relink */
	  enemy->next = unused_enemies;
	  unused_enemies = enemy;

	  /* and update count */
	  delete_enemy( enemy );
	  
	  /* The previous enemy now has the pointer to the enemy
	   * we want to process on the next loop iteration.  If
	   * prev_enemy is NULL, we will later look back at the
	   * head pointer for this timeslot to find the new
	   * first enemy in the list.
	   */
	  enemy = prev_enemy;
	}
      }
      else
	enemy_update_has_collided( enemy, sc );

      prev_enemy = enemy;
      if(enemy)
	enemy = enemy->next;
      else
	enemy = enemy_update_list[i];
    }
  }
}


/* deltas:
 * -------------
 * 0  1  2
 * 3 obj 4
 * 5  6  7
 */

static coordinate deltas[8] = {
  { -1, -1 },
  {  0, -1 },
  {  1, -1 },
  { -1,  0 },
  {  1,  0 },
  { -1,  1 },
  {  0,  1 },
  {  1,  1 }
};


/* dir_pref_list is a list of preferred directions to travel in
 * given one's top choice.  The idea is that if you want to travel
 * northeast, but that square is blocked, north or east are both
 * good second choices.  Whether north or east is the second choice
 * is arbitrary in this list.
 */

static dir_pref_list[8][8] = {
  { 0, 3, 1, 5, 2, 6, 4, 7 },
  { 1, 0, 2, 3, 4, 5, 6, 7 },
  { 2, 1, 4, 7, 0, 3, 6, 5 },
  { 3, 0, 5, 6, 1, 4, 2, 7 },
  { 4, 7, 2, 1, 6, 3, 0, 5 },
  { 5, 3, 6, 0, 7, 1, 4, 2 },
  { 6, 5, 7, 4, 3, 1, 0, 2 },
  { 7, 6, 4, 2, 5, 3, 1, 0 }
};


int get_approximate_direction( coordinate player_delta )
{
  static int dirs[9] = { 0, 3, 5, 1, -1, 6, 2, 4, 7 };

  int dx = SIGN(player_delta.x);
  int dy = SIGN(player_delta.y);
  int idx;

  if( ABS(player_delta.x) > 4*ABS(player_delta.y) )
    dy = 0;
  else if( ABS(player_delta.y) > 4*ABS(player_delta.x) )
    dx = 0;
  
  idx = 3*(dx+1) + (dy+1);

  assert( dirs[idx] != -1 );
  return dirs[idx];
}


int get_different_direction( coordinate player_delta )
{
  int a, d = get_approximate_direction( player_delta );
  while( (a = random()%8) == d );
  return a;
}


int get_opposite_direction( coordinate player_delta )
{
  player_delta.x = -player_delta.x;
  player_delta.y = -player_delta.y;
  return get_approximate_direction( player_delta );
}


/* enemy_get_movement_preferences( struct enemy_struct *enemy )
 * 
 * returns a pointer to a block of 8 integers, which is set to some
 * permutation of the numbers 0-7.  The first number in the list is the
 * direction in which the enemy would most like to move.  The last number
 * is the direction in which the enemy would least like to move.
 */

int* enemy_get_movement_preferences( screen_coords *sc,
				     struct enemy_struct *enemy )
{
  int a;
  int diag_dist, ortho_dist;
  coordinate player_delta;

  /* Sometimes we pick a random direction but most of the time
   * we try to keep moving in the same direction we were going.
   */
  if( --enemy->step_count <= 0 ) {
    find_nearest_player( sc, enemy->position, &player_delta );

    /* Ghosts attempt to follow the nearest player around. */
    if( enemy->ghost ) {
      if( (player_delta.x*player_delta.x + player_delta.y*player_delta.y) <
	  GHOST_VISION_DISTANCE_SQUARE ) {
	enemy->step_count = 2;
	return dir_pref_list[get_approximate_direction(player_delta)];
      }
    }

    /* Non-ghosts attempt to find a good place to shoot from. */
    if( !enemy->ghost ) {
      enemy->has_a_shot = 0;
      
      switch( enemy->movement_mode ) {
	/* If all snipes tracked all the time, you'd get a traffic
	 * jam, so it's better to give up from time to time. */
	case TRACKING:
	  if( random()%10 == 0 )
	    enemy->movement_mode = FORCED_WANDERING;
	    enemy->step_count = 3 + random()%2; /* how long to wander */
	  break;
	case FORCED_WANDERING:
	  if ( enemy->step_count <= 0 )
	    enemy->movement_mode = WANDERING;
	  break;
	default:
	  ;
      }

      /* Is there a player within this snipe's vision range? */
      if( (player_delta.x*player_delta.x + player_delta.y*player_delta.y) <
	  SNIPE_VISION_DISTANCE_SQUARE ) {
	
	/* Should we start tracking that player? */
	/* was %3 */
	if( enemy->movement_mode != FORCED_WANDERING && random()%2 == 0 ) {
	  enemy->movement_mode = TRACKING;
	}
	
	if( enemy->movement_mode == TRACKING ) {
	  /* Don't get too close to the player.  No glomming allowed
	   * unless you are a ghost. */
	  if( ABS(player_delta.x) <= 3 && ABS(player_delta.y) <= 3 ) {
	    enemy->step_count = 2 + random()%2;
	    return dir_pref_list[get_different_direction(player_delta)];
	  }

	  diag_dist  = ABS(ABS(player_delta.x)-ABS(player_delta.y));
	  ortho_dist = MIN(ABS(player_delta.x),ABS(player_delta.y));

	  if( diag_dist == 0 || ortho_dist == 0 ) {
	    /* We have a shot now.  Move toward the player. */
	    enemy->step_count = 1;
	    enemy->has_a_shot = 1;
	  }
	  else if( diag_dist < ortho_dist ) {
	    /* Quicker to get a diagonal shot than to get a vertical or
	     * horizontal shot. */
	    if( ABS(player_delta.x) > ABS(player_delta.y) )
	      player_delta.y = 0;
	    else
	      player_delta.x = 0;
	    enemy->step_count = diag_dist;
	  }
	  else {
	    /* Go for the horizontal or vertical shot. */
	    if( ABS(player_delta.x) > ABS(player_delta.y) )
	      player_delta.x = 0;
	    else
	      player_delta.y = 0;
	    enemy->step_count = ortho_dist;
	  }

	  a = get_approximate_direction(player_delta);
	  return dir_pref_list[a];
	}
	else {
	  enemy->movement_mode = WANDERING;
	}
      }
      else {
	enemy->movement_mode = WANDERING;
      }
    }

    /* How long before we reconsider our direction again? */
    enemy->step_count = 2 + random()%2;

    /* Choose a new direction at random. */
    return dir_pref_list[random()%8];
  }
  else
    return dir_pref_list[enemy->last_dir];
}


void move_enemies( screen_coords *sc )
{
  static int enemy_move_count = 0;
  
  struct enemy_struct *enemy;
  int *dir_prefs;
  char movement;
  int i;
  int temp_prob;

  process_enemies_collisions( sc );

  /* Move the enemies that are ready to move in this iteration. */

  for( enemy=enemy_update_list[enemy_move_count]; enemy; enemy=enemy->next ) {
    if( enemy->has_collided )
      continue;

    /* See if there is an adjacent spot for us to move to. */

    movement = check_adjacent_collisions( enemy->position.x,
					  enemy->position.y,
					  enemy->ghost?1:2, 1,
					  MAZE_OBJECT_SNIPE |
					  MAZE_OBJECT_SNIPE_GHOST |
					  MAZE_OBJECT_SNIPE_MAKER |
					  MAZE_OBJECT_PLAYER_BULLET |
					  MAZE_OBJECT_PLAYER |
					  MAZE_OBJECT_WALL );
    
    /* Find out where the enemy wants to go. */
    dir_prefs = enemy_get_movement_preferences( sc, enemy );

    /* Try to move the enemy. */
    for( i=0; i<8; i++ )
      if( !((movement>>dir_prefs[i]) & 1) ) {
	move_enemy( enemy, deltas[dir_prefs[i]], sc );
	enemy->last_dir = dir_prefs[i];
	break;
      }
    
    /* The enemy may have a shot, but if it does not get to move in
     * its first-choice direction, the shot will not be going in the
     * right direction, so it should be considered as not having a
     * shot.
     */
    if( i != 0 )
      enemy->has_a_shot = 0;

    /* Fire */
    if( !enemy->ghost && enemy->movement_mode == TRACKING ) {
      temp_prob = random()%firing_probability;

      /* Previous code for deciding when to shoot.
       * 
       *    check_line_collisions( enemy->position.x,
       *			   enemy->position.y,
       *			   deltas[enemy->last_dir].x,
       *			   deltas[enemy->last_dir].y,
       *			   SNIPE_VISION_RANGE,
       *			   MAZE_OBJECT_PLAYER )
       */

      /* Fire randomly.  Higher probability if there is a player that we
       * can hit. */
      if( temp_prob == 0 || (temp_prob < 3 && enemy->has_a_shot) )
	enemy_fire( enemy, deltas[enemy->last_dir], sc );
    }
  }

  /* Next time, we'll move a different set of enemies. */
  enemy_move_count = (enemy_move_count+1)%MOVE_DELAY_COUNT;
}
