/*
 * math.c
 *
 * Arithmetic, compare and logical instructions
 *
 */

#include "ztypes.h"

/*
 * add
 *
 * Add two operands
 *
 */

void add (zword_t a, zword_t b)
{
    store_operand (a + b);
}

/*
 * subtract
 *
 * Subtract two operands
 *
 */

void subtract (zword_t a, zword_t b)
{
    store_operand (a - b);
}

/*
 * multiply
 *
 * Multiply two operands
 *
 */

void multiply (zword_t a, zword_t b)
{
    store_operand ((short)a * (short)b);
}

/*
 * divide
 *
 * Divide two operands
 *
 */

void divide (zword_t a, zword_t b)
{
    /* The magic rule is: round towards zero. */
    short sa = (short)a;
    short sb = (short)b;
    short res;
    if (sb < 0) {
        sa = -sa;
	sb = -sb;
    }
    if (sb == 0) {
#ifdef STRICTZ
        report_strictz_error(STRZERR_DIV_ZERO,
            "@div attempted with a divisor of zero.  Result set to 32767.");
#endif
        res=32767;
    }
    else if (sa >= 0) {
        res = sa / sb;
    }
    else {
        res = -((-sa) / (sb));
    }
    store_operand (res);

}

/*
 * remainder
 *
 * Modulus divide two operands
 *
 */

void remainder (zword_t a, zword_t b)
{
    /* The magic rule is: be consistent with divide, because
       (a/b)*a + (a%b) == a. So (a%b) has the same sign as a. */
    short sa = (short)a;
    short sb = (short)b;
    short res;
    if (sb < 0) {
	sb = -sb;
    }
    if (sb == 0) {
#ifdef STRICTZ
        report_strictz_error(STRZERR_DIV_ZERO,
            "@remainder attempted with a divisor of zero.  Result set to 32767.");
#endif
        res=32767;
    }
    if (sa >= 0) {
        res = sa % sb;
    }
    else {
        res = -((-sa) % (sb));
    }
    store_operand (res);

}

/*
 * shift
 *
 * Shift +/- n bits
 *
 */

void shift (zword_t a, zword_t b)
{

    if ((short) b > 0)
        store_operand (a << (short) b);
    else
        store_operand (a >> abs ((short) b));

}


/*
 * arith_shift
 *
 * Aritmetic shift +/- n bits
 *
 */

void arith_shift (zword_t a, zword_t b)
{
    if ((short) b > 0)
        store_operand (a << (short) b);
    else
        if ((short) a > 0)
            store_operand (a >> abs ((short) b));
        else
            store_operand (~((~a) >> abs ((short) b)));
}

/*
 * or
 *
 * Logical OR
 *
 */

void or (zword_t a, zword_t b)
{
    store_operand (a | b);
}

/*
 * not
 *
 * Logical NOT
 *
 */

void not (zword_t a)
{
    store_operand (~a);
}

/*
 * and
 *
 * Logical AND
 *
 */

void and (zword_t a, zword_t b)
{
    store_operand (a & b);
}

/*
 * zip_random
 *
 * Return random number between 1 and operand
 *
 */

void zip_random (zword_t a)
{
    if (a == 0)
        store_operand (0);
    else if (a & 0x8000) { /* (a < 0) - used to set seed with #RANDOM */
        SRANDOM_FUNC ((unsigned int) abs (a));
        store_operand (0);
    } else /* (a > 0) */
        store_operand (((zword_t) RANDOM_FUNC () % a) + 1);
}

/*
 * test
 *
 * Jump if operand 2 bit mask not set in operand 1
 *
 */

void test (zword_t a, zword_t b)
{
    conditional_jump (((~a) & b) == 0);
}

/*
 * compare_zero
 *
 * Compare operand against zero
 *
 */

void compare_zero (zword_t a)
{
    conditional_jump (a == 0);
}

/*
 * compare_je
 *
 * Jump if operand 1 is equal to any other operand
 *
 */

void compare_je (int count, zword_t *operand)
{
    int i;

    for (i = 1; i < count; i++)
        if (operand[0] == operand[i]) {
            conditional_jump (TRUE);
            return;
        }
    conditional_jump (FALSE);
}

/*
 * compare_jl
 *
 * Jump if operand 1 is less than operand 2
 *
 */

void compare_jl (zword_t a, zword_t b)
{
    conditional_jump ((short) a < (short) b);
}

/*
 * compare_jg
 *
 * Jump if operand 1 is greater than operand 2
 *
 */

void compare_jg (zword_t a, zword_t b)
{
    conditional_jump ((short) a > (short) b);
}
