<copyright> Connection class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997-1998 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Connection.t,v 1.14 1998/03/01 23:52:53 tiggr Exp $</id>
    </copyright>

/******************** Connection ********************/

implementation class
Connection: State, DescriptorReadDelegate, Constants
{
  // Connections are not yet removed from this set.
  // Fri Aug  8 11:21:16 1997, tiggr@akebono.ics.ele.tue.nl
  <doc> All connection objects.  </doc>
  static MutableEqSet all_connections;
}

<doc> Store the new connection in the {all_connections}.  </doc>
instance (id)
  alloc
{
  // Ugly syntax!
  // Fri Aug  8 11:16:27 1997, tiggr@akebono.ics.ele.tue.nl
  Connection c = [super (class (State)) alloc];

  if (!all_connections)
    all_connections = [MutableEqSet new];
  [all_connections add c];

  = c;
}

<doc> Pass this message to the {connection}.  </doc>
void
       connection Connection connection
  remoteProxyDead int identity
pre
  !!all_connections[connection]
{
  [connection remoteProxyDead identity];
}

end;

implementation instance
Connection
{
  // Why distinct the root?  Isn't it just the object with identity 0?
  // Mon Jan  6 00:59:46 1997, tiggr@tricky.es.ele.tue.nl
  <doc> The root object of this connection.  </doc>
  public Any root;

  <doc> The Port serving this connection.  </doc>
  public Port port;

  <doc> The set of local proxies, keyed on their local object.  </doc>
  MutableEqDictionary local_objects;

  <doc> The set of local proxies, keyed on their identity.  </doc>
  MutableIntDictionary local_proxies;

  <doc> The set of remote proxies, keyed on their identity.  </doc>
  MutableIntDictionary remote_proxies;

  <doc> The set of remote proxy identities that are dead here and which
      need to be sent to the other side.  </doc>
  MutableIntArray unreported_deaths;

  <doc> The last number used as a local proxy identity.  </doc>
  int last_proxy_ident;
}

<doc> Designated initializer.  </doc>
id (self)
  initWithPort Port p
{
  local_objects = [MutableEqDictionary new];
  local_proxies = [MutableIntDictionary new];
  remote_proxies = [MutableIntDictionary new];
  [remote_proxies setIsContainer YES];
  unreported_deaths = [MutableIntArray new];

  port = p;
}

<doc> Other connections may inspect our proxies.  </doc>
protected IntDictionary
  local_proxies
{
  = local_proxies;
}
  
<doc> Return the local object identified by the {identity} to the other
    side.  </doc>
Any (object)
  localObject int identity
post
  object != nil
{
  = !identity ? root : [local_proxies[identity] original];
}

<doc> Return the local proxy to identify the local {object}.  </doc>
Proxy
  localProxyFor All object
pre
  object != nil
{
  Proxy p = local_objects[object];

  if (!p)
    {
      int identity = object == root ? 0 : ++last_proxy_ident;

      p = [[LocalProxy alloc] initWithConnection self
			      identity identity for object];
      local_objects[object] = p;
      local_proxies[identity] = p;
    }

  = p;
}

<doc> Return the remote object identified by the {identity} by the other
    side.  </doc>
Any
  remoteObject int identity
{
  All object = remote_proxies[identity];

  if (!object)
    {
      object = [[RemoteProxy alloc] initWithConnection self identity identity];
      remote_proxies[identity] = object;
    }

  = Any (object);
}

<doc><h4>Distributed Garbage Collection</h4></doc>

<doc> Be informed that the local proxy with the {identity} has one less
    remote proxy to care for.  If that number reaches zero, the local
    proxy object is removed.  </doc>
void
  localProxyRelease int identity
pre
  !!local_proxies[identity]
{
  LocalProxy lp = local_proxies[identity];

  [local_proxies remove identity];
  [local_objects remove [lp original]];
}

<doc> By informed (by our remote proxy with the {identity}) of the GC
    death of a remote proxy.

    Note that this method is invoked during GC and that no new objects
    should be allocated.  </doc>
void
  remoteProxyDead int identity 
{
  [unreported_deaths add identity];
}

end;

/******************** ServerConnection ********************/

implementation class
ServerConnection: Connection

end;

implementation instance
ServerConnection
{
  <doc> Our {port} is only here for accepting connections.  </doc>
  redeclare ServerPort port;
}

<doc> Designated initializer.  </doc>
id (self)
  initWithPort ServerPort p
{
  [super initWithPort p];
  [p registerForRead self];
}

// CCC This should be a `redeclare mutable All root;' ivar decl.
// Sat Jan  4 21:15:56 1997, tiggr@tricky.es.ele.tue.nl
void
  set_root All r
{
  root = Any (r);
}

<doc> Instantiate another {ConnectedConnection}.  </doc>
void
  readEventOnDescriptor ServerInetPort p
{
  [[ConnectedConnection alloc] initWithPort [p accept] for self];
}

end;

/******************** ConnectedConnection ********************/

implementation class
ConnectedConnection: Connection, DescriptorWriteDelegate, Conditions

end;

implementation instance
ConnectedConnection
{
  <doc> We are actually connected.  </doc>
  redeclare ConnectedPort port;

  <doc> Our decoder.  </doc>
  PortDecoder decoder;

  <doc> Our encoder.  </doc>
  PortEncoder encoder;

  <doc> If we're a slave connection (i.e. the working part for a published
      connection), this is the published server connection.  </doc>
  ServerConnection master;
  
  <doc> Iff {TRUE}, we've lost the connection.  </doc>
  boolean invalid;
}

<doc> Initializer for a client connection.  </doc>
id (self)
  initWithPort ConnectedPort p
{
  [super initWithPort p];

  root = Any ([[RemoteProxy alloc] initWithConnection self identity 0]);

  [self initDetails];
}

<doc> Initializer for a slave connection, i.e. a slave to the {server}
    connection.  </doc>
id (self)
  initWithPort ConnectedPort p
	   for ServerConnection server
{
  [super initWithPort p];

  master = server;

  [self initDetails];
}

<doc> Do part of the work for either initializer.  </doc>
protected void
  initDetails
{
  [port registerForRead self];

  decoder = [[PortDecoder alloc] initWithConnection self];
  encoder = [[PortEncoder alloc] initWithConnection self];
}

void
  invalidate
{
  if (!invalid)
    {
      invalid = YES;
      [[RunLoop current] removeReadDescriptor port];
      [port close];
    }
}

// CCC This should not be necessary.
// Sun Jan  5 17:37:38 1997, tiggr@tricky.es.ele.tue.nl
ConnectedPort
  port
{
  // CCC This should not be necessary.
  // Fri Jul 11 16:16:52 1997, tiggr@natlab.research.philips.com
  = ConnectedPort (port);
}

<doc> Forward to the {master} if we have one.  </doc>
Any
  localObject int identity
{
  = !master ? [super localObject identity] : [master localObject identity];
}

<doc> Forward to the {master} if we have one.  </doc>
Proxy
  localProxyFor All object
{
  = !master ? [super localProxyFor object] : [master localProxyFor object];
}

<doc> Forward to the {master} if we have one.  </doc>
void
  localProxyRelease int identity
pre
  !!master -> !![master local_proxies][identity]
{
  (!master ? [super localProxyRelease identity]
   : [master localProxyRelease identity]);
}

<doc> Forward the {invocation} to the other side.  </doc>
InvocationResult
  forward Invocation invocation
{
  Invocation inv;

  if (catch (self)
        bind ((stream-eos,
	     {
	       [self throw YES];
	       nil;
	     }))
	  ({
	    if ([unreported_deaths length] > 0)
	      {
		[encoder reportDeaths unreported_deaths];
		[unreported_deaths truncate 0];
	      }

	    [encoder encodeRoot invocation];

	    /* Make sure all bytes written end up at the other side.  */
	    [encoder flushOutput];

	    TypeDescription td = [invocation resultTypeDescription];
	    if ([td length] > 0 && [td component 0] != TYPEDESC_VOID)
	      {
		Any reply = [decoder decodeRoot];

		if ([reply invocationp])
		  [self unimplemented cmd
			message: "invocation during synchronous reply wait"];
		else
		  {
		    /* This is the result of our invocation.  Return it.  */
		    = reply;
		  }
	      }
	    NO;
	  }))
    [self invalidate];
}

void
  readEventOnDescriptor ConnectedInetPort p
{
  Invocation inv;

  /* Get the invocation, protecting against stream closes.  */
  if (catch (self)
        bind ((stream-eos, {[self throw YES]; nil;}))
	  ({inv = [decoder decodeRoot]; NO;}))
    {
      [self invalidate];
      return;
    }

  ConditionClass cc = condition;
  InvocationResult res;
  Condition raised;

  /* Execute the invocation, catching all _raised_ conditions.  */
  catch (self)
    bind ((cc,
	   {
	     if ([condition raised])
	       {
		 raised = condition;
		 [self throw void];
	       }
	     nil;
	   }
	   ))
      ({
	res = [inv fire];
       });

  if (raised != nil)
    [self unimplemented cmd
	  message: "condition during remote method execution"];

  /* Return the result if necessary, protecting against stream closes.  */
  TypeDescription td = [res typeDescription];
  if ([td length] > 0 && [td component 0] != TYPEDESC_VOID)
    {
      if (catch (self)
	    bind ((stream-error, {[self throw YES]; nil;}))
	      ({[encoder encodeRoot res]; NO;}))
	[self invalidate];
    }
}

end;
