(*
 *  C2caml : parses c headers and produces appropriate caml bindings for it
 *  Copyright (C) 1999  Sven LUTHER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(* $Id: process.ml,v 1.1 1999/06/19 14:59:20 sven Exp $ *)

open Expr

let cbc b = if b then "Cst_" else ""
let rec string_of_expr e = match e.expr, e.can_be_constant with
  | E_int n, b -> (cbc b) ^ "int : " ^ (string_of_int n)
  | E_float f, b -> (cbc b) ^ "float : " ^ (string_of_float f)
  | E_char c, b -> (cbc b) ^ "char : " ^ (String.make 1 c)
  | E_string s, b -> (cbc b) ^ "string : \"" ^ s ^ "\""
  | E_enum e, b -> (cbc b) ^ "enum : " ^ (string_of_int e)
  | E_ident i, b -> (cbc b) ^ "ident : <" ^ i ^ ">"
  | E_typedef, b -> (cbc b) ^ "typedef : ()"
  | E_liste l, b -> (cbc b) ^ "liste : ()"
  | E_enum_ident, b -> (cbc b) ^ "enum_ident : ()"
  | E_none, b -> (cbc b) ^ "unit : ()"
let cst_expr e = {expr=e; can_be_constant=true}
let no_cst_expr e = {expr=e.expr; can_be_constant=false}

let rec string_of_decl = function
  | D_symbol s -> s
  | D_table (d, e) ->
  (
    match e.expr with
    | E_int n -> (string_of_decl d) ^ "[" ^ (string_of_int n) ^ "]"
    | E_none -> (string_of_decl d) ^ "[]"
    | _ -> (string_of_decl d) ^ "["
  )
  | D_function d -> (string_of_decl d) ^ "()"
  | D_pointer d -> "* " ^ (string_of_decl d)
  | D_none -> ""

type inst =
  | I_goto of string
  | I_continue
  | I_break
  | I_return of expr
type oper =
  | Op_and
  | Op_mult
  | Op_add
  | Op_sub
  | Op_neg
  | Op_not
  | Op_affect
  | Op_mult_affect
  | Op_div_affect
  | Op_mod_affect
  | Op_add_affect
  | Op_sub_affect
  | Op_left_affect
  | Op_right_affect
  | Op_and_affect
  | Op_xor_affect
  | Op_or_affect
  
let rec print_all sot sep = function
  | [] -> ()
  | t::[] -> print_string (sot t)
  | t::q -> print_string ((sot t)^ sep); print_all sot sep q
let process a l =
(
  match a with
    | (T_typedef, _, T_enum (s, el)) ->
	(
      print_string "type ";
      (
        match l with
	| E_liste l -> print_all string_of_expr ", " l
	| _ -> print_string "expression not liste ..."
      );
      print_string " =\n";
	  let string_of_int_option = function
	    | None -> ""
	    | Some n -> " (* = " ^ (string_of_int n) ^ " *)"
	  in let string_of_enum (k,v) = "  | " ^ k ^ (string_of_int_option v) ^ "\n"
	  in try print_all string_of_enum "" el
	    with Not_found -> ()
	)
    | (T_typedef, _, _) ->
	(
      print_string "Following types are defined : ";
      (
        match l with
	| E_liste l -> print_all string_of_expr ", " l
	| _ -> print_string "expression not liste ..."
      );
      print_string " =\n";
	)
    | (_, _, _) ->
	(
      print_string "Following types are declared : ";
      (
        match l with
	| E_liste l -> print_all string_of_expr ", " l
	| _ -> print_string "expression not liste ..."
      );
      print_string "\n";
	)
)
(*
let rec print_all sot sep = function
  | [] -> ()
  | t::[] -> print_string (sot t)
  | t::q -> print_string ((sot t)^ sep); print_all sot sep q
let process a l =
(
  match a with
    | (T_typedef, _, T_enum el) ->
	(
      print_string "type ";
      print_all string_of_decl ", " l;
      print_string " =\n";
	  let string_of_int_option = function
	    | None -> ""
	    | Some n -> " (* = " ^ (string_of_int n) ^ " *)"
	  in let string_of_enum (k,v) = "  | " ^ k ^ (string_of_int_option v) ^ "\n"
	  in try print_all string_of_enum "" el
	    with Not_found -> ()
	)
    | (T_typedef, _, _) ->
	(
      print_string "Following types are defined : ";
      print_all string_of_decl ", " l;
      print_string " =\n";
	)
    | (_, _, _) ->
	(
      print_string "Following types are declared : ";
      print_all string_of_decl ", " l;
      print_string "\n";
	)
)
*)
