%
% Copyright (C) 1997 Thomas Nordin and Alastair Reid
%

\begin{code}

module Main(main, runGreenCard) where

import Package

import GetOpt
import Process( processFile )
import Target( Target(..) )
import ListUtils( split, dropSuffix )

import System  ( getArgs )
import Char    ( toLower )
import Maybe   ( fromMaybe, listToMaybe )

\end{code}


%************************************************************************
%*									*
\subsection{Main program}
%*									*
%************************************************************************


Driver code.

\begin{code}
main = 
  do argv <- getArgs
     greencard (snd (getOpts options [] argv))

-- Entry point when using Hugs interactively
runGreenCard str = greencard (snd (getOpts options [] (words str)))

{- Hugs test harness -}
main2 str = greencard (snd (getOpts options [] (words str)))


tstg file = main2 (unwords [file, "--include-dir .", "--target ghc"])
{- Hugs test harness
tsth file = main2 (unwords [file, "--debug", "--include-dir .", "--target Hugs"])
-}

ifOn :: Bool -> IO () -> IO () -> IO ()
ifOn True a b = a
ifOn _    a b = b


greencard :: [Options] -> IO ()
greencard opts = 
  ifOn optVersion (putStrLn version_msg) $
  ifOn optHelp	  (putStrLn usage_msg)   $
  case optfiles of
    []        -> putStrLn usage_msg
    (fname:_) -> 
      case targets of
        []         -> putStrLn ("Target not specified" ++ targets_supported_msg)
        ["hugs"]   -> greencard' Hugs fname
        ["nhc"]    -> greencard' NHC  fname
        ["ffi"]    -> greencard' FFI  fname
        ["ghc"]    -> greencard' (if optNoInline then GHC_ccall else GHC_casm) fname
	_          -> putStrLn ("Unrecognised target: " ++ (head targets) ++ 
				targets_supported_msg)
  where
    targets_supported_msg = " ( `ffi', `hugs', `ghc' and `nhc' recognised.)"

    greencard' target fname =
      processFile target 
		  optDebug
		  optStubDebug 
		  optVerbose 
		  optSafeCode
		  optHaskell14
		  optMangle
		  (optCallConv, optDllName)
		  includedirs
		  suffixes
		  fname
		  ofile
		  ocfile
      where
       ofile = 
        case optofiles of 
          (x:_) -> x
	  _ -> (case optoprefix of
	         (x:_) -> x
		 _     -> (dropSuffix fname)) ++ ".hs"

       ocfile = 
        case optocfiles of 
          (x:_) -> x
	  _ -> (case optoprefix of
	         (x:_) -> x
		 _     -> (dropSuffix fname)) ++ c_stub_suffix
		    where
		     c_stub_suffix =
		       case target of
		         Hugs -> ".c"
		         NHC  -> ".c"
			 _    -> "_stub.c"

    -- predicates for grabbing various on/off options:n
    optVersion	 = any (DumpVersion==)	opts
    optHelp	 = any (DumpHelp==)	opts
    optDebug	 = any (DumpDebug==)	opts
    optStubDebug = any (OptStubDebug==)	opts
    optVerbose	 = any (DumpVerbose==)	opts
    optSafeCode	 = any (OptSafeCode==)	opts
    optNoInline  = any (OptNoInline==)	opts
    optHaskell14 = any (OptHaskell14==) opts

    targets        = [ map toLower t | OptTarget t <- opts ]
    optfiles	   = [f   | OptFile f <- opts]
    optoprefix     = [pre | OptOutputPrefix pre <- opts] 
    optofiles      = [o   | OptOutputFile o     <- opts] 
    optocfiles     = [oc  | OptOutputCFile oc   <- opts] 
    optDllName	   = fromMaybe "" (listToMaybe [dl  | OptDllName dl <- opts])
    optCallConv    = fromMaybe defCallConv (listToMaybe [ cc  | OptCallConv cc <- opts])

{- BEGIN_GHC_ONLY
#if __GLASGOW_HASKELL__ <= 401
    defCallConv = "_ccall"
#else
    defCallConv = "ccall"
#endif
   END_GHC_ONLY -}
{- BEGIN_NOT_FOR_GHC -}
    defCallConv = "ccall"
{- END_NOT_FOR_GHC -}

    includedirs = 
      (concat [split ':' d | OptIncludeDirs d <- reverse opts]) ++
      default_paths
    suffixes    = 
      (concat [split ':' s | OptSuffix s <- reverse opts]) ++ 
      default_suffixes

    optMangle = 
       case [ m | OptNameMangle m <- opts] of
	 ("classic":_) -> False
         _ -> True  -- i.e., std in the event you haven't
		    --  - supplied it
		    --  - misspelled it.

default_paths :: [String]
default_paths = [""]

default_suffixes :: [String]
default_suffixes = ["","gc"]
\end{code}

The command-line options recognised by Green Card.

\begin{code}

version_msg = 
 unlines
 [ name ++ ", version " ++ version
 , ""
 , "Report bugs to <glasgow-haskell-bugs@dcs.gla.ac.uk>"
 ]

usage_msg   = 
 unlines
 [ "Usage: green-card [OPTION]... SOURCE"
 , ""
 , "Run Green Card, a foreign function interface preprocessor"
 , "for Haskell, over SOURCE"
 , ""
 , " -h, --help      print out this help message and exit"
 , " -v, --version   output version information and exit"
 , " -opre <prefix>  write Green Card Haskell output to <prefix>.{hs,c}"
 , " -o  <file>      write Green Card Haskell output to <file>"
 , " -oc <file>      write Green Card C output to <file>"
 , " -t<target>, --target <target"
 , "                 generate Haskell code for a particular system."
 , "                 Supported targets: ghc, hugs and nhc"
 , " -i<dirs>, --include-dir <dirs>"
 , "                 Add <dirs> to the include search path"
 , "                 (<dirs> is colon separated.)"
 , " --safe-code     call C `safely' (GHC only.)"
 , " --no-inline     put C code in a separate file (GHC only.)"
 , " -d, --debug     output extra debug information"
 , " --stub-debug    include debugging code in generated code"
 , " --dllname=<dll-name> (FFI backend only)"
 , "                 generating 'foreign imports' to dynamic library <dll-name>"
 , " --callconv={stdcall,ccall}"
 , " --haskell1.4    generating code compatible with a Haskell 1.4 system"
 , " --name-mangling-scheme={std,classic} "
 , "                 control how you map external names into"
 , "                 Haskell ones."
 , ""
 , "Green Card home page: "
 , "   http://www.dcs.gla.ac.uk/fp/software/green-card/"
 ]

options = 
  (prefixed "-" $ 
   opts
      [prefixed "-" $
       opts 
        [ "version"	  -= DumpVersion
        , "help"	  -= DumpHelp
        , "debug"	  -= DumpDebug
	, "suffix"        -=== OptSuffix
        , "output-prefix" -=== OptOutputPrefix
        , "no-inline"     -= OptNoInline
	, "stub-debug"    -= OptStubDebug
        , "verbose"	  -= DumpVerbose
        , "safe-code"	  -= OptSafeCode
        , "target"	  -=== OptTarget
        , "name-mangling-scheme="  -== OptNameMangle
        , "include-dir"	  -=== OptIncludeDirs
	, "dllname="		   -== OptDllName
	, "callconv="		   -== OptCallConv
	, "haskell1.4"             -=  OptHaskell14
        ]
      , "h"		-= DumpHelp
      , "d"		-= DumpDebug
      , "v"		-= DumpVerbose
      , "opre"		-=== OptOutputPrefix
      , "oc"		-=== OptOutputCFile
      , "o"		-=== OptOutputFile
      , "t"		-==  OptTarget
      , "s"	        -==  OptSuffix
      , "i"		-==  OptIncludeDirs
     ]) `orOpt`
    ((const True)  -? OptFile)

data Options
 = DumpVersion
 | DumpHelp
 | DumpDebug
 | DumpVerbose

 | OptSafeCode
 | OptNoInline
 | OptStubDebug
 | OptIncludeDirs String
 | OptTarget String
 | OptOutputPrefix String
 | OptOutputFile String
 | OptOutputCFile String
 | OptSuffix String
 | OptNameMangle String
 | OptFile String 
 | OptDllName  String
 | OptCallConv String
 | OptHaskell14
   deriving ( Eq )

\end{code}
