#! /bin/bash

# This program is designed to PGP sign a .dsc and .changes file pair
# in the form needed for a legal Debian upload.  It is based in part
# on dpkg-buildpackage.  It takes one argument: the name of the
# .changes file.  It also takes some options:
#  -m<maintainer>  Sign using key of <maintainer>
#  -k<key>     The PGP/GPG key ID to use; overrides -m
#  -p<type>    <type> is either pgp or gpg to specify which to use
#  -spgp,-sgpg The program takes arguments like pgp or gpg respectively
#  -a<arch>    Debian architecture
#  -t<type>    GNU machine type
#  -r [username@]remotehost  The changes (and dsc) files live on remotehost
#  --help, --version

# Debian GNU/Linux debsign.  Copyright (C) 1999 Julian Gilbey.
# Modifications to work with GPG by Joseph Carter and Julian Gilbey
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Abort if anything goes wrong
set -e

TEXTDOMAIN=devscripts
PKG_VERSION=@VERSION@

# --- Functions

usage () {
    echo \
$"Usage: debsign [options] [changes]
  Options:
    -r [usename@]remotehost
                    The machine on which the changes/dsc files live.
                    A changes file with full pathname (or relative
                    to the remote home directory must be given in
                    such a case
    -sgpg, -spgp    Sign takes options like GPG, PGP respectively
    -pgpg, -ppgp    Sign using GPG, PGP respectively
    -m_maintainer_  Sign using key of _maintainer_
    -k_key_         Sign using key ID _key_; overrides -m option
    -a_arch_        Search for changes file made for Debian build _arch_
    -t_target_      Search for changes file made for GNU _target_ arch
    --help          Show this message
    --version       Show version and copyright information
  If a changes file is specified, it is signed, otherwise debian/changes
  is parsed to find the changes file."
}

version () {
    echo \
$"This is debsign, from the Debian devscripts package, version $PKG_VERSION
This code is copyright 1999 by Julian Gilbey, all rights reserved.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later."
}

mustsetvar () {
    if [ "x$2" = x ]
    then
	echo >&2 $"$0: unable to determine $3"
	exit 1
    else
	# echo $"$0: $3 is $2"
	eval "$1=\"\$2\""
    fi
}

# This takes two arguments: the name of the file to sign and the
# key or maintainer name to use.  NOTE: this usage differs from that
# of dpkg-buildpackage, because we do not know all of the necessary
# information when this function is read first.
signfile () {
    if [ $signinterface = gpg ]
    then
	(cat "$1" ; echo "") | \
	    $signcommand --local-user "$2" --clearsign \
		--armor --textmode --output - - > "$1.asc"
    else
	$signcommand -u "$2" +clearsig=on -fast < "$1" > "$1.asc"
    fi
    echo
    mv -- "$1.asc" "$1"
}

withecho () {
    echo " $@"
    "$@"
}

# --- main script

# For security, even though we know it doesn't work :(
# I guess maintainers will have to be careful, and there's no way around
# this in a shell script.
unset IFS
PATH=/usr/local/bin:/usr/bin:/bin
umask `perl -e 'printf "%03o\n", umask | 022'`

if [ -e $HOME/.gnupg/secring.gpg ] ; then
        signcommand=gpg
else
        signcommand=pgp
fi

while [ $# != 0 ]
do
    value="`echo x\"$1\" | sed -e 's/^x-.//'`"
    case "$1" in
	-spgp)	forcesigninterface=pgp ;;
	-sgpg)	forcesigninterface=gpg ;;
	-p*)	signcommand="$value" ;;
	-m*)	maint="$value" ;;
	-k*)	signkey="$value" ;;
	-a*)	opt_a=1; arch="$value" ;;
	-t*)	targetgnusystem="$value" ;;
	-r*)	if [ -n "$value" ]; then remotehost=$value;
		elif [ $# -lt 1 ]; then
		    echo $"$0: -r option missing argument!" >&2
		    usage >&2; exit 1;
		else shift; remotehost=$1;
		fi ;;
	--help)	usage; exit 0 ;;
	--version)
		version; exit 0 ;;
	-*)	echo $"Unrecognised option: $1" >&2
		usage >&2; exit 1 ;;
	*)	break ;;
    esac
    shift
done

if test -n "$forcesigninterface" ; then
  signinterface=$forcesigninterface
else
  signinterface=$signcommand
fi

# If there is a command-line parameter, it is the name of a .changes file
# If not, we must be at the top level of a source tree and will figure
# out its name from debian/changelog
case $# in
    1)	case "$1" in
	    *.changes) ;;
	    *)	echo $"$0: Only a .changes file is allowed as argument!" >&2
		exit 1 ;;
	esac
	changes=$1
	dsc=`echo $changes | \
	    perl -pe 's/\.changes$/.dsc/; s/(.*)_(.*)_(.*)\.dsc/\1_\2.dsc/'`
	;;

    0)	# We have to parse debian/changelog to find the current version
	if [ -n "$remotehost" ]; then
	    echo $"Need to specify a changes file location with -r!" >&2
	    exit 1
	fi
	if [ ! -r debian/changelog ]; then
	    echo $"$0: Must be run from top of source dir or a .changes file given as arg" >&2
	    exit 1
	fi

	mustsetvar package "`dpkg-parsechangelog | sed -n 's/^Source: //p'`" \
	    $"source package"
	mustsetvar version "`dpkg-parsechangelog | sed -n 's/^Version: //p'`" \
	    $"source version"

	command -v dpkg-architecture > /dev/null 2>&1 && \
	    eval `dpkg-architecture -a${arch} -t${targetgnusystem} -s`
	test "${opt_a}" \
	    || arch=`dpkg-architecture -a${arch} -t${targetgnusystem} \
		-qDEB_HOST_ARCH 2> /dev/null` && test "${arch}" \
	    || mustsetvar arch "`dpkg --print-architecture`" $"build architecture"

	sversion=`echo "$version" | perl -pe 's/^\d+://'`
	pv="${package}_${sversion}"
	pva="${package}_${sversion}${arch:+_${arch}}"
	dsc="../$pv.dsc"
	changes="../$pva.changes"
	;;

    *)	echo $"$0: Only a .changes file is allowed as argument!" >&2
	exit 1 ;;
esac

# Do we have to download the changes file?
if [ -n "$remotehost" ]
then
    mkdir debsign.$$ || { echo $"Can't mkdir!" >&2; exit 1; }
    trap "rm -rf debsign.$$" 0 1 2 3 7 10 13 15
    remotechanges=$changes
    remotedsc=$dsc
    remotedir=`echo $changes | sed -e 's%/[^/]*$%%'`
    changes="debsign.$$/`basename $changes`"
    dsc="debsign.$$/`basename $dsc`"
    withecho scp "$remotehost:$remotechanges" "$changes"
fi

if [ ! -f "$changes" -o ! -r "$changes" ]
then
    echo $"Can't find or can't read changes file $changes!" >&2
    exit 1
fi

if [ -n "$maint" ]
then maintainer="$maint"
else maintainer=`sed -n 's/^Maintainer: //p' $changes`
fi

signas="${signkey:-$maintainer}"

# Is there a dsc file listed in the changes file?
if grep -q "`basename $dsc`" "$changes"
then
    if [ -n "$remotehost" ]
    then
	withecho scp "$remotehost:$remotedsc" "$dsc"
    fi

    if [ ! -f "$dsc" -o ! -r "$dsc" ]
    then
	echo $"Can't find or can't read dsc file $dsc!" >&2
	exit 1
    fi
    withecho signfile "$dsc" "$signas"
    dsc_md5=`md5sum $dsc | cut -d' ' -f1`

    cat $changes |
	perl -pe 'BEGIN {
	    '" \$dsc_file=\"$dsc\"; \$dsc_md5=\"$dsc_md5\"; "'
	    $dsc_size=(-s $dsc_file); ($dsc_base=$dsc_file) =~ s|.*/||;
	    $infiles=0;
	    }
	    /^Files:/ && ($infiles=1);
	    /^\s*$/ && ($infiles=0);
	    if ($infiles &&
		    /^ (\S+) (\d+) (\S+) (\S+) \Q$dsc_base\E\s*$/) {
		$_ = " $dsc_md5 $dsc_size $3 $4 $dsc_base\n";
		$infiles=0;
	    }' > $changes.new

    mv -- "$changes.new" "$changes"
    withecho signfile "$changes" "$signas"

    if [ -n "$remotehost" ]
    then
	withecho scp "$changes" "$dsc" "$remotehost:$remotedir"
    fi

    echo $"Successfully signed dsc and changes files"
else
    withecho signfile "$changes" "$signas"

    if [ -n "$remotehost" ]
    then
	withecho scp "$changes" "$remotehost:$remotedir"
    fi

    echo $"Successfully signed changes file"
fi

exit 0
