/*
 *             Automatically Tuned Linear Algebra Software v3.0Beta
 *                    (C) Copyright 1997 R. Clint Whaley                     
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions, and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. The name of the University, the ATLAS group, or the names of its 
 *      contributers may not be used to endorse or promote products derived
 *      from this software without specific written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE. 
 *
 */
#include <stdio.h>
#include <assert.h>

#define NTIM 3
#define TOLERANCE 1.2
#define BETTER 1.04  /* require at least 4% improvement, see below */

/*
 * We require at least 4% improvement before declaring a longer latency to
 * be better.  This is because longer latencies have less loop overhead, which
 * gives them a possible advantage, depending on the unrollings the compiler
 * does.  When you actually have too small a latency, this event should be
 * relatively catastrophic, with drops greater > 10%, so this is relatively
 * safe assumption.  Also, the smaller latencies require less registers to
 * satisfy, so they tend to be preferable.  - Clint
 */

void emit_muladd(char *type, char pre, int MULADD, int lat, 
                 unsigned long nlma,  /* # muladds in loop */
                 unsigned long mflop)    /* n mflops to time */
{
   int i, j, nama;
   char *ma;
   unsigned long reps;
   double mf;
   FILE *fpout;

   nama = (nlma + lat-1) / lat;  /* # of actual muladds in loop */
   mflop *= 1000000;
   reps = mflop / (2*nama);

   fpout = fopen("muladd.c", "w");
   assert(fpout != NULL);
   if (MULADD) ma = "Combined MULADD";
   else ma = "Separate multiply and add";
   fprintf(fpout, "#include <stdio.h>\n#include<assert.h>\n");
   fprintf(fpout, "double time00();\n");
/*
 * function macase()
 */
   fprintf(fpout, "\nstatic double macase(long nreps, int PRINT)\n{\n");
   fprintf(fpout, "   long i = nreps;\n");
   fprintf(fpout, "   double t0, tim, mf;\n");
   fprintf(fpout, "   register %s c0", type);
   for (i=1; i < lat; i++) fprintf(fpout, ", c%d", i);
   if (!MULADD) for (i=0; i < lat; i++) fprintf(fpout, ", m%d", i);
   fprintf(fpout, ";\n\n");

   fprintf(fpout, "   if (nreps > 0) c0 = 0.0;\n   else c0 = 2.2*nreps;\n");
   if (lat > 1 || !MULADD)
   {
      fprintf(fpout, "   ");
      for (i=1; i < lat; i++) fprintf(fpout, "c%d = ", i);
      if (!MULADD) for (i=0; i < lat; i++) fprintf(fpout, "m%d = ", i);
      fprintf(fpout, "c0;\n");
   }
   fprintf(fpout, "   t0 = time00();\n");
   fprintf(fpout, "   do\n   {\n");
   for (j=0; j < nama; j++)
   {
      for (i=0; i < lat; i++)
      {
         if (MULADD) fprintf(fpout, "      c%d += c%d * c%d;\n", i, i, i);
         else fprintf(fpout,"      c%d += m%d;\n      m%d = m%d * m%d;\n",
                      i, i, i, i, i);
      }
   }
   fprintf(fpout, "   }\n   while(--i);\n");
   fprintf(fpout, "   tim = time00() - t0;\n");
   fprintf(fpout, "   c0 = c0");
   for (i=1; i < lat; i++) fprintf(fpout, " + c%d", i);
   if (!MULADD) for (i=0; i < lat; i++) fprintf(fpout, " + m%d", i);
   fprintf(fpout, ";\n");
   mf = (2.0 * nama)*lat ;
   fprintf(fpout, "   if (tim < 0.0) mf = tim = 0.0;\n");
   fprintf(fpout, "   else mf = (nreps*%lf) / (1000000.0 * tim);\n", mf);
   fprintf(fpout, "   if (PRINT) printf(\"%%.1f:   %s, lat=%d, time=%%f, mflop=%%f\\n\", (float) c0, tim, mf);\n", ma, lat);
   fprintf(fpout, "   else printf(\"      %%.0f: NFLOP=%%.0f, tim=%%f\\n\", (float) c0, nreps*%f, tim);\n", mf);
   fprintf(fpout, "   return(tim);\n");
   fprintf(fpout, "}\n");
/*
 * main()
 */
   fprintf(fpout, "\nmain(int nargs, char **args[])\n{\n");
   fprintf(fpout, "   long nreps = 16000000/%d;\n", (2*nama));
   fprintf(fpout, "   int i, k;\n");
   fprintf(fpout, "   double t0, tim, mf;\n");
   fprintf(fpout, "   FILE *fp;\n");
   fprintf(fpout, "   fp = fopen(\"res/%cmuladd%d_%d\", \"w\");\n", pre, MULADD, lat);
   fprintf(fpout, "   assert(fp != NULL);\n");

   fprintf(fpout, "   fprintf(stdout, \"Finding granularity of timer:\\n\");");
   fprintf(fpout, "   while(macase(nreps, 0) < 0.75) nreps *= 4;\n");
   fprintf(fpout, "   fprintf(stdout, \"Done.\\n\");");

   fprintf(fpout, "   for(k=0; k < 3; k++)\n   {\n");
   fprintf(fpout, "   tim = macase(nreps, 1);\n");
   fprintf(fpout, "   if (tim < 0.0) mf = tim = 0.0;\n");
   fprintf(fpout, "   else mf = (nreps*%lf) / (1000000.0 * tim);\n", mf);
   fprintf(fpout, "   if (fp) fprintf(fp, \"%%f\\n\", mf);\n");
   fprintf(fpout, "   }\n");
   fprintf(fpout, "   fclose(fp);\n");
   fprintf(fpout, "   exit(0);\n");
   fprintf(fpout, "}\n");
   fclose(fpout);
}

double macase(char *type, char pre, int MULADD, int lat, int nlma, int mflops)
{
   char fnam[80], ln[80];
   int i, j;
   double mflop[NTIM], t0, tavg;
   FILE *fp;

   lat++;
   sprintf(fnam, "res/%cmuladd%d_%d", pre, MULADD, lat);
   fp = fopen(fnam, "r");
   if (fp == NULL)
   {
      emit_muladd(type, pre, MULADD, lat, nlma, mflops);
      assert(system("make xmuladd\n") == 0);
      assert( (fp = fopen(fnam, "r")) != NULL );
   }
   t0 = 0.0;
   j = 0;
   for (i=0; i != NTIM; i++)
   {
      assert( fscanf(fp, "%lf", &mflop[i]) );
   }
   fclose(fp);
/*
 * Sort results, largest first
 */
   for (i=0; i != NTIM; i++)
   {
      for (j=i+1; j < NTIM; j++)
      {
         if (mflop[i] < mflop[j])
         {
            t0 = mflop[i];
            mflop[i] = mflop[j];
            mflop[j] = t0;
         }
      }
   }

/*
 * Throw out result if it is outside tolerance; rerun if two mflop not within
 * tolerance
 */
   if (TOLERANCE*mflop[1] < mflop[0])  /* too big a range in results */
   {
      if (TOLERANCE*mflop[2] < mflop[1])
      {
         fprintf(stderr, "MULADD=%d, lat=%d: kill file and rerun with higher reps; variation exceeds tolerence\n", MULADD, lat);
         exit(-1);
      }
      tavg = mflop[1] + mflop[2] / 2.0;
   }
   else if (TOLERANCE*mflop[2] < mflop[0]) tavg = (mflop[0] + mflop[1]) / 2.0;
   else tavg = (mflop[0] + mflop[1] + mflop[2]) / 3.0;

   return(tavg);
}

int FindSafeNumRegs(char pre, char *type, int nlma, int mflops, 
                    int muladd, int lat)
{
   int i, k, lat2, nreg, nrmax, nrmin;
   double mf, mf0;

   if (muladd) nreg = lat;
   else nreg = 2*lat;
   mf0 = macase(type, pre, muladd, lat-1, nlma, mflops);
   for (k=1; k <= nreg; k *= 2);
   k /= 2;

   do
   {
      k *= 2;
      if (muladd) lat2 = k;
      else { lat2 = k / 2; k = lat2 * 2; }

      mf = macase(type, pre, muladd, lat2-1, nlma, mflops);
      fprintf(stdout, "      nreg=%d, mflop = %.2lf (peak %.2lf)\n",k, mf, mf0);
   }
   while (mf*BETTER >= mf0);
   fprintf(stdout, "\n  nreg < %d (drop to %.2f%%)\n\n", k, mf/mf0);

   nrmax = k;
   nrmin = k / 2;
   do
   {
      k = nrmin + (nrmax - nrmin) / 2;
      if (muladd) lat2 = k;
      else lat2 = k / 2;
      mf = macase(type, pre, muladd, lat2-1, nlma, mflops);
      if (mf*BETTER >= mf0) nrmin = k;
      else nrmax = k;
      fprintf(stdout, "      nreg=%d, mflop = %.2lf (peak %.2lf)\n",k, mf, mf0);
   }
   while (nrmax-nrmin > 1);
   return(nrmin);
}

void RunTime(char pre, char *type, int nlma, int mflop)
/*
 * If necessary, fills in mflop field by timing previously chosen case
 */
{
   char fnam[64];
   int muladd, lat, nreg;
   double mf;
   FILE *fp;

   sprintf(fnam, "res/%cMULADD", pre);
   fp = fopen(fnam, "r");
   if (fp == NULL) return;
   fscanf(fp, "%d", &muladd);
   fscanf(fp, "%d", &lat);
   fscanf(fp, "%lf", &mf);
   fscanf(fp, "%d", &nreg);
   fclose(fp);
   if (mf < 0.0)
   {
      mf = macase(type, pre, muladd, lat-1, nlma, mflop);
      fp = fopen(fnam, "w");
      assert(fp != NULL);
      fprintf(fp, "%d\n", muladd);
      fprintf(fp, "%d\n", lat);
      fprintf(fp, "%lf\n", mf);
      fprintf(fp, "%d\n", nreg);
      fclose(fp);
   }
}

main(int nargs, char *args[])
{
   char pre, *type, fnam[64];
   int k, kma, kno, kbest, maxlat, mflop, nlma=1;
   int MULADD=1;
   double mf0, mf, mfma=0, mfno=0;
   FILE *fp;

   if (nargs != 4 && nargs != 5)
   {
      fprintf(stderr, "usage: %s <pre> <maxlat> <mflop> [<nlma>]\n", args[0]);
      exit(-1);
   }
   pre = *args[1];
   if (pre == 'd') type = "double";
   else if (pre == 's') type = "float";
   else if (pre == 'q') type = "long double";
   else
   {
      fprintf(stderr, "\n\nUnknown prefix \'%c\'; aborting.\n\n", pre);
      exit(-1);
   }
   maxlat = atoi(args[2]);
   mflop = atoi(args[3]);  /* this is ignored now */

   if (maxlat <= 0)
   {
      RunTime(pre, type, nlma, mflop);
      exit(0);
   }
   if (nargs == 5) nlma = atoi(args[4]);
 
   kma = kno = kbest = -1;
   mf = mfma = mfno = 0.0;

   for (k=0; k < maxlat; k++)
   {
      mf0 = macase(type, pre, 1, k, nlma, mflop);
      if (mf0 > BETTER*mfma)
      {
         mfma = mf0;
         kma = k;
      }
      fprintf(stdout, "      MULADD=%d, lat=%d, mf=%.2lf\n", 1, k+1, mf0);
   }
   for (k=0; k < maxlat; k++)
   {
      mf0 = macase(type, pre, 0, k, nlma, mflop);
      if (mf0 > BETTER*mfno)
      {
         mfno = mf0;
         kno = k;
      }
      fprintf(stdout, "      MULADD=%d, lat=%d, mf=%.2lf\n", 0, k+1, mf0);
   }
/* 
 * Award ties to separate multiply/add, since it may just be the compiler
 * making the machine behave like it has muladd
 */
   if (BETTER*mfno >= mfma)
   {
      MULADD = 0;
      mf = mfno;
      kbest = kno;
   }
   else
   {
      mf = mfma;
      kbest = kma;
   }
   k = FindSafeNumRegs(pre, type, nlma, mflop, MULADD, kbest+1);
   fprintf(stdout, "\nLower bound on number of registers = %d\n", k);
   sprintf(fnam, "res/%cMULADD", pre);
   fp = fopen(fnam, "w");
   assert(fp != NULL);
   fprintf(fp, "%d\n", MULADD);
   fprintf(fp, "%d\n", kbest+1);
   fprintf(fp, "%lf\n", mf);
   fprintf(fp, "%d\n", k);
   fclose(fp);
   fprintf(stdout,
"\n\nAccording to these timings, I think you need a %d cycle latency,\n", kbest+1);
   if (MULADD) fprintf(stdout, "with a combined multiply/add instruction.\n");
   else fprintf(stdout, "with separate multiply and add instructions.\n\n");
   fprintf(stdout, "These parameters gave a performance of %f MFLOPS.\n\n", mf);
   exit(0);
}

