------------------------------------------------------------------------------
--                                                                          --
--                 ASIS-for-GNAT IMPLEMENTATION COMPONENTS                  --
--                                                                          --
--                          A 4 G . C O N T T . T T                         --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--            Copyright (c) 1995-1999, Free Software Foundation, Inc.       --
--                                                                          --
-- ASIS-for-GNAT is free software; you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIS-for-GNAT is distributed  in the hope  that it will be use- --
-- ful, but WITHOUT ANY WARRANTY; without even the implied warranty of MER- --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License  distributed with ASIS-for-GNAT; see file     --
-- COPYING. If not, write to the Free Software Foundation,  59 Temple Place --
-- - Suite 330,  Boston, MA 02111-1307, USA.                                --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- ASIS-for-GNAT was originally developed  by the ASIS-for-GNAT team at the --
-- Software  Engineering  Laboratory  of  the Swiss  Federal  Institute  of --
-- Technology (LGL-EPFL) in Lausanne,  Switzerland, in cooperation with the --
-- Scientific  Research  Computer  Center of  Moscow State University (SRCC --
-- MSU), Russia,  with funding partially provided  by grants from the Swiss --
-- National  Science  Foundation  and  the  Swiss  Academy  of  Engineering --
-- Sciences.  ASIS-for-GNAT is now maintained by  Ada Core Technologies Inc --
-- (http://www.gnat.com).                                                   --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines for each ASIS Context the corresponding Tree Table,
--  which contains the information about the tree output files needed for
--  handling and swapping the ASTs represented by teh tree output files
--  accessed by ASIS.

with Asis;

package A4G.Contt.TT is              --  Context_Table.Tree_Tables

   ----------------
   -- Tree Table --
   ----------------

   --  The tree table has an entry for each AST ( = tree output file)
   --  created and read at leat once for this run of ASIS application.

   --  The entries in the table are accessing using a Tree_Id which
   --  ranges from Nil_Tree (a special value using for initializing
   --  ASIS Nil_Element and ASIS Nil_Compilation_Unit) to Last_Tree.
   --  Each entry has the following fields:

   ---------------------
   -- Tree Name Table --
   ---------------------

   procedure Get_Name_String (C : Context_Id; Id : Tree_Id);
   --  Get_Name_String is used to retrieve the string associated with
   --  an entry in the name table. The resulting string is stored in
   --  Name_Buffer and Name_Len is set.

   function Allocate_Tree_Entry (C : Context_Id) return Tree_Id; --  #####
   --  Allocates the new entry in the Tree Table for the tree output file
   --  name stored in the A_Name_Buffer (A_Name_Len should be set
   --  in a proper way).

   ------------------------------
   -- Internal Tree Attributes --
   ------------------------------

   --  Each Tree entry contains the following fields, representing the Tree
   --  attributes needed to organize tree processing inside ASIS
   --  implementation:

   --  Enclosing_Lib : Context_Id;                          --##
   --    Context Id of the ASIS Context for which the tree has been
   --    created.

   --  Main_Unit_Id : Unit_Id;
   --    The ASIS Compilation Unit, correspondig to the main unit in
   --    the tree

   --  Main_Top : Node_Id;
   --    The top node (having N_Compilation_Unit Node Kind) of Main_Unit

   --   Units : Elist_Id;
   --     The list of all the Units (or all the Units except Main_Unit?)
   --     which may be processed on the base of this tree, [each Unit
   --     is accompanied by its top node, which it has in the given tree
   --     ??? Not implemented for now!]


   ---------------------------------------------------------------
   --  Internal Tree Unit Attributes Access and Update Routines --
   ---------------------------------------------------------------


   procedure Set_Main_Unit_Id (C : Context_Id; T : Tree_Id; U : Unit_Id);
   procedure Set_Main_Top     (C : Context_Id; T : Tree_Id; N : Node_Id);
   --  Do we really need Set procedures having a Tree (and its "enclosing"
   --  Context) as a parameter? Now it seems, that all settings will be
   --  done for the currently accessing Tree only.

   procedure Set_Main_Unit_Id (U : Unit_Id);
   procedure Set_Main_Top     (N : Node_Id);

   -----------------------------------
   -- Subprograms for Tree Swapping --
   -----------------------------------

   function Unit_In_Current_Tree (C : Context_Id; U : Unit_Id) return Boolean;
   --  Checks if the subtree for a given Unit defined by C and U, is
   --  contained in the currently accessed tree.

   procedure Reset_Tree (Context : Context_Id; Tree : Tree_Id);
   --  Resets the currently accessed tree to the tree identified by
   --  the Context and Tree parameters

   procedure Reset_Tree_For_Unit (C : Context_Id; U : Unit_Id);
   procedure Reset_Tree_For_Unit (Unit : Asis.Compilation_Unit);
   --  Resets the currently accessed tree to some tree containing
   --  the subtree for a given unit. For now, there is no special
   --  strategy for choosing the tree among all the trees containing
   --  the given unit

   procedure Reset_Main_Tree
      (Unit    : Asis.Compilation_Unit;
       Success : out Boolean);
   --  For a given Unit, if the Unit is a main unit in some tree, resets this
   --  tree and sets Success True, othervise sets Success false and does
   --  nothing else

   procedure Append_Tree_To_Unit (C : Context_Id; U : Unit_Id);
   --  Appends the currently accessed tree to the list of the (consistent)
   --  trees containing a given Unit (this tree list belongs to the unit U).

   procedure Set_New_Unit_List;
   --  Creates a new empty Units list for the currently accessed tree

   procedure Append_Unit_To_Tree (U : Unit_Id);
   --  Appends a new Unit to the Unit list of the currently accessed tree

   --  OPEN PROBLEM: do we really need unit lists for trees? If not, we
   --  should get rid of unit lists and related routines everywhere!!!
   --  this should be revised, when migrating to processing pre-created
   --  trees is completed


   ---------------------------------
   -- General-Purpose Subprograms --
   ---------------------------------

   procedure Finalize (C : Context_Id);
   --  Currently this routine is only used to generate debugging output
   --  for the Tree Table of a given Context.

   function Present (Tree : Tree_Id) return Boolean;
   --  Tests given Tree Id for non-equality with No_Tree_Name.
   --  This allows  notations like "if Present (Tree)" as opposed to
   --  "if Tree /= No_Tree_Name"

   function No (Tree : Tree_Id) return Boolean;
   --  Tests given Tree Id for equality with No_Tree_Name. This allows
   --  notations like "if No (Tree)" as opposed to
   --  "if Tree = No_Tree_Name"

   function Last_Tree (C : Context_Id) return Tree_Id;
   --  Returns the Tree_Id of the last tree which has been allocated
   --  in the Tree Table.

   procedure Output_Tree (C : Context_Id; Tree : Tree_Id);
   --  Produces the debug output of the Tree Table entry corresponding
   --  to Tree

   procedure Print_Trees (C : Context_Id);
   --  Produces the debug output from the Tree table for the Context C.

end A4G.Contt.TT;