/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include "loader.h"

/*--------------------------------------------------------------------------*/

typedef struct memfrag_str
{
	unsigned long		size; 		/* size of free block including header  */
	struct memfrag_str	*next;		/* pointer to next memory fragment		*/

} memfrag_t;

/*--------------------------------------------------------------------------*/

static unsigned long	current_file_position = 0;
static const FILEMAP	*current_file_map	  = NULL;
static memfrag_t		*free_list            = NULL;
static unsigned			pcnt_current;
static unsigned			pcnt_total;
static int				pcnt_backspaces;

/*--------------------------------------------------------------------------*/
/* initialise memory allocator
 */

void
malloc_init
(	void			*heap_base,
	unsigned long	heap_size
)
{
	if (free_list == NULL)
	{
		free_list       = heap_base;
		free_list->size = heap_size;
		free_list->next = NULL;
	}
}

/*--------------------------------------------------------------------------*/

void *
malloc
(	unsigned long	size
)
{	memfrag_t		*frag;
	memfrag_t		*last;

	/* long word align and add space to store allocation size */
	size = (size + sizeof(frag->size) + 3) & ~3;

	frag = free_list;
	last = NULL;

	while (frag)
	{
		if (frag->size >= size)
		{
			goto frag_found;
		}

		/* onwards ... */
		last = frag;
		frag = frag->next;
	}

	/* nothing found! */
	return NULL;

	/* A suitable fragment has been found
	 * now we must unlink it from the free list.
	 * If the fragment is large enough just take
	 * the allocation from the end.
	 */
frag_found:
	/* is it big enough to split into two fragments */
	if (frag->size >= (size + sizeof(memfrag_t)))
	{
		/* adjust for amount removed */
		frag->size -= size;

		/* point at allocated chunk at end */
		frag = (memfrag_t *) ((char *)frag + frag->size);

		/* store allocated size */
		frag->size = size;
	}
	else
	{
		/* unlink found fragment */
		if (last)
		{
			last->next = frag->next;
		}
		else
		{
			free_list  = frag->next;
		}
	}

	/* return pointer to allocated memory */
	return &(frag->next);
}

/*--------------------------------------------------------------------------*/

int
free
(	void		*memaddr
)
{	memfrag_t	*last;
	memfrag_t	*frag;
	memfrag_t	*dead;

	if (memaddr == NULL)
	{
		return -1;
	}

	/* recover pointer to base of allocation */
	dead = (memfrag_t *) ((char *)memaddr - sizeof(frag->size));

	frag = free_list;
	last = NULL;

	/* find fragment that will follow us */
	while (frag && (frag < dead))
	{
		last = frag;
		frag = frag->next;
	}

	/* make sure it doesn't overlap previous fragment */
	if (last)
	{
		if ((char *)dead < ((char *)last + last->size))
		{
			return -1;
		}
	}

	/* make sure it doesn't overlap next fragment */
	if (frag)
	{
		if (((char *)dead + dead->size) > (char *)frag)
		{
			return -1;
		}
	}

	/* can we merge with next fragment */
	if (((char *)dead + dead->size) == (char *)frag)
	{
		/* merge with next fragment */
		dead->next  = frag->next;
		dead->size += frag->size;
	}
	else
	{
		/* link to next */
		dead->next = frag;
	}

	/* if there is a previous fragment */
	if (last)
	{
		/* can we merge with previous fragment */
		if (((char *)last + last->size) == (char *)dead)
		{
			/* merge with previous fragment */
			last->next  = dead->next;
			last->size += dead->size;
		}
		else
		{
			/* just link to previous fragment */
			last->next  = dead;
		}
	}

	/* no previous fragment */
	else
	{
		/* make new head of list */
		free_list = dead;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/

void
mem_clear
(	void			*mem,
	unsigned long	count
)
{
	unsigned long	a = (unsigned long) mem;

	while ((a & 3) && count)
	{
		*(char *)a = 0;
		a++;
		count--;
	}
	while (count > 3)
	{
		*(long *)a = 0;
		a     += 4;
		count -= 4;
	}
	while (count)
	{
		*(char *)a = 0;
		a++;
		count--;
	}
}

/*--------------------------------------------------------------------------*/

void
mem_move
(	void			*dest,
	const void		*srce,
	unsigned long	count
)
{
	char			*d = dest;
	const char		*s = srce;

	if (d > s)
	{
		d += count;
		s += count;
		while (count--)
		{
			*--d = *--s;
		}
	}
	else
	{
		while (count--)
		{
			*d++ = *s++;
		}
	}
}

/*--------------------------------------------------------------------------*/

int
mem_cmp
(	void			*mem1,
	void			*mem2,
	unsigned long	count
)
{	unsigned char	c   = 0;
	unsigned char	*p1 = (unsigned char *) mem1;
	unsigned char	*p2 = (unsigned char *) mem2;

	while (c == 0 && count--)
		c = *p1++ - *p2++;

	return (int) c;
}

/*--------------------------------------------------------------------------*/

void
disable_icache
(	unsigned char	cpu_type
)
{
	if (cpu_type == 30)
	{
		/* TODO */
	}
	else
	{
		/* disable and invalidate instruction cache */
		__asm__ __volatile__ (
		"	movec	%%cacr,%%d1
			bclr	#15,%%d1
			movec	%%d1,%%cacr
			.word	0xf498			| cinva	ic
		"
		: /* no outputs */
		: /* no inputs */
		: "d1"
		);
	}
}

/*--------------------------------------------------------------------------*/

void
enable_icache
(	unsigned char	cpu_type	
)
{
	if (cpu_type == 30)
	{
		/* TODO */
	}
	else
	{
		/* enable instruction cache */
		__asm__ __volatile__ (
		"	movec	%%cacr,%%d1
			bset	#15,%%d1
			movec	%%d1,%%cacr
		"
		: /* no outputs */
		: /* no inputs */
		: "d1"
		);
	}
}

/*--------------------------------------------------------------------------*/

#ifdef NOT_USED
void
invalidate_icache
(	unsigned char	cpu_type
)
{
	if (cpu_type == 30)
	{
		/* TODO */
	}
	else
	{
		__asm__ __volatile__ (
		"	.word	0xf498		| cinva ic - invalidate instruction cache
		"
		);
	}
}
#endif /* NOT_USED */

/*--------------------------------------------------------------------------*/
/* Determine CPU type:
 *
 * returns 30 if 68030
 * or      40 if 68040
 * or      60 if 68060
 */

unsigned char
get_cpu_type
(	void
)
{	int		cpu;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d0			| save trace flag
		movec	%%vbr,%%a0			| get vector table address
		move.l	0x10(%%a0),%%d1		| save original illegal inst error vector
		lea.l	1f(%%pc),%%a1		| get new illegal instruction handler
		move.l	%%a1,0x10(%%a0)		| set it
		move.l	%%sp,%%a1			| save stack pointer
		moveq	#60,%%d3			| default: 68060
		dc.l	0x4e7aa803			| movec	%%msp,%%a2
		moveq	#40,%%d3			| default: 68040
		dc.l	0x4e7aa802			| movec %%caar,%%a2
		moveq	#30,%%d3			| default: 68030
1:
		move.l	%%a1,%%sp			| restore stack pointer
		move.l	%%d1,0x10(%%a0)		| restore original handler
		move.w	%%d0,%%sr			| restore trace flag
		move.l	%%d3,%0				| return CPU identifier
	"
	: "=d" (cpu)
	: /* no inputs */
	: "d0", "d1", "d3", "a0", "a1", "a2", "memory"
);

	return cpu;
}

/*--------------------------------------------------------------------------*/
/* Determine FPU type:
 *
 * returns 0 not present
 */

unsigned char
get_fpu_type
(	void
)
{	int		fpu;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d0				| save trace flag
		movec	%%vbr,%%a0				| get vector table address
		move.l	0x2c(%%a0),%%d1			| save original F-line handler
		lea.l	1f(%%pc),%%a1			| get new F-line handler
		move.l	%%a1,0x2c(%%a0)			| set it
		move.l	%%sp,%%a1				| save stack pointer
		moveq	#0,%%d3					| default: no FPU
		dc.l	0xf2800000				| try fnop 
		moveq	#1,%%d3					| got here, must have FPU
1:
		move.l	%%a1,%%sp				| restore stack pointer
		move.l	%%d1,0x2c(%%a0)			| restore original handler
		move.w	%%d0,%%sr				| restore trace flag
		move.l	%%d3,%0					| return FPU fitted status
	"
	: "=d" (fpu)
	: /* no inputs */
	: "d0", "d1", "d3", "a0", "a1", "a2", "memory"
);

	return fpu;
}

/*--------------------------------------------------------------------------*/
/* Probe given address for the existance of RAM
 *
 * Return non-zero if RAM found.
 */

#if MEMORY_SIZE == 0
int
ram_probe
(	unsigned long	where
)
{	int				rv;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d4				| save trace flag
		movec	%%vbr,%%a0				| get vector table ptr
		move.l	0x8(%%a0),%%d5			| save BERR vector
		lea.l	1f(%%pc),%%a1			| new BERR handler
		move.l	%%a1,0x8(%%a0)			| set it
		move.l	%%sp,%%a1				| save stack frame
		moveq	#0,%0					| say no memory here
		move.l	#0x7e57ed17,%%d1		| get pattern
		move.l	(%1),%%d2				| save memory
		move.l	4(%1),%%d3				| save memory + 4
		move.l	%%d1,(%1)				| store pattern
		not.l	%%d1					| new pattern
		move.l	%%d1,4(%1)				| store new pattern, disturb bus
		not.l	%%d1					| old pattern
		cmp.l	(%1),%%d1				| did it store
		bne.s	1f						| branch if not
		move.l	%%d2,(%1)				| restore memory
		move.l	%%d3,4(%1)				| restore memory + 4
		moveq	#1,%0					| say memory found
1:
		move.l	%%a1,%%sp				| restore SP
		move.l	%%d5,0x8(%%a0)			| restore BERR handler
		move.w	%%d4,%%sr				| restore trace flag
	"
	: "=d" (rv)
	: "a" (where)
	: "d1", "d2", "d3", "d4", "d5", "a0", "a1", "memory"
);

	return rv;
}
#endif

/*--------------------------------------------------------------------------*/

int
strlen
(	const char	*s
)
{	const char	*e = s;

	while (*e++)
		;

	return (e - s) - 1;
}

/*--------------------------------------------------------------------------*/

char *
strcpy
(	char		*d,
	const char	*s
)
{	char		*sd = d;

	while ((*d++ = *s++) != '\0')
		;

	return sd;
}

/*--------------------------------------------------------------------------*/

void
strcat
(	char		*d,
	const char	*s
)
{
	while (*d++ != '\0')
		;

	d--;

	while ((*d++ = *s++) != '\0')
		;
}

/*--------------------------------------------------------------------------*/

void
strcatn
(	char			*d,
	const char		*s,
	unsigned long	n
)
{
	while (n && (*d++ != '\0'))
	{
		n--;
	}

	d--;

	while (--n && *s)
	{
		*d++ = *s++;
	}

	*d = '\0';
}

/*--------------------------------------------------------------------------*/

void
strncpy
(	char			*d,
	const char		*s,
	unsigned long	n
)
{	const char		*save;
	unsigned long	max;

	save = s;
	max  = 1;

	while (*s++) max++;

	s = save;

	if (n < max) max = n;

	n -= max;

	while (max--)
	{
		*d++ = *s++;
	}

	while (n--)
	{
		*d++ = '\0';
	}
}

/*--------------------------------------------------------------------------*/
/* printf style output formatter
 */

int
vsoutput
(	void 		(*output)(const int),
	const char	*fmt,
	va_list		va
)
{	int			c;
	int			pos   = 0;
#define			BUFSZ	15

	while ((c = *fmt++))
	{
	    if (c != '%')
	    {
			(*output)(c);
			pos++;
		}
		else
		{
			enum {
				FF_DEFAULT = 0,
				FF_DUMP    = 0x01,	/* dump string to output	*/
				FF_ALT     = 0x02,	/* #, alternate format		*/
				FF_SHORT   = 0x04,	/* h, short arg				*/
				FF_LONG    = 0x08,	/* l, long arg				*/
				FF_ZERO    = 0x10,	/* 0, zero fill				*/
				FF_LEFT    = 0x20,	/* -, left adjust			*/
				FF_PREC    = 0x40,	/* .*, precision			*/
				FF_NEG     = 0x80	/* signed arg				*/
			}				flags   = FF_DEFAULT;
			long			n;
			unsigned long	u;
			char			buf[BUFSZ + 1];
			char			*p    = buf + BUFSZ;
			int				sign  = '-';
			int				width = 0;
			int				prec  = 0;

			*p = '\0';

			/* scan flags */
			while (1)
			{
				switch (c = *fmt++)
				{
					case '0':
					{
						flags |= FF_ZERO;
						break;
					}

				  	case '#':		/* alternate format */
				  	{
						flags |= FF_ALT;
						break;
					}

					case ' ':		/* blank sign */
					{
						sign = ' ';
						break;
					}

					case '+':		/* +/- sign */
					{
						sign = '+';
						break;
					}

				  	case '-':		/* left justified */
				  	{
						flags |= FF_LEFT;
						break;
					}

				  	default:
				  	{
						goto scan_width;
					}
				}
			}

		scan_width:
			/* scan width */
		  	if (c == '*')
		  	{
		  		/* width from arg list */
				if ((width = va_arg(va, int)) < 0)
				{
					width  = -width;
					flags |= FF_LEFT;
				}
				c = *fmt++;
			}
			else
			{
				while ('0' <= c && c <= '9')
				{
					width = (width * 10) + (c - '0');
					c     = *fmt++;
				}
			}

			if (c == '.')
			{
				/* scan precision */
				flags |= FF_PREC;
				c      = *fmt++;

				if (c == '*')
				{
					/* precision from arg list */
					if ((prec = va_arg(va, int)) < 0)
					{
						prec = -prec;
					}
					c = *fmt++;
				}
				else
				{
					while ('0' <= c && c <= '9')
					{
						prec = (prec * 10) + (c - '0');
						c    = *fmt++;
					}
				}
			}

			/* length modifiers */
			if (c == 'h')
			{
				flags |= FF_SHORT;
				c      = *fmt++;
			}
			else if (c == 'l')
			{
				flags |= FF_LONG;
				c      = *fmt++;
			}

			/* do conversion */
			switch (c)
			{
				case '%':		/* %% -> % */
				{
					(*output)(c);
					pos++;
					break;
				}

				case 'n':		/* save position */
				{
					*va_arg(va, int *) = pos;
					break;
				}

				case 'c':		/* character */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
				  	  : va_arg(va, unsigned int);
					*--p = u;
					flags |= FF_DUMP;
					break;
				}

			  	case 's':		/* string */
			  	{
					if ((p = va_arg(va, char *)) == NULL)
					{
						p = "";
					}

					if ((flags & FF_PREC) && strlen(p) > prec)
					{
						pos += prec;

						while (--prec >= 0)
						{
							(*output)(*p++);
						}

						break;
					}

					flags |= FF_DUMP;
					break;
				}

		  		case 'i': case 'd': case 'u': /* decimal */
		  		{
					if (c != 'u')
					{
						/* signed */
						n = (flags & FF_SHORT) ? va_arg(va, short)
						  : (flags & FF_LONG ) ? va_arg(va, long )
						  : va_arg(va, int);

						if (n < 0)
						{
							flags |= FF_NEG;
						}

						u = (n < 0) ? -n : n;
					}
					else
					{
						u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
						  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
						  : va_arg(va, unsigned int);
					}

					do	{
						*--p = '0' + u % 10;
						u   /= 10;
					} while (u != 0);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if (flags & FF_NEG)
					{
						*--p = '-';
					}
					else
					{
						if (sign != '-')
						{
							*--p = (sign == '+') ? '+' : ' ';
						}
					}

					flags |= FF_DUMP;
					break;
				}

				case 'x': case 'X':	/* hex, Hex */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
					  : va_arg(va, unsigned int);
					do	{
						*--p = "0123456789ABCDEF"[u%16];
						u   /= 16;
					} while (u);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if (flags & FF_ALT)
					{
						*--p = 'x';
						*--p = '0';
					}

					flags |= FF_DUMP;
					break;
				}

				case 'o':		/* octal */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
					  : va_arg(va, unsigned int);
					do	{
						*--p = '0' + u % 8;
						u   /= 8;
					} while (u);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if ((flags & FF_ALT) && *p != '0')
					{
						*--p = '0';
					}

					flags |= FF_DUMP;
					break;
				}

				default:		/* error */
				{
					(*output)('%');
					(*output)(c);
					pos += 2;
					break;
				}
			}

			/* copy adjusted string "p" to output */
			if (flags & FF_DUMP)
			{
				int len = strlen(p);
				int pad = width - len;

				if (!(flags & FF_LEFT))
				{
					int padder = (flags & FF_ZERO) ? '0' : ' ';

					while (--pad >= 0)
					{
						(*output)(padder);
					}
				}

				while (*p)
				{
					(*output)(*p++);
				}

				if (flags & FF_LEFT)
				{
					while (--pad >= 0)
					{
						(*output)(' ');
					}
				}

				pos += (len < width) ? width : len;
			}
		}
	}

	return pos;
}

/*--------------------------------------------------------------------------*/
/* Format string
 */

static char *_sprintf_ptr;
static void  _sprintf_out(const int c) {*_sprintf_ptr++ = c;}

int
sprintf
(	char		*buff,
	const char	*fmt,
	...
)
{	va_list		args;
	int			n;

    _sprintf_ptr = buff;

    va_start(args, fmt);
	n = vsoutput(_sprintf_out, fmt, args);
    va_end(args);

	*_sprintf_ptr = '\0';

    return n;
}

/*--------------------------------------------------------------------------*/
/* Print formated string.
 */

int
printf
(	const char	*fmt,
	...
)
{	va_list		args;
	int			n;

    va_start(args, fmt);
	n = vsoutput(put_char, fmt, args);
    va_end(args);

    return n;
}

/*--------------------------------------------------------------------------*/
/* Print unformated string.
 */

int
put_str
(	const char	*str
)
{	int			n = 0;

	while (*str)
	{
		put_char(*str++);
		n++;
	}

	return n;
}

/*--------------------------------------------------------------------------*/
/* Report fatal error
 */

void
panic
(	const char	*fmt,
	...
)
{	va_list		args;

	put_str("\nLILO Panic: ");
    va_start(args, fmt);
    vsoutput(put_char, fmt, args);
    va_end(args);
    put_char('\n');
 
    while (1)
		;
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and second is a prefix of the first
 * while ignoring case.
 */

int
prefix_string
(	const char	*s,						/* string							*/
	const char	*p						/* prefix							*/
)
{	char		sc;
	char		pc;

	if ((s == NULL) || (p == NULL))
	{
		return FALSE;
	}

	/* until we reach end of prefix */
	while (*p)
	{
		sc = *s++;
		pc = *p++;

		/* convert to lower case */
		if ((sc >= 'A') && (sc <= 'Z')) sc += 'a' - 'A';
		if ((pc >= 'A') && (pc <= 'Z')) pc += 'a' - 'A';

		if (sc != pc)
		{
			return FALSE;
		}
	}

	/* make sure we are either at end of source or in white space */
	if (*s >= '!')
	{
		return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal ignoring case.
 */

int
equal_strings
(	const char	*s1,
	const char	*s2
)
{	char		c1;
	char		c2;

	if ((s1 == NULL) || (s2 == NULL))
	{
		return FALSE;
	}

	while (*s1 || *s2)
	{
		c1 = *s1++;
		c2 = *s2++;

		/* convert to lower case */
		if ((c1 >= 'A') && (c1 <= 'Z')) c1 += 'a' - 'A';
		if ((c2 >= 'A') && (c2 <= 'Z')) c2 += 'a' - 'A';

		if (c1 != c2)
		{
			return FALSE;
		}
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal including case
 */

int
case_equal_strings
(	const char	*s1,
	const char	*s2
)
{
	if ((s1 == NULL) || (s2 == NULL))
	{
		return FALSE;
	}

	while (*s1 || *s2)
	{
		if (*s1++ != *s2++)
		{
			return FALSE;
		}
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

int
file_size
(	const char		*path
)
{	const FILEMAP	*map;

	if ((map = find_file_map(path)) == NULL)
	{
		return -1;
	}

	return MAP_FILESIZE(map);
}

/*--------------------------------------------------------------------------*/

int
file_open
(	const char	*path
)
{
    if ((current_file_map = find_file_map(path)) == NULL)
    {
		return -1;
	}

	current_file_position = 0;

	if (v->debug_mode)
	{
		const FILEMAP *p = MAP_FIRSTFRAG(current_file_map);
		unsigned long  n = MAP_NUMFRAGS (current_file_map);

		printf("\nOPEN: file map at %08lx, %ld entries using sector size %d\n",
				(unsigned long) current_file_map, n, SECTOR_SIZE);

		while (n)
		{
			printf("  Frag offset %08lx, length %08lx\n",
				p->offset, p->length);

			p++;
			n--;
		}
	}

    return 0;
}

/*--------------------------------------------------------------------------*/

long
file_tell
(	void
)
{
	return current_file_position;
}

/*--------------------------------------------------------------------------*/

int
file_seek
(	int			where,
	int			whence
)
{	int			filesize;

	/* if no open file */
	if (current_file_map == NULL)
	{
		panic("Internal error: seek requested on closed file");
	}

	/* get file size */
	filesize = MAP_FILESIZE(current_file_map);

	switch (whence)
	{
		case SEEK_SET:	break;

		case SEEK_CUR:	where += current_file_position;
						break;

		case SEEK_END:	where += filesize;
						break;
	}

	/* only allow seek within extents of file */
    if (where < 0 || where > filesize)
    {
		panic("Internal error: seek outside file extents");
	}

	current_file_position = where;

	if (v->debug_mode)
	{
		printf("SEEK: current_file_position %08lx\n",
			current_file_position);
	}

    return where;
}

/*--------------------------------------------------------------------------*/
/* convert requested file position to a logical disk sector number
 * optionally return the number of contiguous sectors that may be read
 */

static
unsigned long
get_sector_address
(	unsigned long	position,			/* required file position			*/
	unsigned long	*extent				/* where to return extent of region	*/
)
{	unsigned long	offset;
	unsigned long	n;
	const FILEMAP	*frag;				/* current file fragment pointer	*/

	/* get pointer to first file fragment */
	frag = MAP_FIRSTFRAG(current_file_map);

	/* find fragment of requested sector */
	while (position >= (n = frag->length * SECTOR_SIZE))
	{
		position -= n;
		frag++;
	}

	/* if not a hole */
	if (frag->offset)
	{
		/* add number of sectors into this chunk */
		offset = frag->offset + (position / SECTOR_SIZE);
  	}
  	else
  	{
  		/* a hole */
  		offset = 0;
  	}

  	/* do they want the number of contiguous sectors */
  	if (extent)
  	{
  		/* return number of contiguous sectors rounded up to next
  		 * whole number of sectors
  		 */

  		/* get number of bytes remaining in fragment */
  		n = (frag->length * SECTOR_SIZE) - position;

  		*extent = (n + SECTOR_SIZE - 1) / SECTOR_SIZE;
  	}

	/* return loader block number or zero if a hole */
	return offset;
}

/*--------------------------------------------------------------------------*/

int
file_read
(	char			*buffer,	/* buffer to read data into					*/
	int				count		/* number of bytes to read					*/
)
{	long			n;
	unsigned long	offset;
	unsigned long	sectors;
	unsigned long	extent;
	unsigned long	sectnum;
	unsigned long	start_file_position;
	int				ret;

	if (v->debug_mode)
	{
		printf("READ: buf=0x%08lx, len=0x%lx, pos=0x%lx\n",
				(unsigned long) buffer,
				(unsigned long) count,
				(unsigned long) current_file_position);
	}

	/* if no open file */
	if (current_file_map == NULL)
	{
		panic("Internal error: read requested on closed file");
	}

	/* if illegal count */
	if (count < 0)
	{
		panic("Internal error: read request with negative count (%d)", count);
	}

	/* get number of bytes to end of file */
	n = MAP_FILESIZE(current_file_map) - current_file_position;

	/* if nothing left */
	if (n < 1)
	{
		/* say nothing read */
		ret = 0;
		goto done;
	}

	/* limit to amount of data left */
    if (count > n)
    {
		count = n;
	}

	/* remember where we started */
	start_file_position = current_file_position;

	/* process partial sector at beginning */
	if ((offset = current_file_position % SECTOR_SIZE) != 0)
	{
		/* get number of bytes required from first sector */
		n = SECTOR_SIZE - offset;

		/* don't copy more than is required */
		if (n > count)
		{
			n = count;
		}

		/* get disk sector number */
		if ((sectnum = get_sector_address(current_file_position, NULL)) == 0)
		{
			/* a hole, fill with zeros */
			mem_clear(buffer, n);
		}
		else
		{
			/* read first sector from disk */
			if (disk_read(v->sector_buffer, sectnum, 1) != 0)
			{
				return -1;
			}

			/* copy partial sector to correct place in callers buffer */
			mem_move(buffer, &v->sector_buffer[offset], n);
		}

		/* update buffer pointer, transfer count and file position */
		current_file_position += n;
		buffer                += n;
		count                 -= n;
	}

	/* get number of whole sectors */
	if ((sectors = count / SECTOR_SIZE) != 0)
	{
		/* transfer complete sectors */
		do	{
			sectnum = get_sector_address(current_file_position, &extent);

			/* don't transfer more than required */
			if (extent > sectors)
			{
				extent = sectors;
			}

			/* get number of bytes to transfer */
			n = extent * SECTOR_SIZE;

			/* if a hole */
			if (sectnum == 0)
			{
				/* fill hole with zeros */
				mem_clear(buffer, n);
			}
			else
			{
				/* read sectors from disk */
				if (disk_read(buffer, sectnum, extent) != 0)	
				{
					return -1;
				}
			}

			/* update buffer pointer, transfer count and file position */
			current_file_position += n;
			buffer                += n;
			count                 -= n;

		/* until no more sectors to transfer */
		} while ((sectors -= extent) != 0);
	}

	/* process partial block at end */
	if (count)
	{
		/* get disk block number */
		if ((sectnum = get_sector_address(current_file_position, NULL)) == 0)
		{
			/* a hole, fill with zeros */
			mem_clear(buffer, count);
		}
		else
		{
			/* read last sector from disk */
			if (disk_read(v->sector_buffer, sectnum, 1) != 0)	
			{
				return -1;
			}
	
			/* copy partial block to callers buffer */
			mem_move(buffer, v->sector_buffer, count);
		}

		/* update file position */
		current_file_position += count;
	}

	/* return number of bytes read */
	ret = current_file_position - start_file_position;

done:
	if (v->debug_mode)
	{
		printf("READ: bytes read = 0x%x\n", ret);
	}

	return ret;	
}

/*--------------------------------------------------------------------------*/

void
file_close
(	void
)
{
	current_file_map = NULL;
}

/*--------------------------------------------------------------------------*/

void
percent_init
(	unsigned long	total
)
{
	pcnt_total      = total;
	pcnt_current    = 999;
	pcnt_backspaces = 0;
}

/*--------------------------------------------------------------------------*/

void
percent_term
(	const char	*msg
)
{
	if (v->debug_mode)
		return;

	/* remove old percentage */
	while (pcnt_backspaces--)
		put_str("\b \b");

	put_str(msg);
}

/*--------------------------------------------------------------------------*/

void
percent_show
(	unsigned long	current
)
{	unsigned 		percentage;

	if (v->debug_mode)
		return;

	/* calculate new percentage */
	percentage = current * 100 / pcnt_total;

	/* if percentage changed */
	if (percentage != pcnt_current)
	{
		/* remove old percentage */
		while (pcnt_backspaces--)
			put_char('\b');

		/* output new percentage */
		printf("%d%% %n", percentage, &pcnt_backspaces);

		/* remember it for next time */
		pcnt_current = percentage;
	}
}

/*-----------------------------< end of file >------------------------------*/
