#include <stdlib.h>
#include <string.h>
#include "misc.h"
#include "misc.m"

PUBLIC ARRAY_OBJ::ARRAY_OBJ()
{
	modified = 0;
}

PUBLIC VIRTUAL ARRAY_OBJ::~ARRAY_OBJ()
{
}
/*
	Reset the modify flags of the element.
*/
PUBLIC void ARRAY_OBJ::rstmodified()
{
	modified = 0;
}
/*
	Reset the modify flags of the element. This function is generally
	used with ARRAY::was_modified().
*/
PUBLIC void ARRAY_OBJ::setmodified()
{
	modified = 1;
}
/*
	Return true if the element was modified.
*/
PUBLIC VIRTUAL int ARRAY_OBJ::was_modified()
{
	return modified;
}

/*
	Edit the object.
	Return 1 if the object must be deleted
	Return 0 if the modification are accepted
	Return -1 if the user has escape away.
*/
PUBLIC VIRTUAL int ARRAY_OBJ::edit()
{
	return 0;
}

/*
	Manage the content of a hosts file (generally /etc/hosts
*/
PUBLIC ARRAY::ARRAY()
{
	tb = NULL;
	nb = 0;
	maxtb = 0;
	modified = 0;
	increm = 100;
	is_owner = 1;
}
/*
	Instruct this array it is not allowed to delete objects.
	We use this function when we intend to share some object between
	several arrays. One array is the owner and the others represents
	various views of the original array (for example).
*/
PUBLIC void ARRAY::neverdelete()
{
	is_owner = 0;
}

/*
	Record the new growth rate of the internal array. Each time
	the array overflow, a larger is realloced. It will be _increm
	item larger.
*/
PUBLIC void ARRAY::setgrowth(int _increm)
{
	increm = _increm;
}
PUBLIC VIRTUAL ARRAY::~ARRAY()
{
	if (is_owner) for (int i=0; i<nb; i++) delete tb[i];
	free (tb);
}

/*
	Make sure there is enough space in the ARRAY_OBJ * table
*/
PROTECTED void ARRAY::grow()
{
	if (nb == maxtb){
		maxtb += increm;
		tb = (ARRAY_OBJ**)realloc(tb,maxtb*sizeof(ARRAY_OBJ*));
		if (tb == NULL){
			xconf_error ("Out of memory\n");
			exit (-1);
		}
	}
}

/*
	Remove one entry from the list.
	The entry is not freed. The caller must do it.

	Return -1 if the entry was not found.
*/
PUBLIC int ARRAY::remove(ARRAY_OBJ *obj)
{
	int ret = -1;
	if (obj != NULL){
		int j=0;
		for (int i=0; i<nb; i++){
			ARRAY_OBJ *tbi = tb[i];
			if (tbi != obj){
				if (j != i) tb[j] = tbi;
				j++;
			}else{
				modified = 1;
				ret = 0;
			}
		}
		nb = j;
	}
	return ret;
}
/*
	Find the occurence of an object in the array.
	Return -1 if not found or the index if found
*/
PUBLIC int ARRAY::lookup(ARRAY_OBJ *o) const
{
	int ret = -1;
	int n = getnb();
	for (int i=0; i<n; i++){
		if (o == tb[i]){
			ret = i;
			break;
		}
	}
	return ret;
}

/*
	Displace an object in the list
*/
PUBLIC void ARRAY::moveto (ARRAY_OBJ *o, int newpos)
{
	remove (o);
	insert (newpos,o);
}

/*
	Remove all entry from the list and destroy the objects if owner
*/
PUBLIC void ARRAY::remove_all()
{
	if (is_owner){
		while (nb > 0) remove_del (0);
	}else{
		nb = 0;
	}
}
/*
	Remove one entry from the list and delete the object.
	If the object is not part of the list, it is not deleted.

	Return -1 if the entry was not found.
*/
PUBLIC int ARRAY::remove_del(ARRAY_OBJ *obj)
{
	int ret = remove(obj);
	if (ret != -1 && is_owner) delete obj;
	return ret;
}

PUBLIC int ARRAY::remove_del(int no)
{
	ARRAY_OBJ *obj = getitem(no);
	return remove_del (obj);
}

PUBLIC void ARRAY::delall()
{
	if (is_owner) for (int i=0; i<nb; i++) delete tb[i];
	nb = 0;
	modified = 1;
}

/*
	Return the number of entry in a table.
*/
PUBLIC int ARRAY::getnb() const
{
	return nb;
}

/*
	Add one item to the in memory hosts table
*/
PUBLIC void ARRAY::add (ARRAY_OBJ *pt)
{
	if (pt != NULL){
		grow();
		tb[nb++] = pt;
		modified = 1;
	}
}

/*
	Insert one item to the table
*/
PUBLIC void ARRAY::insert (int pos, ARRAY_OBJ *pt)
{
	if (pt != NULL){
		if (pos >= nb){
			add (pt);
		}else{
			grow();
			memmove (tb+pos+1,tb+pos,(nb-pos)*sizeof(tb[0]));
			tb[pos] = pt;
			nb++;
		}
		modified = 1;
	}
}
/*
	This function should be duplicate in each sub-class to allow
	proper casting.
*/
PROTECTED ARRAY_OBJ *ARRAY::getitem (int no) const
{
	ARRAY_OBJ *ret = NULL;
	if (no >= 0 && no < nb) ret = tb[no];
	return ret;
}
	

/*
	Reset the modify flags of all elements of the array.
*/
PUBLIC void ARRAY::rstmodified()
{
	int nbo = getnb();
	for (int i=0; i<nbo; i++) tb[i]->rstmodified();
	modified = 0;
}
/*
	Return != if the element was modified.
*/
PUBLIC VIRTUAL int ARRAY::was_modified()
{
	int ret  = modified;
	if (!ret){
		int nbo = getnb();
		for (int i=0; i<nbo; i++){
			int modif = tb[i]->was_modified();
			if (modif){
				ret = true;
				break;
			}
		}
	}
	return ret;
}

static int (*user_cmp) (const ARRAY_OBJ *, const ARRAY_OBJ *);
static int local_cmp (const void *pt1, const void *pt2)
{
	return (*user_cmp)(*(ARRAY_OBJ**)pt1,*(ARRAY_OBJ**)pt2);
}

/*
	Sort the ARRAY with a user function working like the one
	for qsort.

	Use qsort() internally.
*/
PUBLIC void ARRAY::sort (int (*cmp) (const ARRAY_OBJ *, const ARRAY_OBJ *))
{
	user_cmp = cmp;
	qsort (tb,nb,sizeof(ARRAY_OBJ*),local_cmp);
}

/*
	Save the content of the array
	Return -1 if any error.
	The default behavior is to do nothing
*/
PUBLIC VIRTUAL int ARRAY::write()
{
	return -1;
}

/*
	Behavior after the editing of one object
*/
PUBLIC int ARRAY::manage_edit (ARRAY_OBJ *e, int code, int insertpos)
{
	if (code == 0){
		int pos = lookup (e);
		if (pos == -1) add (e);
		if (insertpos != -1) moveto(e,insertpos);
		write();
	}else if (code == 1){
		if (remove_del (e)==-1) delete e;
		write();
	}
	return code;
}

/*
	Behavior after the editing of one object
*/
PUBLIC int ARRAY::manage_edit (ARRAY_OBJ *e, int code)
{
	return manage_edit (e,code,-1);
}
/*
	Edit one object or create a new one and manage the deletion if told
	by the edit function of the object.
*/
PUBLIC int ARRAY::editone(int no)
{
	int ret = -1;
	if (no >= 0 && no < getnb()){
		ARRAY_OBJ *e = getitem(no);
		ret = e->edit();
		manage_edit (e,ret);
	}
	return ret;
}	
/*
	Edit one object or create a new one and manage the deletion if told
	by the edit function of the object.
*/
PUBLIC int ARRAY::editone(ARRAY_OBJ *e)
{
	int ret  = e->edit();
	manage_edit (e,ret);
	return ret;
}	
/*
	Remove the last entries of the array, after the "cut" first items
*/
PUBLIC void ARRAY::remove_last(int cut)
{
	while (getnb() > cut) remove_del(getnb()-1);
}

