/* GMP3 - A front end for mpg123
 * Copyright (C) 1998 Brett Kosinski
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <stdio.h>
#include "rc_parser.h"

void copyFile(char *ifile, char *ofile)
{
  FILE *input, *output;
  char block[1024];
  int block_size;
  
  input = fopen(ifile, "r");
  output = fopen(ofile, "w");
  
  while (! feof(input))
    {
      block_size = fread(block, 1, 1024, input);
      fwrite(block, 1, block_size, output);
    }
}

int parseLine(char *line, tokenInfo *token)
{
  char *buffer = (char *)malloc(strlen(line)+1);
  int pos = 1;
  char *startPos = buffer;

  if (line[0] == '#')
    return 0;
  
  strcpy(buffer, line);
  
  do  
    buffer++;
  while ((buffer) && (buffer[pos-1] != '=') && (buffer[pos] != '\0'));
  
  do
    buffer++;
  while ((buffer) && (buffer[pos-1] == ' ') && (buffer[pos] != '\0'));
   
  pos = strlen(buffer);
  
  while ((buffer[pos] == ' ') || (buffer[pos] == '\n') || (buffer[pos] == '\0'))
    pos--;

  buffer[pos+1] = '\0';  

  if (token->dataBuffer == NULL)
    {
      switch (token->type) 
        {
        case STRING : { token->dataBuffer = (char *)malloc(1023); break; }
        case INTEGER : { token->dataBuffer = (int *)malloc(sizeof(int)); break; }
        case FLOAT : { token->dataBuffer = (int *)malloc(sizeof(float)); break; }
        case BOOLEAN : { token->dataBuffer = (int *)malloc(sizeof(int)); break; }
        }
    }
    
  if (token->type == STRING)
    snprintf(token->dataBuffer, 1023, "%s", buffer);
  else if (token->type == INTEGER)
    sscanf(buffer, "%d", (int *)(token->dataBuffer));
  else if (token->type == FLOAT)
    sscanf(buffer, "%f", (float *)(token->dataBuffer));
  else if (token->type == BOOLEAN)
    {
      if (! strcasecmp(buffer, "TRUE"))
        *(int *)(token->dataBuffer) = 1;
      else if (! strcasecmp(buffer, "FALSE"))
        *(int *)(token->dataBuffer) = 0;
      else  
        {
          buffer = startPos;
          free(buffer);
          return -1;
        }  
    }

  buffer = startPos;    
  free(buffer);
  return 0;  
}

void parseFile(char *file, tokenInfo tokens[], int tokenCount)
{
  FILE *inputFile;
  char *lineBuffer = (char *)malloc(256);
  char *name = (char *)malloc(256);
  int startPos, endPos;
  int i;
  
  inputFile = fopen(file, "r"); 

  while (inputFile && ! feof(inputFile))
    {
      startPos = 0;
      endPos = 0;

      fgets(lineBuffer, 255, inputFile);
      strcpy(name, lineBuffer);
      
      while (name[startPos] == ' ')
        startPos++;
      name += startPos;  
      
      while ((name[endPos] != ' ') && (name[endPos] != '=') &&
             (name[endPos] != '\n'))
        endPos++;        
      name[endPos] = '\0';
      
      for (i = 0; i < tokenCount; i++)
        if (! strcasecmp(name, tokens[i].id_string))
          {             
            parseLine(lineBuffer, &(tokens[i]));
            break;
          }
    }
    
  if (inputFile) fclose(inputFile);
}

int tokenCmp(tokenInfo tok1, tokenInfo tok2)
{
  switch (tok1.type)
    {
    case STRING : 
      {
        if (strcasecmp(tok1.dataBuffer, tok2.dataBuffer) != 0)
          return 1;
        else
          return 0;
        
      }
    case INTEGER : 
      {
        if (*(int *)(tok1.dataBuffer) != *(int *)(tok2.dataBuffer))
          return 1;
        else
          return 0;
      }
    case FLOAT : 
      {
        if (*(float *)(tok1.dataBuffer) != *(float *)(tok2.dataBuffer))
          return 1;
        else
          return 0;

      }
    case BOOLEAN : 
      {
        if (  (*(int *)(tok1.dataBuffer) || *(int *)(tok2.dataBuffer)) &&
            ! (*(int *)(tok1.dataBuffer) && *(int *)(tok2.dataBuffer)))
          return 1;
        else
          return 0;

        break;
      };      
  }
  
  return 0;
}

void writeToken(FILE *outputFile, tokenInfo token)
{
  if ((token.type == STRING) && (((char *)(token.dataBuffer))[0] == '\0'))
    return;

  fprintf(outputFile, "%s", token.id_string);
  switch (token.type)
  {
    case STRING : { fprintf(outputFile, " = %s\n", (char *)(token.dataBuffer)); break; }
    case INTEGER : { fprintf(outputFile, " = %i\n", *(int *)(token.dataBuffer)); break; }
    case FLOAT : { fprintf(outputFile, " = %f\n", *(float *)(token.dataBuffer)); break; }
    case BOOLEAN : 
      {         
        if (*(int *)(token.dataBuffer))
          fprintf(outputFile, " = TRUE\n");
        else
          fprintf(outputFile, " = FALSE\n");
                    
        break;       
      };
   }
}

void saveFile(char *file, tokenInfo tokens[], int tokenCount)
{
  FILE *inputFile, *outputFile;
  char *lineBuffer = (char *)malloc(256);
  char *name = (char *)malloc(256);
  int *written;
  int startPos, endPos;
  int lineStart;
  int i, updated;
  tokenInfo tempToken;

  written = (int *)malloc(sizeof(int)*tokenCount);  
  for (i = 0; i < tokenCount; i++) written[i] = 0;
  
  inputFile = fopen(file, "r");
  outputFile = fopen("/tmp/gmp3", "w+");
  if (inputFile) rewind(inputFile);
  
  while (inputFile && outputFile && ! feof(inputFile))
    {
      startPos = 0;
      endPos = 0;
      updated = 0;

      fgets(lineBuffer, 255, inputFile);
      strcpy(name, lineBuffer);
      lineStart = ftell(inputFile);
      
      while (name[startPos] == ' ') startPos++;
      name += startPos;  
      
      while ((name[endPos] != ' ') && 
             (name[endPos] != '=') && 
             (name[endPos] != '\n'))
        endPos++;        
      name[endPos] = '\0';
      
      for (i = 0; i < tokenCount; i++)
        if (! strcasecmp(name, tokens[i].id_string))
          {
            memcpy(&tempToken, &(tokens[i]), sizeof(tokenInfo));
            tempToken.dataBuffer = NULL; /* Make parser create buffer */
            parseLine(lineBuffer, &(tempToken));
             
            if (tokenCmp(tempToken, tokens[i]) != 0)                         
              { 
                updated = 1;
                written[i] = 1;
                writeToken(outputFile, tokens[i]);
              }

            break;
          }
          
      if ((! updated) && (! written[i]))
        {
          fprintf(outputFile, "%s", lineBuffer);
          written[i] = 1;
        }         
    }
    
  for (i = 0; i < tokenCount; i++)
    if (! written[i])
      writeToken(outputFile, tokens[i]);
    
  if (inputFile) fclose(inputFile);
  if (outputFile) fclose(outputFile);
  
  copyFile("/tmp/gmp3", file);
  unlink("/tmp/gmp3");
  
  free(written);
}
