/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>

#include "Platform_NetLib.h"

// ===== COPIED FROM NETPRV.H =====

// Max # of open built-in sockets at one time.
#define 	netMaxNumSockets		4						// Max # of sockets
#define 	netMinSocketRefNum	(sysFileDescNet)							 // lowest refNum
#define 	netMaxSocketRefNum	(sysFileDescNet+netMaxNumSockets-1) // highest refNum

static long gOpenCount;


Bool Platform::NetLib::Redirecting (void)
{
	return true;
}


//--------------------------------------------------
// Library initialization, shutdown, sleep and wake
//--------------------------------------------------
Err    Platform::NetLib::Open(Word libRefnum, WordPtr netIFErrsP)
{
	if (gOpenCount > 0) {
		++gOpenCount;
		return netErrAlreadyOpen;
	}
	++gOpenCount;
	return errNone;
}

Err    Platform::NetLib::Close(Word libRefnum, Word immediate)
{
	if (gOpenCount <= 0) {
		return netErrNotOpen;
	}
	--gOpenCount;
	return errNone;
}

Err    Platform::NetLib::Sleep(Word libRefnum)
{
	// stub. Do Nothing
	return errNone;
}

Err Platform::NetLib::Wake(Word libRefnum)
{
	// stub. Do Nothing
	return errNone;
}

// This call forces the library to complete a close if it's
//	currently in the close-wait state. Returns 0 if library is closed,
//	Returns netErrFullyOpen if library is still open by some other task.
Err    Platform::NetLib::FinishCloseWait(Word libRefnum)
{
	// stub. Do Nothing
	return errNone;
}

// This call is for use by the Network preference panel only. It
// causes the NetLib to fully open if it's currently in the close-wait 
//	state. If it's not in the close wait state, it returns an error code
Err    Platform::NetLib::OpenIfCloseWait(Word libRefnum)
{
	return netErrNotOpen;
}

// Get the open Count of the NetLib
Err    Platform::NetLib::OpenCount(Word refNum, WordPtr countP)
{
	*countP = gOpenCount;
	return errNone;
}
// Give NetLib a chance to close the connection down in response
// to a power off event. Returns non-zero if power should not be
//	turned off. EventP points to the event that initiated the power off
//	which is either a keyDownEvent of the hardPowerChr or the autoOffChr.
// Don't include unless building for Viewer
Err    Platform::NetLib::HandlePowerOff (Word refNum, EventPtr eventP)
{
	return Platform::NetLib::Close(refNum, true);
}
// Check status or try and reconnect any interfaces which have come down.
// This call can be made by applications when they suspect that an interface
// has come down (like PPP or SLIP). NOTE: This call can display UI 
// (if 'refresh' is true) so it MUST be called from the UI task. 
Err    Platform::NetLib::ConnectionRefresh(Word refNum, 
										   Boolean refresh, 
										   BytePtr allInterfacesUpP, 
										   WordPtr netIFErrP)
{
	*netIFErrP =0;
	return errNone;
}


#if 0
//--------------------------------------------------
// Net address translation and conversion routines.
//--------------------------------------------------

// Convert 32-bit IP address to ascii dotted decimal form. The Sockets glue
//	macro inet_ntoa will pass the address of an application global string in
//	spaceP.
CharPtr Platform::NetLib::AddrINToA(Word libRefnum, 
									NetIPAddr	inet, 
									CharPtr spaceP)
{
	CharPtr piece;
	UInt		i;
	union {
		Byte		bytes[4];
		DWord	wholeThing;
	} inPieces;
	
	
	// Fill in the union with the 32-bit form.
	inPieces.wholeThing = inet;
	piece = spaceP;
	
	// Print ascii format to piece string
	for (i = 0; i < 4; i++) {
		if (inPieces.bytes[i] >= 100)
			*piece++ = (char)('0' + ((inPieces.bytes[i]/100) % 10));
		if (inPieces.bytes[i] >= 10)
			*piece++ = (char)('0' + ((inPieces.bytes[i]/10) % 10));
		*piece++ = (char)('0' + inPieces.bytes[i] % 10);
		if (i == 3)
			*piece = '\0';
		else
			*piece++ = '.';
	}
	
	// Return pointer to string
	return spaceP;
	
}

// Convert a dotted decimal ascii string format of an IP address into
//	a 32-bit value.
NetIPAddr Platform::NetLib::AddrAToIN(Word libRefnum, CharPtr name)
{
	UInt		i;
	CharPtr	piece;
	
	union {
		Byte	bytes[4];
		DWord	wholeThing;
	} inPieces;
	
	
	/* Skip leading whitespace */
	while ((*name == ' ') || (*name == '\t'))
		name++;
	
	for (i = 0; i < 4; i++) {
		piece = name;
		
		// Skip to the end of this dotted decimal
		while ((*name >= '0') && (*name <= '9'))
			name++;
		
		// Drop out if 0 length
		if (piece == name)
			return -1;				   /* zero length piece */
		
		// Convert this section to a byte
		inPieces.bytes[i] = atoi(piece);
		
		// Drop out if last byte
		if (i == 3)
			break;					  /* don't need a . after last piece */
		
		// Make sure it's followed by a '.'
		if (*name != '.')
			return -1;				   /* do need a . after other pieces */
		
		// On to next section
		name++;
	}
	
	return inPieces.wholeThing;
}
#endif


//--------------------------------------------------
// Socket creation and option setting
//--------------------------------------------------

// Create a socket and return a refnum to it. Protocol is normally 0. 
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
NetSocketRef Platform::NetLib::SocketOpen(Word libRefnum, 
										  NetSocketAddrEnum socketDomain, 
										  NetSocketTypeEnum socketType, 
										  SWord socketProtocol, 
										  SDWord timeout, 
										  Err* errP)
{
	int result;
	
	*errP = 0;
	
	result = socket (socketDomain, socketType, socketProtocol);
	
	if (result == -1){
		*errP = errno;
	}
	return result;
}

// Close a socket. 
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketClose(Word libRefnum, 
									NetSocketRef sRef, 
									SDWord timeout, 							
									Err* errP)
{
	int iResult;
	
	*errP = 0;
	
	if (sRef == -1) {
		*errP = netErrSocketNotOpen;
		return -1;
	}
	
	iResult = close(sRef);
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Set a socket option.    Level is usually netSocketOptLevelSocket. Option is one of
//	netSocketOptXXXXX. OptValueP is a pointer to the new value and optValueLen is
//	the length of the option value. 			   
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketOptionSet(Word libRefnum, 
										NetSocketRef sRef,
										NetSocketOptLevelEnum level, 
										NetSocketOptEnum opt, 
										VoidPtr optvalP, 
										Word optlen,
										SDWord timeout, 
										Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = setsockopt(sRef, level, opt, (const char *)optvalP, optlen);
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Get a socket option. 				   
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketOptionGet(Word libRefnum, 
										NetSocketRef sRef,
										NetSocketOptLevelEnum level, 
										NetSocketOptEnum opt,
										VoidPtr optValueP, 
										WordPtr optValueLenP,
										SDWord timeout, 
										Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = getsockopt( sRef, level, opt, (char *)optValueP, (int *)optValueLenP);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
	
	
}

//--------------------------------------------------
// Socket Control
//--------------------------------------------------

// Bind a source address and port number to a socket. This makes the
//	socket accept incoming packets destined for the given socket address.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketBind(Word libRefnum, 
								   NetSocketRef sRef,
								   NetSocketAddrType* sockAddrP, 
								   SWord addrLen, 
								   SDWord timeout, 
								   Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = bind( sRef, (const struct sockaddr *)sockAddrP, addrLen);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}


// Connect to a remote socket. For a stream based socket (i.e. TCP), this initiates
//	a 3-way handshake with the remote machine to establish a connection. For
//	non-stream based socket, this merely specifies a destination address and port
//	number for future outgoing packets from this socket.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketConnect(Word libRefnum, 
									  NetSocketRef sRef,
									  NetSocketAddrType* sockAddrP, 
									  SWord addrLen, 
									  SDWord timeout, 
									  Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = connect( sRef, (const struct sockaddr *)sockAddrP, addrLen);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}
// Makes a socket ready to accept incoming connection requests. The queueLen 
//	specifies the max number of pending connection requests that will be enqueued
//	while the server is busy handling other requests.
//	Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketListen(Word libRefnum, 
									 NetSocketRef sRef,
									 Word	 queueLen, 
									 SDWord timeout, 
									 Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = listen(sRef, queueLen);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Blocks the current process waiting for an incoming connection request. The socket
//	must have previously be put into listen mode through the NetLibSocketListen call.
//	On return, *sockAddrP will have the remote machines address and port number.
//	Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketAccept(Word libRefnum, 
									 NetSocketRef sRef,
									 NetSocketAddrType* sockAddrP, 
									 SWord* addrLenP, 
									 SDWord timeout,
									 Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = accept( sRef, (struct sockaddr *)sockAddrP, (int *)addrLenP);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Shutdown a connection in one or both directions.  
//	Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketShutdown(Word libRefnum, 
									   NetSocketRef sRef, 
									   SWord direction, 
									   SDWord timeout, 
									   Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = shutdown(sRef,direction);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Gets the local and remote addresses of a socket. Useful for TCP sockets that 
//	get dynamically bound at connect time. 
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.
SWord Platform::NetLib::SocketAddr(Word libRefnum, 
								   NetSocketRef sRef,
								   NetSocketAddrType* locAddrP, 
								   SWord* locAddrLenP, 
								   NetSocketAddrType* remAddrP, 
								   SWord* remAddrLenP, 
								   SDWord timeout, 
								   Err* errP)
{
	int iResult;
	
	*errP = 0;
	if ((locAddrP != NULL) && (locAddrLenP != NULL)) {
		iResult = getsockname( sRef, 
			(struct sockaddr *)locAddrP, 
			(int *)locAddrLenP);
	} else { // if ((remAddrP != NULL) && (remAddrLenP != NULL)){
		iResult = getpeername( sRef, 
			(struct sockaddr *)remAddrP,
			(int *)remAddrLenP);
	}
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

//--------------------------------------------------
// Sending and Receiving
//--------------------------------------------------
// Send data through a socket. The data is specified through the NetIOParamType
//	structure.
// Flags is one or more of netMsgFlagXXX.
// Returns # of bytes sent on success, or -1 on error. If error, *errP gets filled 
//	in with error code.
SWord Platform::NetLib::SendPB(Word libRefNum, 
							   NetSocketRef sRef,
							   NetIOParamType* pbP, 
							   Word flags, 
							   SDWord timeout, 
							   Err* errP)
{
	int iResult;
	
	*errP = 0;
	// TODO - what about recvmsg and sendmsg?
	iResult = send( sRef, (const char *)pbP, sizeof(NetIOParamType), flags);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}


// Send data through a socket. The data to send is passed in a single buffer,
//	unlike NetLibSendPB. If toAddrP is not nil, the data will be sent to 
//	address *toAddrP.
// Flags is one or more of netMsgFlagXXX.
// Returns # of bytes sent on success, or -1 on error. If error, *errP gets filled 
//	in with error code.
SWord Platform::NetLib::Send(Word libRefNum, 
							 NetSocketRef sRef,
							 const VoidPtr bufP, 
							 Word bufLen, 
							 Word flags,
							 VoidPtr toAddrP, 
							 Word toLen, 
							 SDWord timeout, 
							 Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = send( sRef, (const char *)bufP, (int)bufLen, (int)flags);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Receive data from a socket. The data is gatthered into buffers specified in the 
//	NetIOParamType structure.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//	with error code.
SWord Platform::NetLib::ReceivePB(Word libRefNum, 
								  NetSocketRef sRef,
								  NetIOParamType* pbP, 
								  Word flags, 
								  SDWord timeout, 
								  Err* errP)
{
	int iResult;
	
	*errP = 0;
	// TODO - what about recvmsg and sendmsg?
	iResult = recv( sRef, (char *)pbP, sizeof(NetIOParamType), flags);
	
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Receive data from a socket. The data is read into a single buffer, unlike
//	NetLibReceivePB. If fromAddrP is not nil, *fromLenP must be initialized to
//	the size of the buffer that fromAddrP points to and on exit *fromAddrP will
//	have the address of the sender in it.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//	with error code.
SWord Platform::NetLib::Receive(Word libRefNum, 
								NetSocketRef sRef,
								VoidPtr bufP, 
								Word bufLen, 
								Word flags, 
								VoidPtr fromAddrP, 
								WordPtr fromLenP, 
								SDWord timeout, 
								Err* errP)
{
	int iResult;
	
	*errP = 0;
	iResult = recvfrom ( sRef, 
		(char * )bufP, 
		(int) bufLen, 
		(int) flags, 
		(struct sockaddr *) fromAddrP, 
		(int *) fromLenP); 
	if (iResult) {
		*errP = errno;
		return -1;
	}
	return 0;
}

// Receive data from a socket directly into a (write-protected) Data Manager 
//	record. 
// If fromAddrP is not nil, *fromLenP must be initialized to
//	the size of the buffer that fromAddrP points to and on exit *fromAddrP will
//	have the address of the sender in it.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//	with error code.
SWord Platform::NetLib::DmReceive(Word libRefNum, 
								  NetSocketRef sRef,
								  VoidPtr recordP, 
								  DWord recordOffset, 
								  Word rcvLen, 
								  Word flags, 
								  VoidPtr fromAddrP, 
								  WordPtr fromLenP, 
								  SDWord timeout, 
								  Err* errP)
{
	return -1;
}

//--------------------------------------------------
// Name Lookups
//--------------------------------------------------
NetHostInfoPtr	  Platform::NetLib::GetHostByName(Word libRefNum, 
												  CharPtr nameP, 
												  NetHostInfoBufPtr bufP, 
												  Long	  timeout, 
												  Err* errP)
{
	struct hostent * pHostEnt;
	
	*errP = 0;
	
	pHostEnt = gethostbyname(nameP);
	if (!pHostEnt){
		*errP = errno;
		return NULL;
	}
	memcpy((void *)&(bufP->hostInfo), pHostEnt, sizeof(struct hostent));
	return &(bufP->hostInfo);
}

NetHostInfoPtr	  Platform::NetLib::GetHostByAddr(Word libRefNum, 
												  BytePtr addrP, 
												  Word len, 
												  Word type,
												  NetHostInfoBufPtr bufP, 
												  Long	  timeout, 
												  Err* errP)
{
	struct hostent * pHostEnt;
	
	*errP = 0;
	pHostEnt = gethostbyaddr( (const char *)addrP, len, type);
	
	if (!pHostEnt){
		*errP = errno;
		return NULL;
	}
	memcpy((void *)&(bufP->hostInfo), pHostEnt, sizeof(struct hostent));
	return &(bufP->hostInfo);
}

NetServInfoPtr	  Platform::NetLib::GetServByName(Word libRefNum, 
												  CharPtr servNameP, 
												  CharPtr protoNameP,  
												  NetServInfoBufPtr bufP, 
												  Long	  timeout, 
												  Err* errP)
{
	struct servent *pServent;
	
	*errP = 0;
	pServent = getservbyname(servNameP, protoNameP);
	
	if (!pServent) {
		*errP = errno;
		return NULL;
	}
	memcpy((void *)&(bufP->servInfo), pServent, sizeof(NetServInfoType));
	return &(bufP->servInfo);
}

// Looks up a mail exchange name and returns a list of hostnames for it. Caller
//	must pass space for list of return names (hostNames), space for 
//	list of priorities for those hosts (priorities) and max # of names to 
//	return (maxEntries).
// Returns # of entries found, or -1 on error. If error, *errP gets filled in
//	with error code.
SWord Platform::NetLib::GetMailExchangeByName(Word libRefNum, CharPtr mailNameP, 
											  Word maxEntries, 
											  Char hostNames[][netDNSMaxDomainName+1], Word priorities[], 
											  Long	timeout, Err* errP)
{
	*errP = 0;
	// TODO
	return -1;
}

#if 0
//--------------------------------------------------
// Interface setup
//--------------------------------------------------
Err Platform::NetLib::IFGet(Word libRefNum, 
							Word index, 
							DWordPtr ifCreatorP, 
							WordPtr ifInstanceP)
{
	return netErrInvalidInterface;
}


Err Platform::NetLib::IFAttach(Word libRefNum, DWord ifCreator, Word ifInstance,
							   SDWord timeout)
{
	// TODO
	// This should probably validate the creator or something
	return netErrInvalidInterface;
}

Err Platform::NetLib::IFDetach(Word libRefNum, DWord ifCreator, Word ifInstance,
							   SDWord timeout)
{
	// TODO
	// This should probably validate the creator or something
	return netErrInvalidInterface;
}

Err Platform::NetLib::IFUp(Word libRefNum, DWord ifCreator, Word ifInstance)
{
	// TODO
	// This should probably validate the creator or something
	return 0;
}

Err Platform::NetLib::IFDown(Word libRefNum, DWord ifCreator, Word ifInstance,
							 SDWord timeout)
{
	// TODO
	// This should probably validate the creator or something
	return 0;
}
#endif



#if 0
//--------------------------------------------------
// Settings
//--------------------------------------------------
// General settings
Err Platform::NetLib::SettingGet(Word libRefNum,
								 Word /*NetSettingEnum*/ setting, 
								 VoidPtr valueP, 
								 WordPtr valueLenP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::SettingSet(Word libRefNum, 
								 Word /*NetSettingEnum*/ setting, 
								 VoidPtr valueP, 
								 Word valueLen)
{
	// TODO
	return -1;
}

// Network interface specific settings.
Err Platform::NetLib::IFSettingGet(Word libRefNum, 
								   DWord ifCreator, 
								   Word ifInstance,
								   Word /*NetIFSettingEnum*/ setting, 
								   VoidPtr valueP, 
								   WordPtr valueLenP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::IFSettingSet(Word libRefNum, 
								   DWord ifCreator, 
								   Word ifInstance,
								   Word /*NetIFSettingEnum*/ setting, 
								   VoidPtr valueP, 
								   Word valueLen)
{
	// TODO
	return -1;
}
#endif

//--------------------------------------------------
// System level
//--------------------------------------------------
SWord Platform::NetLib::Select(Word libRefNum, 
							   Word width, 
							   NetFDSetType* readFDs, 
							   NetFDSetType* writeFDs, 
							   NetFDSetType* exceptFDs,
							   Long    timeout, 
							   Err* errP)
{
	SWord sResult;
	fd_set readfds;
	fd_set writefds;
	fd_set exceptfds;
	timeval timeOut;
	timeOut.tv_usec = timeout;
	timeOut.tv_sec = 0;
	
	
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);
	
	unsigned int fd;
	
	if (readFDs) {
		
		// Check built-in sockets
		for (fd=0; fd <= netMaxSocketRefNum && fd < width; fd++)  {
			if (netFDIsSet(fd, readFDs)) {
				FD_SET(fd, &readfds);
			}
		}
	}
	
	// Check ready outputs
	if (writeFDs) {
		// check built-in sockets. 
		for (fd=0; fd <= netMaxSocketRefNum && fd < width; fd++)  {
			if (netFDIsSet(fd, writeFDs)) {
				FD_SET(fd, &writefds);
			}
		}
	}
	netFDZero(readFDs);
	netFDZero(writeFDs);
	netFDZero(exceptFDs);
	
	*errP = 0;
	sResult = select( width, &readfds, &writefds, &exceptfds, &timeOut);
	
	if (sResult == -1) {
		*errP = errno;
		return -1;
	}
	if (sResult) {
		// Check built-in sockets
		for (fd=0; fd <= netMaxSocketRefNum && fd < width; fd++)  {
			if (FD_ISSET(fd, &readfds)) {
				netFDSet(fd, readFDs);
			}
		}
		// check built-in sockets. 
		for (fd=0; fd <= netMaxSocketRefNum && fd < width; fd++)  {
			if (FD_ISSET(fd, &writefds)) {
				netFDSet(fd, writeFDs);
			}
		}
		// check built-in sockets. 
		for (fd=0; fd <= netMaxSocketRefNum && fd < width; fd++)  {
			if (FD_ISSET(fd, &exceptfds)) {
				netFDSet(fd, writeFDs);
			}
		}
	}
	return sResult;
}


#if 0
//--------------------------------------------------
// Debugging support
//--------------------------------------------------
Err Platform::NetLib::Master(Word libRefNum, 
							 Word cmd, 
							 NetMasterPBPtr pbP,
							 Long timeout)
{
	// TODO
	return 0;
}


Err Platform::NetLib::TracePrintF(Word libRefNum, CharPtr formatStr, ...)
{
	va_list argList;
	char szBuffer[256];
	
	va_start(argList, formatStr);
	sprintf(szBuffer, formatStr, argList);
	// TODO - some poser trace call.
	va_end(argList);
	return 0;
}


Err Platform::NetLib::TracePutS(Word libRefNum, CharPtr strP)
{
	// TODO - some poser trace call.
	return 0;
}
#endif


//--------------------------------------------------
// Configuration Calls
//--------------------------------------------------
Err Platform::NetLib::OpenConfig( Word refNum, 
								 Word configIndex, 
								 DWord openFlags,
								 WordPtr netIFErrP)
{
	int iResult;
	
	// Init return error 
	*netIFErrP = 0;
	// TODO
	return -1;
}

#if 0
Err Platform::NetLib::ConfigMakeActive( Word refNum, Word configIndex)
{
	// TODO
	return 0;
}

Err Platform::NetLib::ConfigList( Word refNum, 
								 NetConfigNameType nameArray[],
								 WordPtr arrayEntriesP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigIndexFromName( Word refNum, 
										  NetConfigNamePtr nameP,
										  WordPtr indexP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigDelete( Word refNum, Word index)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigSaveAs( Word refNum, NetConfigNamePtr nameP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigRename( Word refNum, Word index,
								   NetConfigNamePtr newNameP)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigAliasSet( Word refNum, Word configIndex,
									 Word aliasToIndex)
{
	// TODO
	return -1;
}

Err Platform::NetLib::ConfigAliasGet( Word refNum, Word aliasIndex,
									 WordPtr indexP, BooleanPtr isAnotherAliasP)
{
	// TODO
	return -1;
}
#endif
