/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "UART.h"

#include "Logging.h"
#include "Platform.h"

/*
	This module contains the routines for handling serial I/O.  It
	is responsible for responding to changes in state enacted by
	software (from either the OS or user), and for dealing with
	the actual transmission and reception of serial data.

	There are four ways in which serial activity could occur: something
	could write to a UART register, something could read from a UART
	register, a byte could be received from the host serial port, or
	a byte could be sent out the host serial port.  Here is what
	happens on each of those events.

	Something reads a UART register:
		- If the register is the RX_DATA register, clear the DATA_READY bit
		- Make sure the state is up-to-date (including interrupts)
		- Return the register contents

	Something writes to a UART register:
		- Update the writable parts of the register
		- React to any changes
		- Make sure the state is up-to-date (including interrupts)

	Data appears at the host serial port:
		- Post the byte to the RX FIFO (if there is room)
		- Make sure the state is up-to-date (including interrupts)

	Data needs to be sent to the host serial port:
		- Send the first byte in the TX FIFO
		- Make sure the state is up-to-date (including interrupts)
*/


// ======================================================================
//	Globals and constants
// ======================================================================

static UART::Parity	gLastParity;
static int			gLastStopBits;
static int			gLastDataBits;
static uae_u32		gLastBaud;
static Bool			gLastHwHandshaking;

static UART::State	gCurrentState(UART::kUART_Dragonball);
//static Bool			gStateNeedsUpdating;
static TByteQueue	gRxFIFO;
static TByteQueue	gTxFIFO;


// ======================================================================
//	Private functions
// ======================================================================

static Bool			PrvPinBaud		(uae_u32& newBaud);
static Bool			PrvPinBaud		(uae_u32& newBaud, uae_u32 testBaud);
static void			PrvEmptyQueue	(TByteQueue&);

#define PRINTF	if (!LogSerial ()) ; else LogAppendMsg


/***********************************************************************
 *
 * FUNCTION:	UART::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	type - the type of UART to emulate.  The Dragonball
 *					and DragonballEZ UARTs are similar enough that we
 *					can handle them both here with just a few tests in
 *					the places where they differ.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::Initialize (UART_Type type)
{
	gCurrentState.UART_TYPE = type;
}


/***********************************************************************
 *
 * FUNCTION:	UART::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::Reset (void)
{
	PrvEmptyQueue (gRxFIFO);
	PrvEmptyQueue (gTxFIFO);

	// UART control register bits

	gCurrentState.UART_ENABLE		= 0;
	gCurrentState.RX_ENABLE			= 0;
	gCurrentState.TX_ENABLE			= 0;
	gCurrentState.RX_CLK_CONT		= 0;
	gCurrentState.PARITY_EN			= 0;
	gCurrentState.ODD_EVEN			= 0;
	gCurrentState.STOP_BITS			= 0;
	gCurrentState.CHAR8_7			= 0;
	gCurrentState.GPIO_DELTA_ENABLE	= 0;	// 68328 only
	gCurrentState.OLD_ENABLE		= 0;	// 68EZ328 only
	gCurrentState.CTS_DELTA_ENABLE	= 0;
	gCurrentState.RX_FULL_ENABLE	= 0;
	gCurrentState.RX_HALF_ENABLE	= 0;
	gCurrentState.RX_RDY_ENABLE		= 0;
	gCurrentState.TX_EMPTY_ENABLE	= 0;
	gCurrentState.TX_HALF_ENABLE	= 0;
	gCurrentState.TX_AVAIL_ENABLE	= 0;

	// Baud control register bits

	gCurrentState.GPIO_DELTA		= 0;	// 68328 only
	gCurrentState.GPIO				= 0;	// 68328 only
	gCurrentState.GPIO_DIR			= 0;	// 68328 only
	gCurrentState.GPIO_SRC			= 0;	// 68328 only
	gCurrentState.UCLK_DIR			= 0;	// 68EZ328 only
	gCurrentState.BAUD_SRC			= 0;
	gCurrentState.DIVIDE			= 0;
	gCurrentState.PRESCALER			= 0x3F;

	// Receive register bits

	gCurrentState.RX_FIFO_FULL		= 0;
	gCurrentState.RX_FIFO_HALF		= 0;
	gCurrentState.DATA_READY		= 0;
	gCurrentState.OLD_DATA			= 0;	// 68EZ328 only
	gCurrentState.OVRUN				= 0;
	gCurrentState.FRAME_ERROR		= 0;
	gCurrentState.BREAK				= 0;
	gCurrentState.PARITY_ERROR		= 0;
	gCurrentState.RX_DATA			= 0;

	// Transmitter register bits

	gCurrentState.TX_FIFO_EMPTY		= 0;
	gCurrentState.TX_FIFO_HALF		= 0;
	gCurrentState.TX_AVAIL			= 0;
	gCurrentState.SEND_BREAK		= 0;
	gCurrentState.IGNORE_CTS		= 0;
	gCurrentState.BUSY				= 0;	// 68EZ328 only
	gCurrentState.CTS_STATUS		= 0;
	gCurrentState.CTS_DELTA			= 0;
	gCurrentState.TX_DATA			= 0;

	// Misc register bits

	gCurrentState.BAUD_TEST			= 0;	// 68EZ328 only
	gCurrentState.CLK_SRC			= 0;
	gCurrentState.FORCE_PERR		= 0;
	gCurrentState.LOOP				= 0;
	gCurrentState.BAUD_RESET		= 0;	// 68EZ328 only
	gCurrentState.IR_TEST			= 0;	// 68EZ328 only
	gCurrentState.RTS_CONT			= 0;
	gCurrentState.RTS				= 0;
	gCurrentState.IRDA_ENABLE		= 0;
	gCurrentState.IRDA_LOOP			= 0;
	gCurrentState.RX_POL			= 0;	// 68EZ328 only
	gCurrentState.TX_POL			= 0;	// 68EZ328 only
}


/***********************************************************************
 *
 * FUNCTION:	UART::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	UART::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	UART::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	UART::StateChanged
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::StateChanged (State& newState, Bool sendTxData)
{
	assert (gCurrentState.UART_TYPE == newState.UART_TYPE);

	// The following registers are not referenced by the ROM and
	// are currently not supported.
	//
	//		RX_CLK_CONT
	//		GPIO_DELTA_ENABLE
	//		OLD_ENABLE
	//		CTS_DELTA_ENABLE
	//		RX_FULL_ENABLE
	//		RX_HALF_ENABLE
	//		TX_EMPTY_ENABLE
	//		TX_HALF_ENABLE
	//		TX_AVAIL_ENABLE
	//		GPIO_DELTA
	//		GPIO
	//		GPIO_DIR
	//		GPIO_SRC
	//		UCLK_DIR
	//		BAUD_SRC
	//		BAUD_TEST
	//		CLK_SRC
	//		FORCE_PERR
	//		BAUD_RESET
	//		IR_TEST
	//		IRDA_LOOP
	//		RX_POL
	//		TX_POL
	//		RX_CLK_CONT

	Bool	resetPort = false;
	Bool	openClosePort = false;
	Parity	parity;
	int		stopBits;
	int		dataBits;
	uae_u32	baud;
	Bool	hwHandshaking;


	// ========== RX_ENABLE ==========
	//
	// This bit enables the receiver block. While this bit is low, the receiver is disabled and the
	// receive FIFO is flushed. This bit resets to 0.

	if (gCurrentState.RX_ENABLE != newState.RX_ENABLE)
	{
		if (newState.RX_ENABLE == 0)
		{
			PrvEmptyQueue (gRxFIFO);
		}
	}


	// ========== TX_ENABLE ==========
	//
	// This bit enables the transmitter block. While this bit is low, the transmitter is disabled and
	// the transmit FIFO is flushed. This bit resets to 0.

	if (gCurrentState.TX_ENABLE != newState.TX_ENABLE)
	{
		if (newState.TX_ENABLE)
		{
			PrvEmptyQueue (gTxFIFO);
		}
	}


	// ========== PARITY_EN ==========
	//
	// This bit controls the parity generator in the transmitter and parity checker in the receiver.
	// When this bit is high, they are enabled. When it is low, they are disabled.
	//
	// ========== ODD_EVEN ==========
	//
	// This bit controls the sense of the parity generator and checker. When this bit is high, odd
	// parity is generated and expected. When this bit is low, even parity is generated and
	// expected. This bit has no function if PARITY EN is low.

	if (gCurrentState.PARITY_EN != newState.PARITY_EN ||
		gCurrentState.ODD_EVEN != newState.ODD_EVEN)
	{
		resetPort = true;
	}

	if (newState.PARITY_EN == 0)
	{
		parity = kNoParity;
	}
	else if (newState.ODD_EVEN)
	{
		parity = kOddParity;
	}
	else
	{
		parity = kEvenParity;
	}


	// ========== STOP_BITS =========
	//
	// This bit controls the number of stop bits transmitted after a character. When this bit is high,
	// two stop bits are sent. When this bit is low, one stop bit is sent. This bit has no effect on the
	// receiver, which expects one or more stop bits.

	if (gCurrentState.STOP_BITS != newState.STOP_BITS)
	{
		resetPort = true;
	}

	stopBits = newState.STOP_BITS ? 2 : 1;


	// ========== CHAR8_7 ==========
	//
	// This bit controls the character length. While high, the transmitter and receiver are in 8-bit
	// mode. While low, they are in 7-bit mode. The transmitter then ignores B7 and the receiver
	// sets B7 to 0.

	if (gCurrentState.CHAR8_7 != newState.CHAR8_7)
	{
		resetPort = true;
	}

	dataBits = newState.CHAR8_7 ? 8 : 7;


	// ========== RX_RDY_ENABLE ==========
	//
	// When this bit is high, it enables an interrupt when the receiver has at least one data byte in
	// the FIFO. When it is low, this interrupt is disabled.

	if (gCurrentState.RX_RDY_ENABLE != newState.RX_RDY_ENABLE)
	{
		// Nothing to do here.  Interrupt generated elsewhere.
	}


	// ========== DIVIDE ==========
	//
	// These bits control the clock frequency produced by the baud rate generator.
	//
	//		000 = Divide by 1.
	//		001 = Divide by 2.
	//		010 = Divide by 4.
	//		011 = Divide by 8.
	//		100 = Divide by 16.
	//		101 = Divide by 32.
	//		110 = Divide by 64.
	//		111 = Divide by 128.
	//
	// ========== PRESCALER ==========
	//
	// These bits control the division value of the baud generator prescaler. The division value is
	// determined by the following formula:
	//
	//		Prescaler division value = 65 (decimal) - PRESCALER

	if (gCurrentState.DIVIDE != newState.DIVIDE ||
		gCurrentState.PRESCALER != newState.PRESCALER)
	{
		resetPort = true;
	}

	// Baud rate is sysClockFreq / preScaler / divider

	baud = 1036800 /
		(65 - newState.PRESCALER) /
		(1 << newState.DIVIDE);

	// "newRate" is only approximate to within 0.1%. Pin the value to an
	// exact value.
	//
	// !!! What to do if we can't pin to a valid value?

	(void) PrvPinBaud (baud);


	// ========== IGNORE_CTS ==========
	//
	// When this bit is high, it forces the CTS signal that is presented to the transmitter to always
	// be asserted, which effectively ignores the external pin.

	if (gCurrentState.IGNORE_CTS != newState.IGNORE_CTS)
	{
		resetPort = true;
	}

	hwHandshaking = newState.IGNORE_CTS == 0;


	// ========== RTS_CONT ==========
	//
	// This bit selects the function of the RTS pin.
	//
	//		0 = RTS pin is controlled by the RTS bit.
	//		1 = RTS pin is controlled by the receiver FIFO. When the FIFO is full (one slot is
	//			remaining), RTS is negated.
	//
	// ========== RTS ==========
	//
	// This bit controls the RTS pin while the RTS CONT bit is 0.
	//
	//		0 = RTS pin is 1.
	//		1 = RTS pin is 0.

	if (gCurrentState.RTS_CONT != newState.RTS_CONT ||
		gCurrentState.RTS != newState.RTS)
	{
		// Nothing to do here.  These settings are looked at in the cycle code
		// that generates interrupts. !!! TBD
	}


	// ========== UART_ENABLE ==========
	//
	// This bit enables the UART module. When this bit is low, the UART module is disabled and
	// in low-power mode. While this bit is high, the UART module is active. This bit resets to 0.

	// ========== IRDA_ENABLE ==========
	//
	// This bit enables the IrDA interface.
	//
	//		0 = Normal NRZ operation.
	//		1 = IRDA operation.

	if (gCurrentState.UART_ENABLE != newState.UART_ENABLE ||
		gCurrentState.IRDA_ENABLE != newState.IRDA_ENABLE)
	{
		openClosePort = true;
		resetPort = true;
	}

#if 0
	if (openClosePort)
	{
		Platform::CloseSerialPort ();
	//	Platform::CloseIRPort ();	// !!! TBD

		if (newState.UART_ENABLE)
		{
			if (newState.IRDA_ENABLE == 0)
			{
				Platform::OpenSerialPort ();
			}
			else
			{
	//			Platform::OpenIRPort ();	// !!! TBD
			}
		}
	}
#endif

	PRINTF ("Serial Emulation: Reacting to register changes.");

	if (resetPort && newState.UART_ENABLE)
	{
		PRINTF ("Serial Emulation: Serial control registers have changed and the UART is enabled.");

		if (newState.IRDA_ENABLE == 0)
		{
			PRINTF ("Serial Emulation: IRDA is not enabled.");

			if (openClosePort							||
				gLastParity			!= parity			||
				gLastStopBits		!= stopBits			||
				gLastDataBits		!= dataBits			||
				gLastBaud			!= baud				||
				gLastHwHandshaking	!= hwHandshaking)
			{
				if (openClosePort)
					PRINTF ("Serial Emulation: UART re-enabled, so forcing serial port settings.");
				else
					PRINTF ("Serial Emulation: Settings have changed.");

				PRINTF ("Serial Emulation: Settings:");
				PRINTF ("	parity			= %ld", (long) parity);
				PRINTF ("	stopBits		= %ld", (long) stopBits);
				PRINTF ("	dataBits		= %ld", (long) dataBits);
				PRINTF ("	baud			= %ld", (long) baud);
				PRINTF ("	hwHandshaking	= %ld", (long) hwHandshaking);

				ErrCode	err = Platform::SetSerialPortSettings (parity, stopBits, dataBits, baud, hwHandshaking);

				if (err == errNone)
				{
					gLastParity			= parity;
					gLastStopBits		= stopBits;
					gLastDataBits		= dataBits;
					gLastBaud			= baud;
					gLastHwHandshaking	= hwHandshaking;
				}
			}
			else
			{
				PRINTF ("Serial Emulation: Settings haven't changed and not forcing their establishment.");
			}
		}
		else
		{
			PRINTF ("Serial Emulation: IRDA is enabled -- NOT SUPPORTED.");

	//		Platform::SetIRPortSettings ();	// !!! TBD
		}
	}
	else
	{
		if (!resetPort)
			PRINTF ("Serial Emulation: Serial control registers have not changed.");
		else
			PRINTF ("Serial Emulation: UART is disabled.");
	}


	// ========== SEND_BREAK ==========
	//
	// This bit forces the transmitter to immediately send continuous zeros creating a break
	// character.

	if (gCurrentState.SEND_BREAK != newState.SEND_BREAK)
	{
		// Nothing to do right now.  External cycling will send out the breaks.
		// !!! TBD.
	}


	// ========== TX_DATA ==========
	//
	// These bits are the parallel transmit-data inputs. In 7-bit mode, D7 is ignored and in 8-bit
	// mode, all of the bits are used. Data is transmitted LSB first. A new character is transmitted
	// when these bits are written and have passed through the FIFO.

	if (sendTxData && newState.UART_ENABLE && newState.TX_ENABLE)
	{
		if (newState.LOOP == 0)
		{
			if (Platform::SerialPortOpen ())		// The host serial port is open
			{
				// With or without hardware handshaking, we'll put data
				// in the FIFO, and let the host's handshaking take care
				// of when the data is removed from the FIFO.

				if (gTxFIFO.GetFree () > 0)			// There's room in the FIFO
				{
					if (newState.TX_DATA == 0x7E)
					{
					//	Debugger ();
					}

					gTxFIFO.Put (newState.TX_DATA);	// so add the data

					// Call Platform::TransmitTxFIFO here to send the data we
					// just queued up.  Doing this is important on the Mac in
					// order to send out the data quickly instead of later at
					// idle time.

					Platform::TransmitTxFIFO ();
				}
			}
			else									// The host serial port is NOT open
			{
				if (hwHandshaking)	// Reflects the state of the IGNORE_CTS bit.
				{
					// With hardware handshaking, data is sent only when CTS
					// is asserted.  With no host serial port, we define that
					// CTS is never asserted, so the data clogs up the FIFO.

					if (gTxFIFO.GetFree () > 0)			// There's room in the FIFO
					{
						gTxFIFO.Put (newState.TX_DATA);	// so add the data

						// Serial port is closed, so don't call Platform::TransmitTxFIFO.
					}
				}
				else
				{
					// With no hardware handshaking, data is sent whenever it's
					// ready.  With nowhere to go, we can drop it on the floor.
				}
			}
		}
		else	// We're in loopback mode.
		{
			if (gRxFIFO.GetFree () > 0)
			{
				gRxFIFO.Put (newState.TX_DATA);
			}
		}
	}


	// ========== LOOP ==========
	//
	// This bit controls loopback for system testing purposes. When this bit is high, the receiver
	// input is internally connected to the transmitter and ignores the RXD pin. The TXD pin is
	// unaffected by this bit.

	if (gCurrentState.LOOP != newState.LOOP)
	{
		// Nothing to do here.  This bit is examined in the code
		// that reacts to TX_DATA.
	}


	// Update the state in case any of the above operations have side-effects.

	UpdateState (newState, false);


	// Remember this for next time.

	gCurrentState = newState;
}


/***********************************************************************
 *
 * FUNCTION:	UART::UpdateState
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void UART::UpdateState (State& state, Bool refreshRxData)
{
	assert (gCurrentState.UART_TYPE == state.UART_TYPE);

	// === RX_FIFO_FULL ===
	//
	// This read-only bit indicates that the receiver FIFO is full and may generate an overrun. This
	// bit generates a maskable interrupt.
	//
	// Further, from the overview section of the manual:
	//
	// If your software has a short interrupt
	// latency time, the FIFO FULL interrupt in the Receiver register can be enabled. The FIFO has
	// one remaining space available when this interrupt is generated.

	state.RX_FIFO_FULL = gRxFIFO.GetFree () == 0;	// Interrupt generated in Foo::Cycle.


	// === RX_FIFO_HALF ===
	//
	// This read-only bit indicates that the receiver FIFO has four or fewer slots remaining in the
	// FIFO. This bit generates a maskable interrupt.

	state.RX_FIFO_HALF = gRxFIFO.GetFree () <= 4;	// Interrupt generated in Foo::Cycle.


	// === DATA_READY ===
	//
	// This read-only bit indicates that at least one byte is present in the receive FIFO. The
	// character bits are valid only while this bit is set. This bit generates a maskable interrupt.

	state.DATA_READY = gRxFIFO.GetUsed () > 0;	// Interrupt generated in Foo::Cycle.


	// === OLD_DATA ===	// 68EZ328 only
	//
	// This read-only bit indicates that data in the FIFO is older than 30 bit times. It is useful in
	// situations where the FIFO FULL or FIFO HALF interrupts are used. If there is data in the
	// FIFO, but below the interrupt threshold, a maskable interrupt can be generated to alert the
	// software that unread data is present. This bit clears when the character bits are read.

	// Not supported right now.


	// === OVRUN ===
	//
	// When this read-only bit is high, it indicates that the receiver overwrote data in the FIFO. The
	// character with this bit set is valid, but at least one previous character was lost. In normal
	// circumstances, this bit should never be set. It indicates that your software is not keeping up
	// with the incoming data rate. This bit is updated and valid for each received character.

	// !!! TBD


	// === FRAME_ERROR ===
	//
	// While high, this read-only bit indicates that the current character had a framing error
	// (missing stop bit), indicating the possibility of corrupted data. This bit is updated for each
	// character read from the FIFO.

	// !!! TBD


	// === BREAK ===
	//
	// When this read-only bit is high, it indicates that the current character was detected as a
	// BREAK. The data bits are all 0 and the stop bit was also 0. The FRAME ERROR bit will
	// always be set when this bit is set. If odd parity is selected, PARITY ERROR will also be set
	// along with this bit. This bit is updated and valid with each character read from the FIFO.

	// !!! TBD

	// === PARITY_ERROR ===
	//
	// When this read-only bit is high, it indicates that the current character was detected with a
	// parity error, indicating the possibility of corrupted data. This bit is updated and valid with
	// each character read from the FIFO. While parity is disabled, this bit always reads zero.

	// !!! TBD

	// === RX_DATA ===
	//
	// These read-only bits are the top receive character in the FIFO. They have no meaning if the
	// DATA READY bit is 0. In 7-bit mode, the MSB is forced to 0 and in 8-bit mode, all bits are
	// active.

	if (state.DATA_READY)	// Test against this (which was set above) instead of gRxFIFO.GetUsed()
							// to protect against a byte being added between then and now.
	{
		// !!! Should probably test against RTS, too.  Actually, that test should
		// happen before putting the byte into the Rx FIFO.

		if (state.UART_ENABLE && state.RX_ENABLE && refreshRxData)
		{
			state.RX_DATA = gRxFIFO.Get ();

			// If the Rx FIFO is draining, see if there's anything to
			// refresh it with.  Doing this is important on the Mac.  On
			// the Mac, we try filling the RX FIFO at idle time.  However,
			// idle time may not occur for a while.  During that delay,
			// timeouts may occur (such as inter-character timeouts while
			// reading responses from a modem, which are on the order of
			// 1/32 of a second; this is 2 ticks, while the delay until the
			// next idle may be 10 ticks away).

			if (gRxFIFO.GetUsed () < 4)
			{
				Platform::ReceiveRxFIFO ();
			}
		}
	}


	// === TX_FIFO_EMPTY ===
	//
	// This read-only bit indicates that the transmit FIFO is empty. This bit generates a maskable
	// interrupt.

	state.TX_FIFO_EMPTY = gTxFIFO.GetUsed () == 0;	// Interrupt generated in Foo::Cycle.


	// === TX_FIFO_HALF ===
	//
	// This read-only bit indicates that the transmit FIFO is less than half full. This bit generates a
	// maskable interrupt.

	state.TX_FIFO_HALF = gTxFIFO.GetUsed () < 4;	// Interrupt generated in Foo::Cycle.


	// === TX_AVAIL ===
	//
	// This read-only bit indicates that the transmit FIFO has at least one slot available for data.
	// This bit generates a maskable interrupt.

	state.TX_AVAIL = gTxFIFO.GetFree () > 0;	// Interrupt generated in Foo::Cycle.


	// === BUSY ===	// 68EZ328 only
	//
	// When this bit is high, it indicates that the transmitter is busy sending a character. This signal
	// is asserted while the transmitter state machine is not idle or the FIFO has data in it.

	// Not supported right now.


	// === CTS_STATUS ===
	//
	// This bit indicates the current status of the CTS pin. A "snapshot" of the pin is taken
	// immediately before this bit is presented to the data bus. While the IGNORE CTS bit is high,
	// this bit can serve as a general-purpose input.
	//
	// Note that this pin is ACTIVE LOW!  That's why the Boolean expression is negated below.
	//
	// For now, say that it's clear to send if the FIFO is empty
	//
	// !!! TBD - could be better?

	state.CTS_STATUS = !(gTxFIFO.GetUsed () == 0);


	// === CTS_DELTA ===
	//
	// When this bit is high, it indicates that the CTS pin changed state and generates a maskable
	// interrupt. The current state of the CTS pin is available on the CTS STATUS bit. You can
	// generate an immediate interrupt by setting this bit high. The CTS interrupt is cleared by
	// writing 0 to this bit.

	// Not supported right now.


	// Remember this for next time.

	gCurrentState = state;

	// As long as there's still data going in or out, keep the "needs updating"
	// flag set to true.  Otherwise, set it to false.

//	gStateNeedsUpdating = state.DATA_READY || state.TX_AVAIL;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetRxQueue
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

TByteQueue& UART::GetRxQueue (void)
{
	return gRxFIFO;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetTxQueue
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

TByteQueue& UART::GetTxQueue (void)
{
	return gTxFIFO;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetLastParity
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

UART::Parity UART::GetLastParity (void)
{
	return gLastParity;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetLastStopBits
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

int UART::GetLastStopBits (void)
{
	return gLastStopBits;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetLastDataBits
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

int UART::GetLastDataBits (void)
{
	return gLastDataBits;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetLastBaud
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

uae_u32 UART::GetLastBaud (void)
{
	return gLastBaud;
}


/***********************************************************************
 *
 * FUNCTION:	UART::GetLastHwHandshaking
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	none.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool UART::GetLastHwHandshaking (void)
{
	return gLastHwHandshaking;
}


/***********************************************************************
 *
 * FUNCTION:	PrvPinBaud
 *
 * DESCRIPTION:	Pins the given baud value to the test baud value if the
 *				former is sufficiently close to the latter.
 *
 * PARAMETERS:	newBaud - the value to possibly alter.
 *
 *				testBaud - the value to pin to.
 *
 * RETURNED:	newBaud is changed in place.  If it is changed, the
 *				function returns true.  Otherwise, it returns false.
 *
 ***********************************************************************/

Bool PrvPinBaud (uae_u32& newBaud)
{
	Bool	pinned = false;

	if (!pinned)	pinned = PrvPinBaud (newBaud, 150);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 300);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 600);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 1200);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 1800);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 2400);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 3600);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 4800);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 7200);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 9600);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 14400);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 19200);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 28800);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 38400);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 57600);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 115200);
	if (!pinned)	pinned = PrvPinBaud (newBaud, 230400);

	return pinned;
}


/***********************************************************************
 *
 * FUNCTION:	PrvPinBaud
 *
 * DESCRIPTION:	Pins the given baud value to the test baud value if the
 *				former is sufficiently close to the latter.
 *
 * PARAMETERS:	newBaud - the value to possibly alter.
 *
 *				testBaud - the value to pin to.
 *
 * RETURNED:	newBaud is changed in place.  If it is changed, the
 *				function returns true.  Otherwise, it returns false.
 *
 ***********************************************************************/

Bool PrvPinBaud (uae_u32& newBaud, uae_u32 testBaud)
{
	// Pin to within 2%.  The Dragonball reference says that the uBaud
	// register should be accurate to within 0.1%, but let's give it
	// some slop.

	if (newBaud > (testBaud - (testBaud / 50)) &&
		newBaud < (testBaud + (testBaud / 50)))
	{
		newBaud = testBaud;
		return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	PrvEmptyQueue
 *
 * DESCRIPTION:	Remove all contents of the given queue.
 *
 * PARAMETERS:	q - TByteQueue to erase.
 *
 * RETURNED:	nothing.
 *
 ***********************************************************************/

void PrvEmptyQueue (TByteQueue& q)
{
	while (q.GetUsed () > 0)
	{
		q.Get ();
	}
}
