/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "PreferenceMgr.h"

#include "StringConversions.h"	// ToString, FromString
#include "Logging.h"			// LogUpdateCache
#include "Platform.h"			// GetPreferenceFile, _stricmp
#include "Platform_Files.h" 	// OpenAsFILE, ToPrefString, FromPrefString

#include <algorithm>			// find()

Preferences*			gPrefs;
EmulatorPreferences*	gEmuPrefs;
omni_mutex				Preferences::fgPrefsMutex;


// Define all the keys

#define DEFINE_PREF_KEYS(name, type, init)	const char* kPrefKey##name = #name;
FOR_EACH_PREF(DEFINE_PREF_KEYS)


static Bool PrvFirstBeginsWithSecond (const string& first, const string& second)
{
	// Return true if the first string starts with the second string, and
	// if the point where they both end in common is either the end of
	// the first string or a delimiter indicating that a sub-part follows.

	return
		(first.size () >= second.size ()) &&
		(memcmp (first.c_str (), second.c_str (), second.size ()) == 0) &&
		((first.size () == second.size ()) ||
		 (first[second.size ()] == '.') ||
		 (first[second.size ()] == '['));
}


// ----------------------------------------------------------------------
//	* BasePreference
//
//	Clients of the prefernces sub-system access the data via Preference
//	objects.  These objects are created from the Preference class,
//	which the client templatizes based on the data type in which they
//	want the data returned.
//
//	Preference descends from BasePreference, which implements generic
//	functionality.	If we were to put this functionality into Preference
//	itself, we'd have massive unnecessary code bloat.
// ----------------------------------------------------------------------

/***********************************************************************
 *
 * FUNCTION:	BasePreference constructor
 *
 * DESCRIPTION: Initializes all the data members.
 *
 * PARAMETERS:	name - name of the key used to fetch the data.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

BasePreference::BasePreference (PrefKeyType name, bool acquireLock) :
	fName (name),
	fLoaded (false),
	fChanged (false),
	fAcquireLock (acquireLock)
{
}

BasePreference::BasePreference (long index, bool acquireLock) :
	fName (::ToString (index)),
	fLoaded (false),
	fChanged (false),
	fAcquireLock (acquireLock)
{
}


/***********************************************************************
 *
 * FUNCTION:	BasePreference destructor
 *
 * DESCRIPTION: Destroys all the data members.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

BasePreference::~BasePreference (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	BasePreference::Load
 *
 * DESCRIPTION: Lock the preference data and attempt to load the item
 *				specified by the key (specified to the constructor).
 *				If the data was able to be loaded (which pretty much
 *				means that we were able to convert it into the correct
 *				type, set fLoaded to true.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void BasePreference::Load (void)
{
	if (fAcquireLock)
	{
	// Lock this here.	In case DoLoad sets the prefix, we don't want
	// any other threads accidently using or chaning that prefix.

	omni_mutex_lock lock (Preferences::fgPrefsMutex);

	if (this->DoLoad ())
	{
		fLoaded = true;
	}
	}
	else
	{
	if (this->DoLoad ())
	{
		fLoaded = true;
	}
	}
}


/***********************************************************************
 *
 * FUNCTION:	BasePreference::Save
 *
 * DESCRIPTION: If the data has been change (via operator=()), lock the
 *				preference data and store the entry back out to it.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void BasePreference::Save (void)
{
	if (fChanged)
	{
		// Lock this here.	In case DoSave sets the prefix, we don't want
		// any other threads accidently using or chaning that prefix.

		if (fAcquireLock)
		{
			omni_mutex_lock lock (Preferences::fgPrefsMutex);

			this->DoSave ();
			fChanged = false;
		}
		else
		{
			this->DoSave ();
			fChanged = false;
		}
	}
}


// ----------------------------------------------------------------------
//	* Preference<T>
//
//	Clients of the prefernces sub-system access the data via Preference
//	objects.  These objects are created from the Preference class,
//	which the client templatizes based on the data type in which they
//	want the data returned.
//
//	Once an object has been created, the preference value can be fetched
//	with the * operator:
//
//		Preference<bool>	pref(kPrefKeySomeSetting);
//		if (*pref)
//			...
//
//	The value can also be fetched with the -> operator:
//
//		Preference<RGBType> pref(kPrefKeySomeColor);
//		if (pref->fRed == 0 && pref->fGreen == 0 && pref->fBlue == 0)
//			...
//
//	The value can be changed by simply assigning to the Preference object:
//
//		Preference<string>	pref(kPrefKeySomeString);
//		pref = string ("new string value");
//
//	You can find out if there is a valid value for the desired preference
//	by calling Loaded():
//
//		Preference<bool>		pref1(kPrefKeySomeSetting);
//		pref1 = true;
//
//		Preference<GremlinInfo> pref2(kPrefKeySomeSetting);
//		if (pref2.Loaded ())
//			... fails because we can't convert a bool to a GremlinInfo...
//
//		Preference<bool>		pref3("bogus undefined key);
//		if (pref3.Loaded ())
//			... fails because we key was not found...
//
// ----------------------------------------------------------------------


/***********************************************************************
 *
 * FUNCTION:	Preference<T> constructor
 *
 * DESCRIPTION: Calls BasePreference to initialize the base members,
 *				then calls Load() to load the data.
 *
 * PARAMETERS:	name - name of the key used to fetch the data.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

template <class T>
Preference<T>::Preference (PrefKeyType name, bool acquireLock) :
	BasePreference (name, acquireLock),
	fValue (T())
{
	this->Load ();
}

template <class T>
Preference<T>::Preference (long index, bool acquireLock) :
	BasePreference (index, acquireLock),
	fValue (T())
{
	this->Load ();
}


	
/***********************************************************************
 *
 * FUNCTION:	Preference<T> destructor
 *
 * DESCRIPTION: Ensures that any changes are flushed back to the
 *				preference collection before destructing the base class.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

template <class T>
Preference<T>::~Preference (void)
{
	this->Save ();
}


/***********************************************************************
 *
 * FUNCTION:	Preference<T>::DoLoad
 *
 * DESCRIPTION: Virtual function responsible for the actual conversion
 *				of string information into natural data types.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	True if the function could be loaded and converted.
 *				False otherwise.
 *
 ***********************************************************************/

template <class T>
bool Preference<T>::DoLoad (void)
{
	string	value;

	if (gPrefs->GetPref (fName, value) && ::FromString (value, fValue))
	{
		return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	Preference<T>::DoSave
 *
 * DESCRIPTION: Virtual function responsible for the actual conversion
 *				of our natural data types into strings.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

template <class T>
void Preference<T>::DoSave (void)
{
	gPrefs->SetPref (fName, ::ToString (fValue));
}


// ----------------------------------------------------------------------
//	Instantiations of Preference class for all non-compound types.
// ----------------------------------------------------------------------

template class Preference<bool>;
template class Preference<char>;
template class Preference<signed char>;
template class Preference<unsigned char>;
template class Preference<signed short>;
template class Preference<unsigned short>;
template class Preference<signed int>;
template class Preference<unsigned int>;
template class Preference<signed long>;
template class Preference<unsigned long>;

template class Preference<string>;

//template class Preference<Bool>;
template class Preference<CloseActionType>;
template class Preference<DeviceType>;


// ----------------------------------------------------------------------
//	Specializations of Preference class for compound types.
//
//		template Preference<Bool>;
//		template Preference<CloseActionType>;
//		template Preference<Configuration>;
//		template Preference<DatabaseInfo>;
//		template Preference<DatabaseInfoList>;
//		template Preference<DeviceType>;
//		template Preference<FileReference>;
//		template Preference<FileRefList>;
//		template Preference<GremlinInfo>;
//		template Preference<PointType>;
//		template Preference<RGBType>;
//		template Preference<SkinNameList>;
// ----------------------------------------------------------------------

template <>
bool Preference<Configuration>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("fDeviceType", value)	&& ::FromString (value, fValue.fDeviceType) &&
		gPrefs->GetPref ("fRAMSize", value) 	&& ::FromString (value, fValue.fRAMSize) &&
		gPrefs->GetPref ("fROMFile", value) 	&& ::FromString (value, fValue.fROMFile))
	{
		loaded = true;
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<Configuration>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("fDeviceType", 	::ToString (fValue.fDeviceType));
	gPrefs->SetPref ("fRAMSize",		::ToString (fValue.fRAMSize));
	gPrefs->SetPref ("fROMFile",		::ToString (fValue.fROMFile));

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
bool Preference<DatabaseInfo>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("creator", value)	&& ::FromString (value, fValue.creator) &&
		gPrefs->GetPref ("type", value) 	&& ::FromString (value, fValue.type) &&
		gPrefs->GetPref ("version", value)	&& ::FromString (value, fValue.version) &&
		gPrefs->GetPref ("dbID", value) 	&& ::FromString (value, fValue.dbID) &&
		gPrefs->GetPref ("cardNo", value)	&& ::FromString (value, fValue.cardNo) &&
		gPrefs->GetPref ("modDate", value)	&& ::FromString (value, fValue.modDate) &&
		gPrefs->GetPref ("dbAttrs", value)	&& ::FromString (value, fValue.dbAttrs) &&
		gPrefs->GetPref ("name", value) 	&& ::FromString (value, fValue.name))
	{
		loaded = true;
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<DatabaseInfo>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("creator", 	::ToString (fValue.creator));
	gPrefs->SetPref ("type",		::ToString (fValue.type));
	gPrefs->SetPref ("version", 	::ToString (fValue.version));
	gPrefs->SetPref ("dbID",		::ToString (fValue.dbID));
	gPrefs->SetPref ("cardNo",		::ToString (fValue.cardNo));
	gPrefs->SetPref ("modDate", 	::ToString (fValue.modDate));
	gPrefs->SetPref ("dbAttrs", 	::ToString (fValue.dbAttrs));
	gPrefs->SetPref ("name",		::ToString (fValue.name));

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
bool Preference<DatabaseInfoList>::DoLoad (void)
{
	bool	loaded = true;
	string	value;

	gPrefs->PushPrefix (fName);

	long	ii = 0;
	while (1)
	{
		Preference<DatabaseInfo> pref (ii++, false);
		if (!pref.Loaded ())
			break;

		fValue.push_back (*pref);
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<DatabaseInfoList>::DoSave (void)
{
	gPrefs->DeletePref (fName);

	gPrefs->PushPrefix (fName);

	long	ii = 0;
	DatabaseInfoList::iterator	iter = fValue.begin ();
	while (iter != fValue.end ())
	{
		Preference<DatabaseInfo> pref (ii++, false);
		pref = *iter;

		++iter;
	}

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
bool Preference<FileRefList>::DoLoad (void)
{
	bool	loaded = true;
	gPrefs->PushPrefix (fName);

	long	ii = 0;
	while (1)
	{
		Preference<FileReference>	pref (ii++, false);
		if (!pref.Loaded ())
			break;

		fValue.push_back (*pref);
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<FileRefList>::DoSave (void)
{
	gPrefs->DeletePref (fName);

	gPrefs->PushPrefix (fName);

	long	ii = 0;
	FileRefList::iterator	iter = fValue.begin ();
	while (iter != fValue.end ())
	{
		Preference<FileReference>	pref (ii++, false);
		pref = *iter;

		++iter;
	}

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
bool Preference<GremlinInfo>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("fNumber", value)					&& ::FromString (value, fValue.fNumber) &&
		gPrefs->GetPref ("fSteps", value)					&& ::FromString (value, fValue.fSteps) &&
		gPrefs->GetPref ("fSaveFrequency", value)			&& ::FromString (value, fValue.fSaveFrequency))
	{
		loaded = true;
	}

	if (loaded)
	{
		Preference<DatabaseInfoList> pref ("fAppList", false);
		if (pref.Loaded ())
		{
			fValue.fAppList = *pref;
		}
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<GremlinInfo>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("fNumber", 				::ToString (fValue.fNumber));
	gPrefs->SetPref ("fSteps",					::ToString (fValue.fSteps));
	gPrefs->SetPref ("fSaveFrequency",			::ToString (fValue.fSaveFrequency));

	// Save the fAppList collection.
	{
		Preference<DatabaseInfoList> pref ("fAppList", false);
		pref = fValue.fAppList;
	}

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
bool Preference<HordeInfo>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("fStartNumber", value)			&& ::FromString (value, fValue.fStartNumber) &&
		gPrefs->GetPref ("fStopNumber", value)			&& ::FromString (value, fValue.fStopNumber) &&
		gPrefs->GetPref ("fSaveFrequency", value)		&& ::FromString (value, fValue.fSaveFrequency) &&
		gPrefs->GetPref ("fDepthBound", value)			&& ::FromString (value, fValue.fSwitchDepth) &&
		gPrefs->GetPref ("fMaxDepth", value)			&& ::FromString (value, fValue.fMaxDepth))
	{
		loaded = true;
	}

	if (loaded)
	{
		Preference<DatabaseInfoList> pref ("fAppList", false);
		if (pref.Loaded ())
		{
			fValue.fAppList = *pref;
		}
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<HordeInfo>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("fStartNumber", 				::ToString (fValue.fStartNumber));
	gPrefs->SetPref ("fStopNumber",					::ToString (fValue.fStopNumber));
	gPrefs->SetPref ("fSaveFrequency",				::ToString (fValue.fSaveFrequency));
	gPrefs->SetPref ("fDepthBound",					::ToString (fValue.fSwitchDepth));
	gPrefs->SetPref ("fMaxDepth",					::ToString (fValue.fMaxDepth));

	// Save the fAppList collection.
	{
		Preference<DatabaseInfoList> pref ("fAppList", false);
		pref = fValue.fAppList;
	}

	gPrefs->PopPrefix ();
}


// ----------------------------------------------------------------------

	// It's odd that we have to provide explicit specializations of the
	// constructor and destructor for Preference<PointType>.  I think it
	// has something to do with the fact that PointType doesn't have a
	// constructor of its own.

template <>
Preference<PointType>::Preference (PrefKeyType name, bool acquireLock) :
	BasePreference (name, acquireLock)
	//,	fValue ()
{
	this->Load ();
}

template <>
Preference<PointType>::Preference (long index, bool acquireLock) :
	BasePreference (index, acquireLock)
	//,	fValue ()
{
	this->Load ();
}

template <>
Preference<PointType>::~Preference (void)
{
	this->Save ();
}

template <>
bool Preference<PointType>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("x", value)	&& ::FromString (value, fValue.x) &&
		gPrefs->GetPref ("y", value)	&& ::FromString (value, fValue.y))
	{
		loaded = true;
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<PointType>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("x",	::ToString (fValue.x));
	gPrefs->SetPref ("y",	::ToString (fValue.y));

	gPrefs->PopPrefix ();
}

// ----------------------------------------------------------------------

template <>
Preference<RGBType>::Preference (PrefKeyType name, bool acquireLock) :
	BasePreference (name, acquireLock),
	fValue ()
{
	this->Load ();
}

template <>
Preference<RGBType>::Preference (long index, bool acquireLock) :
	BasePreference (index, acquireLock),
	fValue ()
{
	this->Load ();
}

template <>
Preference<RGBType>::~Preference (void)
{
	this->Save ();
}

template <>
bool Preference<RGBType>::DoLoad (void)
{
	bool	loaded = false;
	string	value;

	gPrefs->PushPrefix (fName);

	if (gPrefs->GetPref ("red", value)		&& ::FromString (value, fValue.fRed) &&
		gPrefs->GetPref ("green", value)	&& ::FromString (value, fValue.fGreen) &&
		gPrefs->GetPref ("blue", value) 	&& ::FromString (value, fValue.fBlue))
	{
		loaded = true;
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<RGBType>::DoSave (void)
{
	gPrefs->PushPrefix (fName);

	gPrefs->SetPref ("red", 	::ToString (fValue.fRed));
	gPrefs->SetPref ("green",	::ToString (fValue.fGreen));
	gPrefs->SetPref ("blue",	::ToString (fValue.fBlue));

	gPrefs->PopPrefix ();
}


// ----------------------------------------------------------------------

template <>
bool Preference<SkinNameList>::DoLoad (void)
{
	bool	loaded = true;
	gPrefs->PushPrefix (fName);

	long	ii = 0;
	while (1)
	{
		Preference<SkinName>	pref (ii++, false);
		if (!pref.Loaded ())
			break;

		fValue.push_back (*pref);
	}

	gPrefs->PopPrefix ();

	return loaded;
}

template <>
void Preference<SkinNameList>::DoSave (void)
{
//	gPrefs->DeletePref (fName);

	gPrefs->PushPrefix (fName);

	long	ii = 0;
	SkinNameList::iterator	iter = fValue.begin ();
	while (iter != fValue.end ())
	{
		Preference<SkinName>	pref (ii++, false);
		pref = *iter;

		++iter;
	}

	gPrefs->PopPrefix ();
}

/***********************************************************************
 *
 * FUNCTION:	Preferences::Preferences
 *
 * DESCRIPTION: Constructor.  Does nothing but construct data members.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

Preferences::Preferences (void) :
	fPreferences ()
{
	if (gPrefs == NULL)
		gPrefs = this;
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::~Preferences
 *
 * DESCRIPTION: Destructor.  Does nothing but destruct data members.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

Preferences::~Preferences (void)
{
	if (gPrefs == this)
		gPrefs = NULL;
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::Load
 *
 * DESCRIPTION: Loads the preferences from the storage medium.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::Load (void)
{
	FileReference	prefRef = Platform::GetPreferenceFile ();
	FILE*			f = prefRef.OpenAsFILE ("r");	// for read, fail on fnf

	if (f == NULL)
		return;

	// If the banner doesn't look right, blow out of here.
	if (!this->ReadBanner (f))
		return;

	string	line;
	int 	ch;

	do {
		ch = fgetc (f);

		if (ch == '\n' || ch == EOF)
		{
			// Skip leading WS
			while ((line[0] == ' ') || (line[0] == '\t'))
			{
				line.erase (0, 1);
			}

			// Skip comment and empty lines
			if ((line[0] == ';' ) || (line[0] == '#' ) || (line[0] == '\0'))
			{
				line.erase ();
				continue;
			}

			// Everything else is data
			string::size_type	p = line.find ('=');
			if (p == string::npos)
			{
				line.erase ();
				continue;
			}

			// Split up the line into key/value sections and
			// add it to our hash table.
			string	key (line, 0, p);
			string	value (line, p + 1, line.size ());
			this->SetPref (key, value);
			
			line.erase ();
		}
		else
		{
			line += ch;
		}
	}
	while (ch != EOF);

	fclose (f);
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::Save
 *
 * DESCRIPTION: Saves the preferences to the storage medium.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::Save (void)
{
	FileReference	prefRef = Platform::GetPreferenceFile ();
	FILE*			f = prefRef.OpenAsFILE ("w");	// for write, destroy old

	if (f == NULL)
		return;

	this->StripUnused ();
	this->WriteBanner (f);

	iterator	iter = fPreferences.begin ();
	while (iter != fPreferences.end ())
	{
		fprintf (f, "%s=%s\n", iter->first.c_str (), iter->second.c_str ());
		++iter;
	}

	fclose (f);
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::GetPref
 *
 * DESCRIPTION: Returns the specified preference.  The given key is
 *				prepended with any prefixes, and the resulting key is
 *				used to look up the value.
 *
 * PARAMETERS:	key - the key used to look up the preference's value.
 *
 *				value - the found value (if any).
 *
 * RETURNED:	True if the key/value could be found, false otherwise.
 *
 ***********************************************************************/

bool Preferences::GetPref (const string& key, string& value)
{
	// Locking done in Preference::DoLoad.
//	omni_mutex_lock lock (fgPrefsMutex);

	string	fullKey = gPrefs->ExpandKey (key);

	iterator	iter = fPreferences.find (fullKey);
	if (iter == fPreferences.end ())
		return false;

	value = iter->second;
	return true;
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::SetPref
 *
 * DESCRIPTION: Store the given key/value pair into our collection.  If
 *				an entry with the given key already exists, its value
 *				is updated with the specified one.
 *
 *				If the value is changed or newly added, then send out
 *				notification to anyone registered for this preference.
 *
 * PARAMETERS:	key - the key used to look up the preference.
 *
 *				value - the value to associate with the key.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::SetPref (const string& key, const string& value)
{
	// Locking done in Preference::DoSave.
//	omni_mutex_lock lock (fgPrefsMutex);

	string	fullKey = gPrefs->ExpandKey (key);
	Bool	doNotify = false;

	iterator	iter = fPreferences.find(fullKey);
	if (iter == fPreferences.end ())
	{
		doNotify = true;
		fPreferences[fullKey] = value;
	}
	else
	{
		doNotify = (iter->second != value);
		iter->second = value;
	}

	if (doNotify)
	{
		this->DoNotify (fullKey);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::DeletePref
 *
 * DESCRIPTION: Delete a preference from our collection.  The delete
 *				operation works on the given key and all sub-keys (that
 *				is, all keys that have "key" as its prefix).
 *
 * PARAMETERS:	key - the key for the value or value to delete.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::DeletePref (const string& key)
{
	// Locking done in Preference::DoSave.
//	omni_mutex_lock lock (fgPrefsMutex);

	string	deleteKey = gPrefs->ExpandKey (key);

	// Iterate over all the entries in the map.

	iterator	iter = fPreferences.begin ();
	while (iter != fPreferences.end ())
	{
		// For each entry, get the key.

		string	thisKey (iter->first.c_str());

		if (::PrvFirstBeginsWithSecond (thisKey, deleteKey))
		{
			fPreferences.erase (iter);
			iter = fPreferences.begin ();
		}
		else
		{
			++iter;
		}
	}

	this->DoNotify (deleteKey);
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::PushPrefix
 *
 * DESCRIPTION: Push a prefix onto our internal stack.	These prefixes
 *				are used later to build a full key with which to fetch
 *				and store preference values.
 *
 * PARAMETERS:	prefix - the subkey to push onto our stack.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::PushPrefix (const string& prefix)
{
	// Locking done in Preference::DoSave/DoLoad.
//	omni_mutex_lock lock (fgPrefsMutex);

	fPrefixes.push_back (prefix);
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::PushPrefix
 *
 * DESCRIPTION: Push a prefix onto our internal stack.	These prefixes
 *				are used later to build a full key with which to fetch
 *				and store preference values.
 *
 * PARAMETERS:	index - the subkey to push onto our stack.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::PushPrefix (long index)
{
	this->PushPrefix (::ToString (index));
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::PopPrefix
 *
 * DESCRIPTION: Pop a prefix off of our internal stack.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::PopPrefix (void)
{
	// Locking done in Preference::DoSave/DoLoad.
//	omni_mutex_lock lock (fgPrefsMutex);

	fPrefixes.pop_back ();
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::ExpandKey
 *
 * DESCRIPTION: Prepend the given key with the prefixes pushed onto
 *				our stack.	Prefixes are prepended in such as way as to
 *				look like a "C" expression.  "Fields" are seperated with
 *				dots, and array subscripts are surround with brackets.
 *
 * PARAMETERS:	name - the part of the key that will appear at the
 *					very end of the expanded key.  If this name is
 *					empty, the expanded key will end with the last
 *					prefix pushed onto the stack.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

string Preferences::ExpandKey (const string& name)
{
	string	key;

	PrefixType::iterator	iter = fPrefixes.begin ();

	if (iter == fPrefixes.end ())
	{
		key = name;
	}
	else
	{
		key = *iter++;

		while (iter != fPrefixes.end ())
		{
			if (!iter->empty ())
				key = this->AppendName (key, *iter);
			
			++iter;
		}

		if (!name.empty ())
		{
			key = this->AppendName (key, name);
		}
	}

	return key;
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::AppendName
 *
 * DESCRIPTION: Append a single prefix to the full expanded key as it
 *				is built up from left to right.
 *
 * PARAMETERS:	key - the expanded key so far.
 *
 *				name - the subkey to append to it.
 *
 * RETURNED:	The catenated value. If name looks like a number, the
 *				catenated result is:
 *
 *					key[name]
 *
 *				Otherwise, we return:
 *
 *					key.name
 *
 ***********************************************************************/

string Preferences::AppendName (string key, const string& name)
{
	if (isdigit (name[0]))
	{
		key = key + "[" + name + "]";
	}
	else
	{
		key = key + "." + name;
	}
	
	return key;
}


/***********************************************************************
 *
 * FUNCTION:    Preferences::AddNotification
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Preferences::AddNotification (PrefNotifyFunc fn, PrefKeyType key)
{
	PrefKeyList	prefList;

	prefList.push_back (key);
	this->AddNotification (fn, prefList);
}


void Preferences::AddNotification (PrefNotifyFunc fn, const PrefKeyList& keyList)
{
	// Find the entry for this function.  If none exists, then
	// operator[] will add one, initializing the "value" for the
	// key with a default constructor.

#if defined(__MWERKS__)
	PrefNotifyList::mapped_type&		entry = fNotifications[fn];
#elif defined (_MSC_VER)
	PrefNotifyList::referent_type&		entry = fNotifications[fn];
#elif defined (__GNUC__)
	PrefNotifyList::data_type&			entry = fNotifications[fn];
#else
	#error "What does *your* compiler use...?"
#endif

	// Add the list of keys for which this function wants notification.
	// (Hmmm...doesn't check for duplicates...should it?)

	entry.insert (entry.end (), keyList.begin (), keyList.end ());
}


/***********************************************************************
 *
 * FUNCTION:    Preferences::RemoveNotification
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Preferences::RemoveNotification (PrefNotifyFunc fn)
{
	PrefNotifyList::iterator	iter = fNotifications.find(fn);

	if (iter != fNotifications.end ())
	{
		fNotifications.erase (iter);
	}
}


void Preferences::RemoveNotification (PrefNotifyFunc fn, PrefKeyType key)
{
	PrefKeyList	prefList;

	prefList.push_back (key);
	this->RemoveNotification (fn, prefList);
}


void Preferences::RemoveNotification (PrefNotifyFunc fn, const PrefKeyList& keyList)
{
	PrefNotifyList::iterator	iter = fNotifications.find(fn);

	if (iter != fNotifications.end ())
	{
		// !!! TBD
	}
}


/***********************************************************************
 *
 * FUNCTION:    DoNotify
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Preferences::DoNotify (const string& key)
{
	// For each entry in the map, get the list of keys for which
	// it wants notification.

	PrefNotifyList::iterator	mapIter = fNotifications.begin ();

	while (mapIter != fNotifications.end ())
	{
		// For each key in the list, see if it matches the specified key.

		PrefKeyList::iterator	keyIter = mapIter->second.begin ();
		while (keyIter != mapIter->second.end ())
		{
			if (::PrvFirstBeginsWithSecond (key, *keyIter))
			{
				// It does, so call the notification function.

				(mapIter->first) (keyIter->c_str ());
			}

			++keyIter;
		}

		++mapIter;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::WriteBanner
 *
 * DESCRIPTION: Write a descriptive banner to the preference file.
 *				Useful so that people will know what the contents of
 *				the file are for (in case they can't glean that from
 *				the name of the file).
 *
 * PARAMETERS:	f - the open FILE to write the banner to.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::WriteBanner (FILE*)
{
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::ReadBanner
 *
 * DESCRIPTION: Read the banner in the given file, possibly validating
 *				it.
 *
 * PARAMETERS:	f - the open FILE containing the banner.
 *
 * RETURNED:	True if it looks like this is our file, false if it
 *				looks like this file contains something else.
 *
 ***********************************************************************/

bool Preferences::ReadBanner (FILE*)
{
	return true;
}


/***********************************************************************
 *
 * FUNCTION:	Preferences::StripUnused
 *
 * DESCRIPTION: Called to remove any obsolete or otherwise stray
 *				settings from our collection.  Usually called some time
 *				before the updated settings are rewritten back to disk.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Preferences::StripUnused (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::EmulatorPreferences
 *
 * DESCRIPTION: Constructor.  Establishes default values for all the
 *				preferences we'll be using.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

EmulatorPreferences::EmulatorPreferences (void) :
	Preferences ()
{
	if (gEmuPrefs == NULL)
		gEmuPrefs = this;

	// Set up default values for all the keys.
	//
	// Warning: this method works implicitly off of "gPrefs", and so will
	// fail when trying to create additional preference objects.

	#define INIT_PREF_KEYS(name, type, init)		\
	{												\
		Preference<type>	pref (kPrefKey##name);	\
		pref = type init;							\
	}

	FOR_EACH_INIT_PREF(INIT_PREF_KEYS)
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::~EmulatorPreferences
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

EmulatorPreferences::~EmulatorPreferences (void)
{
	if (gEmuPrefs == this)
		gEmuPrefs = NULL;
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::Load
 *
 * DESCRIPTION: Install the correct set of logging preferences after the
 *				settings have been loaded from the file.  Also, make
 *				sure the list of skin preferences is large enough so
 *				that we can look up the skin for any device without
 *				having to check the array size all over the place.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void EmulatorPreferences::Load (void)
{
	Preferences::Load ();

	// Install the right set of logging options (normal or gremlins).

	this->UseLoggingOptions (kNormalLogging);

	// Make sure the skin list is large enough.

	Preference<SkinNameList>	skinPrefs (kPrefKeySkins);
	SkinNameList				skins = *skinPrefs;

	while (skins.size () < kDeviceLast)
	{
		skins.push_back ("Default");
	}

	skinPrefs = skins;

	// Let's set up some default configuration in case there isn't
	// one in the prefs file (or it was invalid).

	Preference<Configuration>	prefConfig (kPrefKeyLastConfiguration);
	if (!prefConfig.Loaded())
	{
		prefConfig = Configuration(kDevicePalmIII, 1024, FileReference());
	}
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::GetIndMRU
 *
 * DESCRIPTION: Return a PRC or PSF file from the corresponding list of
 *				files, based on the given index.
 *
 * PARAMETERS:	The index of the file to return.
 *
 * RETURNED:	The nth file in the list.  If index is off the end of
 *				the list, a non-specified FileReference (that is, one
 *				for which the IsSpecified method returns false) is
 *				returned.
 *
 ***********************************************************************/

FileReference EmulatorPreferences::GetIndPRCMRU (int index)
{
	Preference<FileRefList> pref (kPrefKeyPRC_MRU);
	FileRefList mru = *pref;
	return this->GetIndMRU (mru, index);
}


FileReference EmulatorPreferences::GetIndRAMMRU (int index)
{
	Preference<FileRefList> pref (kPrefKeyPSF_MRU);
	FileRefList mru = *pref;
	return this->GetIndMRU (mru, index);
}


FileReference EmulatorPreferences::GetIndMRU (const FileRefList& fileList, int index)
{
	if ((FileRefList::size_type) index < fileList.size ())
		return fileList [index];

	return FileReference();
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::UpdateMRU
 *
 * DESCRIPTION: Update the MRU list for PRC and PSF files.	The given
 *				file is inserted at the beginning of the list.	If
 *				the file previously existed in the list, it is removed
 *				from its old location.	The length of the list is
 *				limited to MRU_COUNT
 *
 * PARAMETERS:	newFile - the file to move to the beginning of the list.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void EmulatorPreferences::UpdatePRCMRU (const FileReference& newFile)
{
	Preference<FileRefList> pref (kPrefKeyPRC_MRU);
	FileRefList mru = *pref;
	this->UpdateMRU (mru, newFile);
	pref = mru;
}


void EmulatorPreferences::UpdateRAMMRU (const FileReference& newFile)
{
	Preference<FileRefList> pref (kPrefKeyPSF_MRU);
	FileRefList mru = *pref;
	this->UpdateMRU (mru, newFile);
	pref = mru;
}


void EmulatorPreferences::UpdateMRU (FileRefList& fileList, const FileReference& newFile)
{
	this->RemoveMRU (fileList, newFile);

	fileList.push_front (newFile);

	while (fileList.size () > MRU_COUNT)
		fileList.pop_back ();
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::RemoveMRU
 *
 * DESCRIPTION: Remove the given PRC or PSF file from its MRU list.
 *
 * PARAMETERS:	oldFile - the file to remove.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void EmulatorPreferences::RemovePRCMRU (const FileReference& oldFile)
{
	Preference<FileRefList> pref (kPrefKeyPRC_MRU);
	FileRefList mru = *pref;
	this->RemoveMRU (mru, oldFile);
	pref = mru;
}


void EmulatorPreferences::RemoveRAMMRU (const FileReference& oldFile)
{
	Preference<FileRefList> pref (kPrefKeyPSF_MRU);
	FileRefList mru = *pref;
	this->RemoveMRU (mru, oldFile);
	pref = mru;
}


void EmulatorPreferences::RemoveMRU (FileRefList& fileList, const FileReference& oldFile)
{
	FileRefList::iterator	iter = find (fileList.begin (), fileList.end (), oldFile);

	if (iter != fileList.end ())
		fileList.erase (iter);
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::UseLoggingOptions
 *
 * DESCRIPTION: Update the logging-related options to mirror the
 *				kNormalLogging or the kGremlinLogging bit up into the
 *				kCurrentLogging bit.  That is, establish what kind of
 *				logging needs to be turned on/off, based on the context
 *				of whether or not Gremlins is running.
 *
 * PARAMETERS:	newType - the context we'll be operating in, either
 *					kNormalLogging or kGremlinLogging.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

static void PrvUpdateOneLoggingOption (PrefKeyType key, int newType)
{
	Preference<uae_u8>	pref(key);

	if ((*pref & newType) != 0)
		pref = *pref | kCurrentLogging;
	else
		pref = *pref & ~kCurrentLogging;
}

int EmulatorPreferences::UseLoggingOptions (int newType)
{
	int oldType = fCurLoggingType;
	fCurLoggingType = newType;

#define UPDATE(name)	\
	::PrvUpdateOneLoggingOption (kPrefKey##name, newType)

	UPDATE (LogErrorMessages);
	UPDATE (LogWarningMessages);
	UPDATE (LogGremlins);
	UPDATE (LogCPUOpcodes);
	UPDATE (LogEnqueuedEvents);
	UPDATE (LogDequeuedEvents);
	UPDATE (LogSystemCalls);
	UPDATE (LogApplicationCalls);
	UPDATE (LogSerial);
	UPDATE (LogSerialData);
	UPDATE (LogNetLib);
	UPDATE (LogNetLibData);
	UPDATE (LogExgMgr);
	UPDATE (LogExgMgrData);
	UPDATE (LogHLDebugger);
	UPDATE (LogHLDebuggerData);
	UPDATE (LogLLDebugger);
	UPDATE (LogLLDebuggerData);

#undef UPDATE

	LogUpdateCache ();

	return oldType;
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::WriteBanner
 *
 * DESCRIPTION: Write out a banner indicating that that is a Poser
 *				preference file.
 *
 * PARAMETERS:	f - the open FILE to write the banner to.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void EmulatorPreferences::WriteBanner (FILE* f)
{
	fprintf (f, "# Palm OS Emulator Preferences\n\n");
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::ReadBanner
 *
 * DESCRIPTION: Do nothing.
 *
 * PARAMETERS:	f - the open FILE containing the banner.
 *
 * RETURNED:	True if it looks like this is our file, false if it
 *				looks like this file contains something else.
 *
 ***********************************************************************/

bool EmulatorPreferences::ReadBanner (FILE*)
{
	return true;
}


/***********************************************************************
 *
 * FUNCTION:	EmulatorPreferences::StripUnused
 *
 * DESCRIPTION: Called to remove any obsolete or otherwise stray
 *				settings from our collection.  Usually called some time
 *				before the updated settings are rewritten back to disk.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

static bool PrvCheckKey (PrefKeyType key)
{
	#define REMOVE_UNUSED(name, type, init) 		\
		if (_stricmp (key, kPrefKey##name) == 0)	\
			return true;
	FOR_EACH_PREF(REMOVE_UNUSED)

	return false;
}

void EmulatorPreferences::StripUnused (void)
{
	// Iterate over all the entries in the map.

	iterator	iter = fPreferences.begin ();
	while (iter != fPreferences.end ())
	{
		// For each entry, get the key, and then just the root part of
		// the key (everything before the first '.' or '[', if any).

		string				keyRoot (iter->first.c_str());

		string::size_type	dotPos = keyRoot.find ('.');
		if (dotPos != string::npos)
			keyRoot.erase (dotPos);

		string::size_type	bracketPos = keyRoot.find ('[');
		if (bracketPos != string::npos)
			keyRoot.erase (bracketPos);

		// See if this key (root) is one that we still support.
		// If not, then remove it.

		if (!::PrvCheckKey (keyRoot.c_str ()))
		{
			// According to MSDN's documentation, erase is supposed to
			// return an iterator to the next element.	MSL doesn't
			// seem to follow that description.
//			iter = fPreferences.erase (iter);

			fPreferences.erase (iter);
			iter = fPreferences.begin ();
		}
		else
		{
			++iter;
		}
	}
}

