/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Platform_NetLib.h"

#include "PreferenceMgr.h"		// Preference
#include "Byteswapping.h"		// Canonical
#include "Miscellaneous.h"		// StMemory
#include "Platform.h"			// AllocateMemory


#if defined (__MACOS__)
#include "sys_errno.h"			// GUSI.h doesn't haul this in, but we need it for error xlation.
#endif

#if defined (UNIX)
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#endif

#if defined(__svr4__)
#include <sys/filio.h>			// FIONBIO
#endif


// ===== COPIED FROM NETPRV.H =====

// Max # of open built-in sockets at one time.
#define		netMaxNumSockets	4								// Max # of sockets
#define		netMinSocketRefNum	(sysFileDescNet)				// lowest libRefNum
#define		netMaxSocketRefNum	(sysFileDescNet+netMaxNumSockets-1) // highest libRefNum

// The sockets maintained by NetLib plug-ins are numbered above the built-in
// sockets.
#define		netMaxNumPISockets	4						// Max # of Plug-In sockets
#define		netMinPISocketRefNum	(netMaxSocketRefNum+1)					// lowest libRefNum
#define		netMaxPISocketRefNum (netMinPISocketRefNum+netMaxNumPISockets-1)


// A NetSocketRef is the index into an array holding the pointers to
// socket data.  Because NetSocketRefs should not be zero, the refNums
// actually start at sysFileDescNet (which happens to be 1).
//
// The NetLib headers are sensitive to this numbering.  In particular,
// netFDSet, et al, assume a certain number range in order to set bits
// in the NetFDSetType object.
//
// Therefore, we can't just return the results of the socket() function as
// the result of NetLibSocketOpen().  Though an opaque type, the range of
// values returned by socket() is outside the range allowed by netFDSet.
// So, we have to map the SOCKET returned by socket() to something allowed
// by NetLib.  We store the SOCKETs in the array below, and return the
// index of the SOCKET, just as NetLib would.

SOCKET	gSockets[netMaxNumSockets] = { INVALID_SOCKET, INVALID_SOCKET, INVALID_SOCKET, INVALID_SOCKET };


// The following are copied from WinSock2.h.  We don't include that file
// here because it can't be included in the same scope as WinSock.h, which
// Windows.h pulls in.

#define SD_RECEIVE		0x00
#define SD_SEND			0x01
#define SD_BOTH			0x02

#if defined (_WINDOWS)
extern WORD	gWinSockVersion;
#endif

static long	gOpenCount;
static Word	gCurConfigIndex;

static Bool	NetLibToSocketsRef			(const NetSocketRef				inSocket,
										 SOCKET&						outSocket);
static Bool	SocketsToNetLibRef			(const SOCKET					inSocket,
										 NetSocketRef&					outSocket);

static Bool	RememberSocketsRef			(const SOCKET					inSocket,
										 NetSocketRef&					outSocket);
static Bool	ForgetSocketsRef			(NetSocketRef					sRef);

static Bool	NetLibToSocketsDomain		(const NetSocketAddrEnum		inDomain,
										 u_short&						outDomain);
static Bool	SocketsToNetLibDomain		(const u_short					inDomain,
										 NetSocketAddrEnum&				outDomain);

static Bool	NetLibToSocketsType			(const NetSocketTypeEnum		inType,
										 int&							outType);
static Bool	NetLibToSocketsProtocol		(const SWord					inProtocol,
										 int&							outProtocol);
static Bool	NetLibToSocketsDirection	(const SWord					inDirection,
										 int&							outDirection);
static Bool NetLibToSocketsFlags		(const Word						inFlags,
										 int&							outFlags);

static Bool	NetLibToSocketsOptions		(const NetSocketOptLevelEnum	inOptLevel,
										 const NetSocketOptEnum			inOptName,
										 const VoidPtr					inOptVal,
										 const Word						inOptValLen,
										 int&							outOptLevel,
										 int&							outOptName,
										 char*&							outOptVal,
										 socklen_t&						outOptValLen);
static Bool	SocketsToNetLibOptions		(const int						inOptLevel,
										 const int						inOptName,
										 /*const*/ char*&					inOptVal,
										 const int						inOptValLen,
										 NetSocketOptLevelEnum			outOptLevel,
										 NetSocketOptEnum				outOptName,
										 VoidPtr						outOptVal,
										 Word							outOptValLen);

static Bool	NetLibToSocketsAddr			(const NetSocketAddrType&		inAddr,
										 const SWord					inAddrLen,
										 sockaddr&						outAddr,
										 int							outAddrLen);
static Bool	SocketsToNetLibAddr			(const sockaddr&				inAddr,
										 int							inAddrLen,
										 NetSocketAddrType&				outAddr,
										 SWord							outAddrLen);

static Bool SocketsToNetLibHostEnt		(const hostent&					inHostEnt,
										 NetHostInfoBufType&			outHostEnt);
static Bool SocketsToNetLibServEnt		(const servent&					inServEnt,
										 NetServInfoBufType&			outServEnt);

static Bool NetLibToSocketsFDSet		(const NetFDSetType*			inFDSet,
										 Word							inWidth,
										 fd_set*&						outFSSet,
										 int&							outWidth);
static Bool SocketsToNetLibFDSet		(const fd_set*					inFDSet,
										 int							inWidth,
										 NetFDSetType*					outFSSet,
										 Word&							outWidth);

static Bool NetLibToSocketsScalarOption (const VoidPtr					inOptVal,
										 const Word						inOptValLen,
										 char*&							outOptVal,
										 socklen_t&						outOptValLen);

static Bool SocketsToNetLibScalarOption (/*const*/ char*&					inOptVal,
										 const int						inOptValLen,
										 VoidPtr						outOptVal,
										 Word							outOptValLen);

static uae_u32	PrvGetError (void);
static Err		PrvGetTranslatedError (void);
static Err		PrvGetTranslatedHError (void);
static Err		PrvTranslateError (uae_u32 err);



Bool Platform_NetLib::Redirecting (void)
{
	Preference<bool>	pref (kPrefKeyRedirectNetLib);
	return *pref;
}


//--------------------------------------------------
// Library initialization, shutdown, sleep and wake
//--------------------------------------------------

Err Platform_NetLib::Open(Word libRefNum, WordPtr netIFErrsP)
{
	UNUSED_PARAM(libRefNum)

	// Init return error 
	*netIFErrsP = 0;

#if defined (_WINDOWS)
	if (gWinSockVersion == 0)
	{
		return netErrInternal;
	}
#endif

	if (gOpenCount > 0)
	{
		++gOpenCount;
		return netErrAlreadyOpen;
	}

	++gOpenCount;
	return errNone;
}

Err Platform_NetLib::OpenConfig(Word libRefNum, Word configIndex, DWord openFlags,
						WordPtr netIFErrP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(openFlags)

	Err		err = errNone;
	Boolean needRefresh = false;

	// Init return error 
	*netIFErrP = errNone;

	//-----------------------------------------------------------------
	// We now have a "lock" on the globals and have prevented all other tasks
	//  from initiating an open or close until we release the lock.
	// Now, see what the current state of the library is and take the
	//  appropriate action
	// If it's already open make sure it's a compatible configuration,
	//  if not, it's an error
	// If we're in the closeWait state see if it's
	//  a compatible configuration, if it is just bump it's open
	//  count and return. If it's not, finish the close before we
	//  change the configuration
	//-----------------------------------------------------------------
	if (gOpenCount > 0)
	{
		// If we can use the current state...
		if (configIndex == netConfigIndexCurSettings || configIndex == gCurConfigIndex)
		{
			gOpenCount++;
			if (gOpenCount > 1)
				err = netErrAlreadyOpen;

			// Refresh connections if necessary 
			needRefresh = true;
			goto Exit;
		}

		err = netErrAlreadyOpenWithOtherConfig;
		goto Exit;
	}


	//-----------------------------------------------------------------
	// Activate the new configuration. This routine will initialize
	//  the config if it's one of our pre-defined ones that we
	//  know how to initialize and it's the first time we've used it. 
	//
	// It will also save the current configuration, update the 
	//  curConfigIndex field of the config table with the new config
	//  index, and  set the restoreConfig index to the old index
	//  so that PrvClose() in the protocol task can restore the 
	//  current configuration when it closes the NetLib. 
	//-----------------------------------------------------------------
	if (configIndex != netConfigIndexCurSettings)
	{
		err = NetLibConfigMakeActive (libRefNum, configIndex);
			// !!! Doesn't do the restoreOldConfigOnClose stuff...
		if (err)
			goto Exit;

		gCurConfigIndex = configIndex;
	}

	
	//-----------------------------------------------------------------
	// Do the guts of the open. This routine will leave the globals
	//  locked. If it's successful, it will return the new "realGlobals"
	//  ptr. If not, it will leave the shadow globals
	//-----------------------------------------------------------------
	err = Platform_NetLib::Open (libRefNum, netIFErrP);

	
	//-----------------------------------------------------------------
	// Exit now..
	//-----------------------------------------------------------------
Exit:

	// See if we should refresh any connections
	if (!err && needRefresh)
	{
		Boolean	allIFsUp;
		err = Platform_NetLib::ConnectionRefresh (libRefNum, true, &allIFsUp, netIFErrP);
	}

	return err;
}

Err Platform_NetLib::Close(Word libRefNum, Word immediate)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(immediate)

	if (gOpenCount <= 0)
	{
		return netErrNotOpen;
	}

	--gOpenCount;

	return errNone;
}


Err Platform_NetLib::Sleep(Word libRefNum)
{
	UNUSED_PARAM(libRefNum)

	// stub. Do Nothing
	return errNone;
}


Err Platform_NetLib::Wake(Word libRefNum)
{
	UNUSED_PARAM(libRefNum)

	// stub. Do Nothing
	return errNone;
}


// This call forces the library to complete a close if it's
//  currently in the close-wait state. Returns 0 if library is closed,
//  Returns netErrFullyOpen if library is still open by some other task.

Err Platform_NetLib::FinishCloseWait(Word libRefNum)
{
	UNUSED_PARAM(libRefNum)

	// stub. Do Nothing
	return errNone;
}


// This call is for use by the Network preference panel only. It
// causes the NetLib to fully open if it's currently in the close-wait
//  state. If it's not in the close wait state, it returns an error code

Err Platform_NetLib::OpenIfCloseWait(Word libRefNum)
{
	UNUSED_PARAM(libRefNum)

	return netErrNotOpen;
}


// Get the open Count of the NetLib

Err Platform_NetLib::OpenCount(Word libRefNum, WordPtr countP)
{
	UNUSED_PARAM(libRefNum)

	*countP = gOpenCount;
	return errNone;
}


// Give NetLib a chance to close the connection down in response
// to a power off event. Returns non-zero if power should not be
//  turned off. EventP points to the event that initiated the power off
//  which is either a keyDownEvent of the hardPowerChr or the autoOffChr.
// Don't include unless building for Viewer

Err Platform_NetLib::HandlePowerOff (Word libRefNum, EventPtr eventP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(eventP)

	return Platform_NetLib::Close (libRefNum, true);
}


// Check status or try and reconnect any interfaces which have come down.
// This call can be made by applications when they suspect that an interface
// has come down (like PPP or SLIP). NOTE: This call can display UI
// (if 'refresh' is true) so it MUST be called from the UI task.

Err Platform_NetLib::ConnectionRefresh(Word libRefNum,
										Boolean refresh,
										BooleanPtr allInterfacesUpP,
										WordPtr netIFErrP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(refresh)

	*allInterfacesUpP = true;
	*netIFErrP =0;
	return errNone;
}


//--------------------------------------------------
// Socket creation and option setting
//--------------------------------------------------

// Create a socket and return a refnum to it. Protocol is normally 0.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

NetSocketRef Platform_NetLib::SocketOpen(Word libRefNum,
										NetSocketAddrEnum socketDomain,
										NetSocketTypeEnum socketType,
										SWord socketProtocol,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	u_short	af;
	int		type;
	int		protocol;

	if (!NetLibToSocketsDomain (socketDomain, af) ||
		!NetLibToSocketsType (socketType, type) ||
		!NetLibToSocketsProtocol (socketProtocol, protocol))
	{
		*errP = netErrParamErr;
		return -1;
	}

	SOCKET	s = socket (af, type, protocol);

	if (s == INVALID_SOCKET)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	NetSocketRef	result;
	if (!RememberSocketsRef (s, result))
	{
		// !!! Need to close the socket.  But we should never
		// really get here...we allow as many sockets at NetLib
		// does.  I guess if NetLib raised the number of sockets
		// and we don't, then we'd get here, though...
		*errP = netErrNoMoreSockets;
		return -1;
	}

	*errP = 0;
	return result;
}


// Close a socket. 
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketClose(Word libRefNum,
									NetSocketRef sRef,
									SDWord timeout,
									Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET	s;
	if (!NetLibToSocketsRef (sRef, s))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result = closesocket (s);

	// Drop our reference to this socket.
	// !!! Do this even if closesocket fails?

	ForgetSocketsRef (sRef);

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


// Set a socket option.  Level is usually netSocketOptLevelSocket. Option is one of
//  netSocketOptXXXXX. OptValueP is a pointer to the new value and optValueLen is
//  the length of the option value.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketOptionSet(Word libRefNum,
										NetSocketRef sRef,
										NetSocketOptLevelEnum optLevel,
										NetSocketOptEnum opt,
										VoidPtr optValueP,
										Word optValueLen,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	int	result;

	if (optLevel == netSocketOptLevelSocket &&
		opt == netSocketOptSockNonBlocking)
	{
		SOCKET	s;
		if (!NetLibToSocketsRef (sRef, s))
		{
			*errP = netErrParamErr;
			return -1;
		}

#if defined (_WINDOWS)
		unsigned long	param;
#else
		long			param;
#endif

		if (optValueLen == 1)
			param = *(char*) optValueP;
		else if (optValueLen == 2)
			param = *(short*) optValueP;
		else if (optValueLen == 4)
			param = *(long*) optValueP;
		else
			param = 1;

		result = ioctlsocket (s, FIONBIO, &param);
	}
	else
	{
		SOCKET		s;
		int			level;
		int			optName;
		char*		optVal;
		socklen_t	optLen;

		if (!NetLibToSocketsRef (sRef, s) ||
			!NetLibToSocketsOptions (optLevel, opt, optValueP, optValueLen, level, optName, optVal, optLen))
		{
			*errP = netErrParamErr;
			return -1;
		}

		result = setsockopt (s, level, optName, optVal, optLen);

		Platform::DisposeMemory (optVal);
	}

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


// Get a socket option.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketOptionGet(Word libRefNum,
										NetSocketRef sRef,
										NetSocketOptLevelEnum optLevel,
										NetSocketOptEnum opt,
										VoidPtr optValueP,
										WordPtr optValueLenP,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET		s;
	int			level;
	int			optName;
	char*		optVal;
	socklen_t	optLen = *optValueLenP;

	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsOptions (optLevel, opt, optValueP, *optValueLenP, level, optName, optVal, optLen))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result	= getsockopt (s, level, optName, optVal, &optLen);

	if (result)
	{
		Platform::DisposeMemory (optVal);
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	if (!SocketsToNetLibOptions (level, optName, optVal, optLen, optLevel, opt, optValueP, *optValueLenP))
	{
		*errP = netErrParamErr;
		return -1;
	}

	*errP = 0;
	return result;
}


//--------------------------------------------------
// Socket Control
//--------------------------------------------------

// Bind a source address and port number to a socket. This makes the
//  socket accept incoming packets destined for the given socket address.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketBind(Word libRefNum,
									NetSocketRef sRef,
									NetSocketAddrType* sockAddrP,
									SWord addrLen,
									SDWord timeout,
									Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET		s;
	sockaddr	name;
	int			namelen = sizeof (name);

	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsAddr (*sockAddrP, addrLen, name, namelen))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result	= bind (s, &name, namelen);

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


// Connect to a remote socket. For a stream based socket (i.e. TCP), this initiates
//  a 3-way handshake with the remote machine to establish a connection. For
//  non-stream based socket, this merely specifies a destination address and port
//  number for future outgoing packets from this socket.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketConnect(Word libRefNum,
										NetSocketRef sRef,
										NetSocketAddrType* sockAddrP,
										SWord addrLen,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET		s;
	sockaddr	name;
	int			namelen = sizeof (name);

	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsAddr (*sockAddrP, addrLen, name, namelen))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result	= connect (s, &name, namelen);

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}

// Makes a socket ready to accept incoming connection requests. The queueLen 
//  specifies the max number of pending connection requests that will be enqueued
//  while the server is busy handling other requests.
//  Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketListen(Word libRefNum,
										NetSocketRef sRef,
										Word queueLen,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET	s;
	if (!NetLibToSocketsRef (sRef, s))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result = listen (s, queueLen);

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}

// Blocks the current process waiting for an incoming connection request. The socket
//  must have previously be put into listen mode through the NetLibSocketListen call.
//  On return, *sockAddrP will have the remote machines address and port number.
//  Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketAccept(Word libRefNum,
									NetSocketRef sRef,
									NetSocketAddrType* sockAddrP,
									SWord* addrLenP,
									SDWord timeout,
									Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET		s;
	sockaddr	addr;
	socklen_t	addrlen;

	if (!NetLibToSocketsRef (sRef, s))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	new_s = accept (s, &addr, &addrlen);

	if (new_s == INVALID_SOCKET)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	NetSocketRef	result;
	if (!RememberSocketsRef (new_s, result))
	{
		// !!! Need to un-accept?
		*errP = netErrParamErr;
		return -1;
	}

	if (!SocketsToNetLibAddr (addr, addrlen, *sockAddrP, *addrLenP))
	{
		// !!! Need to un-accept?
		*errP = netErrParamErr;
		return -1;
	}

	*errP = 0;
	return (SWord) result;
}

// Shutdown a connection in one or both directions.  
//  Only applies to stream based (i.e. TCP) sockets.
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketShutdown(Word libRefNum,
										NetSocketRef sRef,
										SWord direction,
										SDWord timeout,
										Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET	s;
	int		how;
	
	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsDirection (direction, how))
	{
		*errP = netErrParamErr;
		return -1;
	}

	int	result = shutdown (s, how);

	if (result)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


// Gets the local and remote addresses of a socket. Useful for TCP sockets that 
//  get dynamically bound at connect time. 
// Returns 0 on success, -1 on error. If error, *errP gets filled in with error code.

SWord Platform_NetLib::SocketAddr(Word libRefNum,
									NetSocketRef sRef,
									NetSocketAddrType* locAddrP,
									SWord* locAddrLenP,
									NetSocketAddrType* remAddrP,
									SWord* remAddrLenP,
									SDWord timeout,
									Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	int		result = 0;
	SOCKET	s;

	if (!NetLibToSocketsRef (sRef, s))
	{
		*errP = netErrParamErr;
		return -1;
	}

	if (locAddrP && locAddrLenP)
	{
		sockaddr	locName;
		socklen_t	locNamelen = sizeof (locName);

		result = getsockname (s, &locName, &locNamelen);

		// For unbound sockets, NetLib returns 0.0.0.0
		// !!! Assumes AF_INET family!

		if (result == -1 && ::PrvGetError () == WSAEINVAL)
		{
			result = errNone;

			sockaddr_in&	inLocName = (sockaddr_in&) locName;

			memset (&inLocName, 0, sizeof (inLocName));

			inLocName.sin_family = AF_INET;
			inLocName.sin_port = htons(0);
			inLocName.sin_addr.s_addr = htonl(INADDR_ANY);
		}

		if (result)
		{
			*errP = PrvGetTranslatedError ();
			return -1;
		}

		if (!SocketsToNetLibAddr (locName, locNamelen, *locAddrP, *locAddrLenP))
		{
			*errP = netErrParamErr;
			return -1;
		}
	}

	if (remAddrP && remAddrLenP)
	{
		sockaddr	remName;
		socklen_t	remNamelen = sizeof (remName);

		result = getpeername (s, &remName, &remNamelen);

		if (result)
		{
			*errP = PrvGetTranslatedError ();
			return -1;
		}

		if (!SocketsToNetLibAddr (remName, remNamelen, *remAddrP, *remAddrLenP))
		{
			*errP = netErrParamErr;
			return -1;
		}
	}

	*errP = 0;
	return result;
}


//--------------------------------------------------
// Sending and Receiving
//--------------------------------------------------
// Send data through a socket. The data is specified through the NetIOParamType
//  structure.
// Flags is one or more of netMsgFlagXXX.
// Returns # of bytes sent on success, or -1 on error. If error, *errP gets filled 
//  in with error code.

SWord Platform_NetLib::SendPB(Word libRefNum,
								NetSocketRef sRef,
								NetIOParamType* pbP,
								Word sendFlags,
								SDWord timeout,
								Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET	s;
	int		flags;

	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsFlags (sendFlags, flags))
	{
		*errP = netErrParamErr;
		return -1;
	}

	// Collapse the scatter-write array into a single big buffer.

	Word	ii;
	long	bigBufferSize = 0;
	for (ii = 0; ii < pbP->iovLen; ++ii)
	{
		bigBufferSize += pbP->iov[ii].bufLen;
	}

	StMemory	bigBuffer (bigBufferSize);

	long	offset = 0;
	for (ii = 0; ii < pbP->iovLen; ++ii)
	{
		memcpy (bigBuffer.Get() + offset, pbP->iov[ii].bufP, pbP->iov[ii].bufLen);
		offset += pbP->iov[ii].bufLen;
	}

	int	result;

	if (pbP->addrP)
	{
		sockaddr	name;
		int			namelen = sizeof (name);

		if (!NetLibToSocketsAddr (*(NetSocketAddrType*) pbP->addrP, pbP->addrLen, name, namelen))
		{
			*errP = netErrParamErr;
			return -1;
		}

		result = sendto (s, bigBuffer.Get(), bigBufferSize, flags, &name, namelen);
	}
	else
	{	
		result = send (s, bigBuffer.Get(), bigBufferSize, flags);
	}

	if (result == SOCKET_ERROR)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


// Send data through a socket. The data to send is passed in a single buffer,
//  unlike NetLibSendPB. If toAddrP is not nil, the data will be sent to 
//  address *toAddrP.
// Flags is one or more of netMsgFlagXXX.
// Returns # of bytes sent on success, or -1 on error. If error, *errP gets filled 
//  in with error code.

SWord Platform_NetLib::Send(Word libRefNum,
							NetSocketRef sRef,
							const VoidPtr bufP,
							Word bufLen,
							Word sendFlags,
							VoidPtr toAddrP,
							Word toLen,
							SDWord timeout,
							Err* errP)
{
	// Init the iov

	NetIOVecType	iov;
	iov.bufP = (BytePtr) bufP;
	iov.bufLen = bufLen;

	// Form a paramBlock for the Platform_NetLib::SendPB call

	NetIOParamType	pb;
	if (toLen)
	{
		pb.addrP = (BytePtr) toAddrP;
		pb.addrLen = toLen;
	}
	else
	{
		pb.addrP = 0;
		pb.addrLen = 0;
	}

	pb.iov = &iov;
	pb.iovLen = 1;

	// Do the call

	SWord	result = SendPB (libRefNum, sRef, &pb, sendFlags, timeout, errP);

	return result;
}


// Receive data from a socket. The data is gatthered into buffers specified in the 
//  NetIOParamType structure.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//  with error code.

SWord Platform_NetLib::ReceivePB(Word libRefNum,
									NetSocketRef sRef,
									NetIOParamType* pbP,
									Word rcvFlags,
									SDWord timeout,
									Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	SOCKET	s;
	int		flags;
	
	if (!NetLibToSocketsRef (sRef, s) ||
		!NetLibToSocketsFlags (rcvFlags, flags))
	{
		*errP = netErrParamErr;
		return -1;
	}

	// Collapse the gather-read array into a single big buffer.

	Word	ii;
	long	bigBufferSize = 0;
	for (ii = 0; ii < pbP->iovLen; ++ii)
	{
		bigBufferSize += pbP->iov[ii].bufLen;
	}

	StMemory	bigBuffer (bigBufferSize);

	int	result;
	
	if (pbP->addrP)
	{
		sockaddr	name;
		socklen_t	namelen;

		result = recvfrom (s, bigBuffer.Get (), bigBufferSize, flags, &name, &namelen);

		if (!SocketsToNetLibAddr (name, namelen, *(NetSocketAddrType*) pbP->addrP, pbP->addrLen))
		{
			*errP = netErrParamErr;
			return -1;
		}
	}
	else
	{
		result = recv (s, bigBuffer.Get (), bigBufferSize, flags);
	}

	if (result == SOCKET_ERROR)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	// Copy the chunks of the big buffer back into the iov array.
	long	offset = 0;
	for (ii = 0; ii < pbP->iovLen; ++ii)
	{
		memcpy (pbP->iov[ii].bufP, bigBuffer.Get () + offset, pbP->iov[ii].bufLen);
		offset += pbP->iov[ii].bufLen;
	}

	*errP = 0;
	return result;
}


// Receive data from a socket. The data is read into a single buffer, unlike
//  NetLibReceivePB. If fromAddrP is not nil, *fromLenP must be initialized to
//  the size of the buffer that fromAddrP points to and on exit *fromAddrP will
//  have the address of the sender in it.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//  with error code.

SWord Platform_NetLib::Receive(Word libRefNum,
								NetSocketRef sRef,
								VoidPtr bufP,
								Word bufLen,
								Word rcvFlags,
								VoidPtr fromAddrP,
								WordPtr fromLenP,
								SDWord timeout,
								Err* errP)
{
	NetIOParamType		pb;
	NetIOVecType		iov;	

	// Assume no error
	*errP = 0;

	// If no buflen, return 0
	if (!bufLen)
		return 0;

	//-------------------------------------------------------------------------
	// Socket read
	// Form a paramBlock for the NetLibReceivePB call
	//-------------------------------------------------------------------------
	if (fromLenP)
	{
		pb.addrLen = *fromLenP;
		pb.addrP = (BytePtr) fromAddrP;
	}
	else
	{
		pb.addrLen = 0;
		pb.addrP = 0;
	}

	pb.iov = &iov;
	pb.iovLen = 1;

	// Init the iov
	iov.bufP = (BytePtr) bufP;
	iov.bufLen = bufLen;

	// Do the call
	SWord	result = ReceivePB (libRefNum, sRef, &pb, rcvFlags, timeout, errP);

	// Update the address length, if passed
	if (fromLenP)
		*fromLenP = pb.addrLen;

	return result;
}

// Receive data from a socket directly into a (write-protected) Data Manager 
//  record. 
// If fromAddrP is not nil, *fromLenP must be initialized to
//  the size of the buffer that fromAddrP points to and on exit *fromAddrP will
//  have the address of the sender in it.
// Flags is one or more of netMsgFlagXXX.
// Timeout is max # of ticks to wait, or -1 for infinite, or 0 for none.
// Returns # of bytes received, or -1 on error. If error, *errP gets filled in 
//  with error code.

SWord Platform_NetLib::DmReceive(Word libRefNum,
									NetSocketRef sRef,
									VoidPtr recordP,
									DWord recordOffset,
									Word rcvLen,
									Word rcvFlags,
									VoidPtr fromAddrP,
									WordPtr fromLenP,
									SDWord timeout,
									Err* errP)
{
	SWord	result = Receive (libRefNum, sRef, ((char*) recordP) + recordOffset,
								rcvLen, rcvFlags, fromAddrP, fromLenP, timeout, errP);

	return result;
}


//--------------------------------------------------
// Name Lookups
//--------------------------------------------------

NetHostInfoPtr Platform_NetLib::GetHostByName(Word libRefNum,
												CharPtr nameP,
												NetHostInfoBufPtr bufP,
												Long timeout,
												Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	hostent*	pHostEnt = gethostbyname (nameP);

	if (!pHostEnt)
	{
#if defined (__MACOS__)
		*errP = PrvGetTranslatedHError ();
#else
		*errP = PrvGetTranslatedError ();
#endif
		return NULL;
	}

	if (!SocketsToNetLibHostEnt (*pHostEnt, *bufP))
	{
		*errP = netErrParamErr;
		return NULL;
	}

	*errP = 0;
	return &(bufP->hostInfo);
}


NetHostInfoPtr Platform_NetLib::GetHostByAddr(Word libRefNum,
												BytePtr addrP,
												Word len,
												Word type,
												NetHostInfoBufPtr bufP,
												Long timeout,
												Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	hostent*	pHostEnt = gethostbyaddr ((const char*) addrP, len, type);

	if (!pHostEnt)
	{
#if defined (__MACOS__)
		*errP = PrvGetTranslatedHError ();
#else
		*errP = PrvGetTranslatedError ();
#endif
		return NULL;
	}

	if (!SocketsToNetLibHostEnt (*pHostEnt, *bufP))
	{
		*errP = netErrParamErr;
		return NULL;
	}

	*errP = 0;
	return &(bufP->hostInfo);
}


NetServInfoPtr Platform_NetLib::GetServByName(Word libRefNum,
												CharPtr servNameP,
												CharPtr protoNameP,
												NetServInfoBufPtr bufP,
												Long timeout,
												Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(timeout)

	servent*	pServEnt = getservbyname (servNameP, protoNameP);

	if (!pServEnt)
	{
		*errP = PrvGetTranslatedError ();
		return NULL;
	}

	if (!SocketsToNetLibServEnt (*pServEnt, *bufP))
	{
		*errP = netErrParamErr;
		return NULL;
	}

	*errP = 0;
	return &(bufP->servInfo);
}


// Looks up a mail exchange name and returns a list of hostnames for it. Caller
//  must pass space for list of return names (hostNames), space for 
//  list of priorities for those hosts (priorities) and max # of names to 
//  return (maxEntries).
// Returns # of entries found, or -1 on error. If error, *errP gets filled in
//  with error code.

SWord Platform_NetLib::GetMailExchangeByName(Word libRefNum,
												CharPtr mailNameP,
												Word maxEntries,
												Char hostNames[][netDNSMaxDomainName+1],
												Word priorities[],
												Long timeout,
												Err* errP)
{
	UNUSED_PARAM(libRefNum)
	UNUSED_PARAM(mailNameP)
	UNUSED_PARAM(maxEntries)
	UNUSED_PARAM(hostNames)
	UNUSED_PARAM(priorities)
	UNUSED_PARAM(timeout)

	*errP = 0;
	// TODO
	return -1;
}


//--------------------------------------------------
// System level
//--------------------------------------------------

SWord Platform_NetLib::Select(Word libRefNum,
								Word netWidth,
								NetFDSetType* netReadFDs,
								NetFDSetType* netWriteFDs,
								NetFDSetType* netExceptFDs,
								Long netTimeout,
								Err* errP)
{
	UNUSED_PARAM(libRefNum)

	fd_set	hostReadFDs;
	fd_set	hostWriteFDs;
	fd_set	hostExceptFDs;
	int		hostWidth = 0;

	fd_set*	hostReadFDsP = &hostReadFDs;
	fd_set*	hostWriteFDsP = &hostWriteFDs;
	fd_set*	hostExceptFDsP = &hostExceptFDs;

	if (!NetLibToSocketsFDSet (netReadFDs, netWidth, hostReadFDsP, hostWidth) ||
		!NetLibToSocketsFDSet (netWriteFDs, netWidth, hostWriteFDsP, hostWidth) ||
		!NetLibToSocketsFDSet (netExceptFDs, netWidth, hostExceptFDsP, hostWidth))
	{
		*errP = netErrParamErr;
		return -1;
	}

	timeval hostTimeout;
	hostTimeout.tv_usec = netTimeout;
	hostTimeout.tv_sec = 0;

	int	result = select (hostWidth, hostReadFDsP, hostWriteFDsP, hostExceptFDsP, &hostTimeout);

	netWidth = 0;
	if (!SocketsToNetLibFDSet (hostReadFDsP, hostWidth, netReadFDs, netWidth) ||
		!SocketsToNetLibFDSet (hostWriteFDsP, hostWidth, netWriteFDs, netWidth) ||
		!SocketsToNetLibFDSet (hostExceptFDsP, hostWidth, netExceptFDs, netWidth))
	{
		*errP = netErrParamErr;
		return -1;
	}

	if (result == SOCKET_ERROR)
	{
		*errP = PrvGetTranslatedError ();
		return -1;
	}

	*errP = 0;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsRef
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	NetLibToSocketsRef			(const NetSocketRef				inSocket,
									 SOCKET&						outSocket)
{
	if (inSocket < netMinSocketRefNum || inSocket > netMaxSocketRefNum)
		return false;

	outSocket = gSockets [inSocket - netMinSocketRefNum];

	if (outSocket == INVALID_SOCKET)
		return false;

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibRef
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	SocketsToNetLibRef			(const SOCKET					inSocket,
									 NetSocketRef&					outSocket)
{
	// See if this socket is already installed.  If so, return its
	// current index.

	for (int ii = 0; ii < netMaxNumSockets; ++ii)
	{
		if (gSockets[ii] == inSocket)
		{
			outSocket = ii + netMinSocketRefNum;
			return true;
		}
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	RememberSocketsRef
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	RememberSocketsRef			(const SOCKET					inSocket,
									 NetSocketRef&					outSocket)
{
	for (int ii = 0; ii < netMaxNumSockets; ++ii)
	{
		if (gSockets[ii] == INVALID_SOCKET)
		{
			gSockets[ii] = inSocket;
			outSocket = ii + netMinSocketRefNum;
			return true;
		}
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	ForgetSocketsRef
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	ForgetSocketsRef			(NetSocketRef					sRef)
{
	if (sRef >= netMinSocketRefNum && sRef <= netMaxSocketRefNum)
	{
		gSockets [sRef - netMinSocketRefNum] = INVALID_SOCKET;
		return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsDomain
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	NetLibToSocketsDomain	(const NetSocketAddrEnum	inDomain,
								 u_short&					outDomain)
{
	{ char _dummy[(netSocketAddrRaw == AF_UNSPEC) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketAddrINET == AF_INET) ? 1 : 0]; (void) _dummy; }

	switch (inDomain)
	{
		case netSocketAddrRaw:
		case netSocketAddrINET:
			outDomain = (int) inDomain;
			return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	SocketsToNetLibDomain	(const u_short		inDomain,
								 NetSocketAddrEnum&	outDomain)
{
	switch (inDomain)
	{
		case AF_UNSPEC:
		case AF_INET:
			outDomain = (NetSocketAddrEnum) inDomain;
			return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsType
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsType (	const NetSocketTypeEnum	inType,
							int&					outType)
{
	{ char _dummy[(netSocketTypeStream == SOCK_STREAM) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketTypeDatagram == SOCK_DGRAM) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketTypeRaw == SOCK_RAW) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketTypeReliableMsg == SOCK_RDM) ? 1 : 0]; (void) _dummy; }

	switch (inType)
	{
		case netSocketTypeStream:
		case netSocketTypeDatagram:
		case netSocketTypeRaw:
		case netSocketTypeReliableMsg:
			outType = (int) inType;
			return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsProtocol
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsProtocol (	const SWord	inProtocol,
								int&		outProtocol)
{
	{ char _dummy[(netSocketProtoIPICMP == IPPROTO_ICMP) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketProtoIPTCP == IPPROTO_TCP) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketProtoIPUDP == IPPROTO_UDP) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketProtoIPRAW == IPPROTO_RAW) ? 1 : 0]; (void) _dummy; }

	switch (inProtocol)
	{
		// This first case takes care of situations where applications
		// call NetLibOpenSocket, passing in zero for the protocol parameter.
		// Zero doesn't appear to be a valid value when looking at NetMgr.h
		// (it defines only the symbols listed above), but zero *is*
		// defined in WinSock.h, and applications *do* pass it in...
		//
		// Actually, the NetLib.c function comments say:
		//
		// *	protocol	- Ignored for datagram or stream sockets in the 
		// *				INET domain. If there were more than
		// *				one protocol that implements stream oriented services,
		// *				for example, this parameter would indicate which protocol 
		// *				to use.
		//
		// It goes on to describe how this parameter is really only used with
		// "raw" sockets in the INET domain.

		case IPPROTO_IP:

		case netSocketProtoIPICMP:
		case netSocketProtoIPTCP:
		case netSocketProtoIPUDP:
		case netSocketProtoIPRAW:
			outProtocol = (int) inProtocol;
			return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsDirection
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsDirection (	const SWord	inDirection,
								int&		outDirection)
{
	{ char _dummy[(netSocketDirInput == SD_RECEIVE) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketDirOutput == SD_SEND) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketDirBoth == SD_BOTH) ? 1 : 0]; (void) _dummy; }

	switch (inDirection)
	{
		case netSocketDirInput:
		case netSocketDirOutput:
		case netSocketDirBoth:
			outDirection = (int) inDirection;
			return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsFlags
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsFlags (const Word inFlags, int& outFlags)
{
	{ char _dummy[(netIOFlagOutOfBand == MSG_OOB) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netIOFlagPeek == MSG_PEEK) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netIOFlagDontRoute == MSG_DONTROUTE) ? 1 : 0]; (void) _dummy; }

	// Return false if there are any flags set other than the ones we know about.
	if ((inFlags & ~(netIOFlagOutOfBand | netIOFlagPeek | netIOFlagDontRoute)) != 0)
	{
		return false;
	}

	outFlags = (int) inFlags;

	return true;
}

	
/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsOptions
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsOptions (	const NetSocketOptLevelEnum	inOptLevel,
								const NetSocketOptEnum		inOptName,
								const VoidPtr				inOptVal,
								const Word					inOptValLen,
								int&						outOptLevel,
								int&						outOptName,
								char*&						outOptVal,
								socklen_t&					outOptValLen)
{
	Bool	result = true;

	{ char _dummy[(netSocketOptLevelIP == IPPROTO_IP) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptLevelTCP == IPPROTO_TCP) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptLevelSocket == SOL_SOCKET) ? 1 : 0]; (void) _dummy; }

		// IP Level options
	{ char _dummy[(netSocketOptIPOptions == IP_OPTIONS) ? 1 : 0]; (void) _dummy; }

		// TCP Level options
#if defined (_WINDOWS) || defined (UNIX)
	{ char _dummy[(netSocketOptTCPNoDelay == TCP_NODELAY) ? 1 : 0]; (void) _dummy; }
#endif
//	{ char _dummy[(netSocketOptTCPMaxSeg == xxx) ? 1 : 0]; (void) _dummy; }

		// Option flags per-socket
	{ char _dummy[(netSocketOptSockDebug == SO_DEBUG) ? 1 : 0]; (void) _dummy; }

#if !defined (UNIX)
	{ char _dummy[(netSocketOptSockAcceptConn == SO_ACCEPTCONN) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockUseLoopback == SO_USELOOPBACK) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockSndLowWater == SO_SNDLOWAT) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockRcvLowWater == SO_RCVLOWAT) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockSndTimeout == SO_SNDTIMEO) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockRcvTimeout == SO_RCVTIMEO) ? 1 : 0]; (void) _dummy; }
#endif

	{ char _dummy[(netSocketOptSockReuseAddr == SO_REUSEADDR) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockKeepAlive == SO_KEEPALIVE) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockDontRoute == SO_DONTROUTE) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockBroadcast == SO_BROADCAST) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockLinger == SO_LINGER) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockOOBInLine == SO_OOBINLINE) ? 1 : 0]; (void) _dummy; }

	{ char _dummy[(netSocketOptSockSndBufSize == SO_SNDBUF) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockRcvBufSize == SO_RCVBUF) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockErrorStatus == SO_ERROR) ? 1 : 0]; (void) _dummy; }
	{ char _dummy[(netSocketOptSockSocketType == SO_TYPE) ? 1 : 0]; (void) _dummy; }

//	{ char _dummy[(netSocketOptSockNonBlocking == xxx) ? 1 : 0]; (void) _dummy; }
//	{ char _dummy[(netSocketOptSockRequireErrClear == xxx) ? 1 : 0]; (void) _dummy; }
//	{ char _dummy[(netSocketOptSockMultiPktAddr == xxx) ? 1 : 0]; (void) _dummy; }

	switch (inOptLevel)
	{
		case netSocketOptLevelIP:
			outOptLevel = (int) inOptLevel;
//			switch (inOptName)
//			{
//				case some_supported_option:
//					outOptName = (int) inOptName;
//				case netSocketOptIPOptions:				// Not supported by WinSock 1.x
//				default:
					result = false;
//			}
			break;

		case netSocketOptLevelTCP:
			outOptLevel = (int) inOptLevel;
			switch (inOptName)
			{
				case netSocketOptTCPNoDelay:
//				case netSocketOptTCPMaxSeg:				// Not supported by WinSock
					outOptName = (int) inOptName;
					result = NetLibToSocketsScalarOption (inOptVal, inOptValLen, outOptVal, outOptValLen);
					break;

				default:
					result = false;
			}
			break;

		case netSocketOptLevelSocket:
			outOptLevel = (int) inOptLevel;
			switch (inOptName)
			{
				case netSocketOptSockDebug:
				case netSocketOptSockAcceptConn:
				case netSocketOptSockReuseAddr:
				case netSocketOptSockKeepAlive:
				case netSocketOptSockDontRoute:
				case netSocketOptSockBroadcast:
				case netSocketOptSockUseLoopback:
				case netSocketOptSockLinger:
				case netSocketOptSockOOBInLine:
				case netSocketOptSockSndBufSize:
				case netSocketOptSockRcvBufSize:
//				case netSocketOptSockSndLowWater:		// Not supported by WinSock
//				case netSocketOptSockRcvLowWater:		// Not supported by WinSock
//				case netSocketOptSockSndTimeout:		// Not supported by WinSock
//				case netSocketOptSockRcvTimeout:		// Not supported by WinSock
				case netSocketOptSockErrorStatus:
				case netSocketOptSockSocketType:
//				case netSocketOptSockNonBlocking:		// Palm OS Exclusive (special support added in SocketOptionSet)
//				case netSocketOptSockRequireErrClear:	// Palm OS Exclusive
//				case netSocketOptSockMultiPktAddr:		// Palm OS Exclusive
					outOptName = (int) inOptName;

					// Convert the option value. Everything is a simple scalar
					// except for netSocketOptSockLinger.

					if (inOptName != netSocketOptSockLinger)
					{
						result = NetLibToSocketsScalarOption (inOptVal, inOptValLen, outOptVal, outOptValLen);
					}
					else
					{
						if (inOptValLen == sizeof (NetSocketLingerType))
						{
							outOptValLen = sizeof (linger);
							outOptVal = (char*) Platform::AllocateMemory (outOptValLen);

							NetSocketLingerType	val = *(NetSocketLingerType*) inOptVal;

							Canonical (val.onOff);	// Byteswap on LE machines
							Canonical (val.time);	// Byteswap on LE machines

							((linger*) outOptVal)->l_onoff = val.onOff;
							((linger*) outOptVal)->l_linger = val.time;
						}
						else
						{
							result = false;
						}
					}

					break;

				default:
					result = false;
			}
			break;

		default:
			result = false;
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibOptions
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	SocketsToNetLibOptions		(const int				inOptLevel,
									 const int				inOptName,
									 /*const*/ char*&			inOptVal,
									 const int				inOptValLen,
									 NetSocketOptLevelEnum	outOptLevel,
									 NetSocketOptEnum		outOptName,
									 VoidPtr				outOptVal,
									 Word					outOptValLen)
{
	UNUSED_PARAM(outOptLevel)
	UNUSED_PARAM(outOptName)

	Bool	result = true;

	switch (inOptLevel)
	{
		case IPPROTO_IP:
//			switch (inOptName)
//			{
//				case IP_OPTIONS:				// Not supported by WinSock 1.x
//				default:
					result = false;
//			}
			break;

		case IPPROTO_TCP:
			switch (inOptName)
			{
#if defined (_WINDOWS) || defined (UNIX)
				case TCP_NODELAY:
#endif
//				case netSocketOptTCPMaxSeg:				// Not supported by WinSock
					result = SocketsToNetLibScalarOption (inOptVal, inOptValLen, outOptVal, outOptValLen);
					break;

				default:
					result = false;
			}
			break;

		case SOL_SOCKET:
			switch (inOptName)
			{
#if !defined (UNIX)
				case SO_ACCEPTCONN:
				case SO_USELOOPBACK:
#endif

				case SO_DEBUG:
				case SO_REUSEADDR:
				case SO_KEEPALIVE:
				case SO_DONTROUTE:
				case SO_BROADCAST:
				case SO_LINGER:
				case SO_OOBINLINE:
				case SO_SNDBUF:
				case SO_RCVBUF:
//				case SO_SNDLOWAT:		// Not supported by WinSock
//				case SO_RCVLOWAT:		// Not supported by WinSock
//				case SO_SNDTIMEO:		// Not supported by WinSock
//				case SO_RCVTIMEO:		// Not supported by WinSock
				case SO_ERROR:
				case SO_TYPE:
//				case netSocketOptSockNonBlocking:		// Palm OS Exclusive (special support added in SocketOptionGet)
//				case netSocketOptSockRequireErrClear:	// Palm OS Exclusive
//				case netSocketOptSockMultiPktAddr:		// Palm OS Exclusive

					// Convert the option value. Everything is a simple scalar
					// except for netSocketOptSockLinger.

					if (inOptName != SO_LINGER)
					{
						result = SocketsToNetLibScalarOption (inOptVal, inOptValLen, outOptVal, outOptValLen);
					}
					else
					{
						if (inOptValLen == sizeof (linger))
						{
							linger	val = *(linger*) inOptVal;

							Canonical (val.l_onoff);	// Byteswap on LE machines
							Canonical (val.l_linger);	// Byteswap on LE machines

							((NetSocketLingerType*) outOptVal)->onOff = val.l_onoff;
							((NetSocketLingerType*) outOptVal)->time = val.l_linger;

							Platform::DisposeMemory (outOptVal);
						}
						else
						{
							result = false;
						}
					}

					break;

				default:
					result = false;
			}
			break;

		default:
			result = false;
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsAddr
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	NetLibToSocketsAddr			(const NetSocketAddrType&	inAddr,
									 const SWord				inAddrLen,
									 sockaddr&					outAddr,
									 int						outAddrLen)
{
	// We only support netSocketAddrRaw and netSocketAddrINET.  As long
	// as the layout of NetSocketAddrType matches sockaddr, we can just
	// memcpy between the two.

	u_short	sa_family;
	if (!NetLibToSocketsDomain ((NetSocketAddrEnum) inAddr.family, sa_family))
	{
		return false;
	}

	memcpy (&outAddr, &inAddr, inAddrLen < outAddrLen ? inAddrLen : outAddrLen);

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibAddr
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool	SocketsToNetLibAddr			(const sockaddr&	inAddr,
									 int				inAddrLen,
									 NetSocketAddrType&	outAddr,
									 SWord				outAddrLen)
{
	if (!SocketsToNetLibDomain (inAddr.sa_family, (NetSocketAddrEnum&) outAddr.family))
	{
		return false;
	}

	memcpy (&outAddr, &inAddr, inAddrLen < outAddrLen ? inAddrLen : outAddrLen);

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibHostEnt
 *
 * DESCRIPTION:	Convert a sockets hostent to a NetLib NetHostInfoBufType
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool SocketsToNetLibHostEnt (	const hostent&			inHostEnt,
								NetHostInfoBufType&		outHostEnt)
{
	Bool	result = true;

	// hostent.h_name ->	NetHostInfoBufType.name
	//						NetHostInfoBufType.hostInfo.nameP
	strcpy (outHostEnt.name, inHostEnt.h_name);
	outHostEnt.hostInfo.nameP = outHostEnt.name;

	// hostent.h_aliases ->	NetHostInfoBufType.aliasList
	//						NetHostInfoBufType.aliases
	//						NetHostInfoBufType.hostInfo.nameAliasesP
	char*	curSrcAlias;
	char*	curDestAlias = outHostEnt.aliases[0];
	long	index = 0;
	while ((curSrcAlias = inHostEnt.h_aliases[index]) != NULL)
	{
		outHostEnt.aliasList[index] = curDestAlias;
		strcpy (curDestAlias, curSrcAlias);
		curDestAlias += strlen (curDestAlias) + 1;
		if ((((long) curDestAlias) & 1) != 0)
			++curDestAlias;
		++index;

		if (index == netDNSMaxAliases)
			break;
	}

	outHostEnt.aliasList[index] = NULL;
	outHostEnt.hostInfo.nameAliasesP = outHostEnt.aliasList;

	// hostent.h_addrtype ->	NetHostInfoBufType.hostInfo.addrType
	outHostEnt.hostInfo.addrType = inHostEnt.h_addrtype;

	// hostent.h_length ->		NetHostInfoBufType.hostInfo.addrLen
	outHostEnt.hostInfo.addrLen = inHostEnt.h_length;

	// hostent.h_addr_list ->	NetHostInfoBufType.addressList
	//							NetHostInfoBufType.address
	//							NetHostInfoBufType.hostInfo.addrListP
	char*		curSrcAddr;
	NetIPAddr*	curDestAddr = outHostEnt.address;
	/*long*/	index = 0;
	while ((curSrcAddr = inHostEnt.h_addr_list[index]) != NULL)
	{
		outHostEnt.addressList[index] = curDestAddr;
		memcpy (curDestAddr, curSrcAddr, inHostEnt.h_length);
		curDestAddr = (NetIPAddr*) (((char*) curDestAddr) + inHostEnt.h_length);
		++index;

		if (index == netDNSMaxAddresses)
			break;
	}

	outHostEnt.addressList[index] = 0;
	outHostEnt.hostInfo.addrListP = (BytePtr*) &outHostEnt.addressList[0];

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibServEnt
 *
 * DESCRIPTION:	Convert a sockets servent to a NetLib NetServInfoBufType
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool SocketsToNetLibServEnt (	const servent&			inServEnt,
								NetServInfoBufType&		outServEnt)
{
	Bool	result = true;

	// servent.s_name ->	NetServInfoBufType.name
	//						NetServInfoBufType.servInfo.nameP
	strcpy (outServEnt.name, inServEnt.s_name);
	outServEnt.servInfo.nameP = outServEnt.name;

	// servent.s_aliases ->	NetServInfoBufType.aliasList
	//						NetServInfoBufType.aliases
	//						NetServInfoBufType.servInfo.nameAliasesP
	char*	curSrcAlias;
	char*	curDestAlias = outServEnt.aliases[0];
	long	index = 0;
	while ((curSrcAlias = inServEnt.s_aliases[index]) != NULL)
	{
		outServEnt.aliasList[index] = curDestAlias;
		strcpy (curDestAlias, curSrcAlias);
		curDestAlias += strlen (curDestAlias) + 1;
		if ((((long) curDestAlias) & 1) != 0)
			++curDestAlias;
		++index;

		if (index == netServMaxAliases)
			break;
	}

	outServEnt.aliasList[index] = NULL;
	outServEnt.servInfo.nameAliasesP = outServEnt.aliasList;

	// servent.s_port ->	NetServInfoButType.servInfo.port
	outServEnt.servInfo.port = inServEnt.s_port;

	// servent.s_proto ->	NetServInfoBufType.protoName
	//						NetServInfoBufType.servInfo.protoP
	strcpy (outServEnt.protoName, inServEnt.s_proto);
	outServEnt.servInfo.protoP = outServEnt.protoName;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsFDSet
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsFDSet (	const NetFDSetType*	inFDSet,
							Word				inWidth,
							fd_set*&			outFSSet,
							int&				outWidth)
{
	Bool	result = true;
	fd_set*	finalSetPtr = NULL;

	if (inFDSet)
	{
		FD_ZERO (outFSSet);

		if (inWidth > netMaxNumSockets)
		{
			inWidth = netMaxNumSockets;
		}

		for (int ii = 0; ii < inWidth; ++ii)
		{
			NetSocketRef sRef = ii + netMinSocketRefNum;

			if (netFDIsSet (sRef, inFDSet))
			{
				SOCKET	s = gSockets [ii];
				if (s != INVALID_SOCKET)
				{
					FD_SET (s, outFSSet);
					finalSetPtr = outFSSet;

					outWidth = FD_SETSIZE;
				}
			}
		}
	}

	outFSSet = finalSetPtr;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibFDSet
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool SocketsToNetLibFDSet (	const fd_set*	inFDSet,
							int				inWidth,
							NetFDSetType*	outFSSet,
							Word&			outWidth)
{
	Bool	result = true;

	if (inFDSet && outFSSet)
	{
		netFDZero (outFSSet);

		if (inWidth > FD_SETSIZE)
		{
			inWidth = FD_SETSIZE;
		}

		for (int ii = 0; ii < netMaxNumSockets; ++ii)
		{
			SOCKET	s = gSockets[ii];

			if (s != INVALID_SOCKET && FD_ISSET (s, inFDSet))
			{
				netFDSet ((netMinSocketRefNum + ii), outFSSet);

				if (outWidth < ii)
				{
					outWidth = ii;
				}
			}
		}
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	NetLibToSocketsScalarOption
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool NetLibToSocketsScalarOption (	const VoidPtr	inOptVal,
									const Word		inOptValLen,
									char*&			outOptVal,
									socklen_t&		outOptValLen)
{
	Bool	result = true;

	if (inOptVal)
	{
		if (inOptValLen == sizeof (uae_u8))
		{
			outOptValLen = inOptValLen;
			outOptVal = (char*) Platform::AllocateMemory (outOptValLen);
			uae_u8	val = *(uae_u8*) inOptVal;
			Canonical (val);					// Byteswap on LE machines
			*(uae_u8*) outOptVal = val;
		}
		else if (inOptValLen == sizeof (uae_u16))
		{
			outOptValLen = inOptValLen;
			outOptVal = (char*) Platform::AllocateMemory (outOptValLen);
			uae_u16	val = *(uae_u16*) inOptVal;
			Canonical (val);					// Byteswap on LE machines
			*(uae_u16*) outOptVal = val;
		}
		else if (inOptValLen == sizeof (uae_u32))
		{
			outOptValLen = inOptValLen;
			outOptVal = (char*) Platform::AllocateMemory (outOptValLen);
			uae_u32	val = *(uae_u32*) inOptVal;
			Canonical (val);					// Byteswap on LE machines
			*(uae_u32*) outOptVal = val;
		}
		else
		{
			result = false;
		}
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SocketsToNetLibScalarOption
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool SocketsToNetLibScalarOption	(/*const*/ char*&	inOptVal,
									 const int		inOptValLen,
									 VoidPtr		outOptVal,
									 Word			outOptValLen)
{
	Bool	result = true;

	if (inOptVal && outOptVal)
	{
		if (inOptValLen == outOptValLen)
		{
			if (inOptValLen == sizeof (uae_u8))
			{
				uae_u8	val = *(uae_u8*) inOptVal;
				Canonical (val);					// Byteswap on LE machines
				*(uae_u8*) outOptVal = val;
			}
			else if (inOptValLen == sizeof (uae_u16))
			{
				uae_u16	val = *(uae_u16*) inOptVal;
				Canonical (val);					// Byteswap on LE machines
				*(uae_u16*) outOptVal = val;
			}
			else if (inOptValLen == sizeof (uae_u32))
			{
				uae_u32	val = *(uae_u32*) inOptVal;
				Canonical (val);					// Byteswap on LE machines
				*(uae_u32*) outOptVal = val;
			}
			else
			{
				result = false;
			}
		}
		else
		{
			result = false;
		}
	}

	Platform::DisposeMemory (inOptVal);

	return result;
}

										 
/***********************************************************************
 *
 * FUNCTION:	PrvGetError
 *
 * DESCRIPTION:	Returns the most recent sockets error number.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	Sockets error number
 *
 ***********************************************************************/

uae_u32 PrvGetError (void)
{
#if defined (__MACOS__) || defined (UNIX)
	uae_u32	err = errno;
#endif

#if defined (_WINDOWS)
	uae_u32	err = ::WSAGetLastError ();
#endif

	return err;
}


/***********************************************************************
 *
 * FUNCTION:	PrvGetTranslatedError
 *
 * DESCRIPTION:	Returns the most recent sockets error number translated
 *				into a NetLib error number.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	NetLib error number
 *
 ***********************************************************************/

Err PrvGetTranslatedError (void)
{
	uae_u32	err = ::PrvGetError ();

	return PrvTranslateError (err);
}


/***********************************************************************
 *
 * FUNCTION:	PrvGetTranslatedHError
 *
 * DESCRIPTION:	Returns the most recent sockets error number translated
 *				into a NetLib error number.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	NetLib error number
 *
 ***********************************************************************/

#if defined (__MACOS__)
extern int h_errno;
const int kHErrNoBase = 30000;

Err PrvGetTranslatedHError (void)
{
	// For some reason (I don't know...maybe BSD sockets is like this, too),
	// GUSI returns errors from gethostbyname() and gethostbyaddr() in h_errno.

	uae_u32	err = h_errno + kHErrNoBase;

	return PrvTranslateError (err);
}
#endif

	
/***********************************************************************
 *
 * FUNCTION:	PrvTranslateError
 *
 * DESCRIPTION:	Convert a Windows Sockets error number to a NetLib error
 *				number.
 *
 * PARAMETERS:	err - Windows Sockets error number (probably from
 *				WSAGetLastError)
 *
 * RETURNED:	NetLib error number
 *
 ***********************************************************************/

Err PrvTranslateError (uae_u32 err)
{
	switch (err)
	{
#if defined (__MACOS__) || defined (UNIX)
//		case EPERM:				//	1		/* Operation not permitted */
//		case ENOENT:			//	2		/* No such file or directory */
//		case ESRCH:				//	3		/* No such process */
		case EINTR:				//	4		/* Interrupted system call */
			return netErrUserCancel;

//		case EIO:				//	5		/* Input/output error */
//		case ENXIO:				//	6		/* Device not configured */
//		case E2BIG:				//	7		/* Argument list too long */
//		case ENOEXEC:			//	8		/* Exec format error */
//		case EBADF:				//	9		/* Bad file descriptor */
//		case ECHILD:			//	10		/* No child processes */
		case EDEADLK:			//	11		/* Resource deadlock avoided */
					/* 11 was EAGAIN */
			return netErrWouldBlock;

		case ENOMEM:			//	12		/* Cannot allocate memory */
			return netErrOutOfMemory;

		case EACCES:			//	13		/* Permission denied */
			return netErrAuthFailure;

//		case EFAULT:			//	14		/* Bad address */
//		case ENOTBLK:			//	15		/* Block device required */
		case EBUSY:				//	16		/* Device busy */
			return netErrSocketBusy;

//		case EEXIST:			//	17		/* File exists */
//		case EXDEV:				//	18		/* Cross-device link */
//		case ENODEV:			//	19		/* Operation not supported by device */
//		case ENOTDIR:			//	20		/* Not a directory */
//		case EISDIR:			//	21		/* Is a directory */
//		case EINVAL:			//	22		/* Invalid argument */
//		case ENFILE:			//	23		/* Too many open files in system */
//		case EMFILE:			//	24		/* Too many open files */
//		case ENOTTY:			//	25		/* Inappropriate ioctl for device */
//		case ETXTBSY:			//	26		/* Text file busy */
//		case EFBIG:				//	27		/* File too large */
//		case ENOSPC:			//	28		/* No space left on device */
//		case ESPIPE:			//	29		/* Illegal seek */
		case EROFS:				//	30		/* Read-only file system */
			return netErrReadOnlySetting;

//		case EMLINK:			//	31		/* Too many links */
//		case EPIPE:				//	32		/* Broken pipe */
/* math software */
//		case EDOM:				//	33		/* Numerical argument out of domain */
//		case ERANGE:			//	34		/* Result too large */
			// CodeWarrior uses 35 for EFPOS (whatever that is), and so
			// maps EAGAIN (and EWOULDBLOCK) to EDEADLK
////	case EAGAIN:			//	35		/* Resource temporarily unavailable */
////	case EWOULDBLOCK:		//	EAGAIN	/* Operation would block */
		case EINPROGRESS:		//	36		/* Operation now in progress */
			return netErrSocketBusy;

/* These are defined to be the same under QNX/Neutrino */
#if EALREADY != EBUSY
		case EALREADY:			//	37		/* Operation already in progress */
			return netErrAlreadyInProgress;
#endif
/* ipc/network software -- argument errors */
		case ENOTSOCK:			//	38		/* Socket operation on non-socket */
			return netErrNoSocket;

		case EDESTADDRREQ:		//	39		/* Destination address required */
			return netErrIPNoDst;

		case EMSGSIZE:			//	40		/* Message too long */
			return netErrMessageTooBig;

//		case EPROTOTYPE:		//	41		/* Protocol wrong type for socket */
		case ENOPROTOOPT:		//	42		/* Protocol not available */
			return netErrUnknownProtocol;

		case EPROTONOSUPPORT:	//	43		/* Protocol not supported */
			return netErrUnknownProtocol;

		case ESOCKTNOSUPPORT:	//	44		/* Socket type not supported */
			return netErrWrongSocketType;

		case EOPNOTSUPP:		//	45		/* Operation not supported on socket */
			return netErrWrongSocketType;

		case EPFNOSUPPORT:		//	46		/* Protocol family not supported */
			return netErrUnknownService;

		case EAFNOSUPPORT:		//	47		/* Address family not supported by protocol family */
			return netErrUnknownService;

		case EADDRINUSE:		//	48		/* Address already in use */
			return netErrPortInUse;

		case EADDRNOTAVAIL:		//	49		/* Can't assign requested address */
			return netErrPortInUse;

/* ipc/network software -- operational errors */
		case ENETDOWN:			//	50		/* Network is down */
			return netErrUnreachableDest;

		case ENETUNREACH:		//	51		/* Network is unreachable */
			return netErrNoInterfaces;

		case ENETRESET:			//	52		/* Network dropped connection on reset */
		case ECONNABORTED:		//	53		/* Software caused connection abort */
		case ECONNRESET:		//	54		/* Connection reset by peer */
			return netErrSocketClosedByRemote;

		case ENOBUFS:			//	55		/* No buffer space available */
			return netErrNoTCB;

		case EISCONN:			//	56		/* Socket is already connected */
			return netErrSocketAlreadyConnected;

		case ENOTCONN:			//	57		/* Socket is not connected */
			return netErrSocketNotConnected;

		case ESHUTDOWN:			//	58		/* Can't send after socket shutdown */
			return netErrSocketNotOpen;

//		case ETOOMANYREFS:		//	59		/* Too many references: can't splice */
		case ETIMEDOUT:			//	60		/* Connection timed out */
			return netErrTimeout;

		case ECONNREFUSED:		//	61		/* Connection refused */
			return netErrTimeout;

//		case ELOOP:				//	62		/* Too many levels of symbolic links */
//		case ENAMETOOLONG:		//	63		/* File name too long */
		case EHOSTDOWN:			//	64		/* Host is down */
		case EHOSTUNREACH:		//	65		/* No route to host */
			return netErrIPNoRoute;

//		case ENOTEMPTY:			//	66		/* Directory not empty */
//		case EPROCLIM:			//	67		/* Too many processes */
//		case EUSERS:			//	68		/* Too many users */
//		case EDQUOT:			//	69		/* Disc quota exceeded */

/* Network File System */
//		case ESTALE:			//	70		/* Stale NFS file handle */
//		case EREMOTE:			//	71		/* Too many levels of remote in path */
//		case EBADRPC:			//	72		/* RPC struct is bad */
//		case ERPCMISMATCH:		//	73		/* RPC version wrong */
//		case EPROGUNAVAIL:		//	74		/* RPC prog. not avail */
//		case EPROGMISMATCH:		//	75		/* Program version wrong */
//		case EPROCUNAVAIL:		//	76		/* Bad procedure for program */
//		case ENOLCK:			//	77		/* No locks available */
//		case ENOSYS:			//	78		/* Function not implemented */
//		case EFTYPE:			//	79		/* Inappropriate file type or format */
#endif	// __MACOS__ || UNIX

#if defined (__MACOS__)
		case HOST_NOT_FOUND + kHErrNoBase:	//	1		/* Authoritative Answer Host not found */
			return netErrDNSUnreachable;

		case TRY_AGAIN + kHErrNoBase:		//	2		/* Non-Authoritive Host not found, or SERVERFAIL */
			return netErrDNSServerFailure;

		case NO_RECOVERY + kHErrNoBase:		//	3		/* Non recoverable errors, FORMERR, REFUSED, NOTIMP */
			return netErrDNSRefused;

		case NO_DATA + kHErrNoBase:			//	4		/* Valid name, no data record of requested type */
//		case NO_ADDRESS + kHErrNoBase:		//	4		/* no address, look for MX record */
			return netErrDNSNonexistantName;
#endif

#if defined (_WINDOWS)
		case WSAEACCES:
			// (10013)
			// Permission denied.
			// An attempt was made to access a socket in a way forbidden by its access permissions. An example is using a broadcast address for sendto without broadcast permission being set using setsockopt(SO_BROADCAST).
			return netErrWrongSocketType;	// BEST I CAN THINK OF.

		case WSAEADDRINUSE:
			// (10048)
			// Address already in use.
			// Only one usage of each socket address (protocol/IP address/port) is normally permitted. This error occurs if an application attempts to bind a socket to an IP address/port that has already been used for an existing socket, or a socket that wasn't closed properly, or one that is still in the process of closing. For server applications that need to bind multiple sockets to the same port number, consider using setsockopt(SO_REUSEADDR). Client applications usually need not call bind at all - connect will choose an unused port automatically. When bind is called with a wild-card address (involving ADDR_ANY), a WSAEADDRINUSE error could be delayed until the specific address is "committed." This could happen with a call to other function later, including connect, listen, WSAConnect or WSAJoinLeaf.
			return netErrPortInUse;

		case WSAEADDRNOTAVAIL:
			// (10049)
			// Cannot assign requested address.
			// The requested address is not valid in its context. Normally results from an attempt to bind to an address that is not valid for the local machine. This can also result from connect, sendto, WSAConnect, WSAJoinLeaf, or WSASendTo when the remote address or port is not valid for a remote machine (e.g. address or port 0).
			return netErrParamErr;

		case WSAEAFNOSUPPORT:
			// (10047)
			// Address family not supported by protocol family.
			// An address incompatible with the requested protocol was used. All sockets are created with an associated "address family" (i.e. AF_INET for Internet Protocols) and a generic protocol type (i.e. SOCK_STREAM). This error will be returned if an incorrect protocol is explicitly requested in the socket call, or if an address of the wrong family is used for a socket, e.g. in sendto.
			return netErrParamErr;

		case WSAEALREADY:
			// (10037)
			// Operation already in progress.
			// An operation was attempted on a non-blocking socket that already had an operation in progress - i.e. calling connect a second time on a non-blocking socket that is already connecting, or canceling an asynchronous request (WSAAsyncGetXbyY) that has already been canceled or completed.
			return netErrAlreadyInProgress;

		case WSAECONNABORTED:
			// (10053)
			// Software caused connection abort.
			// An established connection was aborted by the software in your host machine, possibly due to a data transmission timeout or protocol error.
			return netErrSocketClosedByRemote;

		case WSAECONNREFUSED:
			// (10061)
			// Connection refused.
			// No connection could be made because the target machine actively refused it. This usually results from trying to connect to a service that is inactive on the foreign host - i.e. one with no server application running.
			return netErrTimeout;			// As near as I can tell, if a connection is refused, NetLib merely closes the connection and eventually times out.

		case WSAECONNRESET:
			// (10054)
			// Connection reset by peer.
			// A existing connection was forcibly closed by the remote host. This normally results if the peer application on the remote host is suddenly stopped, the host is rebooted, or the remote host used a "hard close" (see setsockopt for more information on the SO_LINGER option on the remote socket.) This error may also result if a connection was broken due to "keep-alive" activity detecting a failure while one or more operations are in progress. Operations that were in progress fail with WSAENETRESET. Subsequent operations fail with WSAECONNRESET.
			return netErrSocketClosedByRemote;

		case WSAEDESTADDRREQ:
			// (10039)
			// Destination address required.
			// A required address was omitted from an operation on a socket. For example, this error will be returned if sendto is called with the remote address of ADDR_ANY.
			return netErrParamErr;

		case WSAEFAULT:
			// (10014)
			// Bad address.
			// The system detected an invalid pointer address in attempting to use a pointer argument of a call. This error occurs if an application passes an invalid pointer value, or if the length of the buffer is too small. For instance, if the length of an argument which is a sockaddr is smaller than sizeof(sockaddr).
			return netErrParamErr;

		case WSAEHOSTDOWN:
			// (10064)
			// Host is down.
			// A socket operation failed because the destination host was down. A socket operation encountered a dead host. Networking activity on the local host has not been initiated. These conditions are more likely to be indicated by the error WSAETIMEDOUT.
			return netErrTimeout;

		case WSAEHOSTUNREACH:
			// (10065)
			// No route to host.
			// A socket operation was attempted to an unreachable host. See WSAENETUNREACH
			return netErrUnreachableDest;

		case WSAEINPROGRESS:
			// (10036)
			// Operation now in progress.
			// A blocking operation is currently executing. Windows Sockets only allows a single blocking operation to be outstanding per task (or thread), and if any other function call is made (whether or not it references that or any other socket) the function fails with the WSAEINPROGRESS error.
			return netErrAlreadyInProgress;

		case WSAEINTR:
			// (10004)
			// Interrupted function call.
			// A blocking operation was interrupted by a call to WSACancelBlockingCall.
			return netErrUserCancel;

		case WSAEINVAL:
			// (10022)
			// Invalid argument.
			// Some invalid argument was supplied (for example, specifying an invalid level to the setsockopt function). In some instances, it also refers to the current state of the socket - for instance, calling accept on a socket that is not listening.
			return netErrParamErr;

		case WSAEISCONN:
			// (10056)
			// Socket is already connected.
			// A connect request was made on an already connected socket. Some implementations also return this error if sendto is called on a connected SOCK_DGRAM socket (For SOCK_STREAM sockets, the to parameter in sendto is ignored), although other implementations treat this as a legal occurrence.
			return netErrSocketAlreadyConnected;

		case WSAEMFILE:
			// (10024)
			// Too many open files.
			// Too many open sockets. Each implementation may have a maximum number of socket handles available, either globally, per process or per thread.
			return netErrNoMoreSockets;

		case WSAEMSGSIZE:
			// (10040)
			// Message too long.
			// A message sent on a datagram socket was larger than the internal message buffer or some other network limit, or the buffer used to receive a datagram into was smaller than the datagram itself.
			return netErrMessageTooBig;

		case WSAENETDOWN:
			// (10050)
			// Network is down.
			// A socket operation encountered a dead network. This could indicate a serious failure of the network system (i.e. the protocol stack that the WinSock DLL runs over), the network interface, or the local network itself.
			return netErrUnreachableDest;

		case WSAENETRESET:
			// (10052)
			// Network dropped connection on reset.
			// The connection has been broken due to "keep-alive" activity detecting a failure while the operation was in progress. It can also be returned by setsockopt if an attempt is made to set SO_KEEPALIVE on a connection that has already failed.
			return netErrSocketClosedByRemote;

		case WSAENETUNREACH:
			// (10051)
			// Network is unreachable.
			// A socket operation was attempted to an unreachable network. This usually means the local software knows no route to reach the remote host.
			return netErrUnreachableDest;

		case WSAENOBUFS:
			// (10055)
			// No buffer space available.
			// An operation on a socket could not be performed because the system lacked sufficient buffer space or because a queue was full.
			return netErrOutOfMemory;

		case WSAENOPROTOOPT:
			// (10042)
			// Bad protocol option.
			// An unknown, invalid or unsupported option or level was specified in a getsockopt or setsockopt call.
			return netErrParamErr;

		case WSAENOTCONN:
			// (10057)
			// Socket is not connected.
			// A request to send or receive data was disallowed because the socket is not connected and (when sending on a datagram socket using sendto) no address was supplied. Any other type of operation might also return this error - for example, setsockopt setting SO_KEEPALIVE if the connection has been reset.
			return netErrSocketNotConnected;

		case WSAENOTSOCK:
			// (10038)
			// Socket operation on non-socket.
			// An operation was attempted on something that is not a socket. Either the socket handle parameter did not reference a valid socket, or for select, a member of an fd_set was not valid.
			return netErrParamErr;

		case WSAEOPNOTSUPP:
			// (10045)
			// Operation not supported.
			// The attempted operation is not supported for the type of object referenced. Usually this occurs when a socket descriptor to a socket that cannot support this operation, for example, trying to accept a connection on a datagram socket.
			return netErrWrongSocketType;

		case WSAEPFNOSUPPORT:
			// (10046)
			// Protocol family not supported.
			// The protocol family has not been configured into the system or no implementation for it exists. Has a slightly different meaning to WSAEAFNOSUPPORT, but is interchangeable in most cases, and all Windows Sockets functions that return one of these specify WSAEAFNOSUPPORT.
			return netErrParamErr;

		case WSAEPROCLIM:
			// (10067)
			// Too many processes.
			// A Windows Sockets implementation may have a limit on the number of applications that may use it simultaneously. WSAStartup may fail with this error if the limit has been reached.
			return netErrOutOfResources;

		case WSAEPROTONOSUPPORT:
			// (10043)
			// Protocol not supported.
			// The requested protocol has not been configured into the system, or no implementation for it exists. For example, a socket call requests a SOCK_DGRAM socket, but specifies a stream protocol.
			return netErrUnknownProtocol;

		case WSAEPROTOTYPE:
			// (10041)
			// Protocol wrong type for socket.
			// A protocol was specified in the socket function call that does not support the semantics of the socket type requested. For example, the ARPA Internet UDP protocol cannot be specified with a socket type of SOCK_STREAM.
			return netErrUnknownProtocol;

		case WSAESHUTDOWN:
			// (10058)
			// Cannot send after socket shutdown.
			// A request to send or receive data was disallowed because the socket had already been shut down in that direction with a previous shutdown call. By calling shutdown a partial close of a socket is requested, which is a signal that sending or receiving or both has been discontinued.
			return netErrNotOpen;

		case WSAESOCKTNOSUPPORT:
			// (10044)
			// Socket type not supported.
			// The support for the specified socket type does not exist in this address family. For example, the optional type SOCK_RAW might be selected in a socket call, and the implementation does not support SOCK_RAW sockets at all.
			return netErrParamErr;

		case WSAETIMEDOUT:
			// (10060)
			// Connection timed out.
			// A connection attempt failed because the connected party did not properly respond after a period of time, or established connection failed because connected host has failed to respond.
			return netErrTimeout;

//		case WSATYPE_NOT_FOUND:
			// (10109)
			// Class type not found.
			// The specified class was not found.

		case WSAEWOULDBLOCK:
			// (10035)
			// Resource temporarily unavailable.
			// This error is returned from operations on non-blocking sockets that cannot be completed immediately, for example recv when no data is queued to be read from the socket. It is a non-fatal error, and the operation should be retried later. It is normal for WSAEWOULDBLOCK to be reported as the result from calling connect on a non-blocking SOCK_STREAM socket, since some time must elapse for the connection to be established.
			return netErrWouldBlock;

		case WSAHOST_NOT_FOUND:
			// (11001)
			// Host not found.
			// No such host is known. The name is not an official hostname or alias, or it cannot be found in the database(s) being queried. This error may also be returned for protocol and service queries, and means the specified name could not be found in the relevant database.
			return netErrDNSNonexistantName;

//		case WSA_INVALID_HANDLE:
			// (OS dependent)
			// Specified event object handle is invalid.
			// An application attempts to use an event object, but the specified handle is not valid.

//		case WSA_INVALID_PARAMETER:
			// (OS dependent)
			// One or more parameters are invalid.
			// An application used a Windows Sockets function which directly maps to a Win32 function. The Win32 function is indicating a problem with one or more parameters.

//		case WSAINVALIDPROCTABLE:
			// (OS dependent)
			// Invalid procedure table from service provider.
			// A service provider returned a bogus proc table to WS2_32.DLL. (Usually caused by one or more of the function pointers being NULL.)

//		case WSAINVALIDPROVIDER:
			// (OS dependent)
			// Invalid service provider version number.
			// A service provider returned a version number other than 2.0.

//		case WSA_IO_INCOMPLETE:
			// (OS dependent)
			// Overlapped I/O event object not in signaled state.
			// The application has tried to determine the status of an overlapped operation which is not yet completed. Applications that use WSAGetOverlappedResult (with the fWait flag set to false) in a polling mode to determine when an overlapped operation has completed will get this error code until the operation is complete.

//		case WSA_IO_PENDING:
			// (OS dependent)
			// Overlapped operations will complete later.
			// The application has initiated an overlapped operation which cannot be completed immediately. A completion indication will be given at a later time when the operation has been completed.

//		case WSA_NOT_ENOUGH_MEMORY:
			// (OS dependent)
			// Insufficient memory available.
			// An application used a Windows Sockets function which directly maps to a Win32 function. The Win32 function is indicating a lack of required memory resources.

		case WSANOTINITIALISED:
			// (10093)
			// Successful WSAStartup not yet performed.
			// Either the application hasn't called WSAStartup or WSAStartup failed. The application may be accessing a socket which the current active task does not own (i.e. trying to share a socket between tasks), or WSACleanup has been called too many times.
			return netErrNotOpen;

		case WSANO_DATA:
			// (11004)
			// Valid name, no data record of requested type.
			// The requested name is valid and was found in the database, but it does not have the correct associated data being resolved for. The usual example for this is a hostname -> address translation attempt (using gethostbyname or WSAAsyncGetHostByName) which uses the DNS (Domain Name Server), and an MX record is returned but no A record - indicating the host itself exists, but is not directly reachable.
			return netErrDNSFormat;

		case WSANO_RECOVERY:
			// (11003)
			// This is a non-recoverable error.
			// This indicates some sort of non-recoverable error occurred during a database lookup. This may be because the database files (e.g. BSD-compatible HOSTS, SERVICES or PROTOCOLS files) could not be found, or a DNS request was returned by the server with a severe error.
			return netErrInternal;

//		case WSAPROVIDERFAILEDINIT:
			// (OS dependent)
			// Unable to initialize a service provider.
			// Either a service provider's DLL could not be loaded (LoadLibrary failed) or the provider's WSPStartup/NSPStartup function failed.

//		case WSASYSCALLFAILURE:
			// (OS dependent)
			// System call failure.
			// Returned when a system call that should never fail does. For example, if a call to WaitForMultipleObjects fails or one of the registry functions fails trying to manipulate theprotocol/namespace catalogs.

		case WSASYSNOTREADY:
			// (10091)
			// Network subsystem is unavailable.
			// This error is returned by WSAStartup if the Windows Sockets implementation cannot function at this time because the underlying system it uses to provide network services is currently unavailable. Users should check:
			// * that the appropriate Windows Sockets DLL file is in the current path,
			// * that they are not trying to use more than one Windows Sockets implementation simultaneously. If there is more than one WINSOCK DLL on your system, be sure the first one in the path is appropriate for the network subsystem currently loaded.
			// * the Windows Sockets implementation documentation to be sure all necessary components are currently installed and configured correctly.
			return netErrInternal;

		case WSATRY_AGAIN:
			// (11002)
			// Non-authoritative host not found.
			// This is usually a temporary error during hostname resolution and means that the local server did not receive a response from an authoritative server. A retry at some time later may be successful.
			return netErrDNSTimeout;

		case WSAVERNOTSUPPORTED:
			// (10092)
			// WINSOCK.DLL version out of range.
			// The current Windows Sockets implementation does not support the Windows Sockets specification version requested by the application. Check that no old Windows Sockets DLL files are being accessed.
			return netErrInternal;

		case WSAEDISCON:
			// (10094)
			// Graceful shutdown in progress.
			// Returned by WSARecv and WSARecvFrom to indicate the remote party has initiated a graceful shutdown sequence.
			return netErrSocketNotConnected;

//		case WSA_OPERATION_ABORTED:
			// (OS dependent)
			// Overlapped operation aborted.
			// An overlapped operation was canceled due to the closure of the socket, or the execution of the SIO_FLUSH command in WSAIoctl.
#endif	// _WINDOWS
	}

	return netErrParamErr;
}
