/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_SED1375.h"

#include "Bank_ROM.h"			// ROMBank::IsPCInRAM
#include "Byteswapping.h"		// ByteswapWords
#include "CPU_REG.h"			// Software::BusError, Screen::Palette
#include "DebugMgr.h"			// Debug::CheckStepSpy
#include "RAM_ROM.h"			// gMemoryAccess
#include "SessionFile.h"		// WriteSED1375RegsType, ReadSED1375RegsType, etc.


typedef uae_u32 	(*ReadFunction)(uaecptr address, int size);
typedef void		(*WriteFunction)(uaecptr address, int size, uae_u32 value);

// Macro to return the Dragonball address of the specified register

#define addressof(x)	(kMemoryStart + sed1375RegisterOffset + offsetof(SED1375RegsType, x))


// Macros for reading/writing Dragonball registers.

#define READ_REGISTER(reg)	\
	((sizeof(g1375Regs.reg) == 1) ? do_get_mem_byte(&g1375Regs.reg) :	\
	 (sizeof(g1375Regs.reg) == 2) ? do_get_mem_word(&g1375Regs.reg) :	\
								  do_get_mem_long(&g1375Regs.reg))

#define WRITE_REGISTER(reg, value)	\
	(sizeof(g1375Regs.reg) == 1) ? do_put_mem_byte(&g1375Regs.reg, value) : \
	(sizeof(g1375Regs.reg) == 2) ? do_put_mem_word(&g1375Regs.reg, value) : \
								 do_put_mem_long(&g1375Regs.reg, value)

class SED1375Reg
{
	public:
		static void 		SetHandler			(ReadFunction, WriteFunction, uae_u32 start, int count);

		static uae_u32		UnsupportedRead 	(uaecptr address, int size);
		static void 		UnsupportedWrite	(uaecptr address, int size, uae_u32 value);

		static uae_u32		StdRead 			(uaecptr address, int size);
		static void 		StdWrite			(uaecptr address, int size, uae_u32 value);
		static void 		NullWrite			(uaecptr address, int size, uae_u32 value); // For read-only registers

		static uae_u32		GetVertNonDisplay	(uaecptr address, int size);
		static uae_u32		GetCLUTEntry		(uaecptr address, int size);
		static void 		SetCLUTEntry		(uaecptr address, int size, uae_u32 value);

		static void 		SetCLUTAddr 		(uaecptr address, int size, uae_u32 value);
};


const SED1375RegsType kInitialSED1375RegisterValues =
{
	0x24,		// Byte productRevisionCode;		// 0x00

	0x00,		// Byte mode0;						// 0x01
	0x00,		// Byte mode1;						// 0x02
	0x00,		// Byte mode2;						// 0x03

	0x00,		// Byte horizontalPanelSize;		// 0x04

	0x00,		// Byte verticalPanelSizeLSB;		// 0x05
	0x00,		// Byte verticalPanelSizeMSB;		// 0x06

	0x00,		// Byte FPLineStartPosition;		// 0x07
	0x00,		// Byte horizontalNonDisplayPeriod; // 0x08
	0x00,		// Byte FPFRAMEStartPosition;		// 0x09
	0x00,		// Byte verticalNonDisplayPeriod;	// 0x0A
	0x00,		// Byte MODRate;					// 0x0B

	0x00,		// Byte screen1StartAddressLSB; 	// 0x0C
	0x00,		// Byte screen1StartAddressMSB; 	// 0x0D
	0x00,		// Byte screen1StartAddressMSBit;	// 0x0E

	0x00,		// Byte screen2StartAddressLSB; 	// 0x0F
	0x00,		// Byte screen2StartAddressMSB; 	// 0x10
	0x00,		// Byte screen2StartAddressMSBit;	// 0x11

	0x00,		// Byte memoryAddressOffset;		// 0x12

	0x00,		// Byte screen1VerticalSizeLSB; 	// 0x13
	0x00,		// Byte screen1VerticalSizeMSB; 	// 0x14

	0x00,		// Byte lookUpTableAddress; 		// 0x15
	0x00,		// Byte unused1;					// 0x16
	0x00,		// Byte lookUpTableData;			// 0x17

	0x00,		// Byte GPIOConfigurationControl;	// 0x18
	0x00,		// Byte GPIOStatusControl;			// 0x19

	0x00,		// Byte scratchPad; 				// 0x1A
	0x00,		// Byte portraitMode;				// 0x1B
	0x00,		// Byte lineByteCountRegister;		// 0x1C, for portrait mode only
	0x00,		// Byte unused2;					// 0x01D not used
	0x00,		// Byte unusual;					// 0x1E
	0x00		// Byte testMode;					// 0x1F
};


static AddressBank	gAddressBank =
{
	SED1375Bank::GetLong,
	SED1375Bank::GetWord,
	SED1375Bank::GetByte,
	SED1375Bank::SetLong,
	SED1375Bank::SetWord,
	SED1375Bank::SetByte,
	SED1375Bank::GetRealAddress,
	SED1375Bank::ValidAddress,
	SED1375Bank::GetMetaAddress,
	NULL
};

static const uae_u32	kMemoryStart = sed1375BaseAddress;

static SED1375RegsType	g1375Regs;
static void*			gVideoMem;
static void*			gMetaMem;
static uae_u16			gClutData[256];
static Bool 			gClutDirty;

// These functions provide fetch and store access to the
// emulator's register memory.

static ReadFunction 	gReadFunctions[sizeof(SED1375RegsType)];
static WriteFunction	gWriteFunctions[sizeof(SED1375RegsType)];

#define kCLUTColorIndexMask 	0xf000
#define kCLUTColorsMask 		0x0fff

#define kCLUTRedMask			0x0f00
#define kCLUTGreenMask			0x00f0
#define kCLUTBlueMask			0x000f

#define kCLUTIndexRed			0x4000
#define kCLUTIndexGreen 		0x2000
#define kCLUTIndexBlue			0x1000



#pragma mark -


/***********************************************************************
 *
 * FUNCTION:	SED1375Bank::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SED1375Bank::Initialize (void)
{
	// Memory banks starting at kMemoryStart are managed by the functions in SED1375Bank.

	long	numBanks = (sed1375RegisterOffset + sizeof (SED1375RegsType) + 0x0FFFF) / 0x10000;
	Memory::InitializeBanks (gAddressBank, bankindex (kMemoryStart), numBanks);

	// Allocate a chunk of memory for the VRAM space.

	if (!gVideoMem)
	{
		gVideoMem = Platform::AllocateMemoryClear (sed1375VideoMemSize);
		gMetaMem = Platform::AllocateMemoryClear (sed1375VideoMemSize);
	}

	// Install the handlers for each register.

	SED1375Bank::InstallHandlers ();
}


/***********************************************************************
 *
 * FUNCTION:	SED1375Bank::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SED1375Bank::Reset (void)
{
	g1375Regs = kInitialSED1375RegisterValues;

	Canonical (g1375Regs);
	ByteswapWords (&g1375Regs, sizeof(g1375Regs));

	gClutDirty = true;			// Just make sure the clut gets refreshed after reset.
}


/***********************************************************************
 *
 * FUNCTION:	SED1375Bank::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SED1375Bank::Save (SessionFile& f)
{
	StWordSwapper					swapper1 (&g1375Regs, sizeof(g1375Regs));
//	StCanonical<SED1375RegsType>	swapper2 (g1375Regs);
	f.WriteSED1375RegsType (g1375Regs);
	f.FixBug (SessionFile::kBugByteswappedStructs);

	StWordSwapper	swapper3 (gVideoMem, sed1375VideoMemSize);
	f.WriteSED1375Image (gVideoMem, sed1375VideoMemSize);

	StWordSwapper	swapper4 (gMetaMem, sed1375VideoMemSize);
	f.WriteSED1375MetaImage (gMetaMem, sed1375VideoMemSize);

	StWordSwapper	swapper5 (gClutData, sizeof(gClutData));
	f.WriteSED1375Palette (gClutData);
}


/***********************************************************************
 *
 * FUNCTION:	SED1375Bank::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SED1375Bank::Load (SessionFile& f)
{
	assert (gVideoMem != NULL);
	assert (gMetaMem != NULL);

	// Read in the SED registers, and then byteswap them.

	if (f.ReadSED1375RegsType (g1375Regs))
	{
		// The Windows version of Poser 2.1d29 and earlier did not write
		// out structs in the correct format.  The fields of the struct
		// were written out in Little-Endian format, not Big-Endian.  To
		// address this problem, the bug has been fixed, and a new field
		// is added to the file format indicating that the bug has been
		// fixed.  With the new field (the "bug bit"), Poser can identify
		// old files from new files and read them in accordingly.
		// 
		// With the bug fixed, the .psf files should now be interchangeable
		// across platforms (modulo other bugs...).

		if (!f.IncludesBugFix (SessionFile::kBugByteswappedStructs))
		{
			Canonical (g1375Regs);
		}
		ByteswapWords (&g1375Regs, sizeof(g1375Regs));
	}
	else
	{
		f.SetCanReload (false);
	}

	// Read in the LCD image, and then byteswap it.

	if (f.ReadSED1375Image (gVideoMem))
	{
		ByteswapWords (gVideoMem, sed1375VideoMemSize);
	}
	else
	{
		f.SetCanReload (false);
	}

	// Read in the LCD meta image, and then byteswap it.

	if (f.ReadSED1375MetaImage (gMetaMem))
	{
		ByteswapWords (gMetaMem, sed1375VideoMemSize);
	}
	else
	{
		f.SetCanReload (false);
	}

	// Read in the LCD palette, and then byteswap it.

	if (f.ReadSED1375Palette (gClutData))
	{
		ByteswapWords (gClutData, sizeof (gClutData));
	}
	else
	{
		f.SetCanReload (false);
	}

	gClutDirty = true;	// This variable comments on the state of gClutData.
						// By forcing it to "true", we don't have to save
						// and later reload gClutData.
}


/***********************************************************************
 *
 * FUNCTION:	SED1375Bank::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SED1375Bank::Dispose (void)
{
	Platform::DisposeMemory (gVideoMem);
	Platform::DisposeMemory (gMetaMem);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::InstallHandlers
// ---------------------------------------------------------------------------

void SED1375Bank::InstallHandlers (void)
{
	SED1375Reg::SetHandler (SED1375Reg::UnsupportedRead, SED1375Reg::UnsupportedWrite,
						kMemoryStart + sed1375RegisterOffset, sizeof (SED1375RegsType));

	// Now add standard/specialized handers for the defined registers.

#define INSTALL_HANDLER(read, write, reg) \
	SED1375Reg::SetHandler (SED1375Reg::read, SED1375Reg::write, \
							addressof (reg), sizeof (g1375Regs.reg))


	INSTALL_HANDLER (StdRead,			StdWrite,				productRevisionCode);
	INSTALL_HANDLER (StdRead,			StdWrite,				mode0);
	INSTALL_HANDLER (StdRead,			StdWrite,				mode1);
	INSTALL_HANDLER (StdRead,			StdWrite,				mode2);
	INSTALL_HANDLER (StdRead,			StdWrite,				horizontalPanelSize);
	INSTALL_HANDLER (StdRead,			StdWrite,				verticalPanelSizeLSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				verticalPanelSizeMSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				FPLineStartPosition);
	INSTALL_HANDLER (StdRead,			StdWrite,				horizontalNonDisplayPeriod);
	INSTALL_HANDLER (StdRead,			StdWrite,				FPFRAMEStartPosition);
	INSTALL_HANDLER (GetVertNonDisplay, StdWrite,				verticalNonDisplayPeriod);
	INSTALL_HANDLER (StdRead,			StdWrite,				MODRate);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen1StartAddressLSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen1StartAddressMSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen1StartAddressMSBit);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen2StartAddressLSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen2StartAddressMSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen2StartAddressMSBit);
	INSTALL_HANDLER (StdRead,			StdWrite,				memoryAddressOffset);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen1VerticalSizeLSB);
	INSTALL_HANDLER (StdRead,			StdWrite,				screen1VerticalSizeMSB);
	INSTALL_HANDLER (StdRead,			SetCLUTAddr,			lookUpTableAddress);
	INSTALL_HANDLER (GetCLUTEntry,		SetCLUTEntry,			lookUpTableData);
	INSTALL_HANDLER (StdRead,			StdWrite,				GPIOConfigurationControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				GPIOStatusControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				scratchPad);
	INSTALL_HANDLER (StdRead,			StdWrite,				portraitMode);
	INSTALL_HANDLER (StdRead,			StdWrite,				lineByteCountRegister);
	INSTALL_HANDLER (StdRead,			StdWrite,				unusual);
	INSTALL_HANDLER (StdRead,			StdWrite,				testMode);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLong
// ---------------------------------------------------------------------------

uae_u32 SED1375Bank::GetLong (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375LongRead]++;
	if (iAddress & 2)
		gMemoryAccess[kSED1375LongRead2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETLONG (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = SED1375Reg::UnsupportedRead;

	if (offset < sed1375VideoMemSize)			// Memory access is in VRAM area?
	{
		return do_get_mem_long (((char*) gVideoMem) + offset);
	}

	if (offset >= sed1375RegisterOffset)
	{
		fn = gReadFunctions[offset - sed1375RegisterOffset];
	}

	assert (fn);
	return fn (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetWord
// ---------------------------------------------------------------------------

uae_u32 SED1375Bank::GetWord (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375WordRead]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETWORD (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = SED1375Reg::UnsupportedRead;

	if (offset < sed1375VideoMemSize)			// Memory access is in VRAM area?
	{
		return do_get_mem_word(((char*) gVideoMem) + offset);
	}

	if (offset >= sed1375RegisterOffset)
	{
		fn = gReadFunctions[offset - sed1375RegisterOffset];
	}

	assert (fn);
	return fn (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetByte
// ---------------------------------------------------------------------------

uae_u32 SED1375Bank::GetByte (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375ByteRead]++;
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETBYTE (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	ReadFunction	fn = SED1375Reg::UnsupportedRead;

	if (offset < sed1375VideoMemSize)			// Memory access is in VRAM area?
	{
		return do_get_mem_byte(((char*) gVideoMem) + offset);
	}

	if (offset >= sed1375RegisterOffset)
	{
		fn = gReadFunctions[offset - sed1375RegisterOffset];
	}

	assert (fn);
	return fn (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::SetLong
// ---------------------------------------------------------------------------

void SED1375Bank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375LongWrite]++;
	if (iAddress & 2)
		gMemoryAccess[kSED1375LongWrite2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTLONG (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = SED1375Reg::UnsupportedWrite;

	if (offset < sed1375VideoMemSize)		// Memory access is in VRAM area?
	{
		do_put_mem_long(((char*) gVideoMem) + offset, iLongValue);

		Screen::MarkDirty (iAddress, sizeof (long));
	}
	else if (offset >= sed1375RegisterOffset)
	{
		fn = gWriteFunctions[offset - sed1375RegisterOffset];

		assert (fn);
		fn (iAddress, 4, iLongValue);
	}

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::SetWord
// ---------------------------------------------------------------------------

void SED1375Bank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375WordWrite]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTWORD (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = SED1375Reg::UnsupportedWrite;

	if (offset < sed1375VideoMemSize)			// Memory access is in VRAM area?
	{
		do_put_mem_word(((char*) gVideoMem) + offset, iWordValue);

		Screen::MarkDirty (iAddress, sizeof (short));
	}
	else if (offset >= sed1375RegisterOffset)
	{
		fn = gWriteFunctions[offset - sed1375RegisterOffset];

		assert (fn);
		fn (iAddress, 2, iWordValue);
	}

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::SetByte
// ---------------------------------------------------------------------------

void SED1375Bank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kSED1375ByteWrite]++;
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTBYTE (WAITSTATES_SED1375);
#endif

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = SED1375Reg::UnsupportedWrite;

	if (offset < sed1375VideoMemSize)		// Memory access is in VRAM area?
	{
		do_put_mem_byte(((char*) gVideoMem) + offset, iByteValue);

		Screen::MarkDirty (iAddress, sizeof (char));
	}
	else if (offset >= sed1375RegisterOffset)
	{
		fn = gWriteFunctions[offset - sed1375RegisterOffset];

		assert (fn);
		fn (iAddress, 1, iByteValue);
	}

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::ValidAddress
// ---------------------------------------------------------------------------

int SED1375Bank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	UNUSED_PARAM(iSize)

	uae_u32 		offset = iAddress - kMemoryStart;
	WriteFunction	fn = SED1375Reg::UnsupportedWrite;

	if (offset < sed1375VideoMemSize)			// Memory access is in VRAM area?
		fn = SED1375Reg::StdWrite;
	else if (offset >= sed1375RegisterOffset)
		fn = gWriteFunctions[offset - sed1375RegisterOffset];

	int result = (fn != SED1375Reg::UnsupportedWrite);

	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* SED1375Bank::GetRealAddress (uaecptr iAddress)
{
	uae_u32 offset = iAddress - kMemoryStart;

	if (offset < sed1375RegisterOffset)
		return (uae_u8*) ((uae_u32) gVideoMem + offset);

	return ((uae_u8*) &g1375Regs) + (offset - sed1375RegisterOffset);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetMetaAddress
// ---------------------------------------------------------------------------

uae_u8* SED1375Bank::GetMetaAddress (uaecptr iAddress)
{
	uae_u32 offset = iAddress - kMemoryStart;

	if (offset < sed1375RegisterOffset)
		return (uae_u8*) ((uae_u32) gMetaMem + offset);

	return ((uae_u8*) &g1375Regs) + (offset - sed1375RegisterOffset);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::InvalidAccess
// ---------------------------------------------------------------------------

void SED1375Bank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::PreventedAccess
// ---------------------------------------------------------------------------

void SED1375Bank::PreventedAccess (uaecptr iAddress, long size, Bool forRead)
{
	Memory::PreventedAccess (iAddress, size, forRead, Errors::kRegisterAccess);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDDepth
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDDepth (void)
{
	Byte depth = (READ_REGISTER (mode1) & sed1375BPP256Colors);

	switch (depth)
	{
		case sed1375BPP256Colors:
			return 8;

		case sed1375BPP16Colors:
			return 4;

		case sed1375BPP4Colors:
			return 2;

		default:
			return 1;
	}
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDRowBytes
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDRowBytes (void)
{
	int 	depth = GetLCDDepth ();
	Byte	hSize = GetLCDWidth ();

	return (int) (hSize / (8 / depth));
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDWidth
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDWidth (void)
{
	return (int) ((READ_REGISTER(horizontalPanelSize) + 1) * 8);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDHeight
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDHeight (void)
{
	return (int) ((READ_REGISTER(verticalPanelSizeMSB) << 8) |
						READ_REGISTER(verticalPanelSizeLSB)) + 1;
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDStartAddr
// ---------------------------------------------------------------------------

uaecptr SED1375Bank::GetLCDStartAddr (void)
{
	// The screen address is word-offset based. We must convert to a physical address.

	uae_u32 byteOffset =	(READ_REGISTER(screen1StartAddressMSBit) << 16) |
							(READ_REGISTER(screen1StartAddressMSB) << 8) |
							 READ_REGISTER(screen1StartAddressLSB);

	return kMemoryStart + (byteOffset * 2);
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDStartAddr
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDPalette (RGBType* scrPalette)
{
	int 	depth = SED1375Bank::GetLCDDepth ();
	int 	totalEntries = (1 << depth);

	if (gClutDirty)
	{
		for (int ii = 0; ii < totalEntries; ++ii)
		{
			uae_u16 curEntry = gClutData[ii];
			uae_u8	color;

			color = (uae_u8) ((curEntry & kCLUTRedMask) >> 4);
			scrPalette[ii].fRed = color + (color >> 4);

			color = (uae_u8) ((curEntry & kCLUTGreenMask) >> 0);
			scrPalette[ii].fGreen = color + (color >> 4);

			color = (uae_u8) ((curEntry & kCLUTBlueMask) << 4);
			scrPalette[ii].fBlue = color + (color >> 4);
		}

		gClutDirty = false;
	}

	return totalEntries;
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::GetLCDOffset
// ---------------------------------------------------------------------------

int SED1375Bank::GetLCDOffset (void)
{
	return 0;
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::LCDIsOn
// ---------------------------------------------------------------------------

Bool SED1375Bank::LCDIsOn (void)
{
	return (READ_REGISTER (mode1) & sed1375DisplayBlank) == 0;
}


// ---------------------------------------------------------------------------
//		 SED1375Bank::BacklightIsOn
// ---------------------------------------------------------------------------

Bool SED1375Bank::BacklightIsOn (void)
{
	return true;			// The Backlight is always on for Austin units.
}


#pragma mark -

// ===========================================================================
//		 Register Accessors
// ===========================================================================

// ---------------------------------------------------------------------------
//		 SED1375Reg::SetHandler
// ---------------------------------------------------------------------------

void SED1375Reg::SetHandler (ReadFunction read, WriteFunction write,
							uae_u32 start, int count)
{
	int index = start - (kMemoryStart + sed1375RegisterOffset);
	for (int ii = 0; ii < count; ++ii, ++index)
	{
		gReadFunctions[index] = read;
		gWriteFunctions[index] = write;
	}
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::UnsupportedRead
// ---------------------------------------------------------------------------

uae_u32 SED1375Reg::UnsupportedRead (uaecptr address, int size)
{
	if (!CEnableFullAccess::AccessOK ())
	{
		SED1375Bank::InvalidAccess (address, size, true);
	}

	return ~0;
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::UnsupportedWrite
// ---------------------------------------------------------------------------

void SED1375Reg::UnsupportedWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(value)

	if (!CEnableFullAccess::AccessOK ())
	{
		SED1375Bank::InvalidAccess (address, size, false);
	}
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::StdRead
// ---------------------------------------------------------------------------

uae_u32 SED1375Reg::StdRead (uaecptr address, int size)
{
	if (size == 1)
		return do_get_mem_byte (SED1375Bank::GetRealAddress (address));

	if (size == 2)
		return do_get_mem_word (SED1375Bank::GetRealAddress (address));

	return do_get_mem_long (SED1375Bank::GetRealAddress (address));
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::StdWrite
// ---------------------------------------------------------------------------

void SED1375Reg::StdWrite (uaecptr address, int size, uae_u32 value)
{
	if (size == 1)
		do_put_mem_byte (SED1375Bank::GetRealAddress (address), value);

	else if (size == 2)
		do_put_mem_word (SED1375Bank::GetRealAddress (address), value);

	else
		do_put_mem_long (SED1375Bank::GetRealAddress (address), value);
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::NullWrite
// ---------------------------------------------------------------------------

void SED1375Reg::NullWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	UNUSED_PARAM(value)

	// Do nothing (for read-only registers).
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::GetVertNonDisplay
// ---------------------------------------------------------------------------

uae_u32 SED1375Reg::GetVertNonDisplay (uaecptr address, int size)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	// Always set the vertical non-display status high since in the real
	// hardware, the ROM will check this flag in order to write the CLUT
	// regsiters.

	return READ_REGISTER (verticalNonDisplayPeriod) | sed1375VerticalNonDisplayStatus;
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::SetCLUTAddr
// ---------------------------------------------------------------------------

void SED1375Reg::SetCLUTAddr (uaecptr address, int size, uae_u32 value)
{
	SED1375Reg::StdWrite (address, size, value);

	value &= 0x0FF;

	gClutData[value] &= kCLUTColorsMask;				// Update the rgb index
	gClutData[value] |= kCLUTIndexRed;
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::GetCLUTEntry
// ---------------------------------------------------------------------------

uae_u32 SED1375Reg::GetCLUTEntry (uaecptr address, int size)
{
	assert (size == 1);

	if (size != 1)
		return 0;			// Error case.

	uae_u8	clutIndex = READ_REGISTER (lookUpTableAddress); 	// Get the LUT Addr.
	uae_u16	clutEntry = gClutData[clutIndex];					// Get the entry.
	uae_u8	colorData;

	if ((clutEntry & kCLUTIndexRed) != 0)
	{
		colorData = (uae_u8) ((clutEntry & kCLUTRedMask) >> 4); 	// Get the 4 bits of red.

		gClutData[clutIndex] =										// Update the next rgb index
			(gClutData[clutIndex] & kCLUTColorsMask) | kCLUTIndexGreen;
	}
	else if ((clutEntry & kCLUTIndexGreen) != 0)
	{
		colorData = (uae_u8) (clutEntry & kCLUTGreenMask);			// Get the 4 bits of green

		gClutData[clutIndex] =										// Update the next rgb index
			(gClutData[clutIndex] & kCLUTColorsMask) | kCLUTIndexBlue;
	}
	else
	{
		colorData = (uae_u8) ((clutEntry & kCLUTBlueMask) << 4);	// Get the 4 bits of blue.

		address = (uaecptr) (sed1375RegsAddr + offsetof (SED1375RegsType, lookUpTableAddress));
		SED1375Reg::SetCLUTAddr (address, 1, (clutIndex + 1) & 0xFF);
	}

	return colorData;
}


// ---------------------------------------------------------------------------
//		 SED1375Reg::SetCLUTEntry
// ---------------------------------------------------------------------------

void SED1375Reg::SetCLUTEntry (uaecptr address, int size, uae_u32 value)
{
	assert (size == 1);

	if (size != 1)
		return; 		// Error case.

	uae_u8	clutIndex = READ_REGISTER (lookUpTableAddress); 	// Get the LUT Addr.
	uae_u16	clutEntry = gClutData[clutIndex];					// Get the entry.

	uae_u8	newColor = (uae_u8) (value & 0x00F0);

	if (clutEntry & kCLUTIndexRed)
	{
		gClutData[clutIndex] &= ~kCLUTRedMask;			// Clear out old red bits.
		gClutData[clutIndex] |= newColor << 4;			// Save in new red bits.

		gClutData[clutIndex] =							// Update the rgb index
			(gClutData[clutIndex] & kCLUTColorsMask) | kCLUTIndexGreen;
	}
	else if (clutEntry & kCLUTIndexGreen)
	{
		gClutData[clutIndex] &= ~kCLUTGreenMask;		// Clear out old red bits.
		gClutData[clutIndex] |= newColor;				// Save in new green bits.

		gClutData[clutIndex] =							// Update the rgb index
			(gClutData[clutIndex] & kCLUTColorsMask) | kCLUTIndexBlue;
	}
	else
	{
		gClutData[clutIndex] &= ~kCLUTBlueMask; 		// Clear out old red bits.
		gClutData[clutIndex] |= newColor >> 4;			// Save in new blue bits.

		address = (uaecptr) (sed1375RegsAddr + offsetof (SED1375RegsType, lookUpTableAddress));
		SED1375Reg::SetCLUTAddr (address, 1, (clutIndex + 1) & 0xFF);
	}

	gClutDirty = true;
	Screen::InvalidateAll();
}
