 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Wed Nov 26 1997                               */
 /* --------------------------------------------------------------------  */
 /* Join units by looking at the analysis.                                */
 /*                                                                       */
 /*************************************************************************/



#include "festival.h"
#include "Unit.h"
#include "FramesJoin.h"
#include "FramesUnit.h"
#include "frames_unit_join.h"
#include "module_support.h"

static ModuleDescription description =
{
  "frames_unit_join", 1.0,
  "CSTR",
  "Richard Caley <rjc@cstr.ed.ac.uk>",
  {
    "Look at the analysis to see how to join units.",
    NULL
  },
  {
    { "Segment",	"Contains the segments." },
    { "Unit",		"The units which are to be joined." },
    { NULL,NULL}
  },
  { {NULL,NULL}
  },
  {
    { "Join",		"Final stream of joins." },
    {NULL,NULL}
  },
  {
    { "VerboseJoining",	mpt_bool,	"nil",	"Print trace of Joining process." },
    {NULL,NULL,NULL,NULL}
  }
};

static inline float fabs(float n) { return n>0.0?n:-n; }


// REORG 2 pt - changed coef0 to lpc_0.

static float distance(EST_Track &left, int li,
		      EST_Track &right, int ri)
{
    int order = get_order(left);
    float d = 0.0;

    for(int c=0; c <= order/3; c++)
	d += fabs(left.a(li, channel_lpc_0, c) - right.a(ri, channel_lpc_0, c));

    return d;
}

static void find_join(EST_Track &left, int left_start_frame, int &left_cut,
		      EST_Track &right, int right_end_frame, int &right_cut)

{
  // guess unless we find a better
  left_cut = (left.num_frames()+left_start_frame)/2;
  right_cut = (right_end_frame)/2;

  float best_distance = distance(left, left_cut, right, right_cut);

  int lstart = left_start_frame + (left.num_frames()-left_start_frame)/3;
  int lend = left_start_frame + (left.num_frames()-left_start_frame)*2/3;

  int rstart = right_end_frame/3;
  int rend = right_end_frame*2/3;
  
  for(int li=lstart; li<=lend; li++)
    for(int ri=rstart; ri<=rend; ri++)
      {
	float d = distance(left, li, right, ri);
	if (d < best_distance)
	  {
	    best_distance=d;
	    left_cut = li;
	    right_cut = ri;
	  }
      }
}

LISP frames_unit_join(LISP args)
{
  EST_Utterance *utt;

  EST_String segment_stream_name("Segment");
  EST_String    unit_stream_name("Unit");
  EST_String    join_stream_name("Join");

  EST_Stream *segment_stream=NULL, *unit_stream=NULL, *join_stream=NULL;

  unpack_module_args(args, 
		     utt, 
		     segment_stream_name, segment_stream, sat_existing,
		     unit_stream_name, unit_stream, sat_existing,
		     join_stream_name, join_stream, sat_replace);

  bool verbose = bool_parameter_get("VerboseJoining");

  EST_Stream_Item *unitem, *last;
  int last_endi=0;

  for(last=NULL,unitem=unit_stream->head(); unitem; unitem=next(unitem))
    {
      EST_ORelation *links = &(unitem->rlink(segment_stream_name));
      int starti = links->nth(0);
      int endi = links->nth(1);
      if (last && starti <= last_endi)
	{
	  if (verbose)
	    cout << "overlap " << last_endi << " " << starti << " " << endi << "\n";

	  Unit *left_unit = (Unit *)last->f("Unit").ptr();
	  Unit *right_unit = (Unit *)unitem->f("Unit").ptr();
	  

	  if (verbose)
	    cout << "    units " << left_unit->name() << " " << right_unit->name() << "\n";

	  if (!left_unit->property("frames_subtype"))
	    {
	      cwarn << "bad unit type " << left_unit->subtype_name() <<"\n";
	      continue;
	    }
	  if (!right_unit->property("frames_subtype"))
	    {
	      cwarn << "bad unit type " << right_unit->subtype_name() <<"\n";
	      continue;
	    }


	  EST_Track *left_an = ((FramesUnit *)left_unit)->rfc();
	  EST_Track *right_an = ((FramesUnit *)right_unit)->rfc();

	  EST_TVector<int> *left_segs = ((FramesUnit *)left_unit)->segment_ends();
	  EST_TVector<int> *right_segs = ((FramesUnit *)right_unit)->segment_ends();

	  int left_pos=0, right_pos=0;

	  find_join(*left_an, (*left_segs)(left_segs->length()-2), left_pos,
		    *right_an, (*right_segs)(1), right_pos);

	  if (verbose)
	    cout << "    joins "  
		 << (*left_segs)(left_segs->length()-2) << " <"
		 << left_pos << "> "
		 << (*left_segs)(left_segs->length()-1) << " | "
		 << "0" << " <"
		 << right_pos <<  "> " 
		 << (*right_segs)(1)
		 << "\n";

	  FramesJoin *join = new FramesJoin(left_unit, left_pos,
					    right_unit, right_pos);

	  EST_Stream_Item item;
	  item.init("Join");
	  item.set_name(segment_stream->item(starti)->name());
	  item.f.set("Join", (void *)join, 
		     Unit::decrease_reference_count);

	  ((Join *)join)->gc_ref();
	  link(item, *last);
	  link(item, *unitem);
	  item.f.set("type", join->subtype_name());
	  item.f.set("end_index", join->end_index());
	  item.f.set("start_index", join->start_index());
	  join_stream->append(item);
	}

      last = unitem;
      last_endi = endi;
    }

  return NIL;
}

void frames_unit_join_init(void)

{
  proclaim_module("frames_unit_join", &description);

  init_module_subr("frames_unit_join", frames_unit_join, &description);
}

