//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This class provides a standard API for the screen.  This file contains
  the X Windows implementation of the API.

  @author  Bradford W. Mott
  @version $Id: TermX11.cxx,v 1.2 1997/05/17 19:00:09 bwmott Exp $
*/

#include <assert.h>
#include <stdio.h>
#include <iostream.h>

#include "colors.h"
#include "Term.hxx"
#include "TIA.hxx"

// Indicates if an instance is alive
bool Terminal::ourInstanceCreated = false;

// Contains the state of each of the events based on the keyboard
static volatile uLong theKeyboardEventState[TERMINAL_NUMBER_OF_EVENTS];

// Contains the state of each of the events based on the joystick
static volatile uLong theJoystickEventState[TERMINAL_NUMBER_OF_EVENTS];

#ifdef LINUX_JOYSTICK
  #include <unistd.h>
  #include <fcntl.h>
  #include <linux/joystick.h>

  // File descriptors for the joystick devices
  int theLeftJoystickFd;
  int theRightJoystickFd;
#endif

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

Display* theDisplay;
int theScreen;
Visual* theVisual;
Window theWindow;

// A graphic context for each of the 2600's colors
GC theGCTable[256];

// Enumeration of the possible window sizes
enum WindowSize {Small = 1, Medium = 2, Large = 3} ;

// Indicates the current size of the window
WindowSize theWindowSize = Medium;

// Indicates the width and height based on the property information
uLong theWidth;
uLong theHeight;
uLong theXStart;

// Indicates if the entire frame should be redrawn
bool theRedrawEntireFrameFlag = true;


//============================================================================
// Constructor
//============================================================================
Terminal::Terminal(const Properties& properties)
    : myProperties(&properties)
{
  assert(!ourInstanceCreated);
  ourInstanceCreated = true;

  // Clear the terminal event state
  for(int i = 0;i < TERMINAL_NUMBER_OF_EVENTS; i++)
  {
    theKeyboardEventState[i] = 0;
    theJoystickEventState[i] = 0;
  }

#ifdef LINUX_JOYSTICK
  // Open the joystick devices 
  theLeftJoystickFd = open("/dev/js0", O_RDONLY);
  theRightJoystickFd = open("/dev/js1", O_RDONLY);
#endif

  // Get the desired width and height of the display
  theWidth = myProperties->integer("Display.Width");
  theXStart = myProperties->integer("Display.XStart");
  theHeight = myProperties->integer("Display.Height");

  // Figure out the desired size of the window
  int width = theWidth;
  int height = theHeight;
  if(theWindowSize == Small)
  {
    width *= 2;
  }
  else if(theWindowSize == Medium)
  {
    width *= 4;
    height *= 2;
  }
  else
  {
    width *= 6;
    height *= 3;
  }

  // Open a connection to the X server
  theDisplay = XOpenDisplay(NULL);
  if(theDisplay == NULL)
  {
    cerr << "ERROR: Cannot open X-Windows display.\n";
    exit(1);
  }

  theScreen = DefaultScreen(theDisplay);
  theVisual = DefaultVisual(theDisplay, theScreen);
  Window rootWindow = RootWindow(theDisplay, theScreen);

  theWindow = XCreateSimpleWindow(theDisplay, rootWindow, 0, 0,
      width, height, CopyFromParent, CopyFromParent, 
      BlackPixel(theDisplay, theScreen));

  // Disable the mouse pointer if the paddle is being used
  if(strcmp(myProperties->find("Controller.Paddle0"), "None") != 0)
  {
    static char cursor_bits[] = {0x00};
    Pixmap pixmap = XCreatePixmapFromBitmapData(theDisplay, theWindow, 
       cursor_bits, 1, 1, 0, 0, 1);

    XColor color;
    Cursor cursor = XCreatePixmapCursor(theDisplay, pixmap, pixmap, 
        &color, &color, 0, 0);
    XDefineCursor(theDisplay, theWindow, cursor);
  }

  XSizeHints hints;
  hints.flags = PSize | PMinSize | PMaxSize;
  hints.min_width = hints.max_width = hints.width = width;
  hints.min_height = hints.max_height = hints.height = height;

  // Set window and icon name, size hints and other properties 
  char name[512];
  strcpy(name, "Stella: \"");
  strcat(name, myProperties->find("Cartridge.Name"));
  strcat(name, "\"");

  XSetStandardProperties(theDisplay, theWindow, name, name, None, 0, 0, &hints);

  // Allocate colors in the default colormap
  for(int t = 0; t < 256; t += 2)
  {
    XColor color;

    color.red = (theColorTable[t] & 0x00ff0000) >> 8 ;
    color.green = (theColorTable[t] & 0x0000ff00) ;
    color.blue = (theColorTable[t] & 0x000000ff) << 8;
    color.flags = DoRed | DoGreen | DoBlue;

    XAllocColor(theDisplay, DefaultColormap(theDisplay, theScreen), &color);

    XGCValues values;
    values.foreground = color.pixel;
    theGCTable[t] = XCreateGC(theDisplay, theWindow, GCForeground, &values);
    theGCTable[t + 1] = theGCTable[t];
  }

  XSelectInput(theDisplay, theWindow, ExposureMask);
  XMapWindow(theDisplay, theWindow);

  XEvent event;
  do
  {
    XNextEvent(theDisplay, &event);
  } while (event.type != Expose);

  uLong mask = ExposureMask | KeyPressMask | KeyReleaseMask;

  // Enable mouse events if the paddle is being used
  if(strcmp(myProperties->find("Controller.Paddle0"), "None") != 0)
    mask = mask | (PointerMotionMask | ButtonPressMask | ButtonReleaseMask);

  XSelectInput(theDisplay, theWindow, mask);
}

//============================================================================
// Destructor
//============================================================================
Terminal::~Terminal()
{
#ifdef LINUX_JOYSTICK
  if(theLeftJoystickFd >= 0)
    close(theLeftJoystickFd);

  if(theRightJoystickFd >= 0)
    close(theRightJoystickFd);
#endif

  ourInstanceCreated = false;
}

//============================================================================
// Update the screen with the TIA frame
//============================================================================
void Terminal::update(TIA& tia)
{
  uByte* currentFrame = tia.currentFrameBuffer();
  uByte* previousFrame = tia.previousFrameBuffer();
  uWord screenMultiple = (uWord)theWindowSize;

  struct Rectangle
  {
    uByte color;
    uWord x, y, width, height;
  } rectangles[2][160];

  // This array represents the rectangles that need displaying
  // on the current scanline we're processing
  Rectangle* currentRectangles = rectangles[0];

  // This array represents the rectangles that are still active
  // from the previous scanlines we have processed
  Rectangle* activeRectangles = rectangles[1];

  // Indicates the number of active rectangles
  uWord activeCount = 0;


  // This update procedure requires theWidth and theXStart to be
  // multiples of four.  This is validated when the properties are loaded.
  for(uWord y = 0; y < theHeight; ++y)
  {
    // Indicates the number of current rectangles
    uWord currentCount = 0;

    // Look at four pixels at a time to see if anything has changed
    uLong* current = (uLong*)(currentFrame + theXStart); 
    uLong* previous = (uLong*)(previousFrame + theXStart);

    for(uWord x = 0; x < theWidth; x += 4, ++current, ++previous)
    {
      // Has something changed in this set of four pixels?
      if((*current != *previous) || theRedrawEntireFrameFlag)
      {
        uByte* c = (uByte*)current;
        uByte* p = (uByte*)previous;

        // Look at each of the bytes that make up the uLong
        for(uWord i = 0; i < 4; ++i, ++c, ++p)
        {
          // See if this pixel has changed
          if((*c != *p) || theRedrawEntireFrameFlag)
          {
            // Can we extend a rectangle or do we have to create a new one?
            if((currentCount != 0) && 
               (currentRectangles[currentCount - 1].color == *c) &&
               ((currentRectangles[currentCount - 1].x + 
                 currentRectangles[currentCount - 1].width) == (x + i)))
            {
              currentRectangles[currentCount - 1].width += 1;
            }
            else
            {
              currentRectangles[currentCount].x = x + i;
              currentRectangles[currentCount].y = y;
              currentRectangles[currentCount].width = 1;
              currentRectangles[currentCount].height = 1;
              currentRectangles[currentCount].color = *c;
              currentCount++;
            }
          }
        }
      }
    }

    // Merge the active and current rectangles flushing any that are of no use
    uWord activeIndex = 0;

    for(uWord t = 0; (t < currentCount) && (activeIndex < activeCount); ++t)
    {
      Rectangle& current = currentRectangles[t];
      Rectangle& active = activeRectangles[activeIndex];

      // Can we merge the current rectangle with an active one?
      if((current.x == active.x) && (current.width == active.width) &&
         (current.color == active.color))
      {
        current.y = active.y;
        current.height = active.height + 1;

        ++activeIndex;
      }
      // Is it impossible for this active rectangle to be merged?
      else if(current.x >= active.x)
      {
        // Flush the active rectangle
        XFillRectangle(theDisplay, theWindow, theGCTable[active.color],
           active.x * 2 * screenMultiple, active.y * screenMultiple, 
           active.width * 2 * screenMultiple, active.height * screenMultiple);

        ++activeIndex;
      }
    }

    // Flush any remaining active rectangles
    for(uWord s = activeIndex; s < activeCount; ++s)
    {
      Rectangle& active = activeRectangles[s];

      XFillRectangle(theDisplay, theWindow, theGCTable[active.color],
         active.x * 2 * screenMultiple, active.y * screenMultiple, 
         active.width * 2 * screenMultiple, active.height * screenMultiple);
    }

    // We can now make the current rectangles into the active rectangles
    Rectangle* tmp = currentRectangles;
    currentRectangles = activeRectangles;
    activeRectangles = tmp;
    activeCount = currentCount;
 
    currentFrame += 160;
    previousFrame += 160;
  }

  // Flush any rectangles that are still active
  for(uWord t = 0; t < activeCount; ++t)
  {
    Rectangle& active = activeRectangles[t];

    XFillRectangle(theDisplay, theWindow, theGCTable[active.color],
       active.x * 2 * screenMultiple, active.y * screenMultiple, 
       active.width * 2 * screenMultiple, active.height * screenMultiple);
  }

  // The frame doesn't need to be completely redrawn anymore
  theRedrawEntireFrameFlag = false;

  // Handle any events that have occured
  handleEvents();
}

//============================================================================
// Answer the state of the given event
//============================================================================
uLong Terminal::eventState(Event event)
{
  // If the value is not set by joystick then return the keyboard's value
  if(theJoystickEventState[(int)event] == 0)
    return theKeyboardEventState[(int)event];
  else
    return theJoystickEventState[(int)event];
}

//============================================================================
// Process any events related to the terminal
//============================================================================
void Terminal::handleEvents()
{
  struct Switches
  {
    KeySym scanCode;
    Terminal::Event eventCode;
  };

  static Switches list[] = {
    { XK_Down,        Terminal::LeftJoystickDown },
    { XK_Up,          Terminal::LeftJoystickUp },
    { XK_Left,        Terminal::LeftJoystickLeft },
    { XK_Right,       Terminal::LeftJoystickRight },
    { XK_space,       Terminal::LeftJoystickFire }, 
    { XK_Return,      Terminal::LeftJoystickFire }, 
    { XK_h,           Terminal::RightJoystickDown },
    { XK_H,           Terminal::RightJoystickDown },
    { XK_y,           Terminal::RightJoystickUp },
    { XK_Y,           Terminal::RightJoystickUp },
    { XK_g,           Terminal::RightJoystickLeft },
    { XK_G,           Terminal::RightJoystickLeft },
    { XK_j,           Terminal::RightJoystickRight },
    { XK_J,           Terminal::RightJoystickRight },
    { XK_z,           Terminal::RightJoystickFire }, 
    { XK_Z,           Terminal::RightJoystickFire }, 
    { XK_c,           Terminal::Color },
    { XK_C,           Terminal::Color },
    { XK_b,           Terminal::BlackAndWhite },
    { XK_B,           Terminal::BlackAndWhite },
    { XK_1,           Terminal::LeftDifficultyA },
    { XK_2,           Terminal::LeftDifficultyB },
    { XK_3,           Terminal::RightDifficultyA },
    { XK_4,           Terminal::RightDifficultyB },
    { XK_s,           Terminal::Select },
    { XK_S,           Terminal::Select },
    { XK_r,           Terminal::Reset },
    { XK_R,           Terminal::Reset },
    { XK_p,           Terminal::Pause },
    { XK_P,           Terminal::Pause },
    { XK_q,           Terminal::Quit },
    { XK_Q,           Terminal::Quit }
  };

  XEvent event;

  while(XCheckWindowEvent(theDisplay, theWindow, 
      ExposureMask | KeyPressMask | KeyReleaseMask | ButtonPressMask |
      ButtonReleaseMask | PointerMotionMask, &event))
  {
    char buffer[20];
    KeySym key;
    XComposeStatus compose;

    if((event.type == KeyPress) || (event.type == KeyRelease))
    {
      XLookupString(&event.xkey, buffer, 20, &key, &compose);
      if((key == XK_equal) && (event.type == KeyPress))
      {
        if(theWindowSize == Small)
          theWindowSize = Medium;
        else if(theWindowSize == Medium)
          theWindowSize = Large;
        else
          theWindowSize = Small;

        // Figure out the desired size of the window
        int width = theWidth;
        int height = theHeight;
        if(theWindowSize == Small)
        {
          width *= 2;
        }
        else if(theWindowSize == Medium)
        {
          width *= 4;
          height *= 2;
        }
        else
        {
          width *= 6;
          height *= 3;
        }

        XWindowChanges values;
        values.width = width;
        values.height = height; 
        XConfigureWindow(theDisplay, theWindow, CWWidth | CWHeight, &values);
 
        theRedrawEntireFrameFlag = true;
      }
      else
      { 
        for(unsigned int i = 0; i < sizeof(list) / sizeof(Switches); ++i)
        { 
          if(list[i].scanCode == key)
          {
            theKeyboardEventState[(int)list[i].eventCode] = 
                (event.type == KeyPress) ? 1 : 0;
          }
        }
      }
    }
    else if(event.type == MotionNotify)
    {
      if(theWindowSize == Small)
      {
        int x = event.xmotion.x - 32;

        if(x < 0)
          x = 0;
        else if(x > 255)
          x = 255;

        theKeyboardEventState[(int)Terminal::PaddleZeroPosition] = 512 - x * 2;
      }
      else if(theWindowSize == Medium)
      {
        int x = event.xmotion.x - 64;

        if(x < 0)
          x = 0;
        else if(x > 511)
          x = 511;

        theKeyboardEventState[(int)Terminal::PaddleZeroPosition] = 512 - x;
      }
      else
      {
        int x = event.xmotion.x - 96;

        if(x < 0)
          x = 0;
        else if(x > 768)
          x = 768;

        int X = (int)((float)x / 1.5);
        theKeyboardEventState[(int)Terminal::PaddleZeroPosition] = 512 - X;
      }
    }
    else if(event.type == ButtonPress) 
    {
      theKeyboardEventState[(int)Terminal::PaddleZeroFire] = 1;
    }
    else if(event.type == ButtonRelease)
    {
      theKeyboardEventState[(int)Terminal::PaddleZeroFire] = 0;
    }
    else if(event.type == Expose)
    {
      theRedrawEntireFrameFlag = true;
    }
  }

#ifdef LINUX_JOYSTICK
  // Read joystick values and modify event states
  if(theLeftJoystickFd >= 0)
  {
    struct JS_DATA_TYPE joystick;
    read(theLeftJoystickFd, &joystick, JS_RETURN);

    theJoystickEventState[(int)Terminal::LeftJoystickFire] = 
        (joystick.buttons & 0x01) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickLeft] = 
        (joystick.x < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickRight] = 
        (joystick.x > 750) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickUp] = 
        (joystick.y < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickDown] = 
        (joystick.y > 750) ? 1 : 0;
  }

  if(theRightJoystickFd >= 0)
  {
    struct JS_DATA_TYPE joystick;
    read(theRightJoystickFd, &joystick, JS_RETURN);

    theJoystickEventState[(int)Terminal::RightJoystickFire] = 
        (joystick.buttons & 0x01) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickLeft] = 
        (joystick.x < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickRight] = 
        (joystick.x > 750) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickUp] = 
        (joystick.y < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickDown] = 
      (joystick.y > 750) ? 1 : 0;
  }
#endif
}

