
/*
 * SERVER.C	- /news/dserver.hosts management
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void CheckServerConfig(time_t t, int force);
Prototype void NNArticleRetrieveByMessageId(Connection *conn, const char *msgid);
Prototype void NNServerIdle(Connection *conn);
Prototype void NNServerTerminate(Connection *conn);
Prototype void NNFinishSReq(Connection *conn, const char *ctl);
Prototype void NNServerRequest(Connection *conn, const char *grp, const char *msgid, FILE *cache, int req);

Prototype int ServerTerminated;
Prototype int	NReadServers;
Prototype int	NReadServAct;
Prototype int	NWriteServers;
Prototype int	NWriteServAct;

void QueueServerReadRequest(void);
void QueueServerWriteRequest(void);
int AddServer(const char *host, int type, int port);
void NNServerPrime1(Connection *conn);
void NNServerPrime2(Connection *conn);
void NNServerConnect(Connection *conn);

ServReq	*SReadBase;
ServReq **PSRead = &SReadBase;
ServReq *SWriteBase;
ServReq **PSWrite = &SWriteBase;
int	NReadServers;
int	NReadServAct;
int	NWriteServers;
int	NWriteServAct;
int	ServerTerminated = 0;

/*
 * CheckServerConfig() - determine if configuration file has changed and
 *			 resynchronize with servers list.
 *
 *			 NOTE: a server cannot be destroyed until its current
 *			 pending request completes, but no new requests will
 *			 be queued to it during that time.
 */

void
setServerCloseFlag(ForkDesc *desc)
{
    Connection *conn = desc->d_Data;
    conn->co_Flags |= COF_CLOSESERVER;
    if (conn->co_Func == NNServerIdle)		/* wakeup idle server */
	FD_SET(desc->d_Fd, &RFds);
}

void 
CheckServerConfig(time_t t, int force)
{
    static struct stat St;
    struct stat st;
    char *path = zalloc(&SysMemPool, strlen(NewsHome) + 32);

    sprintf(path, "%s/dserver.hosts", NewsHome);

    printf("CheckServerConfig %d %d\n", force, ServerTerminated);

    if (force)
	ServerTerminated = 0;

    /*
     * Assuming we can stat the file, if we haven't checked the server config
     * before or the server config time is different from when we last checked
     * AND the server config time is not now (so we don't read the file while
     * someone is writing it), then read the configuration file (again).
     * I've also got some reverse-time-index protection in there.
     */

    if (force || (stat(path, &st) == 0 && 
	(St.st_mode == 0 || st.st_mtime != St.st_mtime) && 
	((long)(t - st.st_mtime) > 2 || (long)(t - st.st_mtime) < 0))
    ) {
	FILE *fi;
	if ((fi = xfopen("r", path)) != NULL) {
	    char buf[256];

	    St = st;

	    ScanThreads(THREAD_SPOOL, setServerCloseFlag);
	    ScanThreads(THREAD_POST, setServerCloseFlag);

	    while (fgets(buf, sizeof(buf), fi) != NULL) {
		char *host = strtok(buf, " \t\n");
		char *flags = "";
		char c;
		int addAsServer = 0;
		int addAsPoster = 0;
		int port = 119;

		if (host == NULL || host[0] == '#')
		    continue;
		if ((flags = strtok(NULL, " \t\n")) == NULL)
		    flags = "";
		while ((c = *flags) != 0) {
		    ForkDesc *desc = NULL;

		    switch(c) {
		    case 'p':	/* port			*/
			port = strtol(flags + 1, NULL, 0);
			if (port == 0)
			    port = 119;
			break;
		    case 's':	/* spool		*/
			if ((desc = FindThreadId(THREAD_SPOOL, host)) != NULL) {
			    Connection *conn = desc->d_Data;
			    conn->co_Flags &= ~COF_CLOSESERVER;
			} else {
			    addAsServer = 1;
			}
			break;
		    case 'o':	/* outgoing (post)	*/
			if ((desc = FindThreadId(THREAD_POST, host)) != NULL) {
			    Connection *conn = desc->d_Data;
			    conn->co_Flags &= ~COF_CLOSESERVER;
			} else {
			    addAsPoster = 1;
			}
			break;
		    default:
			/*
			 * ignore unknown flag or number
			 */
			break;
		    }
		    ++flags;
		}
		if (addAsServer) {
		    if (AddServer(host, THREAD_SPOOL, port) == 0) {
			++NReadServers;
			++NReadServAct;
		    }
		}
		if (addAsPoster) {
		    if (AddServer(host, THREAD_POST, port) == 0) {
			++NWriteServers;
			++NWriteServAct;
		    }
		}
	    }
	    fclose(fi);
	}
    }
}

int
AddServer(const char *host, int type, int port)
{
    ForkDesc *desc;
    int fd;
    struct sockaddr_in sin;
    Connection *conn;

    /*
     * connect() to the host (use asynchronous connect())
     */

    bzero(&sin, sizeof(sin));
    {
	struct hostent *he = gethostbyname(host);
        if (he != NULL) {
	    sin.sin_family = he->h_addrtype;
	    memmove(&sin.sin_addr, he->h_addr, he->h_length);
	} else if (strtol(host, NULL, 0) != 0) {
	    sin.sin_family = AF_INET;
	    sin.sin_addr.s_addr = inet_addr(host);
	} else {
	    logit(LOG_ERR, "hostname lookup failure: %s\n", host);
	    return(-1);
	}
    }
    sin.sin_port = htons(port);

    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	logit(LOG_ERR, "socket() call failed on host %s\n", host);
	return(-1);
    }
    {
	int on;
	setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
	/* 
	 * XXX set TxBufSize, RxBufSize, but different from bufsize options
	 * for client connections.. make these bigger.
	 */
    }
    fcntl(fd, F_SETFL, O_NONBLOCK);	/* asynchronous connect() */
    errno = 0;
    if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	if (errno != EINPROGRESS) {
	    close(fd);
	    logit(LOG_ERR, "connect() call failed on host %s: %s\n", host, strerror(errno));
	    return(-1);
	}
    }

    /*
     * add thread.  Preset d_Count to 1 to prevent the server
     * from being allocated for client requests until we get
     * the banner.
     */

    desc = AddThread(host, fd, -1, type, -1);
    FD_SET(desc->d_Fd, &RFds);
    conn = InitConnection(desc, NULL);
    desc->d_Count = 1;
    NNServerConnect(conn);
    return(0);
}

/*
 * QUEUESERVERREADREQUEST() - move requests to the appropriate server
 *
 */

void
QueueServerReadRequest(void)
{
    Connection *conn;
    ServReq    *sreq;
    ForkDesc   *desc;

    if ((desc = FindLeastUsedThread(THREAD_SPOOL, 1)) == NULL) {
	/* XXX panic */
	puts("qsrr panic");
	DumpThreads();
	++NReadServAct;
	return;
    }
    conn = desc->d_Data;
    sreq = SReadBase;
    if ((SReadBase = SReadBase->sr_Next) == NULL)
	PSRead = &SReadBase;
    conn->co_SReq = sreq;		/* server's co_SReq		*/
    conn->co_Desc->d_Count = 1;		/* server in use		*/
    sreq->sr_SConn = conn;		/* sreq assigned to server	*/
    ++NReadServAct;
    NNSpoolCommand1(conn);
}

/*
 * QUEUESERVERWRITEREQUEST() -
 *
 */

void
QueueServerWriteRequest(void)
{
    Connection *conn;
    ServReq    *sreq;
    ForkDesc   *desc;

    if ((desc = FindLeastUsedThread(THREAD_POST, 1)) == NULL) {
	/* XXX panic */
	puts("qswr panic");
	++NWriteServAct;
	return;
    }
    conn = desc->d_Data;
    sreq = SWriteBase;
    if ((SWriteBase = SWriteBase->sr_Next) == NULL)
	PSWrite = &SWriteBase;
    conn->co_SReq = sreq;
    conn->co_Desc->d_Count = 1;		/* server now in use 	*/
    ++NWriteServAct;
    NNPostCommand1(conn);
}

/*
 *
 */

void
FreeSReq(ServReq *sreq)
{

    /*
     * If cache file write in progress, abort it.  allow the
     * fclose() to release the lock after the truncation.
     *
     * We NULL the FILE * out even though we free the structure
     * so potential memory corruption doesn't mess with random 
     * (future) files.
     */
    if (sreq->sr_Cache != NULL) {
	fflush(sreq->sr_Cache);
	AbortCache(fileno(sreq->sr_Cache), sreq->sr_MsgId, 0);
	fclose(sreq->sr_Cache);
	sreq->sr_Cache = NULL;
    }

    zfreeStr(&SysMemPool, &sreq->sr_Group);
    zfreeStr(&SysMemPool, &sreq->sr_MsgId);

    zfree(&SysMemPool, sreq, sizeof(ServReq));
}

void
NNServerRequest(Connection *conn, const char *grp, const char *msgid, FILE *cache, int req)
{
    ServReq *sreq = zalloc(&SysMemPool, sizeof(ServReq));

    sreq->sr_CConn = conn;
    sreq->sr_SConn = NULL;	/* for clarity: not assigned to server yet */
    sreq->sr_Time = time(NULL);
    sreq->sr_Group = grp ? zallocStr(&SysMemPool, grp) : NULL;
    sreq->sr_MsgId = msgid ? zallocStr(&SysMemPool, msgid) : NULL;
    sreq->sr_Cache = cache;	/* may be NULL */

    conn->co_SReq = sreq;	/* client has active sreq		*/

    FD_CLR(conn->co_Desc->d_Fd, &RFds);

    if (req == SREQ_RETRIEVE) {
	*PSRead = sreq;
	PSRead = &sreq->sr_Next;

	while (NReadServAct < NReadServers && SReadBase != NULL) {
	    QueueServerReadRequest();
	}
    } else if (req == SREQ_POST) {
	*PSWrite = sreq;
	PSWrite = &sreq->sr_Next;

	while (NWriteServAct < NWriteServers && SWriteBase != NULL)
	    QueueServerWriteRequest();
    }
}

void
NNFinishSReq(Connection *conn, const char *ctl)
{
    ServReq *sreq;

    if ((sreq = conn->co_SReq)) {
	/*
	 * can be NULL if client was terminated
	 */
	if (sreq->sr_CConn) {
	    if (ctl)
		MBWrite(&sreq->sr_CConn->co_TMBuf, ctl, strlen(ctl));
	    sreq->sr_CConn->co_FCounter = 1;
	    sreq->sr_CConn->co_SReq = NULL;
	    NNCommand(sreq->sr_CConn);
	}
	/*
	 * set FCounter to 1 to prevent further recursion, which might
	 * feed back and screw up the state machine for this connection.
	 */
	conn->co_SReq = NULL;
	FreeSReq(sreq);
    }
    NNServerIdle(conn);
}

/*
 * we have to send a garbage command to prevent INN's nnrpd from timing
 * out in 60 seconds upon initial connect.
 */

void
NNServerPrime1(Connection *conn)
{
    MBPrintf(&conn->co_TMBuf, "mode thisbetterfail\r\n");
    NNServerPrime2(conn);
}

void
NNServerPrime2(Connection *conn)
{
    int len;
    char *buf;

    conn->co_Func = NNServerPrime2;
    conn->co_State = "prime2";

    /*
     * any return code or EOF.  0 means we haven't gotten the
     * return code yet.
     */

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) != 0) {
	NNServerIdle(conn);
    }
}

/*
 * NNSERVERIDLE() - server idle, wait for EOF
 */

void
NNServerIdle(Connection *conn)
{
    conn->co_Func  = NNServerIdle;
    conn->co_State = "sidle";

    /*
     * Check for an unexpected condition on server, i.e. data or
     * EOF on the input where we didn't expect any.
     */

    {
	int len;
	char *buf;

	if ((len = MBReadLine(&conn->co_RMBuf, &buf)) != 0) {
	    if (len > 0)
		logit(LOG_ERR, "Server closed connection: %s", buf);
	    else
		logit(LOG_ERR, "Server closed connection", buf);
	    conn->co_Flags |= COF_CLOSESERVER;
	}
    }

    /*
     * If server is no longer in use, see if we can queue another
     * client request to it.  If we wind up terminating the server,
     * the requests will be requeued so don't worry about that.
     */

    if (conn->co_Desc->d_Count) {
	conn->co_Desc->d_Count = 0;
	if (conn->co_Desc->d_Type == THREAD_POST)
	    --NWriteServAct;
	else
	    --NReadServAct;
        while (NReadServAct < NReadServers && SReadBase != NULL) {
            QueueServerReadRequest();
	}
        while (NWriteServAct < NWriteServers && SWriteBase != NULL) {
            QueueServerWriteRequest();
	}
    }
    if (conn->co_Flags & COF_CLOSESERVER) {
	NNServerTerminate(conn);
    }
}

/*
 * NNServerTerminate() - terminate a server connection.  Usually occurs
 *			 if the server goes down or the related process
 *			 on the server is killed.   Any client request
 *			 queued to the server is requeued to another
 *			 server.
 */

void
NNServerTerminate(Connection *conn)
{
    ServReq *sreq;

    if ((sreq = conn->co_SReq) != NULL) {
	sreq->sr_Next = NULL;
	sreq->sr_SConn = NULL;
	sreq->sr_Time = time(NULL);
	if (conn->co_Desc->d_Type == THREAD_POST) {
	    *PSWrite = sreq;
	    PSWrite = &sreq->sr_Next;
	} else {
	    *PSRead = sreq;
	    PSRead = &sreq->sr_Next;
	}
	conn->co_SReq = NULL;
    }

    /*
     * closeup the server.  If d_Count is non-zero we have
     * to cleanup our reference counts, but then we set d_Count
     * to ensure nothing else gets queued to the server
     * between calling NNTerminate() and the actual termination.
     */

    conn->co_Flags |= COF_CLOSESERVER;
    if (conn->co_Desc->d_Count != 0) {
	conn->co_Desc->d_Count = 0;
	if (conn->co_Desc->d_Type == THREAD_POST)
	    --NWriteServAct;
	else
	    --NReadServAct;
    }
    if (conn->co_Desc->d_Type == THREAD_POST)
	--NWriteServers;
    else
	--NReadServers;
    ServerTerminated = 1;
    conn->co_Desc->d_Count = 999;
    NNTerminate(conn);
}

/*
 * NNSERVERCONNECT() - initial connection, get startup message (co_UCounter
 *			starts out 1, we do not clear it and make the server
 *			available until we get the startup message)
 */

void
NNServerConnect(Connection *conn)
{
    char *buf;
    char *ptr;
    int len;
    int code;

    conn->co_Func  = NNServerConnect;
    conn->co_State = "sconn";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) <= 0) {
	if (len < 0) {
	    logit(LOG_ERR, "connect(%s) failed");
	    NNTerminate(conn);
	}
	return;
    }

    ptr = buf;
    code = strtol(ptr, NULL, 10);
    if (code == 200) {
	logit(LOG_ERR, "connect(%s) %s", conn->co_Desc->d_Id, buf);
    } else if (code == 201) {
	if (conn->co_Desc->d_Type == THREAD_POST)
	    logit(LOG_ERR, "connect(%s) %s", conn->co_Desc->d_Id, buf);
	else
	    logit(LOG_ERR, "connect(%s) %s", conn->co_Desc->d_Id, buf);
    } else {
	logit(LOG_ERR, "connect(%s) unrecognized banner: %s", conn->co_Desc->d_Id, buf);
	NNTerminate(conn);
	return;
    }
    NNServerPrime1(conn);
}

/*
 *  NNARTICLERETRIEVEBYMESSAGEID() - retrieve article by message-id
 *
 *	Retrieve an article by its message id and write the article
 *	to the specified connection.
 *
 *	(a) attempt to fetch the article from cache (positive or negative hit)
 *
 *	(b) initiate the state machine to attempt to fetch the article from a
 *	    remote server.
 *
 *	(c) on remote server fetch completion, cache the article locally
 *
 *	(d) place article in transmission buffer for connection, transmitting
 *	    it, then return to the command state.
 */

void
NNArticleRetrieveByMessageId(Connection *conn, const char *msgid)
{   
    FILE *cache = NULL;

    /*
     * (a) retrieve from cache if caching is enabled
     */

    if (CacheMode) {
	int valid;
	int size;
	int cfd;

	valid = OpenCache(msgid, &cfd, &size);

	if (valid > 0) {
	    /*
	     * good cache
	     */
	    const char *map;
	    printf("good cache\n");
	    if ((map = xmap(NULL, size, PROT_READ, MAP_SHARED, cfd, 0)) != NULL) {
		if (conn->co_ArtMode != COM_BODYNOSTAT) {
		    MBPrintf(&conn->co_TMBuf, "%03d 0 %s %s\r\n", 
			GoodRC(conn),
			msgid,
			GoodResId(conn)
		    );
		}
		if (conn->co_ArtMode != COM_STAT) {
		    DumpArticleFromCache(conn, map, size);
		    MBPrintf(&conn->co_TMBuf, ".\r\n");
		}
		xunmap((void *)map, size);
	    } else {
		if (conn->co_ArtMode == COM_BODYNOSTAT)
		    MBPrintf(&conn->co_TMBuf, "(article not available)\r\n.\r\n");
		else
		    MBPrintf(&conn->co_TMBuf, "430 No such article\r\n");
	    }
	    close(cfd);
	    NNCommand(conn);
	    return;
	} else if (valid == 0) {
	    /*
	     * not in cache, cache file created if cfd >= 0
	     */
	    printf("bad cache\n");
	    if (cfd >= 0) {
		cache = fdopen(cfd, "w");
	    }
	    /* fall through */
	} else if (valid < 0) {
	    /*
	     * negatively cached
	     */
	    printf("neg cache\n");
	    if (conn->co_ArtMode == COM_BODYNOSTAT)
		MBPrintf(&conn->co_TMBuf, "(article not available)\r\n.\r\n");
	    else
		MBPrintf(&conn->co_TMBuf, "430 No such article\r\n");
	    NNCommand(conn);
	    return;
	}
    }

    /*
     * (b)(c)(d)
     */
    NNServerRequest(conn, conn->co_GroupName, msgid, cache, SREQ_RETRIEVE);
    NNWaitThread(conn);
}

