/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <signal.h>
#include <Xm/Xm.h>

#define MAXARGV      50

#ifndef SIGCLD
#define SIGCLD SIGCHLD
#endif

static int child_is_done;
static int child_pid;
static int child_status;

void reaper_for_execute();


/*
 * execute - Execute "command" under the bourne shell.  This function does
 *           not return until the command is complete or until the user
 *           presses the stop button.  Returns -6 for stop button pushed,
 *           -1 for problem running command (including nonzero exit status),
 *           and 0 for success.  The purpose of this function is to act
 *           somewhat like system(), but with stop button processing.
 */
execute(command)
char *command;
{
	char *argv[MAXARGV+1];
	char *cmd;
	int argc = 0;
	int stop_was_pressed = False;

	/* Don't alter input argument */
	cmd = XtNewString(command);

	/* Parse execute line into tokens */
    if ((argv[argc++] = strtok(cmd, " ")) == NULL)
        fatal_error("Trouble in execute()");
    while (argc < MAXARGV && (argv[argc] = strtok(NULL, " ")) != NULL)
        argc++;
    if (argc == MAXARGV)
		fatal_error("Trouble in execute() - Too many tokens");

	/* Temporarily replace current signal handler for reaping children */
	save_sigcld_handler();
	signal(SIGCLD, reaper_for_execute);
	child_is_done = False;

	/* Run the command */
    if ((child_pid = fork()) == 0)
        if (execvp(argv[0], argv) == -1) {
			fprintf(stderr, "\007*** Unable to launch %s: ", argv[1]);
            perror("");
            exit(-1);
        }
	XtFree(cmd);

	/* Wait until command is done or until user presses stop button */
	while (!child_is_done)
		if (rest_stop(1) == -6) {
			kill(child_pid, SIGINT);
			stop_was_pressed = True;
		}

	/* Restore the old signal handler for reaping children */
	restore_sigcld_handler();

	/* Figure out what value to return */
	if (stop_was_pressed)
		return -6;
	else if (child_status)
		return -1;
	else
		return 0;
}


/*
 * reaper_for_execute - Signal handler for processing SIGCLD for function
 *                      execute().
 */
void
reaper_for_execute(signo)
int signo;
{
	int pid;

	while ((pid = simple_iwait(&child_status)) != child_pid) {
		if (pid == -1)
			fatal_error("Problem in reaper_for_execute()");
		signal(SIGCLD, reaper_for_execute);   /* Some systems require this */
	}

	child_is_done = True;
}

