#include	"qpage.h"


/*
** Global variables
*/
#ifndef lint
static char	sccsid[] = "@(#)ixo.c  1.20  06/21/97  tomiii@mtu.edu";
#endif
int		modemfd;


/*
** checksum()
**
** Calculate the checksum of a packet.
**
**	Input:
**		pk - the packet
**
**	Returns:
**		a character representation of the checksum for the packet
**
**	Note:
**		This function came from the "tpage" package, written
**		by Tom Limoncelli <tal@warren.mentorg.com>.
*/
char *
checksum(char *pk)
{
	static char	check[10];
	int		sum;


	for (sum=0;*pk; pk++)
		sum += *pk;

	check[2] = '0' + (sum & 15); sum = sum >> 4;
	check[1] = '0' + (sum & 15); sum = sum >> 4;
	check[0] = '0' + (sum & 15);
	check[3] = 0;

	return(check);
}


/*
** getpacket()
**
** Read a packet (a string terminated by '\r') from the modem.
**
**	Input:
**		seconds - how long to wait before timing out
**
**	Returns:
**		a complete packet or NULL if a timeout occurred
**
**	Note:
**		The timer is reset for each character received.
*/
char *
getpacket(int seconds)
{
	static char	buff[1024];
	static int	got_full_packet;

	struct timeval	timeout;
	fd_set		readfds;
	char		*ptr;
	int		i;


	if (got_full_packet) {
		got_full_packet = 0;
		ptr = buff;
		*ptr = '\0';
	}
	else {
		ptr = buff+strlen(buff);
	}


	do {
#if defined(lint) && defined(SOLARIS)
		/*
		** select.h is broken for Solaris 2.x.  It defines FD_ZERO
		** as a macro which calls memset().  Unfortunately memset()
		** returns a value, which means FD_ZERO returns a value,
		** contrary to the description in the "select" man page.
		**
		** We cannot just cast this to (void) at compile time
		** because it breaks on some platforms (e.g. Linux).
		*/
		(void)FD_ZERO(&readfds);
#else
		FD_ZERO(&readfds);
#endif
		FD_SET(modemfd, &readfds);

		timeout.tv_sec = seconds;
		timeout.tv_usec = 0;

		i = select(FD_SETSIZE, &readfds, 0, 0, &timeout);

		switch (i) {
			case 0: /* timer expired */
				return(NULL);

			case 1: /* modem is waiting to be read */
				errno = 0;
				if ((i = read(modemfd, ptr, 1)) == 1) {

					if (*ptr == '\r') {
						*ptr = '\0';
						got_full_packet++;
					}
					else {
						if (*ptr != '\n')
							ptr++;

						*ptr = '\0';
					}
				}
				else {
					if (i == 0 || errno != EINTR)
						return(NULL);
				}
				break;

			default:
				if (errno == EINTR)
					break;

				qpage_log(LOG_DEBUG, "select() failed: %s",
					strerror(errno));

				return(NULL);
		}

	} while (!got_full_packet);

	if (Debug)
		qpage_log(LOG_DEBUG, "got %d bytes: <%s>", strlen(buff),
			safe_string(buff));

	return(buff);
}


/*
** lookfor()
**
** Read packets from the modem until the packet matches a given
** string or until getpacket() times out.
**
**	Input:
**		str - the string to look for
**		timeout - a timeout value to pass to getpacket()
**
**	Returns:
**		an integer status (1=found, 0=notfound)
*/
int
lookfor(char *str, int timeout)
{
	char	*ptr;


	if (Debug)
		qpage_log(LOG_DEBUG, "looking for <%s>", safe_string(str));

	do {
		ptr = getpacket(timeout);

		if (ptr && !strcmp(ptr, str)) {
			if (Debug) {
				qpage_log(LOG_DEBUG, "found <%s>",
					safe_string(str));
			}

			return(1);
		}

	} while (ptr);

	if (Debug)
		qpage_log(LOG_DEBUG, "didn't find <%s>", safe_string(str));

	return(0);
}


/*
** write_modem()
**
** Write a string to the modem.
**
**	Input:
**		str - the string to write
**
**	Returns:
**		nothing
*/
void
write_modem(char *str)
{
	if (Debug)
		qpage_log(LOG_DEBUG, "sending <%s>", safe_string(str));

	while (write(modemfd, str, strlen(str)) < 0 && errno == EINTR)
		continue;
}


/*
** hangup_modem()
**
** This function attempts to hang up the modem, first by dropping DTR
** and then if that doesn't work, sending +++ followed by ATH0.  The
** modem is unlocked before returning.
**
**	Input:
**		s - the paging service to hang up
**
**	Returns:
**		nothing
*/
void
hangup_modem(service_t *s)
{
	int	i;


	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "hanging up modem");

	i = TIOCM_DTR;
	(void)ioctl(modemfd, TIOCMBIC, &i);

	/*
	** Give the modem a chance to reset and then get the status of
	** the carrier-detect signal.
	*/
	(void)sleep(1);
	if (ioctl(modemfd, TIOCMGET, &i) < 0) {
		/*
		** assume this means the hangup worked
		*/
		(void)close(modemfd);
		unlock_modem(s->device);
		modemfd = 0;
		return;
	}

	if (i & TIOCM_CAR) {
		qpage_log(LOG_INFO, "dropping DTR did *not* hang up the modem!");

		/*
		** try hanging up the old fashioned way
		*/
		write_modem("+++");
		(void)lookfor("OK", 5);
		write_modem("ATH0\r");

		/*
		** read() is probably going to return 0 from now on because
		** a hangup occurred on the stream (see read(2) for details)
		** but we'll give it a try anyway.
		*/
		(void)lookfor("OK", 5);
	}

	(void)close(modemfd);
	unlock_modem(s->device);
	modemfd = 0;
}


/*
** openmodem()
**
** Open and initialize the modem device.
**
**	Input:
**		s - the service specifying the device name, speed, and parity
**
**	Returns:
**		integer status of this function (0=success)
**
**	Side effects:
**		The modem device is opened and initialized with appropriate
**		termio settings.  The resulting file descriptor is assigned
**		to the global variable "modemfd" for use in other functions.
*/
int
openmodem(service_t *s)
{
	struct termios		ti;
	int			flags;


	if (lock_modem(s->device) < 0) {
		modemfd = 0;
		return(-1);
	}

	if (Debug)
		qpage_log(LOG_DEBUG, "opening modem device %s", s->device);

	if ((modemfd = open(s->device, O_RDWR|O_NDELAY, 0)) < 0) {
		qpage_log(LOG_NOTICE, "cannot open modem device: %s",
			strerror(errno));

		unlock_modem(s->device);
		modemfd = 0;
		return(-1);
	}

	/*
	** attempt to flush any data currently in the system buffers
	*/
	(void)tcflush(modemfd, TCIOFLUSH);

	/*
	** Make sure modem isn't already connected
	*/
	if (ioctl(modemfd, TIOCMGET, &flags) < 0) {
		if (Debug || Interactive)
			qpage_log(LOG_DEBUG, "cannot determine modem status, assuming OK.");
	}

	if (flags & TIOCM_CAR) {
		if (Debug || Interactive)
			qpage_log(LOG_DEBUG, "warning: carrier detect is present!");

		hangup_modem(s);
	
		if (lock_modem(s->device) < 0) {
			modemfd = 0;
			return(-1);
		}

		if ((modemfd = open(s->device, O_RDWR|O_NDELAY, 0)) < 0) {
			qpage_log(LOG_NOTICE, "cannot open modem device: %s",
				strerror(errno));

			unlock_modem(s->device);
			modemfd = 0;
			return(-1);
		}
	}

	/*
	** get the file descriptor status flags
	*/
	if ((flags = fcntl(modemfd, F_GETFL, 0)) < 0) {
		qpage_log(LOG_DEBUG, "fcntl(F_GETFL): %s", strerror(errno));
		(void)close(modemfd);
		unlock_modem(s->device);
		modemfd = 0;
		return(-1);
	}

	/*
	** turn off non-blocking mode
	*/
	flags &= ~O_NDELAY;
	if (fcntl(modemfd, F_SETFL, flags) < 0) {
		qpage_log(LOG_DEBUG, "fcntl(F_SETFL): %s", strerror(errno));
		(void)close(modemfd);
		unlock_modem(s->device);
		modemfd = 0;
		return(-1);
	}

	/*
	** get current tty settings
	*/
	if (tcgetattr(modemfd, &ti) < 0) {
		qpage_log(LOG_DEBUG, "tcgetattr(): %s", strerror(errno));
		(void)close(modemfd);
		unlock_modem(s->device);
		modemfd = 0;
		return(-1);
	}

	/*
	** set the desired terminal speed
	*/
	(void)cfsetispeed(&ti, s->baudrate);
	(void)cfsetospeed(&ti, s->baudrate);

	/*
	** set the desired parity
	*/
	switch (s->parity) {
		case 0:
			ti.c_cflag &= ~CSIZE;	/* clear char size bits */
			ti.c_cflag |= CS8;	/* 8-bit bytes */
			ti.c_cflag &= ~PARENB;	/* no parity */
			break;

		case 1:
			ti.c_cflag &= ~CSIZE;	/* clear char size bits */
			ti.c_cflag |= CS7;	/* 7-bit bytes */
			ti.c_cflag |= PARENB;	/* parity enable */
			ti.c_cflag |= PARODD;	/* odd parity */
			break;

		case 2: /* fall through */
		default:
			ti.c_cflag &= ~CSIZE;	/* clear char size bits */
			ti.c_cflag |= CS7;	/* 7-bit bytes */
			ti.c_cflag |= PARENB;	/* parity enable */
			ti.c_cflag &= ~PARODD;	/* even parity */
			break;
	}

	/*
	** For the following code:
	**
	**	|= OPTION	turns the option on
	**	&= ~OPTION	turns the option off
	*/
	ti.c_iflag |= IGNBRK;		/* ignore breaks */
/*	ti.c_iflag |= BRKINT;		signal interrupt on break */
/*	ti.c_iflag |= IGNPAR;		ignore parity */
/*	ti.c_iflag |= PARMRK;		mark parity errors */
	ti.c_iflag &= ~INPCK;		/* enable input parity check */
	ti.c_iflag |= ISTRIP;		/* strip 8th bit */
	ti.c_iflag &= ~INLCR;		/* map nl->cr */
	ti.c_iflag &= ~IGNCR;		/* ignore cr */
	ti.c_iflag &= ~ICRNL;		/* map cp->nl */
/*	ti.c_iflag |= IUCLC;		map uppercase to lowercase */
	ti.c_iflag &= ~IXON;		/* enable start/stop output control */
/*	ti.c_iflag |= IXANY;		enable any char to restart output */
	ti.c_iflag &= ~IXOFF;		/* enable start/stop input control */
	ti.c_iflag |= IMAXBEL;		/* echo BEL on input line too long */

	ti.c_oflag &= ~OPOST;		/* post-process */

	ti.c_cflag &= ~CSTOPB;		/* one stop bit */
/*	ti.c_cflag |= CSTOPB;		two stop bit */

	ti.c_cflag |= HUPCL;		/* hang up on last close */
	ti.c_cflag |= CLOCAL;		/* local line */
/*	ti.c_cflag &= ~CLOCAL;		dial-up line */
/*	ti.c_cflag |= CRTSCTS;		enable RTS/CTS flow control */

	ti.c_lflag |= ISIG;		/* enable signals */
	ti.c_lflag &= ~ICANON;		/* enable erase & kill processing */
/*	ti.c_lflag |= XCASE;		canon upper/lower presentation */
	ti.c_lflag &= ~ECHO;		/* enable echo */
	ti.c_lflag |= ECHOE;		/* echo erase as bs-sp-bs */
/*	ti.c_lflag |= ECHOK;		echo nl after kill */
/*	ti.c_lflag |= ECHONL;		echo nl */
/*	ti.c_lflag |= NOFLSH;		disable flush after interrupt */
/*	ti.c_lflag |= TOSTOP;		send SIGTTOU for background output */
	ti.c_lflag |= ECHOCTL;		/* echo ctrl chars as ^A */
/*	ti.c_lflag |= ECHOPRT;		echo erase as char erased */
	ti.c_lflag |= ECHOKE;		/* bs-sp-bs entire line on line kill */
/*	ti.c_lflag |= FLUSHO;		output is being flushed */
/*	ti.c_lflag |= PENDIN;		retype pending input at next read */
	ti.c_lflag |= IEXTEN;		/* recognize all specials, else POSIX */

        ti.c_cc[VMIN] = 0;		/* must read one char or timeout */
        ti.c_cc[VTIME] = 10;		/* read() times out at 1.0 second */

	/*
	** set new tty settings
	*/
	if (tcsetattr(modemfd, TCSANOW, &ti) < 0) {
		qpage_log(LOG_DEBUG, "tcsetattr(): %s", strerror(errno));
		(void)close(modemfd);
		unlock_modem(s->device);
		modemfd = 0;
		return(-1);
	}

	return(0);
}


/*
** remote_connect()
**
** Dial the remote paging terminal and log in.
**
**	Input:
**		s - the paging service to connect to
**
**	Returns:
**		zero or more flags indicating success or reasons for failure
*/
int
remote_connect(service_t *s)
{
	char		buff[1024];
	char		*ptr;
	int		done;
	char		c;
	int		i;


	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "resetting modem");

	write_modem("ATZ\r");
	if (!lookfor("OK", 5)) {
		qpage_log(LOG_NOTICE, "cannot initialize modem");
		return(F_NOMODEM);
	}

	/*
	** Some braindamaged modems (like the Hayes 1200) return "OK" before
	** they are actually ready to accept commands.
	*/
	(void)sleep(1);

	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "dialing service %s with %s",
			s->name, s->dialcmd);

	/*
	** dial paging terminal
	*/
	write_modem(s->dialcmd);
	write_modem("\r");

	if (!lookfor(s->dialcmd, 2)) {
		qpage_log(LOG_NOTICE, "cannot dial modem");
		return(F_NOMODEM);
	}

	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "waiting for connection");

	/*
	** wait for connection
	*/
	for (;;) {
		/*
		** use a very long timeout here so the modem's S7 register
		** can determine the real timeout
		*/
		if ((ptr = getpacket(255)) == NULL)
			break;

		if (strstr(ptr, "CONNECT")) {
			qpage_log(LOG_INFO, "connected to remote, service=%s",
				s->name);
			break;
		}

		if (strstr(ptr, "BUSY")) {
			qpage_log(LOG_INFO, "remote is busy, service=%s",
				s->name);
			return(F_BUSY);
		}

		if (strstr(ptr, "NO CARRIER")) {
			qpage_log(LOG_NOTICE,
				"no answer from remote, service=%s", s->name);
			return(F_NOCARRIER);
		}

		if (strstr(ptr, "OK")) {
			qpage_log(LOG_NOTICE, "dial string failed, service=%s",
				s->name);
			return(F_NOCARRIER);
		}

	}

	if (!ptr) {
		qpage_log(LOG_NOTICE,
			"timeout waiting for CONNECT, service=%s", s->name);
		return(F_UNKNOWN);
	}

	/*
	** look for "ID=" prompt from paging service
	*/
	for (i=0; i<10; i++) {

		write_modem("\r");
		done = FALSE;
		ptr = "ID=";

		while (read(modemfd, &c, 1) == 1) {
			if (Debug) {
				buff[0] = c;
				buff[1] = '\0';
				qpage_log(LOG_DEBUG, "got 1 byte: <%s>",
					safe_string(buff));
			}

			if (c != *ptr++)
				ptr = "ID=";

			if (*ptr == '\0') {
				done = TRUE;
				break;
			}
		}

		if (done)
			break;
	}

	if (!done) {
		qpage_log(LOG_NOTICE, "no login prompt from %s", s->name);
		return(F_NOPROMPT);
	}

	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "logging into %s", s->name);

	(void)sprintf(buff, "\033PG1%s\r", s->password);
	write_modem(buff);

	for (;;) {
		if ((ptr = getpacket(10)) == NULL)
			break;

		if (strstr(ptr, "\006")) { /* ACK */
			if (Debug || Interactive)
				qpage_log(LOG_DEBUG, "login accepted");

			break;
		}

		if (strstr(ptr, "\025")) { /* NAK */
			if (Debug || Interactive)
				qpage_log(LOG_DEBUG, "login requested again");

			write_modem(buff);
		}

		if (strstr(ptr, "\033\004")) { /* ESC-EOT */
			qpage_log(LOG_NOTICE, "forced disconnect from %s",
				s->name);

			return(F_FORCED);
		}

	}

	if (!ptr) {
		qpage_log(LOG_NOTICE, "login failed (timeout waiting for <ACK>)");
		return(F_UNKNOWN);
	}

	if (!lookfor("\033[p", 5)) {
		qpage_log(LOG_NOTICE, "no go-ahead from %s", s->name);
		return(F_UNKNOWN);
	}

	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "%s says ok to proceed", s->name);

	return(0);
}


/*
** remote_disconnect()
**
** Tell the paging terminal we're all done with this session, then
** hang up the modem.
**
**	Input:
**		s - the paging service to disconnect
**
**	Returns:
**		nothing
*/
void
remote_disconnect(service_t *s)
{
	char		*ptr;


	if (modemfd == 0)
		return;

	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "logging out");

	write_modem("\004\r");

	do {
		ptr = getpacket(5);

		if (ptr && strstr(ptr, "\036")) { /* RS */
			qpage_log(LOG_WARNING, "warning: %s sent <RS>",
				s->name);
		}

	} while (ptr && !strstr(ptr, "\033\004"));

	hangup_modem(s);
}


/*
** xmit_message()
**
** Transmit a page to the remote paging terminal.  This function sends
** an entire message (breaking it into multi-page pieces if necessary)
** to one recipient.
**
**	Input:
**		job - the structure containing the job to transmit
**
**	Returns:
**		an integer status code:
**			 0 = success, send another page if desired
**			 1 = success, hang up now
**			-1 = failure, page not accepted
*/
int
xmit_message(job_t *job)
{
	char	*buf;
	char	*field1;
	char	*field2;
	char	*response;
	char	*countp;
	char	*ptr;
	char	*msg;
	char	*from;
	int	retries;
	int	status;
	int	max_size;
	int	msghdrlen;
	int	field1len;
	int	field2len;
	int	parts;


	field1len = strlen(job->pager->pagerid) + 2;
	field2len = job->service->maxmsgsize + 2;

	field1 = (void *)malloc(field1len);
	field2 = (void *)malloc(field2len);
	buf = (void *)malloc(field1len + field2len + 10);

	if (job->p->from == NULL) {
		if (job->service->identfrom && job->p->ident)
			from = strdup(job->p->ident);
		else
			from = NULL;
	}
	else
		from = strdup(job->p->from);

	/*
	** If "from" is an e-mail address, only use the userid part
	** for the message prefix (i.e. chop off the hostname).
	*/
	if (from && (ptr = strchr(from, '@')) != NULL)
		*ptr = '\0';

	(void)sprintf(field1, "%c%s", CHAR_STX, job->pager->pagerid);

	if (from)
		(void)sprintf(field2, "%s:0: ", from);
	else
		(void)strcpy(field2, "0: ");

	msghdrlen = strlen(field2);
	countp = field2 + msghdrlen - 3;

	max_size = job->service->maxmsgsize - msghdrlen;
	msg = job->p->message;

	parts = 0;
	while (msg) {
		/*
		** increment the message count indicator
		*/
		(*countp)++;
		parts++;

		msg = msgcpy(field2+msghdrlen, msg, max_size);

		if (!msg) {
			if (*countp == '1') {
				/*
				** The whole message fits in one page
				** so let's rewrite field2 to get rid
				** of the split-message indicator.
				*/
				if (from)
					(void)sprintf(field2, "%s: %s", from,
						job->p->message);
				else
					(void)strcpy(field2, job->p->message);
			}
			else {
				/*
				** this is the last page in a series
				*/
				*countp = 'e';

				/*
				** Tack on "-oo-" at the end of the message,
				** but only if it fits in the remaining space.
				** We didn't do this earlier because we don't
				** want to send a whole extra page just for
				** the -oo- indicator.
				*/
				if (job->service->maxmsgsize-strlen(field2) > 5)
					(void)strcat(field2, " -oo-");
			}
		}

		if (msg && (*countp == ('0' + job->service->maxpages))) {
			/*
			** There is more to this message but this
			** is the last piece we're allowed to send.
			** We'll change the split-message indicator
			** to a 't' so they know it's a truncated
			** message.
			*/
			*countp = 't';
			msg = NULL;
		}

		(void)(void)sprintf(buf, "%s\r%s\r%c", field1, field2, CHAR_ETX);

		(void)strcat(buf, checksum(buf));
		(void)strcat(buf, "\r");

		write_modem(buf);

		status = -1; /* assume failure */
		retries = 5;
		response = NULL;
		do {
			if ((ptr = getpacket(15)) == NULL)
				break;

			if (strstr(ptr, "\006")) { /* ACK */
				if (Debug || Interactive)
					qpage_log(LOG_DEBUG, "message accepted");

				status = 0;
				break;
			}

			if (strstr(ptr, "\025")) { /* NAK */
				if (Debug || Interactive)
					qpage_log(LOG_DEBUG, "message requested again");

				if (--retries == 0) {
					qpage_log(LOG_ERR, "too many retries");
					break;
				}

				write_modem(buf);
				status = -1;
				continue;
			}

			if (strstr(ptr, "\036")) { /* RS */
				if (Debug || Interactive)
					qpage_log(LOG_DEBUG, "message rejected");

				status = -1;
				break;
			}

			if (strstr(ptr, "\033\004")) { /* ESC-EOT */
				if (Debug || Interactive)
					qpage_log(LOG_DEBUG, "message not accepted, disconnect now");

				status = 1;
				break;
			}

			/*
			** This must be an informational message from the
			** paging service.  Trim leading whitespace.
			*/
			while (isspace(*ptr))
				ptr++;

			if (*ptr) {
				if (response)
					free(response);

				response = strdup(ptr);
			}

		} while (ptr);


		if (!ptr) {
			if (Debug || Interactive)
				qpage_log(LOG_DEBUG, "no valid response from %s", job->service->name);

			status = -1;
			break;
		}

		/*
		** It's a fatal error if the status is non-zero.  Abort
		** the rest of this page.
		*/
		if (status)
			break;
	}

	switch (status) {
		case 0:
			qpage_log(LOG_NOTICE,
				"page delivered, id=%s, from=%s, to=%s, parts=%d",
				job->p->messageid, from ? from : "[anonymous]",
				job->rcpt->pager, parts);
			break;

		case 1:
			qpage_log(LOG_NOTICE,
				"page status unknown, id=%s, from=%s, to=%s, parts=%d : %s",
				job->p->messageid, from ? from : "[anonymous]",
				job->rcpt->pager, parts,
				response ? response : "<NULL>");
			break;

		default:
			qpage_log(LOG_NOTICE,
				"page failed, id=%s, from=%s, to=%s, parts=%d : %s",
				job->p->messageid, from ? from : "[anonymous]",
				job->rcpt->pager, parts,
				response ? response : "<NULL>");
			break;
	}

	free(field1);
	free(field2);
	free(buf);

	if (response)
		free(response);

	if (from)
		free(from);

	return(status);
}


/*
** send_pages()
**
** Send all the pages in a page list.
**
**	Input:
**		jobs - an ordered linked list of jobs to send
**
**	Returns:
**		nothing
**
**	Side effects:
**		The global file descriptor "modemfd" is modified.
*/
void
send_pages(job_t *jobs)
{
	service_t	*current_service;
	job_t		*tmp;
	int		i;


	current_service = NULL;

	while (jobs) {
		jobs->rcpt->tries++;
		jobs->rcpt->lasttry = time(NULL);
		jobs->rcpt->flags &= ~(CALLSTATUSFLAGS);

		/*
		** Initialize the modem and log into the paging terminal.
		*/
		if ((modemfd == 0) || (current_service != jobs->service)) {
			if (Debug || Interactive)
				qpage_log(LOG_DEBUG, "new service: %s",
					jobs->service->name);

			current_service = jobs->service;

			/*
			** If we're connected to a paging service then we
			** need to let them know we're finished before we
			** can call the next one.
			*/
			remote_disconnect(jobs->service);

			if (openmodem(jobs->service)) {
				jobs->rcpt->flags |= F_NOMODEM;
				tmp = jobs;
				jobs = jobs->next;
				free(tmp);
				continue;
			}

			if ((i = remote_connect(jobs->service)) != 0) {
				/*
				** If we failed for some reason other
				** than resetting the modem or getting
				** a busy signal, it's possible we are
				** paying for a phone call.  Bump up
				** the "good" try counter so we don't
				** get stuck in an endless loop racking
				** up phone charges.
				*/
				if ((i & (F_BUSY|F_NOMODEM)) == 0)
					jobs->rcpt->goodtries++;

				jobs->rcpt->flags |= i;
				(void)close(modemfd);
				unlock_modem(jobs->service->device);
				modemfd = 0;
				tmp = jobs;
				jobs = jobs->next;
				free(tmp);
				continue;
			}
		}

		/*
		** we're definitely using the phone line at this point
		*/
		jobs->rcpt->goodtries++;

		switch (xmit_message(jobs)) {
			case 0: /* success */
				jobs->rcpt->flags |= F_SENT;
				break;

			case 1: /* forced disconnect, hang up now */
				jobs->rcpt->flags |= F_FORCED;
				unlock_modem(jobs->service->device);
				(void)close(modemfd);
				modemfd = 0;
				break;

			default: /* failure, page not accepted */
				jobs->rcpt->flags |= F_REJECT;
				break;
		}

		tmp = jobs;
		jobs = jobs->next;

		/*
		** If this was the last job, tell the paging terminal
		** we're finished.
		*/
		if (jobs == NULL)
			remote_disconnect(tmp->service);

		free(tmp);
	}
}
