/*
 * file devices.c
 *
 * Routines for Pointers device processing
 *
 * Joystick and Mouse
 *
 * original idea from Chris Sharp <sharp@uk.ibm.com>
 *
 */

#define __DEVICES_C_

/* #define USE_X11_JOYEVENTS */ /* just done (if selected) in makefile */

#include "xmame.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifdef svgalib

#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include <signal.h>

#endif

#ifdef ggi

#include <ggi/ggi.h>

#endif

#ifdef USE_JOYSTICK

#ifdef X11_JOYSTICK
/* standard X input extensions based joystick */
#include <X11/extensions/XI.h>
#include <X11/extensions/XInput.h>
void x11_poll_joystick (void);
struct joystick {
	int buttons;
	int x;
	int y;
} joy_data, joy_orig;
XDevice *xdevice;

#elif  LIN_FM_TOWNS
/* FM-TOWNS game pad for linux */
#include <linux/joystick.h>
#include "pad.h"
void fm_town_poll_joystick (void);
extern struct JS_DATA_TYPE joy_data,joy_orig;

#elif  I386_JOYSTICK
void i386_poll_joystick (void);
/* specific joystick for PC clones */
#ifdef netbsd_i386
#include <machine/joystick.h>
extern struct joystick joy_data,joy_orig;
#else
#include <linux/joystick.h>
extern struct JS_DATA_TYPE joy_data,joy_orig;
#endif

#else
#error "USE_JOYSTICK needs a joystick type definition. Abort"
#endif

/* if USE_JOYSTICK */
#endif

#define TAN_22_5	106
#define TAN_67_5	618
#define abs(x) ( ( (x) >= 0 )? (x) : -(x) )

/* joystick specific variables */
#define JOY_THRESHOLD	0
extern int joy_fd;
int joy_up, joy_down, joy_left, joy_right;
int joy_b1, joy_b2, joy_b3, joy_b4;

#ifdef USE_JOYSTICK
/* keep max and min value's for analog joystick */
static int joy_max_x, joy_max_y, joy_min_x, joy_min_y;
void (*poll_joy_func)(void);
#endif

/* mouse variables */
static int mouse_delta_x, mouse_delta_y, current_mouse_x, current_mouse_y;
static int mouse_b1, mouse_b2, mouse_b3;

/*
 ******************************* Mouse routines *********************
 */

/* 
 * NOTE:
 * 
 * Due to the fact that in X-World mouse and trakball movements are 
 * coded as a MotionNotify event, we have a unique, centralized point to
 * evaluate these event. then propper variables are setted according data
 */

#if defined x11 || defined xf86_dga || defined xgl || defined xfx
void process_mouse_event(XEvent *event) {
    switch(event->type)
    {
#ifdef xf86_dga
       case MotionNotify:
          current_mouse_x += event->xmotion.x_root;
          current_mouse_y += event->xmotion.y_root;
          break;
#endif
       case ButtonPress:
          switch(event->xbutton.button)
          {
             case Button1:
                mouse_b1 = 1;
                break;
             case Button2:
                mouse_b2 = 1;
                break;
             case Button3:
                mouse_b3 = 1;
                break;
          }
          break;
       case ButtonRelease:
          switch(event->xbutton.button)
          {
             case Button1:
                mouse_b1 = 0;
                break;
             case Button2:
                mouse_b2 = 0;
                break;
             case Button3:
                mouse_b3 = 0;
                break;
          }
          break;
    }
}
#endif

int sysdep_mouse_init(void) {
        mouse_delta_x = mouse_delta_y = current_mouse_x = current_mouse_y = 0;
	mouse_b1 = mouse_b2 = mouse_b3 = 0;
	if(use_mouse) fprintf(stderr_file,"Mouse/Trakball selected.\n");
	return OSD_OK;
}

void svgalib_mouse_init() {
/* must be invoked in vga init routine */
#ifdef svgalib
   int fd;
   fd=mouse_init_return_fd("/dev/mouse",vga_getmousetype(),MOUSE_DEFAULTSAMPLERATE);
   if (fd<0) {
       perror("mouse_init");
       fprintf(stderr_file,"SVGALib: failed to open mouse device %d\n",fd);
       use_mouse=0;
   }
   /* fix ranges and initial position of mouse */
    mouse_setxrange(-500,500);
    mouse_setyrange(-500,500);
    mouse_setposition(0,0);
#endif
   return;
}

void sysdep_mouse_close(void) {
/* x11 ungrabs the ptr when closing the display */
#ifdef svgalib
	if (use_mouse) mouse_close();
#endif
}

void sysdep_mouse_poll (void)
{
#ifdef svgalib
	int mouse_buttons;
	
	mouse_update();
	
	mouse_delta_x = mouse_getx();
	mouse_delta_y = mouse_gety();
	
	mouse_buttons = mouse_getbutton();
	
	mouse_b1 = (MOUSE_LEFTBUTTON & mouse_buttons)?   1 : 0;
	mouse_b2 = (MOUSE_RIGHTBUTTON & mouse_buttons)?  1 : 0;
	mouse_b3 = (MOUSE_MIDDLEBUTTON & mouse_buttons)? 1 : 0;
	
	mouse_setposition(0,0);
#elif xf86_dga
	mouse_delta_x = current_mouse_x;
	mouse_delta_y = current_mouse_y;
	current_mouse_x = current_mouse_y = 0;
#elif defined x11 || defined xgl || defined xfx
	Window root,child;
	int root_x, root_y, pos_x, pos_y;
	unsigned int keys_buttons;
	
	if (XQueryPointer(display,window, &root,&child, &root_x,&root_y, &pos_x,&pos_y,&keys_buttons) ){
	    mouse_delta_x = pos_x - current_mouse_x;
	    mouse_delta_y = pos_y - current_mouse_y;
	    current_mouse_x = pos_x;
	    current_mouse_y = pos_y;
	}
#endif
}


void osd_trak_read(int *deltax,int *deltay)
{
	*deltax = mouse_delta_x;
	*deltay = mouse_delta_y;
}

/*
 ************************** FM TOWNS PAD specifics routines *************
 */
#ifdef LIN_FM_TOWNS
/*
 * Towns Pad control module for Xmame
 *
 * Author : Osamu KURATI
 * Version : 29 apr 1997 0.000
 */

/*
 * Joy Stick Code
 * Up : 1
 * Down       : 2
 * Left       : 4
 * Right      : 8
 * A  : 10
 * B  : 20
 *
 *
 * PAD bit number
 * up         : 0
 * down               : 1
 * left               : 2
 * right      : 3
 * A          : 4
 * B          : 5
 * RUN                : 6
 * SELECT     : 7
 */

static int iPad;
static unsigned long lPadLastButton = 0;

void PadOpen(char *pdevname)
{
  if ((iPad = open(pdevname, O_NONBLOCK | O_RDONLY)) < 0){
    fprintf(stderr_file,"Unable to open FM-TOWNS game pad. Joystick disabled\n");
    use_joystick=0;
  }
  lPadLastButton = 0;
  return;
}

int Pad()
{
  struct pad_event ev;
  if (read(iPad, &ev, sizeof ev) == sizeof ev){
    lPadLastButton = ev.buttons;
  }
  return((int) lPadLastButton & 0xff);
}

void PadClose() { close(iPad); }

/* TOWNS_PAD */
#endif


/*
 ******************************* JoyStick routines *********************
 */

int  sysdep_joy_initvars(void)
{
    joy_up   = joy_down  = 0;
    joy_left = joy_right = 0 ;
    joy_b1 = joy_b2 = 0;
    joy_b3 = joy_b4 = 0;
    
#ifdef USE_JOYSTICK
    if(!use_joystick)
    {
      analogstick=0; /* disable analogstick if enabled */
      return OSD_OK;
    }
    
    joy_max_x = joy_orig.x + 10; /* + 10 to avoid static behaviour and */
    joy_max_y = joy_orig.y + 10; /* floating point exceptions          */
    joy_min_x = joy_orig.x - 10;
    joy_min_y = joy_orig.y - 10;

#ifdef I386_JOYSTICK
    poll_joy_func = i386_poll_joystick;
#elif LIN_FM_TOWNS
    PadOpen(towns_pad_dev); 
    poll_joy_func = fm_town_poll_joystick;
#elif X11_JOYSTICK
    /* cannot init joystick till display has been opened ... */
    /* x11_joystick_init() */
    poll_joy_func = x11_poll_joystick;
#else
#error "USE_JOYSTICK option needs to specify joystick type"
#endif

#endif
    return OSD_OK;
}

void sysdep_joy_close(void) {
#ifdef LIN_FM_TOWNS
    if (use_joystick)       PadClose();
#else
    /* just done in sysdep_exit() */
#endif
    return;
}


#if defined x11 || defined xf86_dga || defined xgl || defined xfx

void process_x11_joy_event(XEvent *event) {
#ifdef USE_X11_JOYEVENTS
/* does not run yet, don't know why :-( */
#ifdef X11_JOYSTICK
    XDeviceButtonEvent *dbe;
    XDeviceMotionEvent *dme;
    /* remember that event types are not harcoded: we evaluated it in XOpenDevice() */
    /* hack: we'll suppose that:
	 first_axis is allways equals 0. 
	 device_id is joystic's id
       in a real program, should be checked... 
     */
    if (!use_joystick) return;
    if ( (event->type==devicebuttonpress) || (event->type==devicebuttonrelease) ) {
	dbe=(XDeviceButtonEvent *) event;	
	/* evaluate button state */
	joy_data.buttons=dbe->device_state;
	/* and now X-Y position remember that uses relative mode */
	if (! swapjoyaxis) 
	     { 
		joy_data.x = joy_orig.x + dbe->axis_data[0];
		joy_data.y = joy_orig.y + dbe->axis_data[1];
	} else { 
		joy_data.x = joy_orig.x + dbe->axis_data[1];
		joy_data.y = joy_orig.y + dbe->axis_data[0];
	}
    }
    if ( (event->type==devicemotionnotify) ) {
	dme=(XDeviceMotionEvent *) event;	
	/* evaluate button state */
	joy_data.buttons=dme->device_state;
	if (! swapjoyaxis) 
	     { joy_data.x=joy_orig.x+dme->axis_data[0]; joy_data.y=joy_orig.y+dme->axis_data[1]; } 
	else { joy_data.x=joy_orig.x+dme->axis_data[1]; joy_data.y=joy_orig.y+dme->axis_data[0]; }
    }
    joy_b1 = ( joy_data.buttons & Button1Mask ) ? 1 : 0;
    joy_b2 = ( joy_data.buttons & Button2Mask ) ? 2 : 0;
    joy_b3 = ( joy_data.buttons & Button3Mask ) ? 4 : 0;
    joy_b4 = ( joy_data.buttons & Button4Mask ) ? 8 : 0;
#endif
#endif
    return;
}

#endif

int x11_joystick_init() {
#ifndef X11_JOYSTICK
	if (!use_joystick) return OSD_OK;
#else
	int 		i,j,k;
	int 		result;
	XDeviceInfoPtr 	list,slist;
	XAnyClassPtr 	any;
	XButtonInfoPtr 	binfo;
	XValuatorInfoPtr vinfo;
	XAxisInfoPtr    ainfo;
	XInputClassInfo *classptr;
	XEventClass 	xeventlist[8];
	int 		xeventcount;
	

	/* query server for input extensions */
	result =XQueryExtension(display,"XInputExtension",&i,&j,&k);
	if(!result) {
	    fprintf(stderr_file,"Your server doesn't support XInput Extensions\n");
	    fprintf(stderr_file,"X11 Joystick disabled\n");
	    use_joystick=0;
	    return OSD_OK;
	}
	/* now get input device list and locate desired device */
	list = XListInputDevices(display,&result);
	if (!list ) {
	    fprintf(stderr_file,"No extended input devices found !!\n");
	    fprintf(stderr_file,"X11 Joystick disabled\n");
	    use_joystick=0;
	    return OSD_OK;
	}
	slist=list;
	for(i=j=0;i<result;i++,list++) 
		if ( ! strcmp(x11joyname,list->name)  ) { j=1; break; }
	if (!j) {
	    fprintf(stderr_file,"Cannot locate device \"%s\" in available devices\n",x11joyname);
	    fprintf(stderr_file,"X11 Joystick disabled\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	}
	/* test for correct device ( search at least two buttons and two axis */
	any = (XAnyClassPtr)(list->inputclassinfo);
	result=0;
	for(j=0;j<list->num_classes;j++) {
#if defined(__cplusplus) || defined(c_plusplus)
	    switch(any->c_class) {
#else
	    switch(any->class) {
#endif
		case ButtonClass:
			binfo=(XButtonInfoPtr) any;
			if ((num_x_buttons=binfo->num_buttons)>=2) result |= 0x01;
			fprintf(stderr_file,"%s: %d buttons\n",x11joyname,num_x_buttons);
			joy_data.buttons=joy_orig.buttons=0;
			break;
		case ValuatorClass:
			vinfo=(XValuatorInfoPtr) any;
			if (vinfo->num_axes>=2) result |= 0x02;
			fprintf(stderr_file,"%s: %d axes\n",x11joyname,vinfo->num_axes);
			ainfo=vinfo->axes;
			if (!swapjoyaxis ) {
			  joy_data.x=joy_orig.x=(ainfo->max_value-ainfo->min_value)/2;
			  ainfo++; /* next axis */
			  joy_data.y=joy_orig.y=(ainfo->max_value-ainfo->min_value)/2;
			} else {
			  joy_data.y=joy_orig.y=(ainfo->max_value-ainfo->min_value)/2;
			  ainfo++; /* next axis */
			  joy_data.x=joy_orig.x=(ainfo->max_value-ainfo->min_value)/2;
			}
			break;
		case KeyClass: /* no sense to use a extended key device */
			fprintf(stderr_file,"%s: Ingnoring KeyClass info\n",x11joyname);
		default: break;  /* unnknown class: ignore */
	    }
	    any = (XAnyClassPtr) ((char *) any+any->length);
	}
	if (result != 0x03 ) {
	    fprintf(stderr_file,"Your selected X11 device \"%s\"doesn't match X-Mame/X-Mess requirements\n",x11joyname);
	    fprintf(stderr_file,"X11 Joystick disabled\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	}
	/* once located, try to open */	
	if ( ! (xdevice=XOpenDevice(display,list->id) ) ) {
	    fprintf(stderr_file,"XDeviceOpen error\n");
	    use_joystick=0;
	    XFreeDeviceList(slist);
	    return OSD_OK;
	} 
	/* buscamos los eventos asociados que necesitamos */
	/* realmente el bucle for y la sentencia switch no son necesarias, pues
	   en XInput.h se buscan automaticamente los elementos de cada dato, pero
	   lo pongo de ejemplo para si en el futuro se quieren chequear la existencia
           de una determinada clase antes de pedir eventos. Nosotros sabemos a 
	   priori que no deberia fallar....
	*/
	xeventcount=0;
	for (i=0,classptr=xdevice->classes;i<xdevice->num_classes;i++,classptr++ ) {
	    switch(classptr->input_class) {
		case KeyClass: break;
		case ButtonClass:
	    		DeviceButtonPress(xdevice,devicebuttonpress,xeventlist[xeventcount]);
			if (devicebuttonpress) xeventcount++;
	    		DeviceButtonRelease(xdevice,devicebuttonrelease,xeventlist[xeventcount]);
			if (devicebuttonrelease) xeventcount++;
			break;
		case ValuatorClass:
	    		DeviceMotionNotify(xdevice,devicemotionnotify,xeventlist[xeventcount]);
			if (devicemotionnotify) xeventcount++;
	    		DeviceButtonMotion(xdevice,devicebuttonmotion,xeventlist[xeventcount]);
			if (devicebuttonmotion) xeventcount++;
			break;
		case FocusClass: break;
		case ProximityClass: break;
		case OtherClass: break;
		default: break;
	    }
	}
#if 0
	/* 
	NOTE: don't know why but these two items don't work in my linux
	XInputExtension Joystick module. still working on it ...
	*/

	/* force relative motion report */
	XSetDeviceMode(display,xdevice,Relative);
	/* set starting point of joystick (to force joy to be centered) */
	if(!swapjoyaxis) {
	   XSetDeviceValuators(display,xdevice,&(joy_data.x),0,1);
	   XSetDeviceValuators(display,xdevice,&(joy_data.y),1,1);
	}else{
	   XSetDeviceValuators(display,xdevice,&(joy_data.y),0,1);
	   XSetDeviceValuators(display,xdevice,&(joy_data.x),1,1);
	}
#endif
#ifdef USE_X11_JOYEVENTS
	/* not sure why don't recognize all events type. still working... */
	XSelectExtensionEvent(display,window,xeventlist,xeventcount);
	fprintf(stderr_file,"X11PointerDevice: Using X-Window Events\n");
#else
	fprintf(stderr_file,"X11PointerDevice: Using Demand QueryState\n");
#endif
	fprintf(stderr_file,"Found and installed X11 pointer device \"%s\"\n",x11joyname);
	/* and finaly free requested device list */
	XFreeDeviceList(slist);
#endif
	return OSD_OK;
}

void osd_poll_joystick (void)
{
    if (use_mouse)    sysdep_mouse_poll();
#ifdef USE_JOYSTICK
    if (use_joystick) (*poll_joy_func)();
#endif    
}

int osd_joy_pressed (int joycode)
{
    /* if joystick not implemented , all variables set to zero */
    switch (joycode)
    {
    	case OSD_JOY_LEFT:  return joy_left;  break;
    	case OSD_JOY_RIGHT: return joy_right; break;
    	case OSD_JOY_UP:    return joy_up;    break;
    	case OSD_JOY_DOWN:  return joy_down;  break;
    	case OSD_JOY_FIRE1: return (joy_b1 || mouse_b1);    break;
    	case OSD_JOY_FIRE2: return (joy_b2 || mouse_b2);    break;
    	case OSD_JOY_FIRE3: return (joy_b3 || mouse_b3);    break;
    	case OSD_JOY_FIRE4: return joy_b4;    break;
    	case OSD_JOY_FIRE:  return (joy_b1 || joy_b2 || joy_b3 || joy_b4 || mouse_b1 || mouse_b2 || mouse_b3); break;
    	default:            return FALSE;     break;
    }
}

/* return the name of a joystick button */
const char *osd_joy_name(int joycode)
{
  static char *notsupported = "Not supported under Unix";
  static char *joynames[10]   = { "None", "LEFT", "RIGHT", "UP", "DOWN",
  			      "Button A", "Button B", "Button C", "Button D", 
			      "Any Button" };

  if (joycode < 9)                       return (char*)joynames[joycode];
  if (joycode == 15)                     return (char*)joynames[9];
  return (char*)notsupported;
}

/*
 * given a new x an y joystick axis value convert it to a move definition
 */

void evaluate_joy_moves(void) {
#ifdef USE_JOYSTICK
   int x_treshold, y_treshold;

   joy_left = joy_right = joy_up = joy_down = 0;
        
   /* auto calibrate */
   if (joy_data.x > joy_max_x) joy_max_x = joy_data.x;
   if (joy_data.y > joy_max_y) joy_max_y = joy_data.y;
   if (joy_data.x < joy_min_x) joy_min_x = joy_data.x;
   if (joy_data.y < joy_min_y) joy_min_y = joy_data.y;
   
   x_treshold = (joy_max_x - joy_orig.x) >> 1;
   y_treshold = (joy_max_y - joy_orig.y) >> 1;
   
   if      (joy_data.x > (joy_orig.x + x_treshold) ) joy_right = 1;
   else if (joy_data.x < (joy_orig.x - x_treshold) ) joy_left  = 1;
   if      (joy_data.y > (joy_orig.y + y_treshold) ) joy_down  = 1;
   else if (joy_data.y < (joy_orig.y - y_treshold) ) joy_up    = 1;
#endif
   return;
}

#if ( defined USE_JOYSTICK ) && ( defined LIN_FM_TOWNS )
/*
 * Linux FM-TOWNS game pad driver based joystick emulation routine
 */
void fm_town_poll_joystick(void) {
      int res = Pad();

      /* get value of buttons */
      joy_b1 = (res>>4) & 0x01;
      joy_b2 = (res>>4) & 0x02;
      joy_b3 = (res>>4) & 0x04;
      joy_b4 = (res>>4) & 0x08;

      joy_up    = res & 0x01;
      joy_down  = res & 0x02;
      joy_left  = res & 0x04;
      joy_right = res & 0x08;

      evaluate_joy_moves();
}
#endif

#if ( defined USE_JOYSTICK ) && (defined X11_JOYSTICK )
/* 
 * Routine to manage joystick via X-Windows Input Extensions
 * should work in any X-Server that supports them
 */
void x11_poll_joystick(void) {

#ifndef USE_X11_JOYEVENTS
	/* perform a roudtrip query to joy device to ask state */
	XDeviceState    *xstate;
	XInputClass     *any;
	XValuatorState  *vinfo;
	XButtonState    *binfo;
	int j;
	xstate = XQueryDeviceState(display,xdevice);
	any = (XInputClass *)(xstate->data);
	for(j=0;j<xstate->num_classes;j++) {
	    switch(any->class) {
		case ButtonClass:
			binfo=(XButtonState *) any;
			joy_data.buttons=(int)binfo->buttons[0];
			break;
		case ValuatorClass:
			vinfo=(XValuatorState *) any;
			if (!swapjoyaxis ) {
			   joy_data.x = joy_orig.x + *(vinfo->valuators  );
			   joy_data.y = joy_orig.y + *(vinfo->valuators+1);
			} else {
			   joy_data.y = joy_orig.y + *(vinfo->valuators  );
			   joy_data.x = joy_orig.x + *(vinfo->valuators+1);
			}
			break;
		case KeyClass: /* no sense to use a extended key device */
		default: break;  /* unknown class: ignore */
	    }
	    any = (XInputClass *) ((char *) any+any->length);
	}
	XFreeDeviceState(xstate);
/* no idea why, but my Xserver starts in 2nd bit to encoding button masks */
#if 0
        joy_b1=joy_data.buttons & 0x01; joy_b2=joy_data.buttons & 0x02;
        joy_b3=joy_data.buttons & 0x04; joy_b4=joy_data.buttons & 0x08;
#else
        joy_b1=joy_data.buttons & 0x02; joy_b2=joy_data.buttons & 0x04;
        joy_b3=joy_data.buttons & 0x08; joy_b4=joy_data.buttons & 0x10;
#endif

#endif 
	evaluate_joy_moves();
}
#endif

#if ( defined USE_JOYSTICK ) && ( defined  I386_JOYSTICK )
/* 
 * Routine to manage PC clones joystick via standard driver 
 */
void i386_poll_joystick (void)
{
        int res;
#ifdef netbsd_i386
	res = read(joy_fd,&joy_data,sizeof(struct joystick) );
	if (res == sizeof(struct joystick )) {
#else
	res = read(joy_fd,&joy_data,sizeof(struct JS_DATA_TYPE) );
	if (res == sizeof(struct JS_DATA_TYPE )) {
#endif
	/* get value of buttons */
#ifdef netbsd_i386
	    joy_b1 = joy_data.b1;
	    joy_b2 = joy_data.b2;
#else
	    joy_b1 = joy_data.buttons & 0x01;	
	    joy_b2 = joy_data.buttons & 0x02;	
	    joy_b3 = joy_data.buttons & 0x04;	
	    joy_b4 = joy_data.buttons & 0x08;	
#endif
            /* now parse movements */
	    if (swapjoyaxis)
	    {
	        int a      = joy_data.y;
	        joy_data.y = joy_data.x;
	        joy_data.x = a;
	    }
	/* evaluate joystick movements */
	    evaluate_joy_moves();
	} /* if read ok */
	return;
}
#endif

void osd_analogjoy_read(int *analog_x, int *analog_y)
{
#ifdef USE_JOYSTICK
	if (analogstick)
	{
	      if (joy_data.x > joy_orig.x)
	         *analog_x = ( ( 128 * (joy_data.x - joy_orig.x) ) / 
	                     (joy_max_x  - joy_orig.x) );
	       else
	         *analog_x = ( ( 128 * (joy_data.x - joy_orig.x) ) / 
	                     (joy_orig.x - joy_min_x ) );

	      if (joy_data.y > joy_orig.y)
	         *analog_y = ( ( 128 * (joy_data.y - joy_orig.y) ) / 
	                     (joy_max_y  - joy_orig.y) );
	       else
	         *analog_y = ( ( 128 * (joy_data.y - joy_orig.y) ) / 
	                     (joy_orig.y - joy_min_y ) );
	}
	else
#endif
	      *analog_x = *analog_y = 0;
}
