;;;
;;; tm-sgnus.el --- tm-gnus module for September Gnus
;;;
;;; Copyright (C) 1995 Free Software Foundation, Inc.
;;; Copyright (C) 1995,1996 MORIOKA Tomohiko
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Maintainer: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Created: 1995/09/24
;;; Version: $Revision: 7.45 $
;;; Keywords: news, MIME, multimedia, multilingual, encoded-word
;;;
;;; This file is part of tm (Tools for MIME).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Code:

(require 'tl-str)
(require 'tl-list)
(require 'tl-misc)
(require 'tm-view)
(require 'gnus)

(eval-when-compile (require 'cl))


;;; @ version
;;;

(defconst tm-gnus/RCS-ID
  "$Id: tm-sgnus.el,v 7.45 1996/02/09 10:30:48 morioka Exp $")

(defconst tm-gnus/version
  (concat (get-version-string tm-gnus/RCS-ID) " for September"))


;;; @ variables
;;;

(defvar tm-gnus/automatic-mime-preview t
  "*If non-nil, show MIME processed article.
This variable is set to `gnus-show-mime'.")

(setq gnus-show-mime tm-gnus/automatic-mime-preview)


;;; @ command functions
;;;

(defun tm-gnus/view-message (arg)
  "MIME decode and play this message."
  (interactive "P")
  (let ((gnus-break-pages nil))
    (gnus-summary-select-article t t)
    )
  (pop-to-buffer gnus-original-article-buffer t)
  (let (buffer-read-only)
    (if (text-property-any (point-min) (point-max) 'invisible t)
	(remove-text-properties (point-min) (point-max)
				gnus-hidden-properties)
      ))
  (mime/viewer-mode nil nil nil gnus-original-article-buffer
		    gnus-article-buffer)
  )

(defun tm-gnus/summary-scroll-down ()
  "Scroll down one line current article."
  (interactive)
  (gnus-summary-scroll-up -1)
  )

(defun tm-gnus/summary-toggle-header (&optional arg)
  (interactive "P")
  (if tm-gnus/automatic-mime-preview
      (let* ((hidden
	      (save-excursion
		(set-buffer gnus-article-buffer)
		(text-property-any 
		 (goto-char (point-min)) (search-forward "\n\n")
		 'invisible t)
		))
	     (mime-viewer/redisplay t)
	     )
	(gnus-summary-select-article hidden t)
	)
    (gnus-summary-toggle-header arg))
  )

(define-key gnus-summary-mode-map "v" (function tm-gnus/view-message))
(define-key gnus-summary-mode-map
  "\e\r" (function tm-gnus/summary-scroll-down))
(substitute-key-definition
 'gnus-summary-toggle-header
 'tm-gnus/summary-toggle-header gnus-summary-mode-map)


;;; @ for tm-view
;;;

(defun tm-gnus/content-header-filter ()
  (goto-char (point-min))
  (mime-preview/cut-header)
  (tm-gnus/code-convert-region-to-emacs (point-min)(point-max)
					mime/default-coding-system)
  (mime/decode-message-header)
  )

(set-alist 'mime-viewer/content-header-filter-alist
	   'gnus-original-article-mode
	   (function tm-gnus/content-header-filter))

(fset 'tm-gnus/code-convert-region-to-emacs
      (symbol-function 'mime/code-convert-region-to-emacs))

(set-alist 'mime-viewer/code-converter-alist
	   'gnus-original-article-mode
	   (function tm-gnus/code-convert-region-to-emacs))

(defun mime-viewer/quitting-method-for-sgnus ()
  (if (not gnus-show-mime)
      (mime-viewer/kill-buffer))
  (delete-other-windows)
  (gnus-article-show-summary)
  (if (or (not gnus-show-mime)
	  (null gnus-have-all-headers))
      (gnus-summary-select-article nil t)
    ))

(call-after-loaded
 'tm-view
 (lambda ()
   (set-alist 'mime-viewer/quitting-method-alist
	      'gnus-original-article-mode
	      (function mime-viewer/quitting-method-for-sgnus))
   (set-alist 'mime-viewer/show-summary-method
	      'gnus-original-article-mode
	      (function mime-viewer/quitting-method-for-sgnus))
   ))


;;; @ for tm-partial
;;;

(defun tm-gnus/partial-preview-function ()
  (tm-gnus/view-message (gnus-summary-article-number))
  )

(call-after-loaded
 'tm-partial
 (lambda ()
   (set-atype 'mime/content-decoding-condition
	      '((type . "message/partial")
		(method . mime-article/grab-message/partials)
		(major-mode . gnus-original-article-mode)
		(summary-buffer-exp . gnus-summary-buffer)
		))
   (set-alist 'tm-partial/preview-article-method-alist
	      'gnus-original-article-mode
	      'tm-gnus/partial-preview-function)
   ))


;;; @ article filter
;;;

(defun tm-gnus/article-reset-variable ()
  (setq tm-gnus/automatic-mime-preview nil)
  )

(add-hook 'gnus-article-prepare-hook 'tm-gnus/article-reset-variable)

(defun tm-gnus/preview-article ()
  (make-local-variable 'tm:mother-button-dispatcher)
  (setq tm:mother-button-dispatcher
	(function gnus-article-push-button))
  (let ((mime-viewer/ignored-field-regexp "^:$"))
    (mime/viewer-mode nil nil nil gnus-original-article-buffer
		      gnus-article-buffer)
    )
  (setq tm-gnus/automatic-mime-preview t)
  (run-hooks 'tm-gnus/article-prepare-hook)
  )

(setq gnus-show-mime-method (function tm-gnus/preview-article))

(defun tm-gnus/article-decode-encoded-word ()
  (tm-gnus/code-convert-region-to-emacs (point-min)(point-max)
					mime/default-coding-system)
  (mime/decode-message-header)
  (run-hooks 'tm-gnus/article-prepare-hook)
  )

(setq gnus-decode-encoded-word-method
      (function tm-gnus/article-decode-encoded-word))

(if (boundp 'MULE)
    (progn
      (define-service-coding-system gnus-nntp-service nil *noconv*)
      (if (and (boundp 'nntp-server-process)
	       (processp nntp-server-process)
	       )
	  (set-process-coding-system nntp-server-process *noconv* *noconv*)
	)
      
      (or (fboundp 'tm-gnus/original-request-article)
	  (fset 'tm-gnus/original-request-article
		(symbol-function 'gnus-request-article))
	  )
      
      (defun gnus-request-article (article group &optional buffer)
	(let ((file-coding-system-for-read *noconv*))
	  (tm-gnus/original-request-article article group buffer)
	  ))

      (defun tm-gnus/prepare-save-mail-function ()
	(setq file-coding-system *noconv*)
	)
      (add-hook 'nnmbox-prepare-save-mail-hook
		'tm-gnus/prepare-save-mail-function)
      (add-hook 'nnbabyl-prepare-save-mail-hook
		'tm-gnus/prepare-save-mail-function)
      (add-hook 'nnml-prepare-save-mail-hook
		'tm-gnus/prepare-save-mail-function)
      (add-hook 'nnmh-prepare-save-mail-hook
		'tm-gnus/prepare-save-mail-function)
      (add-hook 'nnfolder-prepare-save-mail-hook
		'tm-gnus/prepare-save-mail-function)
      
      (defun tm-gnus/nnheader-find-file-noselect (&rest args)
	(let ((file-coding-system-for-read *noconv*))
	  (apply (function find-file-noselect) args)
	  ))
      (eval-after-load
       "nnheader"
       '(fset 'nnheader-find-file-noselect
	      'tm-gnus/nnheader-find-file-noselect)
       )
      
      (defun tm-gnus/nnmail-get-new-mail (&rest args)
	(let ((file-coding-system-for-read *noconv*))
	  (apply (function tm-gnus/original-nnmail-get-new-mail) args)
	  ))
      (eval-after-load
       "nnmail"
       '(progn (or (fboundp 'tm-gnus/original-nnmail-get-new-mail)
		   (fset 'tm-gnus/original-nnmail-get-new-mail
			 (symbol-function 'nnmail-get-new-mail))
		   )
	       (fset 'nnmail-get-new-mail 'tm-gnus/nnmail-get-new-mail)
	       ))
      ))


;;; @ summary filter
;;;

(cond ((not (boundp 'nnheader-encoded-words-decoding))
       (require 'tm-ew-d)
       
       (defun tm-gnus/decode-summary-from-and-subjects ()
	 (mapcar (lambda (header)
		   (let ((from (mail-header-from header))
			 (subj (mail-header-subject header))
			 )
		     (mail-header-set-from
		      header
		      (if from
			  (mime-eword/decode-string
			   (code-convert-string
			    from mime/default-coding-system *internal*))
			""))
		     (mail-header-set-subject
		      header
		      (if subj
			  (mime-eword/decode-string
			   (code-convert-string
			    subj mime/default-coding-system *internal*))
			""))
		     ))
		 gnus-newsgroup-headers))
       
       (add-hook 'gnus-select-group-hook
		 (function tm-gnus/decode-summary-from-and-subjects))
       ))


;;; @ for tm-edit
;;;

(defun tm-gnus/forward-insert-buffer (buffer)
  (save-excursion
    (save-restriction
      (if gnus-signature-before-forwarded-message
	  (goto-char (point-max))
	(goto-char (point-min))
	(re-search-forward
	 (concat "^" (regexp-quote mail-header-separator) "$"))
	(forward-line 1))
      ;; Narrow to the area we are to insert.
      (narrow-to-region (point) (point))
      ;; Insert the separators and the forwarded buffer.
      (mime-editor/insert-tag "message" "rfc822")
      (insert-buffer-substring buffer)
      ;; Delete any invisible text.
      (goto-char (point-min))
      (let (beg)
	(while (setq beg (next-single-property-change (point) 'invisible))
	  (goto-char beg)
	  (delete-region beg (or (next-single-property-change 
				  (point) 'invisible)
				 (point-max))))))))

(call-after-loaded
 'mime-setup
 (lambda ()
   (cond ((string-match "XEmacs" emacs-version)
	  (require 'gnus-msg)
	  (fset 'gnus-forward-insert-buffer 'tm-gnus/forward-insert-buffer)
	  )
	 (t
	  (eval-after-load
	   "gnus-msg"
	   '(fset 'gnus-forward-insert-buffer 'tm-gnus/forward-insert-buffer)
	   )
	  ))))


;;; @ for BBDB
;;;

(defun tm-gnus/bbdb-setup ()
  (if (memq 'bbdb/gnus-update-record gnus-article-prepare-hook)
      (progn
	(remove-hook 'gnus-article-prepare-hook 'bbdb/gnus-update-record)
	(add-hook 'tm-gnus/article-prepare-hook 'bbdb/gnus-update-record)
	)))

(add-hook 'gnus-startup-hook 'tm-gnus/bbdb-setup t)

(tm-gnus/bbdb-setup)

(call-after-loaded
 'bbdb
 (lambda ()
   (require 'tm-bbdb)
   ))


;;; @ end
;;;

(provide 'tm-sgnus)

;;; tm-sgnus.el ends here
