;==============================================================================

; file: "gsc.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(include "generic.scm")

;------------------------------------------------------------------------------

(set! make-global-environment ; import runtime macros into compilation env
  (lambda ()

    (define (extract-macros cte)
      (if (##cte-top? cte)
        (env-frame #f '())
        (let ((parent-cte (##cte-parent-cte cte)))
          (if (##cte-macro? cte)
            (env-macro (extract-macros parent-cte)
                       (##cte-macro-name cte)
                       (##cte-macro-def cte))
            (extract-macros parent-cte)))))

    (extract-macros (##cte-top-cte ##interaction-cte))))

(define (compile-file-to-c
         filename
         #!optional (options (absent-obj)) (output (absent-obj)))
  (force-vars (filename)
    (check-string filename (compile-file-to-c filename options output)
      (let ((opts
             (if (##eq? options (absent-obj))
               '()
               (force-vars (options)
                 options))))
        (if (##eq? output (absent-obj))
          (c#cf filename #f opts #f)
          (force-vars (output)
            (check-string output (compile-file-to-c filename options output)
              (c#cf filename #f opts output))))))))

(define (compile-file
         filename
         #!optional (options (absent-obj)))
  (force-vars (filename)
    (check-string filename (compile-file filename options)
      (let ((dynamic-cc ##dynamic-cc))
        (if (##procedure? dynamic-cc)

          (let* ((opts
                  (if (##eq? options (absent-obj))
                    '()
                    (force-vars (options)
                      options)))
                 (x
                  (c#cf filename #f opts #f)))
            (if x
              (let ((root
                     (##path-strip-directory
                      (##path-strip-extension filename))))
                (let loop ((version 1))
                  (let* ((root-obj
                          (##string-append
                            root
                            ".o"
                            (##number->string version 10)))
                         (root-obj-port
                          (##open-input-file root-obj)))
                    (if root-obj-port
                      (begin
                        (##close-port root-obj-port)
                        (loop (##fixnum.+ version 1)))
                      (let ((msg (dynamic-cc root root-obj)))
                        (if msg
                          (##runtime-error
                            msg
                            'compile-file
                            (##list filename options))
                          #t))))))
              #f))

          (##runtime-error
            "Dynamic loading is not available on this platform"
            'compile-file
            (##list filename options)))))))

(define (link-incremental
          modules
          #!optional (output (absent-obj)) (base (absent-obj)))
  (if (##not (##pair? modules))
    (check-string modules (link-incremental modules output base)
      #f)
    (let loop ((lst modules) (rev-mods '()))
      (force-vars (lst)
        (if (##pair? lst)
          (let ((s (##car lst)))
            (force-vars (s)
              (check-string s (link-incremental modules output base)
                (loop (##cdr lst) (##cons s rev-mods)))))
          (let* ((baselib
                  (if (##eq? base (absent-obj))
                    (##string-append (or (##path-expand "~~" 'absolute) "")
                                     "_gambc")
                    (force-vars (base)
                      base)))
                 (baselib-and-mods
                  (##cons baselib (##reverse rev-mods))))
            (check-string baselib (link-incremental modules output base)
              (if (##eq? output (absent-obj))
                (c#targ-linker #t
                               baselib-and-mods
                               #f
                               (##string-append (##car rev-mods) "_"))
                (force-vars (output)
                  (check-string output (link-incremental modules output base)
                    (c#targ-linker #t
                                   baselib-and-mods
                                   output
                                   #f)))))))))))

(define (link-flat modules #!optional (output (absent-obj)))
  (if (##not (##pair? modules))
    (check-string modules (link-flat modules output)
      #f)
    (let loop ((lst modules) (rev-mods '()))
      (force-vars (lst)
        (if (##pair? lst)
          (let ((s (##car lst)))
            (force-vars (s)
              (check-string s (link-flat modules output)
                (loop (##cdr lst) (##cons s rev-mods)))))
          (let ((mods (##reverse rev-mods)))
            (if (##eq? output (absent-obj))
              (c#targ-linker #f
                             mods
                             #f
                             (##string-append (##car rev-mods) "_"))
              (force-vars (output)
                (check-string output (link-flat modules output)
                  (c#targ-linker #f
                                 mods
                                 output
                                 #f))))))))))

;------------------------------------------------------------------------------

(define (##main)

  (##define-macro (gsi-or x) x)

  (define (process-initialization-file)

    (define (try filename)
      (let ((init (##open-input-file filename)))
        (if init
          (let* ((x
                  (##read-all-as-a-begin-expr-from-port
                   init
                   ##main-readtable
                   #t))
                 (src
                  (##vector-ref x 1)))
            (##eval-top src ##interaction-cte)
            #t)
          #f)))

    (or (try "gambc.scm")
        (let ((homedir (##path-expand "~" 'absolute)))
          (if homedir
            (try (##string-append homedir "gambc.scm"))))))

  (define (read-source-from-string str)
    (let* ((port
            (##open-input-string str))
           (x
            (##read-all-as-a-begin-expr-from-port
              port
              ##main-readtable
              #f)))
      (##vector-ref x 1)))

  (define (process-evaluate-options options)
    (let loop ((lst options))
      (if (##not (##null? lst))
        (let ((opt (##car lst)))
          (if (##string=? (##car opt) "e")
            (let ((src (read-source-from-string (##cdr opt))))
              (##eval-top src ##interaction-cte)))
          (loop (##cdr lst))))))

  (define (interpreter-batch-mode arguments)
    (let ((loaded-script? #f))
      (let loop ((lst arguments))
        (if (or loaded-script? (##null? lst))
          (##exit 0)
          (let ((filename (##car lst))
                (type-callback
                 (lambda (script?)
                   (if script?
                     (begin
                       (set! loaded-script? #t)
                       (set! ##processed-argv lst))))))
            (if (##string=? filename "-")
              (if (##port-isatty ##stdin)
                (##repl)
                (##load-source-from-port #f type-callback ##stdin))
              (##load filename type-callback #t #f))
            (loop (##cdr lst)))))))

  (define (compiler-batch-mode options arguments)

    (define (scm-file? file)
      (##member (##path-extension file) '(".scm" "")))

    (define (c-file? file)
      (##equal? (##path-extension file) ".c" #f))

    (let* ((opts (##map ##car options))
           (sym-opts (##map ##string->symbol opts)))

      (let loop1 ((lst arguments)
                  (nb-scm-files 0))
        (if (##not (##null? lst))

          (let ((arg (##car lst)))
            (loop1 (##cdr lst)
                   (if (scm-file? arg)
                     (##fixnum.+ nb-scm-files 1)
                     nb-scm-files)))

          (let* ((gen-c?
                  (##member "c" opts))
                 (gen-dynamic?
                  (##member "dynamic" opts))
                 (link?
                  (##not (or gen-c? gen-dynamic?)))
                 (output
                  (let ((x (##assoc "o" options)))
                    (cond ((##not x)
                           #f)
                          (gen-dynamic?
                           (##write-string
                            "*** WARNING -- Dynamic object file: \"o\" option ignored"
                            ##stderr)
                           (##newline ##stderr)
                           #f)
                          ((and (##not link?) (##fixnum.< 1 nb-scm-files))
                           (##write-string
                            "*** WARNING -- Multiple output files: \"o\" option ignored"
                            ##stderr)
                           (##newline ##stderr)
                           #f)
                          (else
                           (##cdr x)))))
                 (pre
                  (##assoc "prelude" options))
                 (post
                  (##assoc "postlude" options)))

            (if (or pre post)
              (set! wrap-program
                (lambda (program)
                  (##cons 'begin
                          (##append
                            (if pre
                              (let ((pre-src
                                     (read-source-from-string
                                      (##cdr pre))))
                                (##list (##desourcify pre-src)))
                              '())
                            (##cons program
                                    (if post
                                      (let ((post-src
                                             (read-source-from-string
                                              (##cdr post))))
                                        (##list (##desourcify post-src)))
                                      '())))))))

            (let loop2 ((lst arguments)
                        (rev-roots '()))
              (if (##not (##null? lst))

                (let* ((file (##car lst))
                       (root (##path-strip-extension file)))
                  (cond ((scm-file? file)
                         (if (##fixnum.< 1 nb-scm-files)
                           (begin
                             (##write-string file ##stdout)
                             (##write-string ":" ##stdout)
                             (##newline ##stdout)))
                         (if gen-dynamic?
                           (if (##not (compile-file file sym-opts))
                             (##exit 1))
                           (let ((out (or (and (##not link?) output)
                                          (absent-obj))))
                             (if (##not (compile-file-to-c file sym-opts out))
                               (##exit 1))))
                       (loop2 (##cdr lst) (##cons root rev-roots)))
                      ((c-file? file)
                       (loop2 (##cdr lst) (##cons root rev-roots)))
                      (else
                       (##write-string file ##stdout)
                       (##write-string ": ignored (file not recognized)"
                                       ##stdout)
                       (##newline ##stdout)
                       (loop2 (##cdr lst) rev-roots))))

              (let* ((flat?
                      (##member "flat" opts))
                     (base
                      (let ((x (##assoc "l" options)))
                        (cond ((##not x)
                               #f)
                              ((or (##not link?) flat?)
                               (##write-string
                                "*** WARNING -- No incremental link: \"l\" option ignored"
                                ##stderr)
                               (##newline ##stderr)
                               #f)
                              (else
                               (##cdr x))))))

                (if link?
                  (if (##not (##null? rev-roots))
                    (let ((roots (##reverse rev-roots)))
                      (if flat?
                        (link-flat
                         roots
                         (or output (absent-obj)))
                        (link-incremental
                         roots
                         (or output (absent-obj))
                         (or base (absent-obj))))))
                  (if flat?
                    (begin
                      (##write-string
                       "*** WARNING -- \"c\" or \"dynamic\" option was specified: \"flat\" option ignored"
                       ##stderr)
                      (##newline ##stderr))))))))))))

  (define (interactive-mode)

    (##define-macro (banner)
      (string-append "Gambit Version " c#compiler-version))

    (##write-string (banner) ##stdout)
    (##newline ##stdout)
    (##newline ##stdout)
    (##repl))

  (define (pipe-mode)
    (let ((port ##stdin))
      (let loop ()
        (let ((src (##read-expr-from-port port ##main-readtable)))
          (if (##eof-object? src)
            (##exit 0)
            (let ((val (##eval-top src ##interaction-cte)))
              (##write val ##stdout ##main-readtable #t)
              (##newline ##stdout)
              (loop)))))))

  (define (##split-command-line options-with-arguments cont)

    (define (option? arg)
      (and (##fixnum.< 1 (##string-length arg))
           (##char=? (##string-ref arg 0) #\-)))

    (define (convert-option arg)
      (##substring arg 1 (##string-length arg)))

    (let loop1 ((arguments (##cdr ##processed-argv))
                (rev-options '()))
      (if (and (##pair? arguments) (option? (##car arguments)))

        (let ((opt (convert-option (##car arguments)))
              (rest (##cdr arguments)))
          (if (##member opt options-with-arguments)
            (if (##pair? rest)
              (loop1 (##cdr rest)
                     (##cons (##cons opt (##car rest)) rev-options))
              (begin
                (##write-string
                  "*** WARNING -- Missing argument for option \""
                  ##stderr)
                (##write-string opt ##stderr)
                (##write-string "\"" ##stderr)
                (##newline ##stderr)
                (loop1 rest rev-options)))
            (loop1 rest
                   (##cons (##cons opt #f) rev-options))))

        (cont (##reverse rev-options) arguments))))

  (define (split-options lst options)
    (if (##null? lst)
      '(())
      (let* ((x (##car lst))
             (opt (##car x)))
        (if (##member opt options)
          (let ((y (split-options (##cdr lst) options)))
            (##cons (##cons x (##car y)) (##cdr y)))
          (##cons '() lst)))))

  (let ((gambcdir (##path-expand "~~" 'absolute)))
    (if gambcdir
      (##load (##string-append gambcdir "gambc") (lambda (script?) #f) #f #f)))

  (##split-command-line
    '("e" "o" "l" "prelude" "postlude")
    (lambda (raw-options arguments)
      (let* ((options1 (split-options raw-options '("f" "i")))
             (main-options (##car options1))
             (skip-initialization-file? (##assoc "f" main-options))
             (force-interpreter? (##assoc "i" main-options))
             (other-options
              (if (gsi-or force-interpreter?)
                (split-options
                 (##cdr options1)
                 '("e"))
                (split-options
                 (##cdr options1)
                 '("c" "flat" "o" "l" "prelude" "postlude" "dynamic"
                   "verbose" "report" "expansion" "gvm"
                   "check" "force" "debug"))))
             (known-options
              (##car other-options))
             (unknown-options
              (##cdr other-options)))
        (if (##not (##null? unknown-options))
          (begin
            (##write-string "*** WARNING -- Unprocessed options: " ##stderr)
            (##write (##map ##car unknown-options)
                     ##stderr
                     ##main-readtable
                     #f)
            (##newline ##stderr)))
        (if (##not skip-initialization-file?)
          (process-initialization-file))
        (process-evaluate-options known-options)
        (cond ((##not (##null? arguments))
               (if (gsi-or force-interpreter?)
                 (interpreter-batch-mode arguments)
                 (compiler-batch-mode known-options arguments)))
              ((##port-isatty ##stdin)
               (interactive-mode))
              (else
               (pipe-mode)))))))

(##namespace (""))

(##main)
