package java.awt;

import java.awt.Image;
import java.awt.image.ImageObserver;

/**
 * trojan horse to enable java.awt.Component derived instances without
 * peer objects. Such objects are no real widgets, but their geometry
 * can be handled like for any other Component (i.e. by LayoutManager).
 * With a little help from a (specialized) Container, Decorations can
 * provide a convenient (and consistent) way to add sophisticated
 * look to this Container (boxed, images, lots of text, lines etc.).
 * And this is even a lot faster than using specialized widgets because
 * it does not involve the GUI except of the drawing itself (no widget
 * creation etc.). Useful for dialogs with a lot of labels, groups etc.
 *
 * However, this class would not be required at all if java.awt.Component
 * would have a public constructor (which could be called outside of
 * the java.awt package)
 *
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
abstract public class Decoration
  extends Component
  implements ImageObserver
{
	protected int X;
	protected int Y;
	protected int Width;
	protected int Height;
/**
 * this field compensates the fact that Component.isShowing() refers to
 * the existence of a peer in order to determine if it is visible. Since
 * we don't have peers, we have to catch the initially invisible state
 * by means of another field
 */
	protected boolean Showing = false;

protected Decoration () {
}

public void blank () {
	Graphics g = getGraphics();
	blank( g, parent.getBackground() );
	g.dispose();
}

public void blank (Graphics g, Color cb) {
	g.setColor( cb );
	g.fillRect( X, Y, Width +1, Height +1);
}

public Graphics getGraphics () {
	return parent.getGraphics();
}

public void hide () {
	visible = Showing = false;
	blank();
}

public boolean imageUpdate ( Image img, int infoflags, int x, int y,
                      int width, int height) {
	if ( (infoflags & ImageObserver.ABORT) > 0 )
		return false;

	boolean ready = ( (infoflags & ImageObserver.ALLBITS) > 0);
	if ( Showing && ready) {
		repaint();
	}
	return !ready;
}

public boolean isShowing () {
	return ( Showing && (parent != null) && (Width > 0) &&
	         (Height > 0) && (parent.isShowing()) );
}

public static void paint ( Container cntr ) {
	Graphics g = cntr.getGraphics();
	if ( g != null ) {
		paint( g, cntr);
		g.dispose();
	}
}

/**
    providing a specialized paint method is the only responsibility of
    Decoration derived classes
    */
abstract public void paint ( Graphics g );

/**
    convenience function for container objects with decorations. Should be called
    from within the containers paint(Graphics) method
    */
public static void paint ( Graphics g, Container cntr ) {
	int i;
	Component c;

	/**** clean, but slow
	  int n = cntr.countComponents();	
	  for ( i=0; i<n; i++ ) {
	  if ( (c = cntr.getComponent( i)) instanceof Decoration )
	  c.paint( g);
	  }
	 *****/

	/**** dirty, but fast ****/
	for ( i=0; i<cntr.ncomponents; i++ ){
		if ( (c = cntr.component[i]) instanceof Decoration )
			c.paint( g);
	}
}

/**
    don't make a general parent repaint because of a single Decoration repaint
    request
    */
public void repaint () {
	Graphics g = getGraphics();
	if ( g != null) {
		paint( g);
		g.dispose();
	}
}

/**
    reshape is resolved to shadow the coordinates of this object. This is done
    not just to make these fields available to derived classes of other packages,
    but also to blank the old contents prior to a reshape (objet might have
    been shifted or moved, parent has to be redrawn correctly)
    */
public void reshape ( int x, int y, int w, int h ) {

	boolean changed = ( (X != x) || (Y != y) || (Width != w) || (Height != h) );

	if ( ! changed)
		return;

	if ( Width > 0 )
		blank();

	/*****************************************************
	  Windows bug workaround ( missing inset offset !!!)
	 *****************************************************/
	if ( System.getProperty( "os.name").startsWith( "Win") ){
		Insets in = parent.insets();
		x -= in.left;
		y -= in.top;
	}

	X = x;
	Y = y;
	Width = w;
	Height = h;

	super.reshape(X, Y, Width, Height);
}

public void show () {
	visible = Showing = true;
	Graphics g = getGraphics();
	paint( g);
	g.dispose();
}

public String toString () {
	return getClass().getName() + " [X: " + X + " ,Y: " + Y +
	           ", Width: " + Width + ", Height: " + Height +
	           ", Showing: " + Showing + "]";
}
}
