// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: columnbar.cc,v 1.10 1998/06/06 03:09:20 jgg Exp $
/* ######################################################################

   Column Bar - A horizontal bar providing titles for columns
   
   This class was designed to be used with a tree to provide it with
   columns. It accepts a string of \t,\e separated names and a list of
   longs indicating the column spacing.
   
   \t in the string indicates the item will be left aligned.
   \e in the string indicates the itme will be centered in the column.
   ie:
   "\eCenter\tLeft"
   Would produce two columns, one centered and one left aligned.
   "Left\tLeft"
   Would produce two columns, bothe centered.
   
   For the positions values a positive value indicates that it is an offset
   from the end of the previous column while a negative value indicates 
   it's size is a percentage of the whole bar, minus any fixed size
   columns.
   
   ##################################################################### */
									/*}}}*/
#include <deity/columnbar.h>
#include <iostream.h>

// ColumnBar::ColumnBar - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
ColumnBar::ColumnBar(string Columns,Widget *Parent) :
           BasicWidget(Parent), Columns(Columns)
{
   Count = 1;
   for (const char *I = Columns.begin(); I != Columns.end(); I++)
      if (*I == '\t' || *I == '\e' && I != Columns.begin())
	 Count++;
      
   Positions = 0;
   if (Count == 0)
      return;
   Positions = new long[Count];

   // Assign some default positions
   for (unsigned long I = 0; I != Count; I++)
      Positions[I] = -100/(signed)Count;

   if (TextGC::GC != 0)
      BorderWidth(0);
   else
      Margins(Point(2,1));
   Flag(Region);
}
									/*}}}*/
// ColumnBar::Render - Draw the bar					/*{{{*/
// ---------------------------------------------------------------------
/* */
void ColumnBar::Render(CombinedGC &GC)
{
   // Draw the basic background and setup for the main draw
   BasicRender(GC,false);
   Point Cur(iBorderWidth,iBorderWidth);
	     
   // Remove fixed size columns from the computation
   long Width = Pos.w;
   for (unsigned long I = 0; I != Count; I++)
      if (Positions[I] > 0) 
	 Width -= Positions[I];

   // Draw each element
   const char *Start = Columns.begin();
   int Index = 0;
   int LineWidth = GC.IsText()?0:2;
   for (const char *I = Columns.begin(); I != Columns.end(); I++)
   {
      if (*I != '\t' && *I != '\e')
	 continue;
      
      // Skip dead items
      if (I == Columns.begin())
      {
	 Start = I + 1;
	 continue;
      }
      
      if (Positions[Index] == -200)
      {
	 Start = I + 1;
	 Index++;
	 continue;
      }
      
      // Advance, -ve positions are % of width value 
      Point Temp = Cur;
      if (Positions[Index] > 0)
	 Cur.x += Positions[Index];
      else
	 Cur.x += Width*(-1)*Positions[Index]/100;

      // We have isolated a complete item
      Rect R = AbsRect(Temp.x,Temp.y,Cur.x - LineWidth,Pos.h - BorderY);
      if (Start != Columns.begin() && Start[-1] == '\e')
	 GC->DrawText(R,Point(R.w/2,iMargins.y),Start,I - Start,GenGC::XCenter);
      else
	 GC->DrawText(R,iMargins,Start,I - Start);
      
      Start = I + 1;

      // Draw divider lines
      if (GC.IsGraphic() == true)
      {
	 GC.gGC->SetColor(iBorderUl);
	 GC.gGC->Line(Point(Cur.x-1,Cur.y),
			    Point(Cur.x-1,Pos.h - iBorderWidth));
	 GC.gGC->SetColor(iBorderLr);
	 GC.gGC->Line(Point(Cur.x-2,Cur.y),
			    Point(Cur.x-2,Pos.h - iBorderWidth));
	 GC.gGC->SetColor(iColor);
      }
      
      Index++;
   }
   
   if (Start >= Columns.end())
      return;

   Rect R = AbsRect(Cur.x,Cur.y,Pos.w - BorderX,Pos.h - BorderY);
   GC->DrawText(R,iMargins,Start,Columns.end() - Start);
}
									/*}}}*/
// ColumnBar::IdealSize - Compute the height of the widget		/*{{{*/
// ---------------------------------------------------------------------
/* */
Point ColumnBar::IdealSize()
{
   return Point(Pos.w,GenGC::GC->ExtentText("").h + 
		2*iBorderWidth + 2*iMargins.y);
}
									/*}}}*/
// ColumnBar::PixelPositions - Return the pixel offsets of each divider	/*{{{*/
// ---------------------------------------------------------------------
/* */
long ColumnBar::PixelPositions(long *Pos, int Max)
{
   Point Cur(iBorderWidth,iBorderWidth);
   
   // Remove fixed size columns from the computation
   long Width = this->Pos.w;
   for (unsigned long I = 0; I != Count; I++)
      if (Positions[I] > 0) 
	 Width -= Positions[I];

   const char *Start = Columns.begin();
   int Index = 0;
   for (const char *I = Columns.begin(); I != Columns.end(); I++)
   {
      if (*I != '\t' && *I != '\e')
	 continue;
      
      Start = I + 1;

      if (Positions[Index] == -200)
      {
	 Pos[Index] = -1;
	 Index++;
	 continue;
      }
	  
      Pos[Index] = Cur.x;
      
      // Advance, -ve positions are % of width value 
      if (Positions[Index] > 0)
	 Cur.x += Positions[Index];
      else
	 Cur.x += Width*(-1)*Positions[Index]/100;
      Index++;

      if (Index >= Max)
	 return Index;
   }
   Pos[Index] = Cur.x;
   
   return Index;
}
									/*}}}*/
// ColumnBar::Position - Set the divider positions			/*{{{*/
// ---------------------------------------------------------------------
/* */
void ColumnBar::Position(long *Pos, int Max)
{
   // Set the positions
   for (int I = 0; I != (signed)Count; I++)
   {
      if (I >= Max)
	 Positions[I] = -100/(signed)Count;
      else
	 Positions[I] = Pos[I];
   }
}
									/*}}}*/
