// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: apt-get.cc,v 1.78 1998/11/20 01:51:08 jgg Exp $
/* ######################################################################
   
   apt-get - Cover for dpkg
   
   This is an allout cover for dpkg implementing a safer front end. It is
   based largely on pkglib.

   The syntax is different, 
      apt-get command args
   Where command is:
      update - Resyncronize the package files from their sources
      upgrade - Smart-Download the newest versions of all packages
      dselect-upgrade - Follows dselect's changes to the Status: field
                       and installes new and removes old packages
      dist-upgrade - Powerfull upgrader designed to handle the issues with
                    a new distribution.
      install - Download and install a given package (by name, not by .deb)
      check - Update the package cache and check for broken packages
      clean - Erase the .debs downloaded to /var/cache/apt/archives and
              the partial dir too

   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#define _GNU_SOURCE 1         // For getopt_long
#define _BSD_SOURCE 1         // For gettimeofday

#include <iostream.h>
#include <pkglib/error.h>
#include <pkglib/aquire.h>
#include <pkglib/sourcelist.h>
#include <pkglib/aquire.h>
#include <pkglib/depcache.h>
#include <pkglib/dpkgpm.h>
#include <pkglib/pkgelement.h>
#include <pkglib/algorithms.h>
#include <pkglib/debadmin.h>
#include <options.h>
#include <strutl.h>
#include <fileutl.h>
#include <system.h>

// For getwinsize
#include <termios.h>
#include <sys/ioctl.h>
#include <signal.h>

#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <signal.h>

#ifdef _GNU_SOURCE
#include <getopt.h>
#endif

									/*}}}*/

// Globals								/*{{{*/
enum {QNoisy = 0, QScript = 1, QReallyQuiet = 2} Quiet = QNoisy;
bool SigStop = false;
bool DownloadOnly = false;
bool NoAct = false;
bool AssumeY = false;
bool FixBroken = false;
bool FixMissing = false;
int NoUpgrade = 0;
int ScreenWidth = 78;
									/*}}}*/
// Prototypes								/*{{{*/
pkgDepCache *Dep = 0;
bool Check();
bool DoCheck(pkgDepCache &Cache);
bool Install(pkgSourceList &List,int argc,char *argv[]);
bool Check(pkgSourceList &List);
bool Update(pkgSourceList &List);
bool Upgrade(pkgSourceList &List);
bool YnPrompt();
void ShowBroken(pkgDepCache &Cache = *Dep);
void DependsDump(pkgCache::VerIterator V);
bool ShowHelp();
									/*}}}*/

// Aquire Class								/*{{{*/
// ---------------------------------------------------------------------
/* */
class Aquire : public pkgAquire
{
   protected:

   vector<Worker *> Active;
   char BlankLine[300];
   char LastLine[300];
   bool NewProgress;
   struct timeval StartTime;
   struct timeval Time;
   unsigned long Bytes;
   double CPS;
   
   virtual bool RegisterWorker(Worker *Work) 
   {
      Active.push_back(Work);
      return true;
   };

   bool Die() {return false;};
   bool Log(LogTypes Type,Item *Itm,string Text);
   void PrintProgress();
   
   public:
   
   bool Run();
};
									/*}}}*/
// Aquire::Run - Run the select loop					/*{{{*/
// ---------------------------------------------------------------------
/* This runs the select loop and monitors the status of all the worker
   threads. */
bool Aquire::Run()
{
   if (pkgAquire::Run() == false)
      return Die();

   struct timeval tv;
   tv.tv_sec = 0;
   tv.tv_usec = 500000;
   LastLine[0] = 0;
   BlankLine[0] = 0;
   Bytes = 0;
   CPS = 0;
   gettimeofday(&Time,0);
   StartTime = Time;
   while (Active.empty() == false)
   {
      // Configure the select loop.
      int n = 0;
      fd_set rfds;
      FD_ZERO(&rfds);
      
      for (vector<Worker *>::iterator I = Active.begin(); I != Active.end(); I++)
      {
	 FD_SET((*I)->Fd,&rfds);
	 n = MAX(n,(*I)->Fd);
      }
      
      int Res;
      if ((Res = select(n+1,&rfds,0,0,&tv)) < 0 && errno != EINTR)
      {
	 _error->Errno("select","Error during select");
	 return Die();
      }
      
      /* Run the processes, carefull, Read will change the contents of 
         active and invalidate the iterator */
      for (unsigned int Cur = 0; Cur < Active.size(); Cur++)
      {
	 vector<Worker *>::iterator I = Active.begin() + Cur;
	 
	 if (!FD_ISSET((*I)->Fd,&rfds))
	    continue;

	 if ((*I)->Read() == false)
	 {
	    I = Active.begin() + Cur;
	    delete *I;
	    Active.erase(I);
	    Cur--;
	 }
      }

      if (Res == 0 || NewProgress == true)
      {
	 tv.tv_sec = 0;
	 tv.tv_usec = 500000;
	 PrintProgress();
      }      
   }

   if (Quiet < 1 && SigStop == false)
      cout << '\r' << BlankLine << '\r' << flush;

   struct timeval NewTime;
   gettimeofday(&NewTime,0);

   unsigned long Total = 0;
   for (vector<Item *>::iterator I = List.begin(); I != List.end(); I++)
      if ((*I)->ErrorText.empty() == true)
	 Total += (*I)->ExpectedSize;

   if (Total != 0)
   {
      // Compute the delta time with full accuracy
      long usdiff = NewTime.tv_usec - StartTime.tv_usec;
      long sdiff = NewTime.tv_sec - StartTime.tv_sec;
      
      // Borrow
      if (usdiff < 0)
      {
	 usdiff += 1000000;
	 sdiff--;
      }
      if (sdiff == 0)
	 usdiff += 5000;
      
      // Compute the CPS value
      CPS = Total/(sdiff + usdiff/1000000.0);

      if (Quiet < 2 && SigStop == false)
	 cout << "Fetched " << SizeToStr(Total) << " in " << TimeToStr(sdiff) <<
	        " (" << SizeToStr(CPS) << "/s)" << endl;
   }
   
   bool Ok = true;
   for (vector<Item *>::iterator I = List.begin(); I != List.end(); I++)
   {
      if ((*I)->Status == pkgAquire::Item::Error)
      {
	 if (FixMissing == false)
	 {
	    cerr << "ERROR " << (*I)->URI << endl;
	    cerr << "  " << (*I)->ErrorText << endl;
	 }
	 Ok = false;
      }
   }

   return Ok;
};
									/*}}}*/
// Aquire::Log - Called when things happen				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Aquire::Log(LogTypes Type,Item *Cur,string Text)
{
   if (Cur == 0)
      return true;

   // We block off signals while we fiddle this
   sigset_t Sigs,OldSigs;
   sigemptyset(&Sigs);
   sigaddset(&Sigs,SIGTSTP);
   sigaddset(&Sigs,SIGCONT);
   sigprocmask(SIG_BLOCK,&Sigs,&OldSigs);

   if (SigStop == true)
   {    
      sigprocmask(SIG_UNBLOCK,&OldSigs,0);
      return true;
   }
   
   if (Type == StatNewFile && Cur->Status == pkgAquire::Item::Downloading)
   {
      if (Quiet < 1)
	 cout << '\r' << BlankLine << '\r';
      if (Quiet < 2)
      {
	 if (Cur->ExpectedSize != 0)
	    cout << "Get " << Cur->GetInfo << " [" <<
	         SizeToStr(Cur->ExpectedSize) << ']' << endl;
	 else
	    cout << "Get " << Cur->GetInfo << endl;
      }
      
      cout << LastLine << flush;
      NewProgress = true;
   }

   if (Type == StatError)
   {
      if (Quiet < 1)
	 cout << '\r' << BlankLine << '\r';
      cout << "Error " << Cur->GetInfo << endl << ' ' << Text << endl;

      cout << LastLine << flush;
      NewProgress = true;
   }
   sigprocmask(SIG_UNBLOCK,&OldSigs,0);
   
   return true;
}
									/*}}}*/
// Aquire::PrintProgress - Called to output progress info		/*{{{*/
// ---------------------------------------------------------------------
/* */
void Aquire::PrintProgress()
{
   if (Quiet >= 1 || SigStop == true)
   {
      NewProgress = false;
      return;
   }
   
   // Compute the total expected bytes and total downloaded bytes
   unsigned int Unknown = 0;
   unsigned long Complete = 0;
   unsigned long Total = 0;
   for (vector<Item *>::iterator I = List.begin(); I != List.end(); I++)
   {
      if ((*I)->ExpectedSize == 0)
      {
	 if ((*I)->EstimatedSize == 0)
	    Unknown++;
	 else
	    Total += (*I)->EstimatedSize;
	 continue;
      }
      
      Total += (*I)->ExpectedSize;
      if ((*I)->Status != Item::Waiting && (*I)->Status != Item::Downloading)
	 Complete += (*I)->ExpectedSize;
   }
   
   // Normalize the figures and account for unknown size downloads
   if (Total <= 0)
      Total = 1;
   if (Unknown == List.size())
      Total = Unknown;
   else
      Total += Total/(List.size() - Unknown)*Unknown;

   string TmpBlank = BlankLine;
   strcpy(BlankLine,"   ");
   BlankLine[4] = 0;
   BlankLine[5] = 0;
   char Tmp[300];
   for (vector<Worker *>::iterator I = Active.begin(); I != Active.end(); I++)
   {
      char *Mode[] = {"Err:","Wgt:","","Cmp:","Prt:","Dne:"};
      
      if ((*I)->Cur != 0)
      {
	 Item &Cur = *(*I)->Cur;

	 // Grab the current file size
	 struct stat Buf;
	 if ((*I)->OutputFile.empty() == true ||
	     stat((*I)->OutputFile.c_str(),&Buf) != 0)
	    Buf.st_size = 0;
	 Complete += Buf.st_size;
	 
	 // Info string
	 const char *Info = "";
	 if ((*I)->Info.empty() == false)
	    Info = (*I)->Info.c_str();
	 if (strcmp(Info,"Downloading") == 0)
	    Info = "";
	 
	 // File
	 string::size_type End = Cur.GetInfo.rfind(' ');
	 if (End == string::npos)
	    End = 0;
	 else
	    End++;
	 const char *File = Cur.GetInfo.c_str() + End;

	 if (Cur.ExpectedSize != 0)
	 {
	    if (Info[0] != 0)
	    {
	       if (Cur.ExpectedSize == (unsigned)Buf.st_size)
		  sprintf(Tmp," [%s%s `%s' %s]",Mode[Cur.Status],
			  File,Info,SizeToStr(Cur.ExpectedSize).c_str());
	       else
		  sprintf(Tmp," [%s%s `%s' %lu/%s %li%%]",Mode[Cur.Status],
			  File,Info,Buf.st_size,
			  SizeToStr(Cur.ExpectedSize).c_str(),
			  long(double(Buf.st_size*100.0)/double(Cur.ExpectedSize)));
	    }
	    else
	       sprintf(Tmp," [%s%s %lu/%s %li%%]",Mode[Cur.Status],
		       File,Buf.st_size,
		       SizeToStr(Cur.ExpectedSize).c_str(),
		       long(double(Buf.st_size*100.0)/double(Cur.ExpectedSize)));
	 }
	 else
	    sprintf(Tmp," [%s%s `%s' %lu]",Mode[Cur.Status],
		    File,Info,Buf.st_size);
      }
      else
	 sprintf(Tmp," [Forking]");
      strcat(BlankLine,Tmp);
      
      // Just in case we have a -large- number of sources.
      if (strlen(BlankLine) > sizeof(BlankLine) - 70)
	 break;
   }

   struct timeval NewTime;
   gettimeofday(&NewTime,0);
   if (NewTime.tv_sec - Time.tv_sec == 6 && NewTime.tv_usec > Time.tv_usec ||
       NewTime.tv_sec - Time.tv_sec > 6)
   {
      // Compute the delta time with full accuracy
      long usdiff = NewTime.tv_usec - Time.tv_usec;
      long sdiff = NewTime.tv_sec - Time.tv_sec;

      // Borrow
      if (usdiff < 0)
      {
	 usdiff += 1000000;
	 sdiff--;
      }

      // Compute the CPS value
      CPS = (Complete - Bytes)/(sdiff + usdiff/1000000.0);
      Bytes = Complete;
      Time = NewTime;
   }

   BlankLine[sprintf(BlankLine,"%li%%",long(double(Complete*100.0)/double(Total)))] = ' ';

   if (CPS != 0)
   {
      unsigned long ETA = (unsigned long)((Total - Complete)/CPS);
      sprintf(Tmp," %s/s %s",SizeToStr(CPS).c_str(),TimeToStr(ETA).c_str());
      int Len = strlen(BlankLine);
      int LenT = strlen(Tmp);
      if (Len + LenT < ScreenWidth)
      {
	 memset(BlankLine + Len,' ',ScreenWidth - Len);
	 strcpy(BlankLine + ScreenWidth - LenT,Tmp);
      }
   }
   
   BlankLine[ScreenWidth] = 0;

   // Dont redraw duplicates
   if (strcmp(BlankLine,LastLine) == 0 && NewProgress == false)
   {
      memset(BlankLine,' ',strlen(BlankLine));
      return;
   }
   
   NewProgress = false;
   strcpy(LastLine,BlankLine);

   // We block off signals while we fiddle this
   sigset_t Sigs,OldSigs;
   sigemptyset(&Sigs);
   sigaddset(&Sigs,SIGTSTP);
   sigaddset(&Sigs,SIGCONT);
   sigprocmask(SIG_BLOCK,&Sigs,&OldSigs);
   if (SigStop == false)
      cout << '\r' << TmpBlank << '\r' << BlankLine << flush;
   sigprocmask(SIG_UNBLOCK,&OldSigs,0);
   
   memset(BlankLine,' ',strlen(BlankLine));
}
									/*}}}*/

// YnPrompt - Yes No Prompt.						/*{{{*/
// ---------------------------------------------------------------------
/* Returns true on a Yes.*/
bool YnPrompt()
{
   if (AssumeY == true)
   {
      cout << 'Y' << endl;
      return true;
   }
   
   char C = 0;
   char Jnk = 0;
   read(STDIN_FILENO,&C,1);
   while (C != '\n' && Jnk != '\n') read(STDIN_FILENO,&Jnk,1);
   
   if (!(C == 'Y' || C == 'y' || C == '\n' || C == '\r'))
      return false;
   return true;
}
									/*}}}*/
// Stats - Show some statistics						/*{{{*/
// ---------------------------------------------------------------------
/* */
void Stats()
{
   unsigned long Upgrade = 0;
   unsigned long Install = 0;
   for (pkgCache::PkgIterator I = Dep->PkgBegin(); I.end() == false; I++)
   {
      if ((*Dep)[I].NewInstall() == true)
	 Install++;
      else
	 if ((*Dep)[I].Upgrade() == true)
	    Upgrade++;
   }   

   cout << Upgrade << " packages upgraded, " << 
      Install << " newly installed, " <<
      Dep->DelCount() << " to remove and " << 
      Dep->KeepCount() << " not upgraded." << endl;

   if (Dep->BadCount() != 0)
      cout << Dep->BadCount() << " packages not fully installed or removed." << endl;
}
									/*}}}*/
// ShowList - Show a list						/*{{{*/
// ---------------------------------------------------------------------
/* This prints out a string of space seperated words with a title and 
   a two space indent line wraped to the current screen width. */
void ShowList(string Title,string List)
{
   if (List.empty() == true)
      return;

   // Acount for the leading space
   int ScreenWidth = ::ScreenWidth - 3;
      
   cout << Title << endl;
   string::size_type Start = 0;
   while (Start < List.size())
   {
      string::size_type End;
      if (Start + ScreenWidth >= List.size())
	 End = List.size();
      else
	 End = List.rfind(' ',Start+ScreenWidth);

      if (End == string::npos || End < Start)
	 End = Start + ScreenWidth;
      cout << "  " << string(List,Start,End - Start) << endl;
      Start = End + 1;
   }   
}
									/*}}}*/
// ShowBroken - Debugging aide						/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowBroken(pkgDepCache &Cache)
{
   cout << "Sorry, but the following packages are broken - this means they have unmet" << endl;
   cout << "dependencies:" << endl;
   pkgCache::PkgIterator I = Cache.PkgBegin();
   for (;I.end() != true; I++)
   {
      if (Cache[I].InstBroken() == true)
      {
	 // Print out each package and the failed dependencies
	 cout <<"  " <<  I.Name() << ":";
	 for (pkgCache::DepIterator D = Cache[I].InstVerIter(Cache).DependsList(); D.end() == false; D++)
	 {
	    if (Dep->IsImportantDep(D) == false || (Cache[D] &
						    pkgDepCache::DepInstall) != 0)
	       continue;
	    
	    if (D->Type == pkgDEP_Conflicts)
	       cout << " Conflicts:" << D.TargetPkg().Name();
	    else
	       cout << " Depends:" << D.TargetPkg().Name();
	 }	    
	 cout << endl;
      }   
   }   
}
									/*}}}*/
// ShowNew - Show packages to newly install				/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowNew()
{
   /* Print out a list of packages that are going to be removed extra
      to what the user asked */
   pkgCache::PkgIterator I = Dep->PkgBegin();
   string List;
   for (;I.end() != true; I++)
      if ((*Dep)[I].NewInstall() == true)
	 List += string(I.Name()) + " ";
   ShowList("The following NEW packages will be installed:",List);
}
									/*}}}*/
// ShowDel - Show packages to delete					/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowDel()
{
   /* Print out a list of packages that are going to be removed extra
      to what the user asked */
   pkgCache::PkgIterator I = Dep->PkgBegin();
   string List;
   for (;I.end() != true; I++)
      if ((*Dep)[I].Delete() == true)
	 List += string(I.Name()) + " ";
   ShowList("The following packages will be REMOVED:",List);
}
									/*}}}*/
// ShowKept - Show kept packages					/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowKept()
{
   pkgCache::PkgIterator I = Dep->PkgBegin();
   string List;
   for (;I.end() != true; I++)
   {	 
      // Not interesting
      if ((*Dep)[I].Upgrade() == true || (*Dep)[I].Upgradable() == false ||
	  I->CurrentVer == 0 || (*Dep)[I].Delete() == true)
	 continue;
      
	 List += string(I.Name()) + " ";
   }
   ShowList("The following packages have been kept back",List);
}
									/*}}}*/
// ShowHold - Show held but changed packages				/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowHold()
{
   pkgCache::PkgIterator I = Dep->PkgBegin();
   string List;
   for (;I.end() != true; I++)
   {
      if ((*Dep)[I].InstallVer != (pkgCache::Version *)I.CurrentVer() &&
	  I->SelectedState == pkgSTATE_Hold)
	 List += string(I.Name()) + " ";
   }

   ShowList("The following held packages will be changed:",List);
}
									/*}}}*/
// ShowEssential - Show an essential package warning			/*{{{*/
// ---------------------------------------------------------------------
/* */
void ShowEssential()
{
   pkgCache::PkgIterator I = Dep->PkgBegin();
   string List;
   for (;I.end() != true; I++)
   {
      if ((I->Flags & pkgFLAG_Essential) != pkgFLAG_Essential)
	 continue;
      
      if ((*Dep)[I].Delete() == true)
	 List += string(I.Name()) + " ";
      
      if (I->CurrentVer == 0)
	 continue;

      // Print out any essential package depenendents that are to be removed
      for (pkgDepCache::DepIterator D = I.CurrentVer().DependsList(); D.end() == false; D++)
      {
	 if ((*Dep)[D.SmartTargetPkg()].Delete() == true)
	    List += string(D.SmartTargetPkg().Name()) + " ";
      }      
   }
   
   if (List.empty() == false)
	    cout << "WARNING: The following essential packages will be removed" << endl;
   ShowList("This should NOT be done unless you know exactly what you are doing!",List);
}
									/*}}}*/
// DoArchives - Fetch all of the required archives and do the install	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool DoArchives(pkgSourceList &List,bool Ask)
{
   if (NoAct == true)
   {
      pkgSimulate PM(*Dep);
      return PM.DoInstall();
   }
   
   pkgDPkgPM PM(*Dep);
   Aquire Get;
   
   // Generate the archive list
   if (PM.GetArchives(List,Get) == false)
      return false;
   
   cout << "Need to get ";
   if ((unsigned)Dep->DebSize() != Get.FetchBytes())
      cout << SizeToStr(Get.FetchBytes()) << '/' << SizeToStr(Dep->DebSize());
   else
      cout << SizeToStr(Dep->DebSize());
      
   cout << " of archives. After unpacking ";
   
   if (Dep->UsrSize() >= 0)
      cout << SizeToStr(Dep->UsrSize()) << " will be used." << endl;
   else
      cout << SizeToStr(-1*Dep->UsrSize()) << " will be freed." << endl;

   if (_error->PendingError() == true)
      return false;

   if (Ask == true && (Quiet < 2 || AssumeY == false))
   {
      cout << "Do you want to continue? [Y/n] " << flush;
      if (YnPrompt() == false)
	 return false;
   }      
   
   // Download the archives
   bool Result = Get.Run();
   if (Result == false && FixMissing == false)
      return false;

   // We block off signals while we fiddle this
   sigset_t Sigs,OldSigs;
   sigemptyset(&Sigs);
   sigaddset(&Sigs,SIGTSTP);
   sigaddset(&Sigs,SIGCONT);
   sigprocmask(SIG_BLOCK,&Sigs,&OldSigs);
   if (SigStop == true)
   {
      sigprocmask(SIG_UNBLOCK,&OldSigs,0);
      return false;
   }
   sigprocmask(SIG_UNBLOCK,&OldSigs,0);
   
   if (FixMissing == true && PM.FixMissing() == false)
      return _error->Error("Unable to fix missing files");
   
   // Do the installation
   if (DownloadOnly == false)
      return PM.DoInstall() && Result;
   
   return Result;
}
									/*}}}*/
// GetScreenWidth - Determine the size of the screen			/*{{{*/
// ---------------------------------------------------------------------
/* This is ripped from slang */
void GetScreenWidth()
{
#ifdef TIOCGWINSZ
   struct winsize wind_struct;

   do
   {
      if ((ioctl(1,TIOCGWINSZ,&wind_struct) == 0)
	  || (ioctl(0,TIOCGWINSZ,&wind_struct) == 0)
	  || (ioctl(2,TIOCGWINSZ,&wind_struct) == 0))
      {
	 // Some terminals do not like drawing right to the edge (qansi)
	 ScreenWidth = wind_struct.ws_col - 1; 
	 break;
      }
   }
   while (errno == EINTR);
#endif   
}
									/*}}}*/
// SigWinch - Handle changing the screen width				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SigWinch(int)
{
   GetScreenWidth();
}
									/*}}}*/
// SigTstp - Terminal Stop						/*{{{*/
// ---------------------------------------------------------------------
/* */
void SigTstp(int)
{
   SigStop = true;

   kill(getpid(), SIGSTOP);
}
									/*}}}*/
// SigCont - Terminal Stop						/*{{{*/
// ---------------------------------------------------------------------
/* */
void SigCont(int)
{
   if (tcgetpgrp(STDOUT_FILENO) == getpgrp())
   {
      SigStop = false;
      const char *Text = "Continued, resuming previous operation\n";
      write(STDOUT_FILENO,Text,strlen(Text));
   }
   else
   {
      const char *Text = "Backgrounded, resuming in silent download only mode\n";
      write(STDOUT_FILENO,Text,strlen(Text));
   }      
}
									/*}}}*/

// Update - Download newer package files				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Update(pkgSourceList &List)
{
   FixMissing = false;
   
   Aquire Get;
   if (pkgUpdateMeta(List,Get) == false)
      return false;

   if (Get.Clean() == false)
      return false;
   
   if (Get.Run() == false)
      return false;

   pkgDebAdmin DebAdmin;
   if (DebAdmin.Open() == false || DebAdmin.ReadLock() == false)
      return false;

   if (Quiet < 2 && SigStop == false)
      cout << "Updating package file cache..." << flush;
   if (pkgMakeSrcCache(List) == false)
   {
      if (Quiet < 2 && SigStop == false)
	 cout << endl;
      return false;
   }

   if (Quiet < 2 && SigStop == false)
      cout << "done" << endl;

   Check();
   return true;
}
									/*}}}*/
// Check - Perform a full check of all caches				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Check(pkgSourceList &List)
{
   if (Quiet < 2 && SigStop == false)
      cout << "Updating package file cache..." << flush;
   if (pkgMakeSrcCache(List) == false)
   {
      if (Quiet < 2 && SigStop == false)
	 cout << endl;
      return false;
   }

   if (Quiet < 2 && SigStop == false)
      cout << "done" << endl;
   return Check();
}
									/*}}}*/
// Check - Check the system for problems				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Check()
{
   if (Dep != 0)
      return true;

   // Grab locks for the state and cache directories.
   if (GetLock(PKG_DEB_CACHE "lock") == -1)
      return _error->Error("Couldn't lock the cache dir, %s another process "
			   "is using it",PKG_DEB_CACHE);
   if (GetLock(PKG_DEB_STATE "lock") == -1)
      return _error->Error("Couldn't lock the state dir, %s another process "
			   "is using it",PKG_DEB_STATE);
   pkgDebAdmin DebAdmin;
   if (DebAdmin.Open() == false || DebAdmin.ReadLock() == false)
      return false;
   
   if (Quiet < 2 && SigStop == false)
      cout << "Updating package status cache..." << flush;
   if (pkgMakeStatusCache() == false)
   {
      _error->Error("You should probably re-run apt-get update");
      return false;
   }

   if (Quiet < 2 && SigStop == false)
      cout << "done" << endl << "Checking system integrity..." << flush;

   Dep = new pkgDepCache(true,true);
   if (_error->PendingError() == true)
   {
      delete Dep;
      Dep = 0;
      if (Quiet < 2 && SigStop == false)
	 cout << "Bad Cache" << endl;
      return false;
   }

   // Check and apply status transforms
   if (DoCheck(*Dep) == false || pkgApplyStatus(*Dep) == false)
   {
      if (Quiet < 2 && SigStop == false)
      {
	 cout << "dependency error" << endl;
	 if (FixBroken == false)
	 {
	    cout << "You might want to run `apt-get -f install' to correct these." << endl;
	    ShowBroken();
	 }	 
      }
      
      if (FixBroken == true)
      {	    
	 if (Quiet < 2 && SigStop == false)
	    cout << "Correcting dependencies..." << flush;
	 if (pkgFixBroken(*Dep) == false)
	 {
	    if (Quiet < 2 && SigStop == false)
	       cout << "failed"<< endl;
	    return false;
	 }
      }
      else
	 return false;
   }
   
   if (Quiet < 2 && SigStop == false)
      cout << "ok" << endl;
   
   return true;
}
									/*}}}*/
// DoCheck - Check the system for problems				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool DoCheck(pkgDepCache &Cache)
{   
   if (Cache.DelCount() != 0 || Cache.InstCount() != 0)
      return _error->Error("Internal Error, non-zero counts");
   
   // Nothing is broken
   if (Cache.BrokenCount() == 0)
      return true;
   return false;
}
									/*}}}*/
// Install - Install Packages						/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Install(pkgSourceList &List,int argc,char *argv[])
{
   // Setup Dep
   if (Check() == false || Dep == 0)
      return false;
   
   int ExpectedInst = 0;
   pkgProblemResolver Fix(*Dep);
   
   for (int I = 0; I != argc; I++)
   {
      if (argv[I][0] == 0 || argv[I][0] == '-')
	 continue;
      
      // Duplicate the string
      unsigned int Length = strlen(argv[I]);
      char S[300];
      if (Length >= sizeof(S))
	 continue;
      strcpy(S,argv[I]);
      
      // See if we are removing the package
      bool Remove = false;
      if (S[Length - 1] == '-')
      {
	 Remove = true;
	 S[--Length] = 0;
      }
      
      // Locate the package
      pkgCache::PkgIterator Pkg = Dep->FindPkg(S);
      if (Pkg.end() == true)
	 return _error->Error("Couldn't find package %s",S);
      
      // Handle the no-upgrade case
      if (NoUpgrade == 1 && Pkg->CurrentVer != 0)
      {
	 if (Quiet < 2)
	   cout << "Skipping " << Pkg.Name() << ", it is already installed and no-upgrade is set." << endl;
	 continue;
      }
      
      // Check if there is something new to install
      pkgDepCache::StateCache &State = (*Dep)[Pkg];
      if (State.CandidateVer == 0)
	 return _error->Error("Package %s has no installation candidate",S);
      
      Fix.Protect(Pkg);
      if (Remove == true)
      {
	 Dep->MarkDelete(Pkg);
	 continue;
      }
      
      // Install it
      Dep->MarkInstall(Pkg,false);
      if (State.Install() == false)
      {
	 if (Quiet < 2)
	    cout << "Sorry, " << S << " is already the newest version"  << endl;
      }
      else
	 ExpectedInst++;

      // Install it with autoinstalling enabled.
      if (State.InstBroken() == true)
	 Dep->MarkInstall(Pkg,true);
   }

   // Call the scored problem resolver
   if (Fix.Resolve(true) == false)
      _error->Discard();
      
   /* Print out a list of packages that are going to be installed extra
      to what the user asked */
   if (Dep->InstCount() != ExpectedInst && Quiet < 2)
   {
      string List;
      pkgCache::PkgIterator I = Dep->PkgBegin();
      for (;I.end() != true; I++)
      {
	 if ((*Dep)[I].Install() == false)
	    continue;

	 int J;
	 for (J = 0; J != argc; J++)
	 {
	    if (argv[J][0] == 0 || argv[J][0] == '-')
	       continue;
	    if (strcmp(argv[J],I.Name()) == 0)
		break;
	 }
	 
	 if (J == argc)
	    List += string(I.Name()) + " ";
      }
      
      ShowList("The following extra packages will be installed:",List);
   }

   if (Quiet < 2)
   {
      ShowDel();
      ShowNew();
      ShowHold();
      ShowEssential();
      Stats();
   }
   
   // Now we check the state of the packages,
   if (Dep->BrokenCount() != 0)
   {
      if (Quiet < 2)
      {
	 ShowBroken();
	 return false;
      }
      else
	 return _error->Error("Sorry, broken packages");
   }   

   if (Dep->DelCount() == 0 && Dep->InstCount() == 0)
      return true;
   
   // Check, just to be sure.
   if (Dep->InstCount() != ExpectedInst || Dep->DelCount() != 0)
      return DoArchives(List,true);
   else
      return DoArchives(List,false);
}
									/*}}}*/
// Upgrade - Update to the newest packages				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Upgrade(pkgSourceList &List)
{
   // Setup Dep
   if (Check() == false || Dep == 0)
      return false;

   bool PreBroke = Dep->BrokenCount() != 0;
   
   // Do the upgrade
   Dep->AllUpgrade();
   
   // Safety check
   if (Dep->BrokenCount() != 0)
   {
      if (Quiet < 2)
      {
	 ShowBroken();
	 if (PreBroke == true)
	    return true;
      }
      
      return _error->Error("Internal Error, AllUpgrade broke stuff");
   }
   
   if (Quiet < 2)
   {
      ShowDel();
      ShowNew();
      ShowKept();
      ShowHold();
      ShowEssential();
      Stats();
   }
   
   if (Dep->DelCount() == 0 && Dep->InstCount() == 0 && 
       Dep->BadCount() == 0)
      return true;
   
   if (Quiet < 2 || AssumeY == false)
      return DoArchives(List,true);
   else
      return DoArchives(List,false);
}
									/*}}}*/
// DselectUpgrade - Upgrade using dselect new install information	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool DselectUpgrade(pkgSourceList &List)
{
   // Setup Dep
   if (Check() == false || Dep == 0)
      return false;

   // Install everything with the install flag set
   pkgCache::PkgIterator I = Dep->PkgBegin();
   for (;I.end() != true; I++)
   {
      /* Install the package only if it is a new install, the autoupgrader
         will deal with the rest */
      if (I->SelectedState == pkgSTATE_Install)
	 Dep->MarkInstall(I,false);
   }

   /* Now install their deps too, if we do this above then order of
      the status file is significant for | groups */
   for (;I.end() != true; I++)
   {
      /* Install the package only if it is a new install, the autoupgrader
         will deal with the rest */
      if (I->SelectedState == pkgSTATE_Install)
	 Dep->MarkInstall(I);
   }
   
   // Apply erasures now, they override everything else.
   for (I = Dep->PkgBegin();I.end() != true; I++)
   {
      // Remove packages 
      if (I->SelectedState == pkgSTATE_DeInstall ||
	  I->SelectedState == pkgSTATE_Purge)
	 Dep->MarkDelete(I);
   }

   /* Use updates smart upgrade to do the rest, it will automatically
      ignore held items */
   return Upgrade(List);
}
									/*}}}*/
// DistUpgrade - Perform a distribution upgrade				/*{{{*/
// ---------------------------------------------------------------------
/* */
bool DistUpgrade(pkgSourceList &List)
{
   // Setup Dep
   if (Check() == false || Dep == 0)
      return false;
   
   if (pkgDistUpgrade(*Dep) == false)
      return false;
   
   if (Quiet < 2)
   {
      ShowDel();
      ShowNew();
      ShowKept();
      ShowHold();
      Stats();
   }

   if (Dep->DelCount() == 0 && Dep->InstCount() == 0 && Dep->BadCount() == 0)
      return true;
   
   if (Quiet < 2 || AssumeY == false)
      return DoArchives(List,true);
   else
      return DoArchives(List,false);
}
									/*}}}*/
// Clean - Clean the archive directory					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Clean()
{
   Aquire Get;
   Get.OutputDir(PKG_DEB_CA_ARCHIVES);
   if (Get.Clean() == false)
      return false;
   return true;
}
									/*}}}*/

// ProcessOptions - Process the command line options			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ProcessOptions(int argc,char *argv[])
{
   opterr = 0;
   int Opt;

#ifdef _GNU_SOURCE
   static struct option long_options [] = 
     {{"help",no_argument,0,'h'},
      {"silent",no_argument,0,'q'},
      {"download-only",no_argument,0,'d'},
      {"simulate",no_argument,0,'s'},
      {"just-print",no_argument,0,'s'},
      {"dry-run",no_argument,0,'s'},
      {"recon",no_argument,0,'s'},
      {"no-act",no_argument,0,'s'},
      {"yes",no_argument,0,'y'},
      {"assume-yes",no_argument,0,'y'},
      {"fix-broken",no_argument,0,'f'},
      {"ignore-missing",no_argument,0,'m'},
      {"no-upgrade",no_argument,&NoUpgrade,1},      
      {0,0,0,0}};
      
   while ((Opt = getopt_long(argc,argv,"hqsdyfm",long_options,0)) != EOF)
#else      
   while ((Opt = getopt(argc,argv,"hqsdyfm")) != EOF)
#endif
   {
      switch (Opt)
      {
	 case 'h':
	 ShowHelp();
	 return false;
	 
	 case 'q':
	 if (Quiet == 0)
	    Quiet = QScript;
	 else
	    Quiet = QReallyQuiet;
	 break;
	 
	 case 'd':
	 DownloadOnly = true;
	 break;
	 
	 case 's':
	 NoAct = true;
	 break;
	 
	 case 'y':
	 AssumeY = true;
	 break;
	 
	 case 'f':
	 FixBroken = true;
	 break;
	 
	 case 'm':
	 FixMissing = true;
	 break;
	 
	 case '?':
	 if (optind > 1)
	    return _error->Error("Unknown option %s",argv[optind-1]);
	 else
	    return _error->Error("Unknown option %s",argv[1]);
      }
   }
   
   return true;
}
									/*}}}*/
// ShowHelp - Show the help text					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ShowHelp()
{
   cout << PACKAGE << ' ' << VERSION << " for " << PKG_DEB_ARCH <<
       " compiled on " << __DATE__ << "  " << __TIME__ << endl;

   if (sizeof(INT32) != 4 || sizeof(INT16) != 2)
      return _error->Error("Incorrect integer sizes");
   INT32 Jnk = 0x12345678;
   
   #ifdef WORDS_BIGENDIAN
   if (((unsigned char *)(&Jnk))[0] != 0x12)
      return _error->Error("Incorrect byte ordering");
   #else
   if (((unsigned char *)(&Jnk))[0] != 0x78)
      return _error->Error("Incorrect byte ordering");
   #endif
      
   cout << "Usage: apt-get [-hqq] update" << endl;
   cout << "       apt-get [-hqqdn] upgrade" << endl;
   cout << "       apt-get [-hqqdn] install pkg [pkg ...]" << endl;
   cout << "       apt-get [-hqqdn] dist-upgrade" << endl;
   cout << "       apt-get [-hqqdn] dselect-upgrade" << endl;
   cout << "" << endl;
   cout << "The most frequently used commands are update and install." << endl;
   cout << "" << endl;
   cout << "Commands:" << endl;
   cout << "       update - Retrieve new lists of packages" << endl;
   cout << "       upgrade - Perform an upgrade" << endl;
   cout << "       install - Install new packages (pkg is libc6 not libc6.deb)" << endl;
   cout << "       dist-upgrade - Distribution upgrade, see apt-get(8)" << endl;
   cout << "       dselect-upgrade - Follow dselect selections" << endl;
   cout << "       clean - Erase downloaded archive files" << endl;
   cout << "" << endl;
   cout << "Options:" << endl;
   cout << "  -h  This help text" << endl;
   cout << "  -q  Loggable output - no progress indicator" << endl;
   cout << "  -qq No output except for errors" << endl;
   cout << "  -d  Download only - do NOT install or unpack archives" << endl;
   cout << "  -s  No-act. Perform ordering simulation" << endl;
   cout << "  -y  Assume Yes to all queries and do not prompt" << endl;
   cout << "  -f  Attempt to continue if the integrity check fails" << endl;
   cout << "  -m  Attempt to continue if archives are unlocatable" << endl;
   cout << "" << endl;
   cout << "See the apt-get(8) and sources.list(8) manual pages for more information." << endl;

   return true;
}
									/*}}}*/

int main(int argc,char *argv[])
{
   if (argc <= 1)
   {
      ShowHelp();
      _error->DumpErrors();
      return 100;
   }
   
   GetScreenWidth();
   signal(SIGWINCH,SigWinch);
   signal(SIGCONT,SigCont);
   signal(SIGTSTP,SigTstp);

   if (ProcessOptions(argc,argv) == false)
   {
      _error->DumpErrors();
      return 100;
   }
   
   int Cmd = 1;
   for (;Cmd < argc && (argv[Cmd][0] == 0 || argv[Cmd][0] == '-'); Cmd++);
   if (Cmd >= argc)
   {
      _error->Error("You need to specify a command");
      ShowHelp();
      _error->DumpErrors();
      return 100;
   }
   
   // We will need the sourcelist
   pkgSourceList List;
   
   int Res = -1;
   while (Cmd < argc)
   {
      if (List.ReadMainList() == false)
      {
	 Res = 0;
	 _error->Error("Unable to read the main source list %s",PKG_DEB_CF_SOURCELIST);
	 break;
      }
      
      if (List.empty() == true)
      {
	 Res = 0;
	 _error->Error("Your %s file is empty.",PKG_DEB_CF_SOURCELIST);
	 break;
      }
      
      if (strcmp(argv[Cmd],"update") == 0)
	 Res = Update(List);
      
      if (strcmp(argv[Cmd],"check") == 0)
	 Res = Check(List);

      if (strcmp(argv[Cmd],"install") == 0)
	 Res = Install(List,argc-1-Cmd,argv+1+Cmd);
      
      if (strcmp(argv[Cmd],"upgrade") == 0)
	 Res = Upgrade(List);

      if (strcmp(argv[Cmd],"dselect-upgrade") == 0)
	 Res = DselectUpgrade(List);

      if (strcmp(argv[Cmd],"dist-upgrade") == 0)
	 Res = DistUpgrade(List);
      
      if (strcmp(argv[Cmd],"clean") == 0)
	 Res = Clean();
      
      break;
   }

   if (Res == -1)
   {
      _error->Error("You need to specify a command");
      ShowHelp();
      _error->DumpErrors();
      return 100;
   }
   
   // An error
   if (Res == 0)
   {
      if (_error->empty())
	  return 1;
      
      _error->DumpErrors();
      return 100;
   }     

   return 0;   
}
