##############################################################################
#
# Copyright (c) 1996-1998, Digital Creations, Fredericksburg, VA, USA.
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
# 
#   o Redistributions of source code must retain the above copyright
#     notice, this list of conditions, and the disclaimer that follows.
# 
#   o Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions, and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
# 
#   o Neither the name of Digital Creations nor the names of its
#     contributors may be used to endorse or promote products derived
#     from this software without specific prior written permission.
# 
# 
# THIS SOFTWARE IS PROVIDED BY DIGITAL CREATIONS AND CONTRIBUTORS *AS IS*
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
# PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL DIGITAL
# CREATIONS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
# OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
# TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
# USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
# DAMAGE.
#
# 
# If you have questions regarding this software, contact:
#
#   Digital Creations, L.C.
#   910 Princess Ann Street
#   Fredericksburge, Virginia  22401
#
#   info@digicool.com
#
#   (540) 371-6909
#
##############################################################################
__doc__="""
Mapping objects that store persistent pickles

$Id: PickleDictionary.py,v 1.22 1998/10/23 21:41:50 jim Exp $""" #'
__version__='$Revision: 1.22 $'[11:-2]

import marshal,time,Persistence,SimpleDB, PickleJar

class PickleDictionary:
    """\
    Mapping objects that store persistent pickles

    Pickle dictionaries provide for storage of pickles in files.  Any
    picklable object may be stored and any marshalable object may be used
    as a key.

    If the objects stored are instances of classes that are subclasses of
    class Persistent, then nested objects will be handled
    intelligently. In particular, if you have objects that contain
    persistent objects, then when you reload your objects from the
    database, the persistent objects won\'t be loaded until they are
    accessed.
    """ #'

    def __init__(self,
                 file_name=None, create=None, revision=None,
                 db=None,
                 DBType=SimpleDB.Default,
                 cache_age=1000,
                 cache_size=1000,
                 oid=-1,
                 ):
        """\
        Create a Pickle Dictionary

        The most comment usage:

           PickleDictionary(file_name)

              Creates a pickle dictionary using data stored in the
              file named by file_name.  If the file exists, then
              existing data in the file will be used. If the file
              doesn\'t exist, then it will be created.

           PickleDictionary(file_name,create=1)

              Creates a pickle dictionary using data stored in the
              file named by file_name. A new file will be created,
              overwriting the existing file if it exists.

        In addition, the following optional arguments may be used:

           file_name --  Specify the name of a simple MultipleRevision
                         database to be opened.
 
           create --     Specify whether to create a new database.
     
           revision --   If specified, inactive record revisions older than
                         revision seconds will be removed from the database.
     
           db --         Specify a simple database to be used.  If this
                         argument is specified, then none of the above
                         arguments has any effect.

           cache_size -- If specified, then this will be the cache
                         size used for the dictionary.
        """

        if db is None:
            db=DBType(
                file_name=file_name, revision_time=revision,
                create=create)
            self.file_name=file_name

        jar=PickleJar.PickleJar(db,cache_size,cache_age=cache_age)
        self._jar=jar
        self._oid=oid

        try: index=jar[oid]
        except KeyError:
            # New database
            index=Root()
            index._init(jar, oid)
        except:
            # Probably marshalled version of data
            index=Root()
            index._init(jar, oid, marshal.loads(db[oid]).items())
        self._index=index

    def __len__(self): return len(self._index)
    def __getitem__(self,key): return self._index[key]
    def __setitem__(self,key,object): self._index[key]=object
    def __delitem__(self,key): del self._index[key]
    def keys(self): return self._index.keys()
    def values(self): return self._index.values()
    def items(self): return self._index.items()
    def has_key(self, key): return self._index.has_key(key)

    def pack(self,start_time=None):
        if start_time==None: start_time=time.time()
        self._jar.db.pack(start_time)

    def transactions(self):
        return self._jar.db.transactions()

    def oid(self, key): return self._index[key]._p_oid

class Root(Persistence.Persistent):

    def _init(self, jar, oid, items=()):
        self._p_jar=jar
        self._p_oid=oid
        d=self._d={}
        for key, oid in items: d[key]=jar[oid]
        self._p_changed=1
        jar.store(self)

    def __getstate__(self): return self._d
    def __setstate__(self, state): self._d=state

    def __getitem__(self, key): return self._d[key]

    def __setitem__(self, key, v):
        self._d[key]=v
        self.__changed__(1)

    def __delitem__(self, key):
        del self._d[key]
        self.__changed__(1)

    def __len__(self): return len(self._d)
    def keys(self): return self._d.keys()
    def items(self): return self._d.items()
    def values(self): return self._d.values()
    def has_key(self, v): return self._d.has_key(v)
