/* 
** Copyright (C) 1994, 1995 Enterprise Integration Technologies Corp.
**         VeriFone Inc./Hewlett-Packard. All Rights Reserved.
** Kevin Hughes, kev@kevcom.com 3/11/94
** Kent Landfield, kent@landfield.com 4/6/97
** 
** This program and library is free software; you can redistribute it and/or 
** modify it under the terms of the GNU (Library) General Public License 
** as published by the Free Software Foundation; either version 2 
** of the License, or any later version. 
** 
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of 
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
** GNU (Library) General Public License for more details. 
** 
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
*/

#define MAIN_FILE 1

#include "hypermail.h"

static char *savstrval(char *d, char *envvar)
{
    char *ep;

    if ((ep = getenv(envvar)) != NULL)
    	return(strsav(ep));
    else 
    	return(strsav(d));
}

static void setstrval(char *s, char *d, char *envvar)
{
    char *ep;

    if ((ep = getenv(envvar)) != NULL)
    	strcpy(s, ep);
    else 
        strcpy(s, d);
}

static void setintval(int *s, int d, char *envvar)
{
    char *ep;

    if ((ep = getenv(envvar)) != NULL)
    	*s = atoi(ep);
    else 
        *s = d;
}

static void setoctalval(int *s, long d, char *envvar)
{
    char *ep;

    if ((ep = getenv(envvar)) != NULL)
    	*s = strtol(ep, (char **) NULL, 0);
    else
        *s = d;
}

static struct hmlist *savlist(struct hmlist *listname, 
                                char *d, char *envvar)
{
    char *ep;

    if ((ep = getenv(envvar)) != NULL)
    	return(add_2_list(listname, ep));
    else 
    	return(add_2_list(listname, d));
}

/* Print out the version number and die. */

void version(void)
{
    printf("%s: Version: %s Patchlevel: %s\n",PROGNAME,VERSION,PATCHLEVEL);
    exit(0);
}

/*
** Prints the usage.
*/

void usage(void)
{
    printf("\nUsage: %s -ipvVux [-m \"mbox\"] [-l \"label\"] [-d \"dir\"]\n",
                   PROGNAME);
    printf("                         [-a \"URL\"] [-b \"URL\"] [-c \"file\"]\n");
    printf("Options:\n");
    printf("  -a URL  : URL to other archives\n");
    printf("  -b URL  : URL to archive information\n");
    printf("  -c file : Configuration file to read in\n");
    printf("  -d dir  : The directory to save HTML files in\n");
    printf("  -i      : Read messages from standard input\n");
    printf("  -l label: What to name the output archive\n");
    printf("  -m mbox : Mail archive to read in\n");
    printf("  -p      : Show progress\n");
    printf("  -v      : Show configuration variables and exit\n");
    printf("  -V      : Show version information and exit\n");
    printf("  -u      : Update archive by one article\n");
    printf("  -x      : Overwrite previous messages\n\n");
    printf("Version   : %s\n", VERSION);
    printf("Patchlevel: %s\n", PATCHLEVEL);
    printf("      Docs: %s\n\n", HMURL);

    exit(1);
}

int main(int argc, char **argv)
{
    extern char *optarg;
    int i, overwrite, use_stdin, use_mbox, increment, show_variables;
    char mbox[MAXLINE], label[MAXLINE], dir[MAXLINE];
    char configfile[MAXLINE], defaultindex[MAXLINE];

    /* temporary cmdline option variables */
    char *cmd_archives, *cmd_about, *cmd_dir;
    char *cmd_label, *cmd_mbox, *cmd_hmail;
    int cmd_use_stdin, cmd_showprogress, cmd_usetable;
    int cmd_show_variables, cmd_increment, cmd_overwrite;

    use_stdin = 0;
    overwrite = 0;
    use_mbox = 0;
    show_variables = 0;
    firstdatenum = lastdatenum = 0;

    prefered_types = NULL;
    ignore_types = NULL;
    inline_types = NULL;
    show_headers = NULL;

    /*
    ** Load the hard-wired defaults, then the environment variables.
    */

    setstrval(mbox, MBOX, "HM_MBOX");
    setstrval(archives, ARCHIVES, "HM_ARCHIVES");
    setstrval(about, ABOUT, "HM_ABOUT");
    setstrval(label, LABEL, "HM_LABEL");
    setstrval(dir, (char *)dirpath(DIR), "HM_DIR");
    setstrval(defaultindex, DEFAULTINDEX, "HM_DEFAULTINDEX");
    setstrval(configfile, CONFIGFILE, "HM_CONFIGFILE");
    setstrval(mailto, MAILTO, "HM_MAILTO");
    setstrval(hmail, HMAIL, "HM_HMAIL");
    setstrval(mailcommand, MAILCOMMAND, "HM_MAILCOMMAND");
    setstrval(domainaddr, DOMAINADDR, "HM_DOMAINADDR");
    setstrval(htmlbody, BODY, "HM_BODY");

    setintval(&showprogress, PROGRESS, "HM_PROGRESS");
    setintval(&overwrite, OVERWRITE, "HM_OVERWRITE");
    setintval(&increment, INCREMENT, "HM_INCREMENT");
    setintval(&reverse, REVERSE, "HM_REVERSE");
    setintval(&showheaders, SHOWHEADERS, "HM_SHOWHEADERS");
    setintval(&showhtml, SHOWHTML, "HM_SHOWHTML");
    setintval(&thrdlevels, THRDLEVELS, "HM_THRDLEVELS");
    setintval(&showbr, SHOWBR, "HM_SHOWBR");
    setintval(&showhr, SHOWHR, "HM_SHOWHR");
    setintval(&iquotes, IQUOTES, "HM_IQUOTES");
    setintval(&eurodate, EURODATE, "HM_EURODATE");
    setintval(&showreplies, SHOWREPLIES, "HM_SHOWREPLIES");
    setintval(&show_msg_links, SHOW_MSG_LINKS, "HM_SHOW_MSG_LINKS");
    setintval(&usetable, USETABLE, "HM_USETABLE");

    setoctalval(&dirmode,  DIRMODE,  "HM_DIRMODE");
    setoctalval(&filemode, FILEMODE, "HM_FILEMODE");

    ihtmlheader = savstrval(IHTMLHEADERFILE,"HM_IHTMLHEADERFILE");
    ihtmlfooter = savstrval(IHTMLFOOTERFILE,"HM_IHTMLFOOTERFILE");
    mhtmlheader = savstrval(MHTMLHEADERFILE,"HM_MHTMLHEADERFILE");
    mhtmlfooter = savstrval(MHTMLFOOTERFILE,"HM_MHTMLFOOTERFILE");

    show_headers = savlist(show_headers,SHOW_HEADERS,"HM_SHOW_HEADERS");
    inline_types = savlist(inline_types,INLINE_TYPES,"HM_INLINE_TYPES");
    ignore_types = savlist(ignore_types,IGNORE_TYPES,"HM_IGNORE_TYPES");

    if (!strcasecmp(domainaddr, "NONE"))
    	use_domainaddr = 0;
    else
    	use_domainaddr = 1;

    if (!strcasecmp(mailto, "NONE"))
    	use_mailto = 0;
    else
    	use_mailto = 1;

    if (!strcasecmp(mailcommand, "NONE"))
        use_mailcommand = 0;
    else
        use_mailcommand = 1;

    /*
    ** Here is where things need a bit of explanation... The command line 
    ** options need to override what is in the configuration file. In the
    ** past the config file values overrode the command line values because 
    ** the command line needed to be processed to get the name of the config
    ** file to read.  Now the state of the command line variables is saved 
    ** and used later, after the config file has been read.
    */

    cmd_archives = cmd_about = cmd_dir = NULL;
    cmd_label = cmd_mbox = cmd_hmail = NULL;
    cmd_use_stdin = cmd_showprogress = cmd_show_variables = 0;
    cmd_increment = cmd_overwrite =  cmd_usetable = 0;

    while ((i = (int) getopt(argc, argv, "a:b:c:d:hil:m:ptuvVxz?")) != -1) {
        switch((char) i) {
            case 'a': cmd_archives = strsav(optarg);
                      break;
            case 'b': cmd_about = strsav(optarg);
                      break;
            case 'c': strcpy(configfile, optarg);
                      break;
            case 'd': cmd_dir = strsav(dirpath(optarg));
                      break;
            case 'i': cmd_use_stdin = 1;
                      break;
            case 'l': cmd_label = strsav(optarg);
                      break;
            case 'm': cmd_mbox = strsav(optarg);
                      break;
            case 'n': cmd_hmail = strsav(optarg);
                      break;
            case 'p': cmd_showprogress = 1;
                      break;
            case 't': cmd_usetable = 1;
                      break;
            case 'v': cmd_show_variables = 1;
                      break;
            case 'V': version();
                      break;
            case 'u': cmd_increment = 1;
                      break;
            case 'x': cmd_overwrite = 1;
                      break;
            case 'h':
            case 'z':
            case '?': 
            default : usage();
                      break;
        }
    }

    /* 
    ** ...then read the configuration file.
    */

    readconfigs(configfile, mbox, label, dir, archives, about, 
                &overwrite, &increment, defaultindex);

    /*
    ** Now override the configuration file variables with any explicitly
    ** passed on the command line. This way you need not change the
    ** configuration file settings for a minor change in a single run.
    */

    if (cmd_archives != NULL) {
        strcpy(archives, cmd_archives);
        free(cmd_archives);
    }

    if (cmd_about != NULL) {
        strcpy(about, cmd_about);
        free(cmd_about);
    }

    if (cmd_dir != NULL) {
        strcpy(dir, cmd_dir);
        free(cmd_dir);
    }

    if (cmd_hmail != NULL) {
        strcpy(hmail, cmd_hmail);
        free(cmd_hmail);
    }

    if (cmd_label != NULL) {
        strcpy(label, cmd_label);
        free(cmd_label);
    }

    if (cmd_mbox != NULL) {
        strcpy(mbox, cmd_mbox);
        use_mbox = 1;
        free(cmd_mbox);
    }

    if (cmd_use_stdin == 1)      use_stdin = 1;
    if (cmd_showprogress == 1)   showprogress = 1;
    if (cmd_show_variables == 1) show_variables = 1;
    if (cmd_increment == 1)      increment = 1;
    if (cmd_overwrite == 1)      overwrite = 1;
    if (cmd_usetable == 1)       usetable = 1;

    /*
    ** Did they decide to use stdin by specifying it on the command line ?
    ** If not then check and see if that is the default they wanted from
    ** the options.h or environment values.
    */
    if (!use_stdin) {
        if (!strcasecmp(mbox, "NONE")) 
    	    use_stdin = 1;
        else
    	    use_stdin = 0;
    }
    if (use_stdin)   /* so it won't be displayed */
        mbox[0] = '\0';

    /* 
    ** If you specify a mailbox you are indicating you 
    ** don't want to increment only a single message.
    */
    if (use_mbox)
        increment = 0;

    /*
    ** Determine if headers and footers disabled.
    */

    if (ihtmlheader && !strcasecmp(ihtmlheader, "NONE")) {
        free(ihtmlheader);
        ihtmlheader = NULL;
    }
    if (ihtmlfooter && !strcasecmp(ihtmlfooter, "NONE")) {
        free(ihtmlfooter);
        ihtmlfooter = NULL;
    }
    if (mhtmlheader && !strcasecmp(mhtmlheader, "NONE")) {
        free(mhtmlheader);
        mhtmlheader = NULL;
    }
    if (mhtmlfooter && !strcasecmp(mhtmlfooter, "NONE")) {
        free(mhtmlfooter);
        mhtmlfooter = NULL;
    }

    /*
    ** Default names for directories and labels need to be figured out.
    */

    if (use_stdin && !strcasecmp(dir, "NONE"))
    	strcpy(dir, DIRNAME);

    if (!strcasecmp(dir, "NONE"))
    	strcpy(dir,(strrchr(mbox,'/')) ? (char *)strrchr(mbox,'/') + 1 : mbox);

    if (!strcasecmp(label, "NONE"))
    	strcpy(label,(strrchr(mbox,'/')) ? (char *)strrchr(mbox,'/')+1 : mbox);

    /*
    ** Which index file will be called "index.html"?
    */

    strcpy(datename,(!strcmp(defaultindex, "date")) ? INDEXNAME : DATENAME);
    strcpy(thrdname,(!strcmp(defaultindex, "thread")) ? INDEXNAME : THRDNAME);
    strcpy(subjname,(!strcmp(defaultindex, "subject")) ? INDEXNAME : SUBJNAME);
    strcpy(authname,(!strcmp(defaultindex, "author")) ? INDEXNAME : AUTHNAME);

    /* 
    ** A little performance speed up.  The following was being done
    ** over and over in the write functions. This way it is done once.
    ** A bigger win on larger archives.
    */

    if (!strcasecmp(archives, "NONE"))
        archives[0] = '\0';

    if (!strcasecmp(about, "NONE"))
        about[0] = '\0';

    if (!strcasecmp(hmail, "NONE"))
        hmail[0] = '\0';

    if (!strcasecmp(htmlbody, "NONE"))
        htmlbody[0] = '\0';

    /* Simply show what the values of the variables are and exit */

    if (show_variables) {
        dump_config(mbox, label, dir, archives, about, 
                    overwrite, increment, defaultindex);

        printf("\nCommand and Control Variables:\n");
        printf("use_stdin = %s\n", use_stdin ? "Yes" : "No");
        printf("use_mbox = %s\n", use_mbox ? "Yes" : "No");
        printf("use_mailto = %s\n", use_mailto ? "Yes" : "No");
        printf("use_mailcommand = %s\n", use_mailcommand ? "Yes" : "No");
        printf("use_domainaddr = %s\n", use_domainaddr ? "Yes" : "No");
        printf("usetable = %s\n", usetable ? "Yes" : "No");
        if (!*about)
            printf("About the archive not used\n");
        if (!*archives)
            printf("Other archives not used\n");
        if (!*hmail)
            printf("hmail address not used\n");
        if (!*htmlbody)
            printf("Builtin <BODY> statement used\n");

        return(0);
    }

    /* Injecting a little sanity... */

    if (use_mbox && use_stdin)
    	progerr("Can't read from file and stdin at once.");

    gettimezone();
    getthisyear();

    /*
    ** MIME processing requires the files be created as they
    ** are read in loadheaders() so assure the directories are 
    ** there first...
    */

    checkdir(dir); 

    /*
    ** Let's do it.
    */

    if (increment) {
    	loadoldheaders(dir);
    	loadheaders(mbox,use_stdin,1,dir);
    	writearticles(dir,label,about,overwrite,bignum);
        if (show_msg_links) {
    	    fixnextheader(dir,bignum - 1);
            if (showreplies)
                fixreplyheader(dir,bignum);
    	    fixthreadheader(dir,bignum);
        }
    }
    else {
    	loadheaders(mbox,use_stdin,0,dir);
    	writearticles(dir,label,about,overwrite,0);
    }

    /*
    ** Always write the index files
    */

    writedates(dir,label,archives,about);
    writethreads(dir,label,archives,about);
    writesubjects(dir,label,archives,about);
    writeauthors(dir,label,archives,about);

    /* cleanup */
    if (ihtmlheader) free(ihtmlheader);
    if (ihtmlfooter) free(ihtmlfooter);
    if (mhtmlheader) free(mhtmlheader);
    if (mhtmlfooter) free(mhtmlfooter);

    return(0);
}

