/*
 * Spline routine for equally spaced points
 * A. Schiffler, 1995, andreas@karlsberg.usask.ca
 *
 * It was modified for gus library by Jaroslav Kysela
 * Rewrote 'oversample' routines - A.S., 16 Sept. 1995 - unchecked
 * Checked 'oversample' routines - J.K., 21 Sept. 1995 - debugged
 *
 */

#include <asm/byteorder.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include "libgus_local.h"

/*
 *  defines
 */

#define SFLOAT		float	/* data type for calculations */
#define SNUMPOINTS	64	/* number of data points to process at once */
#define COPYPOINTS	2

/*
 *  global variables
 */

static SFLOAT spline_y[SNUMPOINTS];	/* spline input */

static SFLOAT spline_b[SNUMPOINTS];	/* spline coefficients */
static SFLOAT spline_c[SNUMPOINTS];
static SFLOAT spline_d[SNUMPOINTS];

/*
 *  calculate the coefficients
 *  expects y values in spline_y[] up to and including 'numpoints'
 */
 
static void create_spline( int numpoints )
{
  int index;
  SFLOAT l[ SNUMPOINTS ];
  SFLOAT mu[ SNUMPOINTS ];
  SFLOAT z[ SNUMPOINTS ];
 
  /* calculate c */
  l[ 0 ] = mu[ 0 ] = z[ 0 ] = 0.0;
  for ( index = 1; index < numpoints; index++ ) 
    {
      l[ index ] = 4.0-mu[index-1];
      mu[ index ] = 1.0/l[index];
      z[ index ] = (3.0*(spline_y[index+1]-2.0*spline_y[index]+spline_y[index-1])
      		   -z[index-1])/l[index];
   }
  spline_c[ numpoints ] = 0.0;
  for ( index = (numpoints - 1); index >= 0; index-- )
    spline_c[ index ] = z[index]-mu[index]*spline_c[index+1];
  /* calculate b and d */
  for ( index = (numpoints - 1); index >= 0; index-- ) 
    {
      spline_b[index] = (spline_y[index+1]-spline_y[index])
   		        -(spline_c[index+1]+2.0*spline_c[index])/3.0;
      spline_d[index] = (spline_c[index+1]-spline_c[index])/3.0;	
    }
}

/*
 *  calculate spline interpolator at position 'xpt'
 */

static SFLOAT spline( SFLOAT xpt ) 
{
  int loc;
  SFLOAT cx;
 
  loc=(int)xpt;
  cx=xpt-(SFLOAT)loc;
  return( ((spline_d[loc]*cx+spline_c[loc])*cx+spline_b[loc])*cx+spline_y[loc]);
}

/*
 *
 */
 
void gus_convert_spline_oversampling_8bit(
		signed char *dest, signed char *src, 
		size_t size, float factor, int unsigned_ )
{
  int i,j,c,first;
  float position,step,maximum;
  SFLOAT res;
  
  unsigned_ = unsigned_ ? 0x80 : 0;
  position=0.0;  /* x-position for interpolation */
  step=1.0/factor;  /* x-step for interpolation */
  i=0; /* Sample byte position */
  first=1;
  while (i<size) {
   if ( !first )
     {
       for ( c = 0; c < COPYPOINTS; c++ )
         spline_y[ c ] = spline_y[ ( SNUMPOINTS - 2 * COPYPOINTS ) + c ];
     }
   j = first ? 0 : COPYPOINTS;
   position += (float)j;
   while ((i<size) && (j<SNUMPOINTS-COPYPOINTS)) {
    spline_y[ j++ ] = (signed char)(((unsigned char)*(src++)) ^ unsigned_);  /* Load spline array */
    i++;
   }
   for ( c = 0; (c < COPYPOINTS) && (i + c < size); c++ ) {
    spline_y[ j++ ] = (signed char)(((unsigned char)*(src + c)) ^ unsigned_);
   }
   /* Interpolate */
   create_spline (j);
   if ( j > SNUMPOINTS - COPYPOINTS ) j = SNUMPOINTS - COPYPOINTS;
   maximum=(float)j; /* Maximum floating point number for this chunk */
   while (position<maximum) {
    /* Calculate interpolation value */
    res = spline( position );
    if ( res < -128 ) *(dest++) = -128; else
    if ( res > 127 ) *(dest++) = 127; else
    *(dest++) = _p_rint( res );
    position += step; /* Update position */
   }
   position -= maximum; /* Subtract last maximum */
   first = 0;
  }
}

void gus_convert_spline_oversampling_16bit(
		signed short *dest, signed short *src,
		size_t size, float factor, int unsigned_ )
{
  int i,j,c,first;
  float position,step,maximum;
  SFLOAT res;
  
#ifdef __LITTLE_ENDIAN
  unsigned_ = unsigned_ ? 0x8000 : 0;
#else
  unsigned_ = unsigned_ ? 0x0080 : 0;
#endif
  position=0.0;  /* x-position for interpolation */
  step=1.0/factor;  /* x-step for interpolation */
  i=0; /* Sample byte position */
  first=1;
  while (i<size) {
   if ( !first )
     {
       for ( c = 0; c < COPYPOINTS; c++ )
         spline_y[ c ] = spline_y[ ( SNUMPOINTS - 2 * COPYPOINTS ) + c ];
     }
   j = first ? 0 : COPYPOINTS;
   position += (float)j;
   while ((i<size) && (j<SNUMPOINTS-COPYPOINTS)) {
    spline_y[ j++ ] = (signed short)(((unsigned short)*(src++)) ^ unsigned_);  /* Load spline array */
    i++;
   }
   for ( c = 0; (c < COPYPOINTS) && (i + c < size); c++ ) {
    spline_y[ j++ ] = (signed short)(((unsigned short)*(src + c)) ^ unsigned_);
   }
   /* Interpolate */
   create_spline (j);
   if ( j > SNUMPOINTS - COPYPOINTS ) j = SNUMPOINTS - COPYPOINTS;
   maximum=(float)j; /* Maximum floating point number for this chunk */
   while (position<maximum) {
    /* Calculate interpolation value */
    res = spline( position );
    if ( res < -32768 ) *(dest++) = -32768; else
    if ( res > 32767 ) *(dest++) = 32767; else
    *(dest++) = _p_rint( res );
    position += step; /* Update position */
   }
   position -= maximum; /* Subtract last maximum */
   first = 0;
  }
}

void gus_convert_spline_oversampling_8bit_to_16bit(
		signed short *dest, signed char *src,
		size_t size, float factor, int unsigned_ )
{
  int i,j,c,first;	
  float position,step,maximum;
  SFLOAT res;
  
  unsigned_ = unsigned_ ? 0x80 : 0;
  position=0.0;  /* x-position for interpolation */
  step=1.0/factor;  /* x-step for interpolation */
  i=0; /* Sample byte position */
  first=1;
  while (i<size) {
   if ( !first )
     {
       for ( c = 0; c < COPYPOINTS; c++ )
         spline_y[ c ] = spline_y[ ( SNUMPOINTS - 2 * COPYPOINTS ) + c ];
     }
   j = first ? 0 : COPYPOINTS;
   position += (float)j;
   while ((i<size) && (j<SNUMPOINTS-COPYPOINTS)) {
    spline_y[ j++ ] = (signed char)(((unsigned char)*(src++)) ^ unsigned_) * 256;  /* Load spline array */
    i++;
   }
   for ( c = 0; (c < COPYPOINTS) && (i + c < size); c++ ) {
    spline_y[ j++ ] = (signed char)(((unsigned char)*(src + c)) ^ unsigned_) * 256;
   }
   /* Interpolate */
   create_spline (j);
   if ( j > SNUMPOINTS - COPYPOINTS ) j = SNUMPOINTS - COPYPOINTS;
   maximum=(float)j; /* Maximum floating point number for this chunk */
   while (position<maximum) {
    /* Calculate interpolation value */
    res = spline( position );
    if ( res < -32768 ) *(dest++) = -32768; else
    if ( res > 32767 ) *(dest++) = 32767; else
    *(dest++) = _p_rint( res );
    position += step; /* Update position */
   }
   position -= maximum; /* Subtract last maximum */
   first = 0;
  }
}

