/*
 * main.c  -  Utility program to create a bootrom image
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "nblib.h"
#include "makerom.h"



/*
 * Path name definitions
 */
#define KERNELDIR	"//binaries"
#define LOADERDIR	"//binaries"
#define NETDRVDIR	"//netdrvr"
#define UTILSDIR	"//utils"



/*
 * Global variables
 */
char *kerneldir = NULL;		/* name of directory with kernel binaries */
char *loaderdir = NULL;		/* name of directory with loader binaries */
char *netdrvdir = NULL;		/* name of directory with network drivers */
char *utilsdir = NULL;		/* name of utilities directory */
struct desclist *kerneldesc = NULL;	/* list of kernel descriptions */
struct desclist *netdrvdesc = NULL;	/* list of network driver descriptions */



/*
 * Private variables
 */
static char *batchname = NULL;		/* name of system to batch process */
static int accesserror = 0;		/* non-zero if unable to access a file */



/*
 * Command line options
 */
static struct cmdopt opts[] = {
	{ "batch-sys", 'b', strval, {(char **)&batchname}, NULL,
	  "name of system to process", "SYSTEM"				},
	{ NULL, 0, noval, {NULL}, NULL, NULL, NULL			}
};



/*
 * Definition of makerom:general section in configuration file
 */
static struct paramdef general_params[] = {
	{ "kerneldir",	par_string,	NULL,	{&kerneldir}},
	{ "loaderdir",	par_string,	NULL,	{&loaderdir}},
	{ "netdrvdir",	par_string,	NULL,	{&netdrvdir}},
	{ "utilsdir",	par_string,	NULL,	{&utilsdir}},
	{ NULL,		par_null,	NULL,	{NULL}}
};



/*
 * Define kernel parameters in configuration file
 */
static struct {
	char *description;
	char *filename;
	char *flags;
} kernel_vars = {NULL, NULL, NULL};

static struct paramdef kernel_params[] = {
  { "description",	par_string,	NULL,	{&kernel_vars.description}},
  { "file",		par_string,	NULL,	{&kernel_vars.filename}},
  { "flags",		par_string,	NULL,	{&kernel_vars.flags}},
  { NULL,		par_null,	NULL,	{NULL}}
};



/*
 * Define network driver parameters in configuration file
 */
char *drtype[] = {
  "dos program", "packet driver", "ndis driver", "linux driver", NULL
};

static struct {
	char *description;
	char *filename;
	int   drivertype;
	char *parameters;
	char *cmdline;
} netdrv_vars = {NULL, NULL, 0, NULL, NULL};

static struct paramdef netdrv_params[] = {
  { "description",	par_string,	NULL,	{&netdrv_vars.description}},
  { "file",		par_string,	NULL,	{&netdrv_vars.filename}},
  { "type",		par_string,	drtype,	{(char **)&netdrv_vars.drivertype}},
  { "parameters",	par_string,	NULL,	{&netdrv_vars.parameters}},
  { "cmdline",		par_string,	NULL,	{&netdrv_vars.cmdline}},
  { NULL,		par_null,	NULL,	{NULL}}
};



/*
 * List of sections to read from configuration file at startup
 */
static char *handle_kernel_sect __P((char *sectname, struct sectdef *cursect));
static char *handle_netdrv_sect __P((char *sectname, struct sectdef *cursect));

static struct sectdef sects[] = {
	{ "makerom:general",	general_params,	NULL,	NULL },
	{ "makerom:kernel:*",	kernel_params,	NULL,	&handle_kernel_sect },
	{ "makerom:netdrv:*",	netdrv_params,	NULL,	&handle_netdrv_sect },
	{ NULL,			NULL,		NULL,	NULL}
};



/*
 * List of recognized options for binary files
 */
struct optlist {
	char *optname;
	int   optval;
};

static struct optlist netoptlist[] = {
	{ "HW_IRQ",	HW_IRQ },
	{ "IO_ADDR",	IO_ADDR },
	{ "BASE_MEM",	BASE_MEM },
	{ "AUI_TYPE",	AUI_TYPE },
	{ "DMA_NUM",	DMA_NUM },
	{ NULL,		0 }
};

static struct optlist kernoptlist[] = {
	{ "MINIMAL",	K_MINIMAL },
	{ "X86",	K_X86 },
	{ NULL,		0 }
};



/*
 * Generate an option value from an option string
 */
static int getoption(optstr, optlist)
char *optstr;
struct optlist *optlist;
{
  char *cp;
  int i, optval = 0;

  cp = strtok(optstr, ",");
  while (cp != NULL) {
	for (i = 0; optlist[i].optname != NULL; i++)
		if (!strcmp(optlist[i].optname, cp))
			break;
	if (optlist[i].optname == NULL)
		return(-1);
	optval |= optlist[i].optval;
	cp = strtok(NULL, ",");
  }
  return(optval);
}



/*
 * Handle processing of parameters for a kernel description section. This
 * routine gets called after a kernel description section has been read.
 */
static char *handle_kernel_sect(sectname, cursect)
char *sectname;
struct sectdef *cursect;
{
  struct desclist *dp;
  char *cp;
  int optval = 0;

  /* Analyze kernel parameters */
  if (kernel_vars.flags != NULL) {
	if ((optval = getoption(kernel_vars.flags, kernoptlist)) < 0)
		return("invalid kernel flag");
	free(kernel_vars.flags);
  }
  if (kernel_vars.filename == NULL)
	return("missing kernel file name");

  /* Add new item to kernel description list */
  dp = nbmalloc(sizeof(struct desclist));
  dp->filename = kernel_vars.filename;
  dp->descript = kernel_vars.description;
  dp->drivertype = TYPE_NONE;
  dp->cmdline = NULL;
  dp->options = optval;
  dp->progname = NULL;
  if ((cp = strrchr(sectname, ':')) == NULL)
	copystr(&(dp->progname), sectname);
  else
	copystr(&(dp->progname), (cp + 1));
  dp->next = kerneldesc;
  kerneldesc = dp;

  /* Clear out temporary space for next round */
  memset(&kernel_vars, 0, sizeof(kernel_vars));
  return(NULL);
}



/*
 * Handle processing of parameters for a network driver description section.
 * This routine gets called after a driver description section has been read.
 */
static char *handle_netdrv_sect(sectname, cursect)
char *sectname;
struct sectdef *cursect;
{
  struct desclist *dp;
  char *cp;
  int optval = 0;

  /* Analyze network driver parameters */
  if (netdrv_vars.parameters != NULL) {
	if ((optval = getoption(netdrv_vars.parameters, netoptlist)) < 0)
		return("invalid network driver flag");
	free(netdrv_vars.parameters);
  }
  if (!netdrv_vars.drivertype)
	return("missing network driver type");
  if (netdrv_vars.filename == NULL)
	return("missing network driver file name");
  if (netdrv_vars.cmdline == NULL)
	return("missing network driver command line");

  /* Add new item to kernel description list */
  dp = nbmalloc(sizeof(struct desclist));
  dp->filename = netdrv_vars.filename;
  dp->descript = netdrv_vars.description;
  dp->drivertype = netdrv_vars.drivertype;
  dp->cmdline = netdrv_vars.cmdline;
  dp->options = optval;
  dp->progname = NULL;
  if ((cp = strrchr(sectname, ':')) == NULL)
	copystr(&(dp->progname), sectname);
  else
	copystr(&(dp->progname), (cp + 1));
  dp->next = netdrvdesc;
  netdrvdesc = dp;

  /* Clear out temporary space for next round */
  memset(&netdrv_vars, 0, sizeof(netdrv_vars));
  return(NULL);
}



/*
 * Check that we have read access to the specified filed
 */
static void checkaccess(fname)
char *fname;
{
  if (access(fname, R_OK) < 0) {
	fprintf(stderr, "%s: unable to access file %s\n", progname, fname);
	accesserror++;
  }
}



/*
 * Print boot definition record for debugging purposes
 */
static void printbootdef(bp)
struct bootdef *bp;
{
  int i;

  if (bp->name != NULL)
	printf("Bootrom building definitions for %s\n", bp->name);
  else
	printf("Bootrom building definitions\n");
  printf("Kernel file        = %s (int %dh)\n", bp->kernelname,
			bp->use_int18 ? 18 : 19);

  for (i = 0; i < MAXLOADERS; i++)
	if (bp->loadernames[i] != NULL) {
		printf("Loader binary %d    = %s\n", i, bp->loadernames[i]);
		printf("Output file %d      = %s\n", i, bp->outnames[i]);
	}
  printf("\n");

  for (i = 0; i < MAXDRIVERS; i++)
	if (bp->drivernames[i] != NULL) {
		printf("Driver %d program   = %s\n", i, bp->drivernames[i]);
		printf("Driver %d arguments = %s\n", i, bp->driverargs[i]);
	}
  printf("\n");
}



/*
 * Main program
 */
void main(argc, argv)
int argc;
char **argv;
{
  struct bootdef *bp;
  char *tmpfname;
  int i, tmpfile;

  /* Parse options and read configuration file */
  nbsetup(argc, argv, opts, sects);
  setpath(&kerneldir, KERNELDIR);
  setpath(&loaderdir, LOADERDIR);
  setpath(&netdrvdir, NETDRVDIR);
  setpath(&utilsdir, UTILSDIR);

  if (verbose > 0) {
	printf("Bootrom configuration program, " VERSION "\n" COPYRIGHT "\n");
	printf("Database file           = %s\n", dbname);
	printf("Kernel directory        = %s\n", kerneldir);
	printf("Loader directory        = %s\n", loaderdir);
	printf("Packet driver directory = %s\n", netdrvdir);
	printf("Utilities directory     = %s\n\n", utilsdir);
  }

  /* Get bootrom specification either from the user or the database */
  if (batchname != NULL) {
	bp = getdb(batchname);
  } else {
	if (!verbose)
		printf("\nBootrom configuration program, "
					VERSION "\n" COPYRIGHT "\n\n");
	bp = getuser();
  }

  /* Expand filenames */
  setpath(&(bp->kernelname), NULL);
  for (i = 0; i < MAXLOADERS; i++)
	setpath(&(bp->loadernames[i]), NULL);
  for (i = 0; i < MAXDRIVERS; i++)
	setpath(&(bp->drivernames[i]), NULL);
  if (verbose > 1)
	printbootdef(bp);

  /* Check that all input files are readable */
  checkaccess(bp->kernelname);
  for (i = 0; i < bp->loadernum; i++)
	checkaccess(bp->loadernames[i]);
  for (i = 0; i < bp->drivernum; i++)
	checkaccess(bp->drivernames[i]);
  if (accesserror > 0)
	exit(EXIT_ACCESS);

  /* Open temporary file and call the different passes successively */
  if ((tmpfname = tempnam(NULL, "mkrom")) == NULL) {
	fprintf(stderr, "%s: unable to generate temporary file name\n",
								progname);
	exit(EXIT_TEMPNAME);
  }
  if ((tmpfile = open(tmpfname, O_CREAT | O_RDWR | O_TRUNC | O_BINARY,
								0644)) < 0) {
	fprintf(stderr, "%s: unable to open temporary file %s\n",
							progname, tmpfname);
	exit(EXIT_CREATE);
  }
  if (verbose < 2)
	unlink(tmpfname);

  pass1(bp, tmpfile);
  for (i = 0; i < bp->loadernum; i++)
	pass2(bp, tmpfile, i);

  close(tmpfile);
  exit(EXIT_SUCCESS);
}

