/*
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the Licence, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>

#include "../common/cmalloc.h"
#include "xfonts.h"	/* from common */
#include <X11/xpm.h>

#include "TextentryP.h"

#define offset(field) XtOffsetOf(TextentryRec, textentry.field)
static XtResource resources[] = {
	{
		XtNcursor,
		XtCCursor,
		XtRCursor,
		sizeof(Cursor),
		offset(cursor),
		XtRString,
		"xterm"
	}, {
		XtNinsertPosition,
		XtCTextPosition,
		XtRInt,
		sizeof(int),
		offset(insert_pos),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNdisplayCaret,
		XtCOutput,
		XtRBoolean,
		sizeof(Boolean),
		offset(display_caret),
		XtRImmediate,
		(XtPointer)True
	}, {
		XtNstring,
		XtCString,
		XtRString,
		sizeof(char *),
		offset(string),
		XtRString,
		NULL
	}, {
		XtNforeground,
		XtCForeground,
		XtRPixel,
		sizeof(Pixel),
		offset(foreground),
		XtRString,
		XtDefaultForeground
	}, {
		XtNfont,
		XtCFont,
		XtRFontStruct,
		sizeof(XFontStruct *),
		offset(font),
		XtRString,
		XtDefaultFont
	}, {
		XtNtextentryTop,
		XtCTextentryTop,
		XtRInt,
		sizeof(int),
		offset(top),
		XtRImmediate,
		(XtPointer)0
	}
};
#undef offset

/* methods */
static void Realize();
static void Destroy();
static void GetValuesHook();
static void Redisplay(Widget, XEvent *, Region);
static Boolean SetValues();

/* actions */
static void BeginningOfLine(Widget, XEvent *, String *, Cardinal *);
static void BackwardCharacter(Widget, XEvent *, String *, Cardinal *);
static void DeleteNextCharacter(Widget, XEvent *, String *, Cardinal *);
static void EndOfLine(Widget, XEvent *, String *, Cardinal *);
static void ForwardCharacter(Widget, XEvent *, String *, Cardinal *);
static void DeletePreviousCharacter(Widget, XEvent *, String *, Cardinal *);
static void KillToEndOfLine(Widget, XEvent *, String *, Cardinal *);
static void RedrawDisplay(Widget, XEvent *, String *, Cardinal *);
static void TransposeCharacters(Widget, XEvent *, String *, Cardinal *);
static void BeginningOfLine(Widget, XEvent *, String *, Cardinal *);
static void EndOfLine(Widget, XEvent *, String *, Cardinal *);
static void InsertChar(Widget, XEvent *, String *, Cardinal *);
static void NoOp(Widget, XEvent *, String *, Cardinal *);
static void Click(Widget, XEvent *, String *, Cardinal *);

/* Names of actions have been chosen to match those used by the
   Athena Text widget.
*/
static XtActionsRec actions[] =
{
	{"beginning-of-line", BeginningOfLine},
	{"backward-character", BackwardCharacter},
	{"delete-next-character", DeleteNextCharacter},
	{"end-of-line", EndOfLine},
	{"forward-character", ForwardCharacter},
	{"delete-previous-character", DeletePreviousCharacter},
	{"kill-to-end-of-line", KillToEndOfLine},
	{"redraw-display", RedrawDisplay},
	{"transpose-characters", TransposeCharacters},
	{"beginning-of-file", BeginningOfLine},
	{"end-of-file", EndOfLine},
	{"insert-char", InsertChar},
	{"no-op", NoOp},
	{"click", Click},
};

/* translations */
/* Names of translations have been chosen to match those used by the
   Athena Text widget.
*/
static char translations[] =
"Ctrl<Key>A:	beginning-of-line()\n"
"Ctrl<Key>B:	backward-character()\n"
"Ctrl<Key>D:	delete-next-character()\n"
"Ctrl<Key>E:	end-of-line()\n"
"Ctrl<Key>F:	forward-character()\n"
"Ctrl<Key>H:	delete-previous-character()\n"
"Ctrl<Key>K:	kill-to-end-of-line()\n"
"Ctrl<Key>L:	redraw-display()\n"
"Ctrl<Key>T:	transpose-characters()\n"
":Meta<Key>\\<:	beginning-of-file()\n"
":Meta<Key>\\>:	end-of-file()\n"
"<Key>Home:	beginning-of-file()\n"
":<Key>KP_Home:	beginning-of-file()\n"
"<Key>End:	end-of-file()\n"
":<Key>KP_End:	end-of-file()\n"
"<Key>Right:	forward-character()\n"
":<Key>KP_Right: forward-character()\n"
"<Key>Left:	backward-character()\n"
":<Key>KP_Left:	backward-character()\n"
"<Key>Delete:	delete-previous-character()\n"
":<Key>KP_Delete: delete-previous-character()\n"
"<Key>BackSpace: delete-previous-character()\n"
":<Btn1Down>:	click()\n"
"<Key>:		insert-char()";

TextentryClassRec textentryClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Textentry",
    /* widget_size		*/	sizeof(TextentryRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	GetValuesHook,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* textentry fields */
    /* empty			*/	0
  }
};

WidgetClass textentryWidgetClass = (WidgetClass)&textentryClassRec;

static XComposeStatus compose_status = {NULL, 0};


static void BeginningOfLine(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	tw->textentry.insert_pos = 0;
	Redisplay(w, NULL, None);
}

static void BackwardCharacter(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;

	if (tw->textentry.insert_pos > 0) {
		tw->textentry.insert_pos--;
		Redisplay(w, NULL, None);
	}
}

static void DeleteNextCharacter(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	int i = tw->textentry.insert_pos;

	memmove(tw->textentry.text+i,
		tw->textentry.text+i+1,
		strlen(tw->textentry.text+i));
	Redisplay(w, NULL, None);
}

static void EndOfLine(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	tw->textentry.insert_pos = strlen(tw->textentry.text);
	Redisplay(w, NULL, None);
}

static void ForwardCharacter(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	if (tw->textentry.insert_pos < strlen(tw->textentry.text)) {
		tw->textentry.insert_pos++;
		Redisplay(w, NULL, None);
	}
}

static void DeletePreviousCharacter(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	if (tw->textentry.insert_pos > 0) {
		tw->textentry.insert_pos--;
		DeleteNextCharacter(w, event, params, n);
	}
}

static void KillToEndOfLine(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	tw->textentry.text[tw->textentry.insert_pos] = '\0';
	Redisplay(w, NULL, None);
}

static void RedrawDisplay(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	Redisplay(w, NULL, None);
}

static void TransposeCharacters(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	if (strlen(tw->textentry.text) >= 2 &&
		tw->textentry.insert_pos < strlen(tw->textentry.text)) {
		int c = tw->textentry.text[tw->textentry.insert_pos];
		tw->textentry.text[tw->textentry.insert_pos] =
			tw->textentry.text[tw->textentry.insert_pos+1];
		tw->textentry.text[tw->textentry.insert_pos+1] = c;
		Redisplay(w, NULL, None);
	}
}

static void insert_text(TextentryWidget tw, int i, char *p, int length)
{
	tw->textentry.text = crealloc(tw->textentry.text,
				strlen(tw->textentry.text)+length+1);
	memmove(tw->textentry.text+i+length,
		tw->textentry.text+i,
		strlen(tw->textentry.text+i)+1);
	memcpy(tw->textentry.text+i, p, length);
	tw->textentry.insert_pos = i+length;
}

static void InsertChar(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	int i;
	char strbuf[1024];
	int length;
	KeySym keysym;
	TextentryWidget tw = (TextentryWidget)w;
	i = tw->textentry.insert_pos;
	length = XLookupString((XKeyEvent *)event, strbuf, 1024,
		&keysym, &compose_status);
	if (length == 0 || ((unsigned char)strbuf[0]) < ' ') return;

	insert_text(tw, i, strbuf, length);
	Redisplay(w, NULL, None);
}

static void NoOp(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	if (*n == 1 && (**params == 'R' || **params == 'r'))
		XBell(XtDisplay(w), 0);
}

static void x2c(TextentryWidget tw, int x, int *col)
{
	char *p = tw->textentry.text;
	int i = 0, n = strlen(p);
	XFontStruct *fs = tw->textentry.font;
	x += tw->textentry.top;
	while (i < n && x > 0) {
		x -= XTextWidth(fs, p+i, 1);
		i++;
	}
	if (i < 0) i = 0;
	else if (i > strlen(p)) i = strlen(p);
	*col = i;
}

static void Click(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	TextentryWidget tw = (TextentryWidget)w;
	int col, x;
	x = event->xbutton.x;
	x2c(tw, x, &col);
	tw->textentry.insert_pos = col;
	Redisplay(w, NULL, None);
}


static GC get_gc(Widget w, unsigned long fg, unsigned long bg)
{
        unsigned long valuemask = 0;
        XGCValues values;
        GC gc = XCreateGC(XtDisplay(w), XtWindow(w),
                                valuemask, &values);
        XSetForeground(XtDisplay(w), gc, fg);
        XSetBackground(XtDisplay(w), gc, bg);
        return gc;
}

#define superclass (&coreClassRec)
static void Realize(Widget w, XtValueMask *valueMask,
                XSetWindowAttributes *attributes)
{
        TextentryWidget tw = (TextentryWidget) w;
        unsigned long fg, bg, grey;
	XColor screen_color, exact_color;

        (*superclass->core_class.realize) (w, valueMask, attributes);
        fg = BlackPixelOfScreen(XtScreen(w));
        bg = tw->core.background_pixel;
	XAllocNamedColor(XtDisplay(w),
		DefaultColormap(XtDisplay(w), DefaultScreen(XtDisplay(w))),
		"grey", &screen_color, &exact_color);
	grey = screen_color.pixel;
        tw->textentry.clear_gc = get_gc(w, bg, fg);
        tw->textentry.draw_gc = get_gc(w, fg, grey);
	XSetFont(XtDisplay(w), tw->textentry.draw_gc, tw->textentry.font->fid);
	tw->textentry.cursor_gc = get_gc(w, fg, bg);
	tw->textentry.text = cstrdup("");
	XDefineCursor(XtDisplay(w), XtWindow(w), tw->textentry.cursor);
	tw->textentry.sel_top = 0;
	tw->textentry.sel_bot = -1;
}

static void Destroy(Widget w)
{
        TextentryWidget tw = (TextentryWidget) w;

        XFreeGC(XtDisplay(w), tw->textentry.clear_gc);
	XFreeGC(XtDisplay(w), tw->textentry.draw_gc);
	XFreeGC(XtDisplay(w), tw->textentry.cursor_gc);
	cfree(tw->textentry.text);
}

static void Redisplay(Widget w, XEvent *event, Region r)
{
	TextentryWidget tw = (TextentryWidget) w;
	Pixmap scribble;
	XFontStruct *fs = tw->textentry.font;
	char *p = tw->textentry.text;
	int i = tw->textentry.insert_pos;
	int top = tw->textentry.top;
	Dimension width, height;
	int n, x;

	width = tw->core.width;
	height = tw->core.height;
	if (width > 2000 || height > 2000) return;

	/* figure out width of the string up to caret */
	if (!p) p = "";
	n = XTextWidth(fs, p, i);
	/* adjust to the left... */
	if (n-top < 2) top = n-2;
	/* and to the right */
	if (n-top > width-2) top = n-width+2;
	tw->textentry.top = top;

	scribble = XCreatePixmap(XtDisplay(w), XtWindow(w),
		width, height, tw->core.depth);
	if (scribble == None) return;
	XFillRectangle(XtDisplay(w), scribble, tw->textentry.clear_gc,
		0, 0, width, height);
#if 1
	x = -top;
	for (i = 0; p[i]; p++) {
		if (i >= tw->textentry.sel_top && i <= tw->textentry.sel_bot)
			XDrawImageString(XtDisplay(w), scribble,
				tw->textentry.draw_gc,
				x, height-5, p+i, 1);
		else
			XDrawString(XtDisplay(w), scribble,
				tw->textentry.draw_gc,
				x, height-5, p+i, 1);
		x += XTextWidth(fs, p+i, 1);
	}
#else
	XDrawString(XtDisplay(w), scribble, tw->textentry.draw_gc,
		-top, height-5, p, strlen(p));
#endif
	if (tw->textentry.display_caret) {
		/* draw cursor */;
		XDrawLine(XtDisplay(w), scribble,
			tw->textentry.cursor_gc,
			n-top, 2, n-top, height-2);
	}
	XCopyArea(XtDisplay(w), scribble, XtWindow(w),
		tw->textentry.draw_gc, 0, 0, width, height, 0, 0);
	XFreePixmap(XtDisplay(w), scribble);
}

static void GetValuesHook(Widget w, ArgList args, Cardinal *num_args)
{
	TextentryWidget tw = (TextentryWidget)w;
	int i;

	for (i = 0; i < *num_args; i++) {
		if (!strcmp(args[i].name, XtNstring)) {
			*((char **)args[i].value) =
				tw->textentry.text;
		}
	}
}

static Boolean SetValues(Widget current, Widget request, Widget new)
{
	Boolean do_redisplay = False;
	TextentryWidget ctw = (TextentryWidget)current;
	TextentryWidget ntw = (TextentryWidget)new;
	int length;

	if (ntw->textentry.string) {
		ntw->textentry.text = cstrdup(ntw->textentry.string);
		ntw->textentry.string = NULL;
		if (XtIsRealized(current)) {
			if (strcmp(ntw->textentry.text, ctw->textentry.text))
				do_redisplay = True;
			cfree(ctw->textentry.text);
		}
	}
	if (XtIsRealized(new) && ntw->textentry.text)
		length = strlen(ntw->textentry.text);
	else length = 0;
	if (ntw->textentry.insert_pos > length) {
		ntw->textentry.insert_pos = length;
	}
	if (ntw->textentry.insert_pos < 0) {
		ntw->textentry.insert_pos = 0;
	}
	if (ntw->textentry.insert_pos != ctw->textentry.insert_pos)
		do_redisplay = True;
	if (ntw->textentry.display_caret != ctw->textentry.display_caret)
		do_redisplay = True;
	if (do_redisplay) {
		Redisplay(new, NULL, None);
		do_redisplay = False;
	}
	return do_redisplay;
}

