# file MASS/stepAIC.q
# copyright (C) 1994-8 W. N. Venables and B. D. Ripley
#
"stepAIC"<-
function(object, scope, scale = 0, direction = c("both", "backward", 
	"forward"), trace = 1, keep = NULL, steps = 1000, screen = F,
         use.start = F, k = 2, ...)
{
  fixFormulaObject <- function(object) {
    tmp <- attr(terms(object), "term.labels")
    formula(paste(deparse(formula(object)[[2]]), "~", paste(tmp, collapse=" + ")))
  }
  update.formula <-
    function (old, new) {
      tmp <-.Internal(update.formula(as.formula(old), as.formula(new)))
      class(tmp ) <- "formula"
      tmp
    }
  cut.string <- function(string)
    {
      if(length(string) > 1)
        string[-1] <- paste("\n", string[-1], sep = "")
      string
    }
  AIC.drop1 <- function(fit, Terms, scale, trace, k = 2, ...)
    {
      n <- length(Terms)
      ans <- matrix(nrow = n + 1, ncol = 2)
      dimnames(ans) <- list(c("<none>", paste("-", Terms, sep = "")), 
			    c("df", "AIC"))
      ans[1,  ] <- extractAIC(fit, scale, k = k, ...)
      if(n == 0) return(ans)
      i <- 2
      for(tt in Terms) {
        if(trace > 1) cat("trying -", tt, "\n")
        nfit <- update(fit, as.formula(paste("~ . -", tt)))
        ans[i,  ] <- extractAIC(nfit, scale, k = k, ...)
        i <- i + 1
      }
      ans
    }

  AIC.add1 <- function(fit, Terms, scale, trace, screen, k = 2, ...)
    {
      if(screen) {
        lmscale <- switch(family(fit)$family,
                          poisson = 1,
                          binomial = 1,
                          if(scale > 0) scale else deviance(fit)/fit$df.resid)
        Ad <- add1.lm(fit, Terms, scale = lmscale)
        Ad$Df[1] <- 0
        Cp <- Ad$RSS/lmscale + k * Ad$Df
        Cp <- Cp - Cp[1]
        Cp <- Cp[-1]
        Terms <- Terms[Cp < 1 & Cp < min(Cp) + 4 & Ad$Df[-1] > 0]
      }
      n <- length(Terms)
      ans <- matrix(nrow = n + 1, ncol = 2)
      t2 <- if(length(Terms)) paste("+", Terms, sep = "") else NULL
      dimnames(ans) <- list(c("<none>", t2), c("df", "AIC"))
      ans[1,  ] <- extractAIC(fit, scale, k = k, ...)
      if(n == 0) return(ans)
      i <- 2
      for(tt in Terms) {
        if(trace > 1) cat("trying +", tt, "\n")
        nfit <- update(fit, as.formula(paste("~ . +", tt)))
        ans[i,  ] <- extractAIC(nfit, scale, k = k, ...)
        i <- i + 1
      }
      ans
    }

  AIClm.drop1 <- function(fit, Terms, scale, trace, k = 2, ...)
    {
      n <- length(Terms)
      ans <- matrix(nrow = n + 1, ncol = 2)
      dimnames(ans) <- list(c("<none>", paste("-", Terms, sep = "")), 
			    c("df", "AIC"))

      Ad <- drop1.lm(fit, Terms, scale = scale)
      nn <- length(fit$residuals)
      edf <- nn - fit$df.residual
      ans[, 1] <- edf - c(0, Ad$Df[-1])
      ans[, 2] <- if(scale > 0) Ad$RSS/scale + k * ans[, 1] - nn
        else nn * log(Ad$RSS/nn) + k * ans[, 1]
      ans
    }

  AIClm.add1 <- function(fit, Terms, scale, k = 2, ...)
    {
      n <- length(Terms)
      ans <- matrix(nrow = n + 1, ncol = 2)
      t2 <- if(length(Terms)) paste("+", Terms, sep = "") else NULL
      dimnames(ans) <- list(c("<none>", t2), c("df", "AIC"))
      if(n == 0) return(ans)
      Ad <- add1.lm(fit, Terms, scale = scale)
      nn <- length(fit$residuals)
      edf <- nn - fit$df.residual
      ans[, 1] <- edf + c(0, Ad$Df[-1])
      ans[, 2] <- if(scale > 0) Ad$RSS/scale + k * ans[, 1] - nn
        else nn * log(Ad$RSS/nn) + k * ans[, 1]
      ans
    }

  re.arrange <- function(keep)
    {
      namr <- names(k1 <- keep[[1]])
      namc <- names(keep)
      nc <- length(keep)
      nr <- length(k1)
      array(unlist(keep, recursive = F), c(nr, nc), list(namr, namc))
    }

  make.step <- function(models, fit, object)
    {
      change <- sapply(models, "[[", "change")
      rd <- sapply(models, "[[", "deviance")
      dd <- c(NA, diff(rd))
      rdf <- sapply(models, "[[", "df.resid")
      ddf <- c(NA, diff(rdf))
      AIC <- sapply(models, "[[", "AIC")
      heading <- c("Stepwise Model Path \nAnalysis of Deviance Table",
		   "\nInitial Model:", deparse(as.vector(formula(object))),
		   "\nFinal Model:", deparse(as.vector(formula(fit))), 
		   "\n")
      aod <- data.frame(Step = change, Df = ddf, Deviance = dd, 
			"Resid. Df" = rdf, "Resid. Dev" = rd, AIC = AIC, 
			check.names = F)
      attr(aod, "heading") <- heading
      attr(aod, "class") <- c("anova", "data.frame")
      fit$anova <- aod
      fit
    }

  # need to fix up . in formulae in R
  object$formula <- fixFormulaObject(object)
  Terms <- object$formula
  object$call$formula <- object$formula
  attributes(Terms) <- attributes(object$terms)
  object$terms <- Terms
  cl <- class(object)[1]
  LM <- (cl == "lm" || cl == "aov" ||
         (cl == "glm" && object$family$family == "Gaussian" && object$family$link == "Identity: mu"))
  LM <- F # for now
  if(!LM && use.start) 
    if(is.null(object$linear.predictors)) {
      use.start <- F
      warning(paste("cannot use start with object of class", class(object)))
    } else {
      assign(".eta", object$linear.predictors)
      object$call$start <- .eta
    }
  if(missing(direction)) direction <- "both"
  else direction <- match.arg(direction)
  backward <- direction == "both" | direction == "backward"
  forward <- direction == "both" | direction == "forward"
  if(missing(scope)) {
    fdrop <- numeric(0)
    fadd <- NULL
  } else {
    if(is.list(scope)) {
      fdrop <- if(!is.null(fdrop <- scope$lower))
	attr(terms(update.formula(object, fdrop)), "factors")
	else numeric(0)
      fadd <- if(!is.null(fadd <- scope$upper))
	attr(terms(update.formula(object, fadd)), "factors")
    } else {
      fadd <- if(!is.null(fadd <- scope))
	attr(terms(update.formula(object, scope)), "factors")
      fdrop <- numeric(0)
    }
  }
  if(is.null(fadd)) {
    backward <- T
    forward <- F
  }
  models <- vector("list", steps)
  if(!is.null(keep)) {
    keep.list <- vector("list", steps)
    nv <- 1
  }
  n <- length(object$residuals)
  fit <- object
  cf <- attributes(coef(object))	
  #check if any terms have zero df
  if(!is.null(cf$singular) && cf$singular > 0) {
    TT <- !match(TL <- attr(object$terms, "term.labels"), names(cf$assign), F)
    if(any(TT)) {
      upd <- eval(parse(text = paste(c(".~.", TL[TT]), collapse = "-")))
      fit <- update(fit, upd)
    }
  }
  bAIC <- extractAIC(fit, scale, k = k, ...)
  edf <- bAIC[1]
  bAIC <- bAIC[2]
  nm <- 1
  Terms <- fit$terms
  if(trace)
    cat("Start:  AIC=", format(round(bAIC, 2)), "\n",
        cut.string(deparse(as.vector(formula(fit)))), "\n\n")

  models[[nm]] <- list(deviance = deviance(fit), df.resid = n - edf, 
		       change = "", AIC = bAIC)
  if(!is.null(keep)) keep.list[[nm]] <- keep(fit, bAIC)
  AIC <- bAIC + 1
  while(bAIC < AIC & steps > 0) {
    steps <- steps - 1
    AIC <- bAIC
    bfit <- fit
    ffac <- attr(Terms, "factors")
    scope <- findScope(ffac, list(add = fadd, drop = fdrop))
    aod <- NULL
    change <- NULL
    if(backward && (ndrop <- length(scope$drop))) {
      aod <- if(LM) AIClm.drop1(fit, scope$drop, scale = scale,
                                trace = trace, k = k, ...)
      else AIC.drop1(fit, scope$drop, scale = scale, trace = trace, k = k, ...)
    }
    if(forward && (nadd <- length(scope$add))) {
      aodf <- if(LM) AIClm.add1(fit, scope$add, scale = scale, k = k, ...)
      else AIC.add1(fit, scope$add, scale = scale, trace = trace,
                    screen = screen, k = k,, ...)
      if(is.null(aod)) aod <- aodf
      else aod <- rbind(aod, aodf[-1, , drop=F])
    }
    if(is.null(aod)) break
    o <- order(aod[, "AIC"])
    if(trace) print(aod[o,  ])
    if(o[1] == 1) break
    change <- dimnames(aod)[[1]][o[1]]
    fit <- update(fit, eval(parse(text = paste("~ .", change))))
    fit$formula <- fixFormulaObject(fit)
    Terms <- fit$formula
    attributes(Terms) <- attributes(fit$terms)
    fit$terms <- Terms
    bAIC <- extractAIC(fit, scale, k = k, ...)
    edf <- bAIC[1]
    bAIC <- bAIC[2]
    if(trace)
      cat("\nStep:  AIC=", format(round(bAIC, 2)), "\n", 
          cut.string(deparse(as.vector(formula(fit)))), "\n\n")
    if(bAIC >= AIC) break
    nm <- nm + 1
    edf <- 
    models[[nm]] <- list(deviance = deviance(fit), df.resid = n - edf,
			 change = change, AIC = bAIC)
    if(!is.null(keep)) keep.list[[nm]] <- keep(fit, bAIC)
  if(use.start) assign(".eta", fit$linear.predictors)
  }
  if(!is.null(keep)) fit$keep <- re.arrange(keep.list[seq(nm)])
  if(!LM && use.start) fit$call$start <- NULL
  make.step(models = models[seq(nm)], fit, object)
}

"findScope"<-
function(factor, scope)
{
  drop <- scope$drop
  add <- scope$add

  if(length(factor) && !is.null(drop)) {# have base model
    if(length(drop)) {
      nmdrop <- dimnames(drop)[[2]]
      nmfac <- dimnames(factor)[[2]]
      where <- match(nmdrop, nmfac, 0)
      if(any(!where))
        stop("lower scope is not included in model")
      nmdrop <- nmfac[-where]
      facs <- factor[, -where, drop = F]
    } else {
      nmdrop <- dimnames(factor)[[2]]
      facs <- factor
    }
    if(ncol(facs) > 1) {
      # now check no interactions will be left without margins.
      keep <- rep(T, ncol(facs))
      f <- crossprod(facs > 0)
      for(i in seq(keep)) keep[i] <- max(f[i,  - i]) != f[i, i]
      nmdrop <- nmdrop[keep]
    }
  } else nmdrop <- character(0)

  facs <- dimnames(factor)[[2]]
  if(is.null(add)) nmadd <- character(0)
  else {
    nmadd <- dimnames(add)[[2]]
    where <- match(dimnames(factor)[[2]], nmadd, 0)
    if(any(!where))
      stop("upper scope does not include model")
    nmadd <- nmadd[-where]
    facs <- add[, -where, drop = F]
    if(ncol(facs) > 1) {
      # now check marginality: 
      keep <- rep(T, ncol(facs))
      f <- crossprod(facs > 0)
      for(i in seq(keep)) keep[-i] <- keep[-i] & (f[i, -i] < f[i, i])
      nmadd <- nmadd[keep]
    }    
  }
  list(drop = nmdrop, add = nmadd)
}


extractAIC <- function(fit, scale, k = 2, ...) UseMethod("extractAIC")

extractAIC.coxph <- function(fit, scale, k = 2, ...)
{
  edf <- length(fit$coef)
  c(edf, -2 * fit$loglik[2] + k * edf)
}
extractAIC.survreg <- function(fit, scale, k = 2, ...)
{
  n <- length(fit$residuals)
  edf <- n  - fit$df.residual
  c(edf, -2 * fit$loglik[2] + k * edf)
}
extractAIC.glm <- function(fit, scale = 0, k = 2, ...)
{
  n <- length(fit$residuals)
  edf <- n  - fit$df.residual
  dev <- fit$deviance
  if(scale > 0) dev <- dev/scale
  if(scale == 0 && fit$family$family == "Gaussian") dev <- n * log(dev/n)
  c(edf,  dev + k * edf)
}
extractAIC.lm <- function(fit, scale = 0, k = 2, ...)
{
  n <- length(fit$residuals)
  edf <- n  - fit$df.residual
  RSS <- deviance.lm(fit)
  dev <- if(scale > 0) RSS/scale else n * log(RSS/n)
  c(edf, dev + k * edf)
}
extractAIC.aov <- function(fit, scale = 0, k = 2, ...)
{
  n <- length(fit$residuals)
  edf <- n - fit$df.residual
  RSS <- deviance.lm(fit)
  dev <- if(scale > 0) RSS/scale - n else n * log(RSS/n)
  c(edf, dev + k * edf)
}
extractAIC.negbin <- function(fit, scale, k = 2, ...)
{
  n <- length(fit$residuals)
  edf <- n - fit$df.residual
  c(edf,  -fit$twologlik + k * edf)
}

deviance.default <- function(x, ...) extractAIC(x, k=0)[2]
